/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Diff       diff            Compare two datasets
*/

#include <cdi.h>

#include <map>
#include "cdo_int.h"
#include "param_conversion.h"
#include "cdo_math.h"

#include "mpmo_color.h"
#include "cdo_options.h"
#include "printinfo.h"
#include "pmlist.h"

static void
diffGetParameter(double &abslim, double &abslim2, double &rellim, int &mapflag)
{
  const int pargc = operatorArgc();
  if (pargc)
    {
      const char **pargv = (const char **) operatorArgv();

      KVList kvlist;
      kvlist.name = "DIFF";
      if (kvlist.parseArguments(pargc, pargv) != 0) cdoAbort("Parse error!");
      if (Options::cdoVerbose) kvlist.print();

      for (const auto &kv : kvlist)
        {
          const char *key = kv.key.c_str();
          if (kv.nvalues > 1) cdoAbort("Too many values for parameter key >%s<!", key);
          if (kv.nvalues < 1) cdoAbort("Missing value for parameter key >%s<!", key);
          const char *value = kv.values[0].c_str();

          // clang-format off
          if      (cstrIsEqual(key, "abslim"))  abslim = parameter2double(value);
          else if (cstrIsEqual(key, "abslim2")) abslim2 = parameter2double(value);
          else if (cstrIsEqual(key, "rellim"))  rellim = parameter2double(value);
          else if (cstrIsEqual(key, "names"))
            {
              if      (cstrIsEqual(value, "left"))      mapflag = 1;
              else if (cstrIsEqual(value, "right"))     mapflag = 2;
              else if (cstrIsEqual(value, "intersect")) mapflag = 3;
              else cdoAbort("Invalid value for key >%s< (names=<left/right/intersect>)", key, value);
            }
          else cdoAbort("Invalid parameter key >%s<!", key);
          // clang-format on
        }
    }
}

void *
Diff(void *process)
{
  bool lhead = true;
  int nrecs, nrecs2;
  int varID1, varID2 = -1;
  int levelID;
  size_t nmiss1, nmiss2;
  int ndrec = 0, nd2rec = 0, ngrec = 0;
  char varname[CDI_MAX_NAME];
  char paramstr[32];
  char vdatestr[32], vtimestr[32];
  double abslim = 0., abslim2 = 1.e-3, rellim = 1.0;

  cdoInitialize(process);

  // clang-format off
  const int DIFF  = cdoOperatorAdd("diff",  0, 0, nullptr);
  const int DIFFP = cdoOperatorAdd("diffp", 0, 0, nullptr);
  const int DIFFN = cdoOperatorAdd("diffn", 0, 0, nullptr);
  const int DIFFC = cdoOperatorAdd("diffc", 0, 0, nullptr);
  // clang-format on

  const int operatorID = cdoOperatorID();

  int mapflag = 0;

  diffGetParameter(abslim, abslim2, rellim, mapflag);

  if (rellim < -1.e33 || rellim > 1.e+33) cdoAbort("Rel. limit out of range!");
  if (abslim < -1.e33 || abslim > 1.e+33) cdoAbort("Abs. limit out of range!");
  if (abslim2 < -1.e33 || abslim2 > 1.e+33) cdoAbort("Abs2. limit out of range!");

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);

  const int nvars = vlistNvars(vlistID1);
  std::map<int, int> mapOfVarIDs;

  if (mapflag == 0)
    {
      vlistCompare(vlistID1, vlistID2, CMP_ALL);
      for (int varID = 0; varID < nvars; ++varID) mapOfVarIDs[varID] = varID;
    }
  else
    {
      vlistMap(vlistID1, vlistID2, CMP_ALL, mapflag, mapOfVarIDs);
    }

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  std::vector<double> array1(gridsizemax);
  std::vector<double> array2(gridsizemax);
  std::vector<double> work(vlistNumber(vlistID1) != CDI_REAL ? 2 * gridsizemax : 0);

  const int taxisID = vlistInqTaxis(vlistID1);

  int indg = 0;
  int tsID = 0;
  while (true)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs > 0)
        {
          date2str(taxisInqVdate(taxisID), vdatestr, sizeof(vdatestr));
          time2str(taxisInqVtime(taxisID), vtimestr, sizeof(vtimestr));
        }

      nrecs2 = cdoStreamInqTimestep(streamID2, tsID);

      if (nrecs == 0 || nrecs2 == 0) break;

      int recID2next = 0;

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID1, &levelID);

          auto it = mapOfVarIDs.find(varID1);
          if (it == mapOfVarIDs.end())
            {
              if (mapflag == 2 || mapflag == 3) continue;
              cdoAbort("Internal problem: varID1=%d not found!", varID1);
            }

          for (; recID2next < nrecs2; ++recID2next)
            {
              cdoInqRecord(streamID2, &varID2, &levelID);
              if (it->second == varID2)
                {
                  ++recID2next;
                  break;
                }
            }

          if (it->second != varID2 && recID2next == nrecs2)
            cdoAbort("Internal problem: varID2=%d not found in second stream!", it->second);

          indg += 1;

          const int param = vlistInqVarParam(vlistID1, varID1);
          const int code = vlistInqVarCode(vlistID1, varID1);
          const int gridID = vlistInqVarGrid(vlistID1, varID1);
          const int zaxisID = vlistInqVarZaxis(vlistID1, varID1);
          const size_t gridsize = gridInqSize(gridID);
          const double missval1 = vlistInqVarMissval(vlistID1, varID1);
          const double missval2 = vlistInqVarMissval(vlistID2, varID2);

          // checkrel = gridInqType(gridID) != GRID_SPECTRAL;
          const bool checkrel = true;

          cdiParamToString(param, paramstr, sizeof(paramstr));

          if (vlistInqVarNumber(vlistID1, varID1) == CDI_COMP)
            {
              cdoReadRecord(streamID1, work.data(), &nmiss1);
              for (size_t i = 0; i < gridsize; ++i) array1[i] = work[i * 2];
            }
          else
            cdoReadRecord(streamID1, array1.data(), &nmiss1);

          if (vlistInqVarNumber(vlistID2, varID2) == CDI_COMP)
            {
              cdoReadRecord(streamID2, work.data(), &nmiss2);
              for (size_t i = 0; i < gridsize; ++i) array2[i] = work[i * 2];
            }
          else
            cdoReadRecord(streamID2, array2.data(), &nmiss2);

          int ndiff = 0;
          bool dsgn = false;
          bool zero = false;
          double absm = 0.0;
          double relm = 0.0;
          double absdiff;

          for (size_t i = 0; i < gridsize; i++)
            {
              if ((DBL_IS_NAN(array1[i]) && !DBL_IS_NAN(array2[i])) || (!DBL_IS_NAN(array1[i]) && DBL_IS_NAN(array2[i])))
                {
                  ndiff++;
                  relm = 1.0;
                }
              else if (!DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array2[i], missval2))
                {
                  absdiff = std::fabs(array1[i] - array2[i]);
                  if (absdiff > 0.) ndiff++;

                  absm = cdo::max(absm, absdiff);

                  if (array1[i] * array2[i] < 0.)
                    dsgn = true;
                  else if (IS_EQUAL(array1[i] * array2[i], 0.))
                    zero = true;
                  else
                    relm = cdo::max(relm, absdiff / cdo::max(std::fabs(array1[i]), std::fabs(array2[i])));
                }
              else if ((DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array2[i], missval2))
                       || (!DBL_IS_EQUAL(array1[i], missval1) && DBL_IS_EQUAL(array2[i], missval2)))
                {
                  ndiff++;
                  relm = 1.0;
                }
            }

          if (!Options::silentMode || Options::cdoVerbose)
            {
              if (absm > abslim || (checkrel && relm >= rellim) || Options::cdoVerbose)
                {
                  if (lhead)
                    {
                      lhead = false;

                      set_text_color(stdout, BRIGHT, BLACK);
                      fprintf(stdout, "               Date     Time   Level Gridsize    Miss ");
                      fprintf(stdout, "   Diff ");
                      fprintf(stdout, ": S Z  Max_Absdiff Max_Reldiff : ");

                      if (operatorID == DIFFN)
                        fprintf(stdout, "Parameter name");
                      else if (operatorID == DIFF || operatorID == DIFFP)
                        fprintf(stdout, "Parameter ID");
                      else if (operatorID == DIFFC)
                        fprintf(stdout, "Code number");
                      reset_text_color(stdout);

                      fprintf(stdout, "\n");
                    }

                  if (operatorID == DIFFN) vlistInqVarName(vlistID1, varID1, varname);

                  set_text_color(stdout, BRIGHT, BLACK);
                  fprintf(stdout, "%6d ", indg);
                  reset_text_color(stdout);
                  set_text_color(stdout, BLACK);
                  fprintf(stdout, ":");
                  reset_text_color(stdout);

                  set_text_color(stdout, MAGENTA);
                  fprintf(stdout, "%s %s ", vdatestr, vtimestr);
                  reset_text_color(stdout);
                  set_text_color(stdout, GREEN);
                  double level = cdoZaxisInqLevel(zaxisID, levelID);
                  fprintf(stdout, "%7g ", level);
                  fprintf(stdout, "%8zu %7zu ", gridsize, cdo::max(nmiss1, nmiss2));
                  fprintf(stdout, "%7d ", ndiff);
                  reset_text_color(stdout);

                  set_text_color(stdout, BLACK);
                  fprintf(stdout, ":");
                  reset_text_color(stdout);
                  fprintf(stdout, " %c %c ", dsgn ? 'T' : 'F', zero ? 'T' : 'F');
                  set_text_color(stdout, BLUE);
                  fprintf(stdout, "%#12.5g%#12.5g", absm, relm);
                  set_text_color(stdout, BLACK);
                  fprintf(stdout, " : ");
                  reset_text_color(stdout);

                  set_text_color(stdout, BRIGHT, GREEN);
                  if (operatorID == DIFFN)
                    fprintf(stdout, "%-11s", varname);
                  else if (operatorID == DIFF || operatorID == DIFFP)
                    fprintf(stdout, "%-11s", paramstr);
                  else if (operatorID == DIFFC)
                    fprintf(stdout, "%4d", code);
                  reset_text_color(stdout);

                  fprintf(stdout, "\n");
                }
            }

          ngrec++;
          if (absm > abslim || (checkrel && relm >= rellim)) ndrec++;
          if (absm > abslim2 || (checkrel && relm >= rellim)) nd2rec++;
        }
      tsID++;
    }

  if (ndrec > 0)
    {
      Options::cdoExitStatus = 1;

      set_text_color(stdout, BRIGHT, RED);
      fprintf(stdout, "  %d of %d records differ", ndrec, ngrec);
      reset_text_color(stdout);
      fprintf(stdout, "\n");

      if (ndrec != nd2rec && abslim < abslim2) fprintf(stdout, "  %d of %d records differ more than %g\n", nd2rec, ngrec, abslim2);
      //  fprintf(stdout, "  %d of %d records differ more then one thousandth\n", nprec, ngrec);
    }

  if (nrecs == 0 && nrecs2 > 0) cdoWarning("stream2 has more time steps than stream1!");
  if (nrecs > 0 && nrecs2 == 0) cdoWarning("stream1 has more time steps than stream2!");

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
