/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Pressure    pressure_fl          Pressure on full hybrid levels
      Pressure    pressure_hl          Pressure on half hybrid levels
      Pressure    deltap               Difference of two half hybrid levels
*/

#include <cdi.h>

#include "cdo_int.h"
#include "after_vertint.h"
#include "stdnametable.h"
#include "util_string.h"
#include "const.h"

void *
Pressure(void *process)
{
  ModelMode mode(ModelMode::UNDEF);
  int nrecs;
  int k;
  int varID, levelID;
  int zaxisIDp, zaxisIDh = -1;
  int nhlevf = 0, nhlevh = 0, nlevel = 0;
  int nvct = 0;
  size_t nmiss;
  int psID = -1, lnpsID = -1;
  char paramstr[32];
  char varname[CDI_MAX_NAME];
  double minval, maxval;
  double *pout = nullptr;
  gribcode_t gribcodes;
  memset(&gribcodes, 0, sizeof(gribcode_t));

  cdoInitialize(process);

  // clang-format off
  int PRESSURE_FL = cdoOperatorAdd("pressure_fl", 0, 0, nullptr);
  int PRESSURE_HL = cdoOperatorAdd("pressure_hl", 0, 0, nullptr);
  int DELTAP      = cdoOperatorAdd("deltap",      0, 0, nullptr);
  // clang-format on

  int operatorID = cdoOperatorID();

  CdoStreamID streamID1 = cdoOpenRead(0);

  int vlistID1 = cdoStreamInqVlist(streamID1);

  size_t gridsize = vlist_check_gridsize(vlistID1);

  int nhlev;
  double *vct = vlist_read_vct(vlistID1, &zaxisIDh, &nvct, &nhlev, &nhlevf, &nhlevh);

  bool l3Dvars = (zaxisIDh != -1 && gridsize > 0);
  if (!l3Dvars) cdoAbort("No 3D variable with hybrid sigma pressure coordinate found!");

  std::vector<double> ps_prog(gridsize);
  std::vector<double> deltap(gridsize * nhlevf);
  std::vector<double> full_press(gridsize * nhlevf);
  std::vector<double> half_press(gridsize * nhlevh);

  if (operatorID == PRESSURE_FL || operatorID == DELTAP)
    {
      if (Options::cdoVerbose) cdoPrint("Creating ZAXIS_HYBRID .. (nhlevf=%d)", nhlevf);
      zaxisIDp = zaxisCreate(ZAXIS_HYBRID, nhlevf);
    }
  else
    {
      if (Options::cdoVerbose) cdoPrint("Creating ZAXIS_HYBRID_HALF .. (nhlevh=%d)", nhlevh);
      zaxisIDp = zaxisCreate(ZAXIS_HYBRID_HALF, nhlevh);
    }

  std::vector<double> level(nhlevh);
  for (int l = 0; l < nhlevh; l++) level[l] = l + 1;
  zaxisDefLevels(zaxisIDp, level.data());

  zaxisDefVct(zaxisIDp, 2 * nhlevh, vct);

  int nvars = vlistNvars(vlistID1);

  bool useTable = false;
  for (varID = 0; varID < nvars; varID++)
    {
      int tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));
      if (tableNum > 0 && tableNum != 255)
        {
          useTable = true;
          break;
        }
    }

  if (Options::cdoVerbose && useTable) cdoPrint("Use code tables!");

  for (varID = 0; varID < nvars; varID++)
    {
      int zaxisID = vlistInqVarZaxis(vlistID1, varID);
      int nlevel = zaxisInqSize(zaxisID);
      int instNum = institutInqCenter(vlistInqVarInstitut(vlistID1, varID));
      int tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));

      int code = vlistInqVarCode(vlistID1, varID);
      int param = vlistInqVarParam(vlistID1, varID);

      cdiParamToString(param, paramstr, sizeof(paramstr));

      if (useTable)
        {
          if (tableNum == 2)
            {
              mode = ModelMode::WMO;
              wmo_gribcodes(&gribcodes);
            }
          else if (tableNum == 128)
            {
              mode = ModelMode::ECHAM;
              echam_gribcodes(&gribcodes);
            }
          //  KNMI: HIRLAM model version 7.2 uses tableNum=1    (LAMH_D11*)
          //  KNMI: HARMONIE model version 36 uses tableNum=1   (grib*)
          //  (opreational NWP version) KNMI: HARMONIE model version 38 uses
          //  tableNum=253 (grib,grib_md) and tableNum=1 (grib_sfx) (research
          //  version)
          else if (tableNum == 1 || tableNum == 253)
            {
              mode = ModelMode::HIRLAM;
              hirlam_harmonie_gribcodes(&gribcodes);
            }
        }
      else
        {
          mode = ModelMode::ECHAM;
          echam_gribcodes(&gribcodes);
        }

      if (Options::cdoVerbose)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cdoPrint("Mode = %d  Center = %d TableNum =%d Code = %d Param = %s Varname = %s varID = %d", (int) mode, instNum,
                   tableNum, code, paramstr, varname, varID);
        }

      if (code <= 0)
        {
          vlistInqVarName(vlistID1, varID, varname);

          cstrToLowerCase(varname);

          //                       ECHAM                         ECMWF
          if (cstrIsEqual(varname, "geosp") || cstrIsEqual(varname, "z"))
            code = 129;
          else if (cstrIsEqual(varname, "st") || cstrIsEqual(varname, "t"))
            code = 130;
          else if (cstrIsEqual(varname, "aps") || cstrIsEqual(varname, "sp"))
            code = 134;
          else if (cstrIsEqual(varname, "ps"))
            code = 134;
          else if (cstrIsEqual(varname, "lsp") || cstrIsEqual(varname, "lnsp"))
            code = 152;
          /* else if (cstrIsEqual(varname, "geopoth") == 0 ) code = 156; */
        }

      if (mode == ModelMode::ECHAM)
        {
          if (code == gribcodes.ps && nlevel == 1)
            psID = varID;
          else if (code == gribcodes.lsp && nlevel == 1)
            lnpsID = varID;
        }
      else if (mode == ModelMode::WMO || mode == ModelMode::HIRLAM)
        {
          if (code == gribcodes.ps && nlevel == 1) psID = varID;
        }
    }

  int pvarID = lnpsID;
  if (zaxisIDh != -1 && lnpsID != -1)
    {
      int gridID = vlistInqVarGrid(vlistID1, lnpsID);
      if (gridInqType(gridID) == GRID_SPECTRAL)
        {
          lnpsID = -1;
          cdoWarning("Spectral LOG(%s) not supported - using %s!", var_stdname(surface_air_pressure),
                     var_stdname(surface_air_pressure));
        }
    }

  if (zaxisIDh != -1 && lnpsID == -1)
    {
      pvarID = psID;
      if (psID == -1) cdoAbort("%s not found!", var_stdname(surface_air_pressure));
    }

  int gridID = vlistInqVarGrid(vlistID1, pvarID);
  if (gridInqType(gridID) == GRID_SPECTRAL)
    cdoAbort("%s on spectral representation not supported!", var_stdname(surface_air_pressure));

  std::vector<double> array(gridsize);

  int vlistID2 = vlistCreate();
  varID = vlistDefVar(vlistID2, gridID, zaxisIDp, TIME_VARYING);
  vlistDefVarParam(vlistID2, varID, cdiEncodeParam(1, 255, 255));
  vlistDefVarName(vlistID2, varID, "pressure");
  vlistDefVarStdname(vlistID2, varID, "air_pressure");
  vlistDefVarUnits(vlistID2, varID, "Pa");

  int taxisID1 = vlistInqTaxis(vlistID1);
  int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (varID == pvarID)
            {
              cdoReadRecord(streamID1, array.data(), &nmiss);
              if (nmiss) cdoAbort("Missing valus unsupported!");
            }
        }

      if (zaxisIDh != -1)
        {
          if (lnpsID != -1)
            for (size_t i = 0; i < gridsize; i++) ps_prog[i] = exp(array[i]);
          else if (psID != -1)
            arrayCopy(gridsize, array.data(), ps_prog.data());

          // check range of ps_prog
          arrayMinMax(gridsize, ps_prog.data(), minval, maxval);
          if (minval < MIN_PS || maxval > MAX_PS) cdoWarning("Surface pressure out of range (min=%g max=%g)!", minval, maxval);

          presh(full_press.data(), half_press.data(), vct, ps_prog.data(), nhlevf, gridsize);
        }

      if (operatorID == PRESSURE_FL)
        {
          nlevel = nhlevf;
          pout = full_press.data();
        }
      else if (operatorID == DELTAP)
        {
          nlevel = nhlevf;
          for (k = 0; k < nhlevf; ++k)
            for (size_t i = 0; i < gridsize; ++i)
              {
                deltap[k * gridsize + i] = half_press[(k + 1) * gridsize + i] - half_press[k * gridsize + i];
              }

          pout = deltap.data();
        }
      else if (operatorID == PRESSURE_HL)
        {
          nlevel = nhlevh;
          pout = half_press.data();
        }

      varID = 0;
      for (levelID = 0; levelID < nlevel; levelID++)
        {
          cdoDefRecord(streamID2, varID, levelID);
          size_t offset = levelID * gridsize;
          cdoWriteRecord(streamID2, pout + offset, 0);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  if (vct) Free(vct);

  cdoFinish();

  return nullptr;
}
