/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Rotuv      rotuvb          Backward rotation
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"

#include <mpim_grid.h>

static void
rot_uv_back(int gridID, double *us, double *vs)
{
  double xpole = 0, ypole = 0, angle = 0;
  if (gridInqType(gridID) == GRID_PROJECTION && gridInqProjType(gridID) == CDI_PROJ_RLL)
    gridInqParamRLL(gridID, &xpole, &ypole, &angle);

  const size_t nlon = gridInqXsize(gridID);
  const size_t nlat = gridInqYsize(gridID);

  std::vector<double> xvals(nlon);
  std::vector<double> yvals(nlat);
  gridInqXvals(gridID, xvals.data());
  gridInqYvals(gridID, yvals.data());

  /* Convert lat/lon units if required */
  char units[CDI_MAX_NAME];
  gridInqXunits(gridID, units);
  grid_to_degree(units, 1, &angle, "angle");
  grid_to_degree(units, 1, &xpole, "xpole");
  grid_to_degree(units, nlon, xvals.data(), "grid center lon");
  gridInqYunits(gridID, units);
  grid_to_degree(units, 1, &ypole, "ypole");
  grid_to_degree(units, nlat, yvals.data(), "grid center lat");

  if (xpole > 180) xpole -= 360;
  if (angle > 180) angle -= 360;

  for (size_t ilat = 0; ilat < nlat; ilat++)
    for (size_t ilon = 0; ilon < nlon; ilon++)
      {
        const size_t i = ilat * nlon + ilon;
        const double xval = lamrot_to_lam(yvals[ilat], xvals[ilon], ypole, xpole, angle);
        const double yval = phirot_to_phi(yvals[ilat], xvals[ilon], ypole, angle);
        usvs_to_uv(us[i], vs[i], yval, xval, ypole, xpole, &us[i], &vs[i]);
      }
}

#define MAXARG 16384

void *
Rotuv(void *process)
{
  int varID, levelID;
  int varID1, varID2;
  int nrecs;
  int chcodes[MAXARG];
  char *chvars[MAXARG];
  char varname[CDI_MAX_NAME];
  char varname2[CDI_MAX_NAME];
  double *single, *usvar = nullptr, *vsvar = nullptr;

  cdoInitialize(process);

  operatorInputArg("pairs of u and v in the rotated system");

  int nch = operatorArgc();
  if (nch % 2) cdoAbort("Odd number of input arguments!");

  bool lvar = false;  // We have a list of codes
  int len = (int) strlen(operatorArgv()[0]);
  int ix = (operatorArgv()[0][0] == '-') ? 1 : 0;
  for (int i = ix; i < len; ++i)
    if (!isdigit(operatorArgv()[0][i]))
      {
        lvar = true;  // We have a list of variables
        break;
      }

  if (lvar)
    {
      for (int i = 0; i < nch; i++) chvars[i] = operatorArgv()[i];
    }
  else
    {
      for (int i = 0; i < nch; i++) chcodes[i] = parameter2int(operatorArgv()[i]);
    }

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int nvars = vlistNvars(vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  size_t **varnmiss = (size_t **) Malloc(nvars * sizeof(size_t *));
  double **vardata = (double **) Malloc(nvars * sizeof(double *));

  bool lfound[MAXARG];
  for (int i = 0; i < nch; i++) lfound[i] = false;

  if (lvar)
    {
      for (varID = 0; varID < nvars; varID++)
        {
          vlistInqVarName(vlistID2, varID, varname);
          for (int i = 0; i < nch; i++)
            if (strcmp(varname, chvars[i]) == 0) lfound[i] = true;
        }
      for (int i = 0; i < nch; i++)
        if (!lfound[i]) cdoAbort("Variable %s not found!", chvars[i]);
    }
  else
    {
      for (varID = 0; varID < nvars; varID++)
        {
          const int code = vlistInqVarCode(vlistID2, varID);
          for (int i = 0; i < nch; i++)
            if (code == chcodes[i]) lfound[i] = true;
        }
      for (int i = 0; i < nch; i++)
        if (!lfound[i]) cdoAbort("Code %d not found!", chcodes[i]);
    }

  for (varID = 0; varID < nvars; varID++)
    {
      const int gridID = vlistInqVarGrid(vlistID1, varID);
      if (!(gridInqType(gridID) == GRID_PROJECTION && gridInqProjType(gridID) == CDI_PROJ_RLL))
        cdoAbort("Only rotated lon/lat grids supported!");

      const size_t gridsize = gridInqSize(gridID);
      const int nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      varnmiss[varID] = (size_t *) Malloc(nlevel * sizeof(size_t));
      vardata[varID] = (double *) Malloc(gridsize * nlevel * sizeof(double));
    }

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          recList[recID].varID = varID;
          recList[recID].levelID = levelID;
          recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;

          const size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
          const size_t offset = gridsize * levelID;
          single = vardata[varID] + offset;
          cdoReadRecord(streamID1, single, &varnmiss[varID][levelID]);
          if (varnmiss[varID][levelID]) cdoAbort("Missing values unsupported for this operator!");
        }

      for (int i = 0; i < nch; i += 2)
        {
          for (varID = 0; varID < nvars; varID++)
            {
              if (lvar)
                {
                  vlistInqVarName(vlistID2, varID, varname);
                  if (strcmp(varname, chvars[i]) == 0) break;
                }
              else
                {
                  const int code = vlistInqVarCode(vlistID2, varID);
                  if (code == chcodes[i]) break;
                }
            }

          if (varID == nvars) cdoAbort("u-wind not found!");

          usvar = vardata[varID];
          varID1 = varID;

          for (varID = 0; varID < nvars; varID++)
            {
              if (lvar)
                {
                  vlistInqVarName(vlistID2, varID, varname);
                  if (strcmp(varname, chvars[i + 1]) == 0) break;
                }
              else
                {
                  const int code = vlistInqVarCode(vlistID2, varID);
                  if (code == chcodes[i + 1]) break;
                }
            }

          if (varID == nvars) cdoAbort("v-wind not found!");

          vsvar = vardata[varID];
          varID2 = varID;

          if (Options::cdoVerbose)
            {
              if (lvar)
                {
                  vlistInqVarName(vlistID2, varID1, varname);
                  vlistInqVarName(vlistID2, varID2, varname2);
                  cdoPrint("Using var %s [%s](u) and var %s [%s](v)", varname, chvars[i], varname2, chvars[i + 1]);
                }
              else
                cdoPrint("Using code %d [%d](u) and code %d [%d](v)", vlistInqVarCode(vlistID1, varID1), chcodes[i],
                         vlistInqVarCode(vlistID1, varID2), chcodes[i + 1]);
            }

          const int gridID = vlistInqVarGrid(vlistID1, varID);
          const size_t gridsize = gridInqSize(gridID);
          const int nlevel1 = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID1));
          const int nlevel2 = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID2));

          if (nlevel1 != nlevel2) cdoAbort("u-wind and v-wind have different number of levels!");

          for (levelID = 0; levelID < nlevel1; levelID++)
            {
              size_t offset = gridsize * levelID;
              rot_uv_back(gridID, usvar + offset, vsvar + offset);
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          varID = recList[recID].varID;
          levelID = recList[recID].levelID;
          const size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
          const size_t offset = gridsize * levelID;
          single = vardata[varID] + offset;

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, single, varnmiss[varID][levelID]);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  for (varID = 0; varID < nvars; varID++)
    {
      Free(varnmiss[varID]);
      Free(vardata[varID]);
    }

  cdoFinish();

  return nullptr;
}
