/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <float.h>
#include <fenv.h>
#include <assert.h>

#include "compare.h"
#include "array.h"

//#pragma STDC FENV_ACCESS ON

const char *
fpe_errstr(int fpeRaised)
{
  const char *errstr = nullptr;

  // clang-format off
  if      (fpeRaised & FE_DIVBYZERO) errstr = "division by zero";
  else if (fpeRaised & FE_INEXACT)   errstr = "inexact result";
  else if (fpeRaised & FE_INVALID)   errstr = "invalid result";
  else if (fpeRaised & FE_OVERFLOW)  errstr = "overflow";
  else if (fpeRaised & FE_UNDERFLOW) errstr = "underflow";
  // clang-format on

  return errstr;
}

void
arrayMinMax(const size_t len, const double *array, double &rmin, double &rmax)
{
  double zmin = DBL_MAX;
  double zmax = -DBL_MAX;

  // #pragma omp parallel for default(none) shared(min, max, array, gridsize)
  // reduction(+:mean) #pragma omp simd reduction(+:mean) reduction(min:min)
  // reduction(max:max) aligned(array:16)
  for (size_t i = 0; i < len; ++i)
    {
      if (array[i] < zmin) zmin = array[i];
      if (array[i] > zmax) zmax = array[i];
    }

  rmin = zmin;
  rmax = zmax;
}

size_t
arrayMinMaxMV(const size_t len, const double *array, const double missval, double &rmin, double &rmax)
{
  double zmin = DBL_MAX;
  double zmax = -DBL_MAX;

  size_t nvals = 0;
  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(array[i], missval))
        {
          if (array[i] < zmin) zmin = array[i];
          if (array[i] > zmax) zmax = array[i];
          nvals++;
        }
    }

  rmin = zmin;
  rmax = zmax;

  return nvals;
}

void
arrayMinMaxSum(const size_t len, const double *array, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  // #pragma omp parallel for default(none) shared(min, max, array, gridsize)
  // reduction(+:mean) #pragma omp simd reduction(+:mean) reduction(min:min)
  // reduction(max:max) aligned(array:16)
  for (size_t i = 0; i < len; ++i)
    {
      if (array[i] < rmin) rmin = array[i];
      if (array[i] > rmax) rmax = array[i];
      rsum += array[i];
    }
}

void
arrayMinMaxSum_f(const size_t len, const float *array, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  // #pragma omp parallel for default(none) shared(min, max, array, gridsize)
  // reduction(+:mean) #pragma omp simd reduction(+:mean) reduction(min:min)
  // reduction(max:max) aligned(array:16)
  for (size_t i = 0; i < len; ++i)
    {
      if (array[i] < rmin) rmin = array[i];
      if (array[i] > rmax) rmax = array[i];
      rsum += array[i];
    }
}

size_t
arrayMinMaxSumMV(const size_t len, const double *array, const double missval, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  size_t nvals = 0;
  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(array[i], missval))
        {
          if (array[i] < rmin) rmin = array[i];
          if (array[i] > rmax) rmax = array[i];
          rsum += array[i];
          nvals++;
        }
    }

  if (nvals == 0 && IS_EQUAL(rmin,  DBL_MAX)) rmin = missval;
  if (nvals == 0 && IS_EQUAL(rmax, -DBL_MAX)) rmax = missval;

  return nvals;
}

size_t
arrayMinMaxSumMV_f(const size_t len, const float *array, const double missval, double &rmin, double &rmax, double &rsum)
{
  // rmin, rmax and rsum will be initialized in Info

  size_t nvals = 0;
  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(array[i], missval))
        {
          if (array[i] < rmin) rmin = array[i];
          if (array[i] > rmax) rmax = array[i];
          rsum += array[i];
          nvals++;
        }
    }


  if (nvals == 0 && IS_EQUAL(rmin,  DBL_MAX)) rmin = missval;
  if (nvals == 0 && IS_EQUAL(rmax, -DBL_MAX)) rmax = missval;

  return nvals;
}

void
arrayMinMaxMean(const size_t len, const double *array, double &rmin, double &rmax, double &rmean)
{
  rmin = DBL_MAX;
  rmax = -DBL_MAX;
  double sum = 0.0;
  arrayMinMaxSum(len, array, rmin, rmax, sum);

  rmean = len ? sum / (double) len : 0;
}

size_t
arrayMinMaxMeanMV(const size_t len, const double *array, const double missval, double &rmin, double &rmax, double &rmean)
{
  rmin = DBL_MAX;
  rmax = -DBL_MAX;
  double sum = 0.0;
  size_t nvals = arrayMinMaxSumMV(len, array, missval, rmin, rmax, sum);

  rmean = nvals ? sum / (double) nvals : missval;

  return nvals;
}

void
arrayMinMaxMask(const size_t len, const double *array, int *mask, double &rmin, double &rmax)
{
  double zmin = DBL_MAX;
  double zmax = -DBL_MAX;

  if (mask)
    {
      for (size_t i = 0; i < len; ++i)
        {
          if (!mask[i])
            {
              if (array[i] < zmin) zmin = array[i];
              if (array[i] > zmax) zmax = array[i];
            }
        }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        {
          if (array[i] < zmin) zmin = array[i];
          if (array[i] > zmax) zmax = array[i];
        }
    }

  rmin = zmin;
  rmax = zmax;
}

void
arrayAddArray(const size_t len, double *restrict array1, const double *restrict array2)
{
  //#ifdef  _OPENMP
  //#pragma omp parallel for default(none) shared(array1,array2)
  //#endif
  for (size_t i = 0; i < len; ++i) array1[i] += array2[i];
}

void
arrayAddArrayMV(const size_t len, double *restrict array1, const double *restrict array2, const double missval)
{
  if (DBL_IS_NAN(missval))
    {
      for (size_t i = 0; i < len; i++)
        if (!DBL_IS_EQUAL(array2[i], missval))
          {
            if (!DBL_IS_EQUAL(array1[i], missval))
              array1[i] += array2[i];
            else
              array1[i] = array2[i];
          }
    }
  else
    {
      for (size_t i = 0; i < len; i++)
        if (IS_NOT_EQUAL(array2[i], missval))
          {
            if (IS_NOT_EQUAL(array1[i], missval))
              array1[i] += array2[i];
            else
              array1[i] = array2[i];
          }
    }
}

size_t
arrayNumMV(const size_t len, const double *restrict array, const double missval)
{
  size_t nmiss = 0;

  if (DBL_IS_NAN(missval))
    {
      for (size_t i = 0; i < len; ++i)
        if (DBL_IS_EQUAL(array[i], missval)) nmiss++;
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        if (IS_EQUAL(array[i], missval)) nmiss++;
    }

  return nmiss;
}

size_t
arrayNumMV_f(const size_t len, const float *restrict array, const double missval)
{
  size_t nmiss = 0;

  if (DBL_IS_NAN(missval))
    {
      for (size_t i = 0; i < len; ++i)
        if (DBL_IS_EQUAL(array[i], missval)) nmiss++;
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        if (IS_EQUAL(array[i], missval)) nmiss++;
    }

  return nmiss;
}

double
arrayMin(const size_t len, const std::vector<double> &v)
{
  assert(v.size() >= len);

  double min = v[0];

  for (size_t i = 0; i < len; ++i)
    if (v[i] < min) min = v[i];

  return min;
}

double
arrayMax(const size_t len, const std::vector<double> &v)
{
  assert(v.size() >= len);

  double max = v[0];

  for (size_t i = 0; i < len; ++i)
    if (v[i] > max) max = v[i];

  return max;
}

double
arrayRange(const size_t len, const std::vector<double> &v)
{
  assert(v.size() >= len);

  double min = v[0];
  double max = v[0];

  for (size_t i = 0; i < len; ++i)
    {
      if (v[i] < min) min = v[i];
      if (v[i] > max) max = v[i];
    }

  return (max - min);
}

double
arrayMinMV(const size_t len, const std::vector<double> &v, const double missval)
{
  assert(v.size() >= len);

  double min = DBL_MAX;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval))
      if (v[i] < min) min = v[i];

  if (IS_EQUAL(min, DBL_MAX)) min = missval;

  return min;
}

double
arrayMaxMV(const size_t len, const std::vector<double> &v, const double missval)
{
  assert(v.size() >= len);

  double max = -DBL_MAX;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval))
      if (v[i] > max) max = v[i];

  if (IS_EQUAL(max, -DBL_MAX)) max = missval;

  return max;
}

double
arrayRangeMV(const size_t len, const std::vector<double> &v, const double missval)
{
  assert(v.size() >= len);

  double min = DBL_MAX;
  double max = -DBL_MAX;

  for (size_t i = 0; i < len; ++i)
    {
      if (!DBL_IS_EQUAL(v[i], missval))
        {
          if (v[i] < min) min = v[i];
          if (v[i] > max) max = v[i];
        }
    }

  double range;
  if (IS_EQUAL(min, DBL_MAX) && IS_EQUAL(max, -DBL_MAX))
    range = missval;
  else
    range = max - min;

  return range;
}

double
arraySum(const size_t len, const std::vector<double> &v)
{
  assert(v.size() >= len);

  double sum = 0;

  for (size_t i = 0; i < len; ++i) sum += v[i];

  return sum;
}

double
arraySumMV(const size_t len, const std::vector<double> &v, const double missval)
{
  assert(v.size() >= len);

  double sum = 0;
  size_t nvals = 0;

  if (DBL_IS_NAN(missval))
    {
      for (size_t i = 0; i < len; ++i)
        if (!DBL_IS_EQUAL(v[i], missval))
          {
            sum += v[i];
            nvals++;
          }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        if (IS_NOT_EQUAL(v[i], missval))
          {
            sum += v[i];
            nvals++;
          }
    }

  if (!nvals) sum = missval;

  return sum;
}

double
arrayMean(const size_t len, const std::vector<double> &v)
{
  assert(v.size() >= len);

  const double sum = arraySum(len, v);

  return sum / len;
}

double
arrayMeanMV(const size_t len, const std::vector<double> &v, const double missval)
{
  assert(v.size() >= len);

  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval))
      {
        sum += v[i];
        sumw += 1;
      }

  double missval1 = missval, missval2 = missval;
  return DIVMN(sum, sumw);
}

double
arrayWeightedMean(const size_t len, const std::vector<double> &v, const std::vector<double> &w, const double missval)
{
  assert(v.size() >= len);
  assert(w.size() >= len);

  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    {
      sum += w[i] * v[i];
      sumw += w[i];
    }

  return IS_EQUAL(sumw, 0.) ? missval : sum / sumw;
}

double
arrayWeightedMeanMV(const size_t len, const std::vector<double> &v, const std::vector<double> &w, const double missval)
{
  assert(v.size() >= len);
  assert(w.size() >= len);

  const double missval1 = missval, missval2 = missval;
  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(v[i], missval1) && !DBL_IS_EQUAL(w[i], missval1))
      {
        sum += w[i] * v[i];
        sumw += w[i];
      }

  return DIVMN(sum, sumw);
}

double
arrayAvgMV(const size_t len, const std::vector<double> &v, const double missval)
{
  assert(v.size() >= len);

  const double missval1 = missval, missval2 = missval;
  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    {
      sum = ADDMN(sum, v[i]);
      sumw += 1;
    }

  return DIVMN(sum, sumw);
}

double
arrayWeightedAvgMV(const size_t len, const std::vector<double> &v, const std::vector<double> &w, const double missval)
{
  assert(v.size() >= len);
  assert(w.size() >= len);

  const double missval1 = missval, missval2 = missval;
  double sum = 0, sumw = 0;

  for (size_t i = 0; i < len; ++i)
    if (!DBL_IS_EQUAL(w[i], missval))
      {
        sum = ADDMN(sum, MULMN(w[i], v[i]));
        sumw = ADDMN(sumw, w[i]);
      }

  return DIVMN(sum, sumw);
}
