/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifndef PROCESS_INT_H
#define PROCESS_INT_H

#include "processManager.h"
static ProcessManager g_processManager;
#include "process.h"
#include "cdo_defaultValues.h"

int cdoFiletype(void);
void cdoCloseAllStreams();

/**
 * Sets the underlying Process threadID and calls omp_set_num_threads
 */

void cdoRun(int processed_argc, const char **processed_argv);
void cdoInitialize(void *process);
void cdoFinish();

/**
 * Add operator to process.
 *
 * @param name name of the operator
 * @param f1 value for use in operator(e.g function ID from field.cc)
 * @param f2 value for use in operator
 * @param enter string for inquiring user input
 * @return operatorID
 */
int cdoOperatorAdd(const char *name, int f1, int f2, const char *enter);
int cdoOperatorID(void);
int cdoOperatorF1(int operID);
int cdoOperatorF2(int operID);

/**
 * Returns registered operator name
 */
const char *cdoOperatorName(int operID);

/**
 * Returns operator name
 */
const char *cdoOperatorEnter(int operID);

/**
 * Returns a stream name for given index.
 * if the given index is smaller than the current count of input files
 * the returned name will be a input stream.
 * Otherwise a name for an output stream will be returned.
 * Aborts if the streamID is larger than inStreamCnt + outStreamCnt
 */
const char *cdoGetStreamName(int p_streamIndex);
const char *cdoGetCommandFromInStream(int streamID);

/**
 *  Returns the basis name for output files.
 *  If no obase was found Cdo will exit
 */
const char *cdoGetObase();

/**
 * Returns the number type for the operator used in the current process
 */
int cdoStreamNumber();

/**
 * Returns the current number of in and out streams that were added to this cdoProcess
 */
int cdoStreamCnt(void);

/**
 * Returns the stream ID for \p inStreamIDX. If the input is from another
 * process, a new thread will be started and the found process will run in that
 * thread. Is the ID bigger thatn the count of input streams cdo will exit.
 */
int cdoStreamOpenRead(int inStreamIDX);
int cdoStreamOpenWrite(int outStreamIDX, int filetype = CDI_UNDEFID);
int cdoStreamOpenWrite(std::string p_filename, int filetype = CDI_UNDEFID);

/**
 * returns pstreamID for ID \p outStreamIDX.
 *
 */
int cdoStreamOpenAppend(int outStreamIDX);

/**
 * Checks if this process has \p numargs arguments
 */
void operatorCheckArgc(int numargs);

/**
 * Asks user for input.
 * @param enter string that will be displayed when asking for input from the
 * user
 */
void operatorInputArg(const char *enter);
/**
 * Returns pointer to first element of operator arguments
 */
char **operatorArgv(void);

/**
 * Returns the number of operator arguments
 */
int operatorArgc(void);

void processDefVarNum(int nvars);

/**
 * Returns number of variables
 */
int processInqVarNum();
/**
 * Returns c string representation of this process
 */
const char *processInqPrompt(void);

/**
 * Returns the surrounding process
 */
Process &processSelf(void);

/**
 * Returns vlistID for pstream with ID = \p pstreamID
 * cdo will exit if no stream for \p pstreamID is found
 */
int cdoStreamInqVlist(int pstreamID);

/**
 * Checks whether pstream with \p pstreamID is a file or a pipe.
 * @return true if pipe otherwise false
 */
bool cdoAssertFilesOnly();

int cdoStreamInqTimestep(int pstreamID, int tsID);

void cdoKillProcesses();

void cdoStreamClose(int pstreamID);

/**
 * Define the variable list.
 *
 *     @param  streamID Stream ID, from a previous call to
 * cdoStreamOpenRead(int) or cdoStreamOpenWrite(int,int).
 *     @param  vlistID  Variable list ID, from a previous call to
 * cdoInqVlist(int).
 *
 * The function cdoDefVlist(int, int) defines the variable list of a stream.
 *
 * To safeguard against errors by modifying the wrong vlist object,
 * this function makes the passed vlist object immutable.
 * All further vlist changes have to use the vlist object returned by
 * cdoInqVlist().
 *
 */
void cdoDefVlist(int streamID, int vlistID);

/**
 * Inquires record and sets \p varID.
 * @param streamID
 * @param varID
 * @param levelID
 * If \p streamID represents a pipe \p varID will be -1 if an error occured.
 * If \p streamID represents a file either varID will be set or CDO exits with
 * an error message.
 **/
void cdoInqRecord(int streamID, int *varID, int *levelID);
/**
 *Define the next record.
 *The function cdoDefRecord defines the meta-data of the next record.
 *    @param  streamID  Stream ID, from a previous call to cdoStreamOpenRead()
 *    or cdoStreamOpenWrite().
 *    @param  varID     Variable identifier.
 *    @param  levelID   Level identifier.
 */
void cdoDefRecord(int streamID, int varID, int levelID);

void cdoDefTimestep(int streamID, int tsID);

void cdoInqGRIBinfo(int streamID, int *intnum, float *fltnum, off_t *bignum);
int cdoInqFiletype(int streamID);
int cdoInqByteorder(int streamID);
int cdoFileID(int streamID);

void cdoReadRecord(int streamID, double *data, size_t *nmiss);
void cdoReadRecordF(int streamID, float *data, size_t *nmiss);
void cdoCopyRecord(int streamIDdest, int streamIDsrc);

void cdoWriteRecord(int streamID, double *data, size_t nmiss);
void cdoWriteRecordF(int streamID, float *data, size_t nmiss);

void cdoVlistCopyFlag(int vlistID2, int vlistID1);
bool unchangedRecord();

/****** NEW FUNCTION DEFINITIONS FOR POINTER BASED INTERFACE *******/

int cdoFiletype(void);
void cdoCloseAllStreams();

/**
 * Sets the underlying Process threadID and calls omp_set_num_threads
 */

void cdoRun(int processed_argc, const char **processed_argv);
void cdoInitialize(void *process);
void cdoFinish();

/**
 * Add operator to process.
 *
 * @param name name of the operator
 * @param f1 value for use in operator(e.g function ID from field.cc)
 * @param f2 value for use in operator
 * @param enter string for inquiring user input
 * @return operatorID
 */
int cdoOperatorAdd(const char *name, int f1, int f2, const char *enter);
int cdoOperatorID(void);
int cdoOperatorF1(int operID);
int cdoOperatorF2(int operID);

/**
 * Returns registered operator name
 */
const char *cdoOperatorName(int operID);

/**
 * Returns operator name
 */
const char *cdoOperatorEnter(int operID);

/**
 * Returns a stream name for given index.
 * if the given index is smaller than the current count of input files
 * the returned name will be a input stream.
 * Otherwise a name for an output stream will be returned.
 * Aborts if the streamID is larger than inStreamCnt + outStreamCnt
 */
const char *cdoGetStreamName(int p_streamIndex);
const char *cdoGetCommandFromInStream(int p_streamIndex);

/**
 *  Returns the basis name for output files.
 *  If no obase was found Cdo will exit
 */
const char *cdoGetObase();

/**
 * Returns the number type for the operator used in the current process
 */
int cdoStreamNumber();

/**
 * Returns the current number of in and out streams that were added to this cdoProcess
 */
int cdoStreamCnt(void);

/**
 * Returns the stream ID for \p inStreamIDX. If the input is from another
 * process, a new thread will be started and the found process will run in that
 * thread. Is the ID bigger thatn the count of input streams cdo will exit.
 */
CdoStreamID cdoOpenRead(int inStreamIDX);
CdoStreamID cdoOpenWrite(int outStreamIDX, int filetype = CDI_UNDEFID);
CdoStreamID cdoOpenWrite(std::string p_filename, int filetype = CDI_UNDEFID);
CdoStreamID cdoOpenAppend(int outStreamIDX);

/**
 * returns pstreamID for ID \p outStreamIDX.
 *
 */
void operatorCheckArgc(int numargs);
void operatorInputArg(const char *enter);
char **operatorArgv(void);
int operatorArgc(void);
void processDefVarNum(int nvars);
int processInqVarNum();
const char *processInqPrompt(void);
Process &processSelf(void);
void cdoKillProcesses();
/****************************************************/
int cdoStreamInqVlist(CdoStreamID streamID);
bool cdoStreamIsPipe(CdoStreamID streamID);
int cdoStreamInqTimestep(CdoStreamID streamID, int tsID);
void cdoStreamClose(CdoStreamID streamID);
void cdoDefVlist(CdoStreamID streamID, int vlistID);
void cdoInqRecord(CdoStreamID streamID, int *varID, int *levelID);
void cdoDefRecord(CdoStreamID streamID, int varID, int levelID);
void cdoDefTimestep(CdoStreamID streamID, int tsID);
void cdoInqGRIBinfo(CdoStreamID streamID, int *intnum, float *fltnum, off_t *bignum);
int cdoInqFiletype(CdoStreamID streamID);
int cdoInqByteorder(CdoStreamID streamID);
int cdoFileID(CdoStreamID streamID);
void cdoReadRecord(CdoStreamID streamID, double *data, size_t *nmiss);
void cdoReadRecordF(CdoStreamID streamID, float *data, size_t *nmiss);
void cdoCopyRecord(CdoStreamID streamIDdest, CdoStreamID streamIDsrc);
void cdoWriteRecord(CdoStreamID streamID, double *data, size_t nmiss);
void cdoWriteRecordF(CdoStreamID streamID, float *data, size_t nmiss);
void cdoVlistCopyFlag(int vlistID2, int vlistID1);

void cdoDefCompType(CdoStreamID p_streamID, int p_cdi_compression_type);

#endif
