/* cdw
 * Copyright (C) 2002 Varkonyi Balazs, 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "cdw_tmp_files.h"

int cdw_tmpfile;
char tmp_file_name[100];




/*
 * Create some temporary files used by cdw to store various data
 *
 * It will be long function implementing safe opening of random files
 *
 * \returns 0
 */
int tmpfiles_init(void)
{
	strcpy(tmp_file_name, "/tmp/cdw_tmpfile");
	return 0;
}

void filelist_clean(void)
{
	int i;
	for (i = 1; i < FILELIST_SIZE; i++) {
		if (filelist[i] != NULL) {
			free(filelist[i]);
		}
	}

	return;
}


/* 
 * Add given path to set of file paths
 *
 * Add file path to list of unique file paths.
 *
 * \param char *path - path to file or directory that will be added to list
 *
 * \returns 0 - if file was not added (already on list); -1 - on memory allocation error;
 * 	-2 - if no place left on list; value > 0 - when path succesfully added
 */
int filelist_add(char *path)
{
	
	int i;
	for (i = 1; i < FILELIST_SIZE; i++) {
		if ( filelist[i] != NULL ) {
			if ( ! (strcmp(path, filelist[i])) ) { /* file is already on the list */
				return 0; /* file not added, already exists, no error */
			}
		}
	}

	for (i = 1; i < FILELIST_SIZE; i++) {
		if (filelist[i] == NULL) { /* empty place - add here */
			filelist[i] = (char *)  malloc( strlen(path) + 1);
			if (filelist[i] == NULL) {
				return -1; /* file not added - memory alloc error */
			}
			strcpy(filelist[i], path);
			return i; /* success */

		}
	}

	return -2; /* error - no place on list */
}




/*
 * Remove given path from list of files
 *
 * Remove given path from list of files 
 * that are going to be put on CD.
 *
 * \param char *path - path to file/dir that has to be removed ftom list
 *
 * \returns 0 on success, -1 when no path found on list, -2 when other error
 */
int filelist_remove(char *path)
{
	int i;

	// we are starting from 1 because adding to list was started from 1 too
	for (i = 1; i < FILELIST_SIZE; i++) {
		if ( filelist[i] != NULL ) {
			if ( !(strcmp(path, filelist[i])) ) { /* path matched */
				free(filelist[i]);  /* remove path */
				filelist[i] = NULL;  /* mark position as free for use */
				return 0;
			}
		}
	}

	return -1; /* no such path */
}




/* 
 * Put content of list of files to temporary file
 *
 * Put content of list of files (that are going to be put to
 * CD or iso image) to temporary file that will be used by
 * prgram creating iso image.
 *
 * \returns -1 on tmp file open error; -2 on write error; 0 on success
 */
int filelist_to_file(void)
{	
	int rv;
	int i;
	char tmp[300];
	cdw_tmpfile = open(tmp_file_name, O_CREAT | O_TRUNC | O_RDWR, S_IRWXU);
	if (cdw_tmpfile == -1) {
		return -1;
	}

	for (i = 1; i < FILELIST_SIZE; i++) {
		if (filelist[i] != NULL) {
			/* determine last dir's name */
			int len = strlen(filelist[i]);
			char tmp_path[255];
			strcpy(tmp_path, filelist[i]);
			int k = 0;
			for (k = len; k > 0; k--){
				if (tmp_path[k] == '/') {
					break;
				}
			}
			/* now last dir in path will be top level dir on CD;
			 * +1 skips slash in path;
			 * this code works for paths ending with filenames, too (somehow) */
			sprintf(tmp, "%s=%s\n", tmp_path + k + 1, filelist[i]);
			rv = write(cdw_tmpfile, tmp, strlen(tmp)); 

			if (rv == -1) {
				return -2;
			}
		}
	}
	close(cdw_tmpfile);
	fsync(cdw_tmpfile);

	return 0;
}




/*
 * Return number of items on this list
 *
 * \returns 0 if no files, number of files > 0  if any file on the list, -1 on error
 */
int filelist_items(char ***filetable)
{
	int n = 0; /* number of items on list */
	int fli;   /* file list index */
	int fti;   /* file table index */

	/* first - get number of non-empty elements */
	for (fli = 1; fli < FILELIST_SIZE; fli++) {
		if (filelist[fli] != NULL) {
			n++;
		}
	}

	/* copy n non-empty elements to freshly allocated table */
	if (n != 0) {
		*filetable = (char **) malloc(n * sizeof(char *));

		/* <low-level>table for pointers to path strings is allocated, paths themselves also
		 * exist (and so do poiners to them)</low-level>
		 *
		 * <high-level>let's copy all strings from list to table</high-level> */
		fti = 0;	
		for (fli = 1; fli < FILELIST_SIZE; fli++) {
			if (filelist[fli] != NULL) {
				/* this line assumes that filelist outlives filetable
				 * and pointes from filelist always belong to filelist,
				 * eg. you cannot free them when cleannin filetable */
				*(*(filetable) + fti) = filelist[fli];
				fti++;
			}
		}
	} else {
		*filetable = NULL;
	}
	
	return n;
}




/*
 * Check whether list of files to write to CD is empty
 *
 * \returns 0 if filelist not empty, 1 otherwise 
 */
int filelist_empty(void)
{
	int i;
	for (i = 1; i < FILELIST_SIZE; i++) {
		if (filelist[i] != NULL) { /* empty elements are set to NULL */
			return 0; /* filelist not empty */
		}
	}

	return 1; /* filelist empty */
}




/*
 * Initialize filelist data structure
 *
 * Set all values of this list to NULL - NULL means that entry 
 * is empty and may be used
 */
void filelist_init(void)
{
	int i;
	for (i = 1; i < FILELIST_SIZE; i++) {
		filelist[i] = NULL;
	}

	return;
}
