/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>

#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_regex_dispatch.h"
#include "cdw_debug.h"
#include "cdw_dvd_rw_mediainfo_regex.h"
#include "cdw_config.h"


/* plus_16 comment:
 * somewhere below you can find comment with plus_16 in it, commenting
 * line, where mysterious value of 16 is added to other value. 16 was guessed
 * after analyzing outputs from k3b and dvd+rw-mediainfo, and was confirmed
 * after reading k3b sources. The value appears in sources of k3b in
 * k3b-1.0.4/libk3b/projects/datacd/k3bmsinfofetcher.cpp in definition of
 * K3bMsInfoFetcher::slotMediaDetectionFinished().
 *
 * For some reason the value is added to value of starting address of last
 * track on dvd disc. I'm sure that there is some good explanation for this,
 * I'm just too lazy to read T10 specs ;-)
 */


extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
/* extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1]; */



extern cdw_config_t global_config;
extern cdw_disc_t current_disc;


/* there is a special way of checking if DVD+RW is empty, even if
   "Disc status" line in dvd+rw-mediainfo says otherwise; this is a flag that
   is set to true if DVD+RW is REALLY empty, and its state should override
   value of "Disc status" */

static regex_t regex1, regex2, regex3;
/* static regex_t regex5, regex4,  regex6; */
static regex_t regex7, regex8, regex9;
static regmatch_t *matches1 = NULL, *matches2 = NULL, *matches3 = NULL;
/* static regmatch_t *matches4 = NULL, *matches5 = NULL, *matches6 = NULL; */
static regmatch_t *matches7 = NULL, *matches8 = NULL, *matches9 = NULL;



static void cdw_dvd_rw_mediainfo_pr_handle_disc_status(regex_t *regex, regmatch_t *matches);
static void cdw_dvd_rw_mediainfo_pr_handle_dvd_write_speed(regex_t *regex, regmatch_t *matches);
static void cdw_dvd_rw_mediainfo_pr_handle_mounted_media(regex_t *regex, regmatch_t *matches);
static void cdw_dvd_rw_mediainfo_pr_handle_current_write_speed(regex_t *regex, regmatch_t *matches);
static void cdw_dvd_rw_mediainfo_pr_handle_read_capacity(regex_t *regex, regmatch_t *matches);

/* valid but unused */
/*
static void handle_dvd_toc(regex_t *regex, regmatch_t *matches);
static void handle_track_start_address(regex_t *regex, regmatch_t *matches);
static void handle_next_writable_address(regex_t *regex, regmatch_t *matches); */


void dvd_rw_mediainfo_stdout_regexp_prepare(void)
{
	int rv;

	/*  "Write Speed #0:        4.0x1385=5540KB/s" */
	rv = regcomp(&regex1, "Write Speed #([0-9]+):([\t ]+)([0-9]+).([0-9]+)x", REG_EXTENDED);
	matches1 = (regmatch_t *) calloc(1, (regex1.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex1, 1);
	}

	/* "Disc status:           complete" (appendable/blank/other) */
	rv = regcomp(&regex2, "Disc status:([\t ]+)(complete|appendable|blank|other)", REG_EXTENDED);
	matches2 = (regmatch_t *) calloc(1, (regex2.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex2, 2);
	}

	/* "Mounted Media:         1Ah, DVD+RW" */
	/* "Mounted Media:         1Ah, DVD-RW Sequential" */
	rv = regcomp(&regex3, "Mounted Media:([ ]+)([a-zA-Z0-9]+),([ ]+)([a-zA-Z +-]*)", REG_EXTENDED);
	matches3 = (regmatch_t *) calloc(1, (regex3.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex3, 3);
	}
#if 0
	/* " Track#2  :             14@145056" - information about DVD TOC,
	   last number is start of track; track number may be also AA, whatever
	   it means in DVD context */
	rv = regcomp(&regex4, "Track#([0-9A]+)([ ]*):([ ]*)([0-9]*)@([0-9]+)", REG_EXTENDED);
	matches4 = (regmatch_t *) calloc(1, (regex4.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex4, 4);
	}

	/* "Track Start Address:   666512" - capture start addresses of DVD tracks*/
	rv = regcomp(&regex5, "Track Start Address:([ ]*)([0-9]+)", REG_EXTENDED);
	matches5 = (regmatch_t *) calloc(1, (regex5.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex5, 5);
	}

	/* "Next Writable Address: 470672*2KB" start address for next session */
	rv = regcomp(&regex6, "Next Writable Address:([ ]*)([0-9]+)", REG_EXTENDED);
	matches6 = (regmatch_t *) calloc(1, (regex6.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex6, 6);
	}
#endif
	/* "Current Write Speed:   16.0x1385=22160KB/s" */
	rv = regcomp(&regex7, "Current Write Speed:([\t ]*)([0-9]+)", REG_EXTENDED);
	matches7 = (regmatch_t *) calloc(1, (regex7.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex7, 7);
	}

	/* "READ CAPACITY:          0*2048=0" */
	/* read capacity for empty disc is 0 */
	rv = regcomp(&regex8, "READ CAPACITY:([\t ]+)([0-9]+)", REG_EXTENDED);
	matches8 = (regmatch_t *) calloc(1, (regex8.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex8, 8);
	}

	/* "DVD-ROM media detected, exiting..." */
	rv = regcomp(&regex9, "DVD-ROM media detected", REG_EXTENDED);
	matches9 = (regmatch_t *) calloc(1, (regex9.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex9, 9);
	}

	return;
}





void dvd_rw_mediainfo_stdout_regexp_execute(void)
{
	int rv = 0;

	/*  "Write Speed #0:        4.0x1385=5540KB/s" */
	rv = regexec(&regex1, stdout_pipe_buffer, regex1.re_nsub + 1, matches1, 0);
	if (rv == 0) {
		cdw_dvd_rw_mediainfo_pr_handle_dvd_write_speed(&regex1, matches1);
	}

	/* "Disc status:           xxxxxxx" */
	rv = regexec(&regex2, stdout_pipe_buffer, regex2.re_nsub + 1, matches2, 0);
	if (rv == 0) {
		cdw_dvd_rw_mediainfo_pr_handle_disc_status(&regex2, matches2);
	}

	/* "Mounted Media:         1Ah, DVD+RW" */
	rv = regexec(&regex3, stdout_pipe_buffer, regex3.re_nsub + 1, matches3, 0);
	if (rv == 0) {
		cdw_dvd_rw_mediainfo_pr_handle_mounted_media(&regex3, matches3);
	}
#if 0
	/* " Track#2  :             14@145056" */
	rv = regexec(&regex4, stdout_pipe_buffer, regex4.re_nsub + 1, matches4, 0);
	if (rv == 0) {
		/* valid but unused */
		/* handle_dvd_toc(&regex4, matches4); */
	}

	/* " Track Start Address:   666512" */
	rv = regexec(&regex5, stdout_pipe_buffer, regex5.re_nsub + 1, matches5, 0);
	if (rv == 0) {
		handle_track_start_address(&regex5, matches5);
	}

	/* "Next Writable Address: 470672*2KB" */
	rv = regexec(&regex6, stdout_pipe_buffer, regex6.re_nsub + 1, matches6, 0);
	if (rv == 0) {
		handle_next_writable_address(&regex6, matches6);
	}
#endif
	/* "Current Write Speed:   16.0x1385=22160KB/s" */
	rv = regexec(&regex7, stdout_pipe_buffer, regex7.re_nsub + 1, matches7, 0);
	if (rv == 0) {
		cdw_dvd_rw_mediainfo_pr_handle_current_write_speed(&regex7, matches7);
	}

	/* "READ CAPACITY:          0*2048=0" */
	rv = regexec(&regex8, stdout_pipe_buffer, regex8.re_nsub + 1, matches8, 0);
	if (rv == 0) {
		cdw_sdm ("dispatching to cdw_dvd_rw_mediainfo_pr_handle_read_capacity()\n");
		cdw_dvd_rw_mediainfo_pr_handle_read_capacity(&regex8, matches8);
	}

	/* "DVD-ROM media detected, exiting..." */
	rv = regexec(&regex9, stdout_pipe_buffer, regex9.re_nsub + 1, matches9, 0);
	if (rv == 0) {
		cdw_vdm ("handling \"DVD-ROM media detected, exiting...\"\n");
		/* this isn't necessary, because
		   cdw_dvd_rw_mediainfo_pr_handle_mounted_media()
		   can recognize DVD-ROM, but it doesn't hurt either */
		current_disc.type = CDW_DVD_ROM;
	}

	return;
}





void dvd_rw_mediainfo_stdout_regexp_destroy(void)
{
	regfree(&regex1);
	regfree(&regex2);
	regfree(&regex3);
	/* regfree(&regex4);
	   regfree(&regex5);
	   regfree(&regex6); */
	regfree(&regex7);
	regfree(&regex8);
	regfree(&regex9);

	free(matches1);
	free(matches2);
	free(matches3);
	/* free(matches4);
	   free(matches5);
	   free(matches6); */
	free(matches7);
	free(matches8);
	free(matches9);

	return;
}





/**
   \brief Parse writing speeds reported by dvd+rw-mediainfo

   dvd+rw-mediainfo writes to stdout writing speeds in following format:
   \verbatim
Write Speed #0:        16.0x1385=22160KB/s
Write Speed #1:        12.0x1385=16620KB/s
Write Speed #2:        10.0x1385=13850KB/s
Write Speed #3:        8.0x1385=11080KB/s
Write Speed #4:        4.0x1385=5540KB/s
\endverbatim

   All that needs to be done is to convert #0 (#1, #2...) into number,
   use the number to index current_disc.write_speeds.speeds[] table,
   and store writing speed in the table.
*/
void cdw_dvd_rw_mediainfo_pr_handle_dvd_write_speed(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing write speeds is
                            1        2       3        4
           "Write Speed #([0-9]+):([\t ]+)([0-9]+).([0-9]+)x */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);
	int speed_i = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 1) {
			/* number of available speed */
			speed_i = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed index = \"%s\" -> %d\n", i, submatch, speed_i);

			/* this will work in both cases if dvd+rw-mediainfo
			   would list speed indexes from highest to lowest and
			   from lowest to highest */
			if (speed_i + 1 > current_disc.write_speeds.n_speeds
			    && speed_i < CDW_DISC_N_SPEEDS_MAX) {

				current_disc.write_speeds.n_speeds = speed_i + 1; /* number of elements in table */
			}
			cdw_vdm ("INFO: speed_i = %d, n_speeds = %d, CDW_DISC_N_SPEEDS_MAX = %d\n",
				 speed_i, current_disc.write_speeds.n_speeds, CDW_DISC_N_SPEEDS_MAX);
		} else if (i == 3) {
			/* speed value; write_speeds[] has limited size */
			if (speed_i < CDW_DISC_N_SPEEDS_MAX) {
				current_disc.write_speeds.speeds[speed_i] = atoi(submatch);
				cdw_vdm ("INFO: submatch %d: speed %d = \"%s\" -> %d\n",
					 i, speed_i, submatch, current_disc.write_speeds.speeds[speed_i]);
			}
		} else if (i == 4) {
			cdw_vdm ("INFO: submatch %d: remainder of regexp = \"%s\"\n", i, submatch);
			/* fractional part of writing speed, not too
			   interesting, since writing speeds are integers;
			   there may be DVD discs with 2.5 speed, but I haven't
			   seen one yet, so I can't test this */
		}
	}

	return;
}





/**
   \brief Handle string describing status of disc

   Handle string printed by dvd+rw-mediainfo that has form of:
   "Disc status:           xxxxxxx"

   \verbatim
"Disc status:           complete"
\endverbatim
 *
 * Disc status for DVD can be one of following:
 * "blank", "appendable", "complete", "other"
 */
void cdw_dvd_rw_mediainfo_pr_handle_disc_status(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing disc status:
                           1                   2
	   "Disc status:([\t ]+)(complete|appendable|blank|other) */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	unsigned int sub = 2; /* select second regexp submatch */
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return;
	} else {
		/* yes, using only first letter as information about
		   status reported by dvd+rw-mediainfo - all statuses
		   have distinct first letters;
		   assigning values to disc->state_* has been moved to
		   dvd_rw_mediainfo_interface.c */
		current_disc.dvd_rw_mediainfo_info.disc_status = submatch[0];
		cdw_vdm ("INFO: disc status \"%s\" saved as '%c'\n",
			 submatch,
			 current_disc.dvd_rw_mediainfo_info.disc_status);
		return;
	}
}





/**
   \brief Extract information about media type

   Extract information (media type) from dvd+rw-mediainfo
   string which looks like this:

   \verbatim
" Mounted Media:         1Bh, DVD+R"

                 1         2          3        4
"Mounted Media:([ ]+)([a-zA-Z0-9]+),([ ]+)([\-a-zA-Z +]*)"
\endverbatim

   This function sets current_disc.type
*/
void cdw_dvd_rw_mediainfo_pr_handle_mounted_media(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing media type:
                            1        2           3         4
	   "Mounted Media:([ ]+)([a-zA-Z0-9]+),([ ]+)([a-zA-Z +-]*)"
	   2 - coded value
	   4 - human-readable name */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);

	unsigned int sub = 4;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return;
	}

	cdw_vdm ("INFO submatch %d, media type = \"%s\"\n", sub, submatch);

	/* convert string value into int value so that
	   higher level code doesn't have to compare strings */
	if (!strncmp(submatch, "DVD-R Sequential", 16)) {
		current_disc.type = CDW_DVD_R_SEQ;

	} else if (!strncmp(submatch, "DVD-R Restricted", 16)) {
		current_disc.type = CDW_DVD_R_RES;

	} else if (!strcmp(submatch, "DVD-R")) {
		current_disc.type = CDW_DVD_R;

	} else if (!strncmp(submatch, "DVD-RW Sequential", 17)) {
		current_disc.type = CDW_DVD_RW_SEQ;

	} else if (!strncmp(submatch, "DVD-RW Restricted", 17)) {
		current_disc.type = CDW_DVD_RW_RES;

	} else if (!strncmp(submatch, "DVD-RW", 6)) {
		current_disc.type = CDW_DVD_RW;

	} else if (!strcmp(submatch, "DVD+R")) {
		current_disc.type = CDW_DVD_RP;

	} else if (!strcmp(submatch, "DVD+RW")) {
		current_disc.type = CDW_DVD_RWP;

	} else if (!strcmp(submatch, "DVD-ROM")) {
		current_disc.type = CDW_DVD_ROM;

	} else if (!strcmp(submatch, "DVD+R Double Layer")
		   && global_config.support_dvd_rp_dl) {
		current_disc.type = CDW_DVD_RP_DL;

	} else {
		current_disc.type = CDW_DISC_TYPE_UNKNOWN;
		cdw_vdm ("WARNING: unknown disc type, submatch was \"%s\"\n", submatch);
	}

	return;
}





void cdw_dvd_rw_mediainfo_pr_handle_current_write_speed(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing current write speed is
                                   1       2
	   "Current Write Speed:([\t ]*)([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	unsigned int sub = 2;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return;
	}

	/* current write speed */
	current_disc.write_speeds.drive_default_speed = atoi(submatch);
	cdw_vdm ("INFO: submatch %d, current write speed = \"%s\" -> %d\n",
		 sub, submatch, current_disc.write_speeds.drive_default_speed);

	return;
}





/**
 \verbatim
"READ CAPACITY:          0*2048=0"
\endverbatim
*/
void cdw_dvd_rw_mediainfo_pr_handle_read_capacity(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing read capacity of DVD disc is
                             1       2
	   "READ CAPACITY:([\t ]+)([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	unsigned int sub = 2;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return;
	}
	current_disc.dvd_rw_mediainfo_info.read_capacity = (ssize_t) atoll(submatch);

	/* since regexp code matched this submatch as ([0-9]+), then value
	   '0' possibly returned by atoi() is not a sign of incorrectly
	   formed string in its argument, but it means that the arguments
	   translates to zero */
	cdw_vdm ("INFO: submatch %d, read capacity = \"%s\" -> %zd\n",
		 sub, submatch, current_disc.dvd_rw_mediainfo_info.read_capacity);
	cdw_vdm ("INFO: read capacity = %zd\n", current_disc.dvd_rw_mediainfo_info.read_capacity);

	return;
}




/* *** valid, but unused code below */

#if 0

/**
 * Handle output of dvd+rw-mediainfo when it prints TOC table
 *
 * TOC printed by dvd+rw-mediainfo looks like this:
 * \verbatim
FABRICATED TOC:
 Track#1  :             14@0
 Track#2  :             14@145056
 Track#3  :             14@309712
 Track#4  :             14@470672
 Track#AA :             14@601376
\endverbatim
 *
 * Note that there is a track number, so we can use it to index writing speed
 * table. Note also that tracks are numbered from 1.
 *
 * The function has similar effect to handle_track_start_address() but it does
 * not capture start of blank area. However it does capture start of AA area.
 */
void handle_dvd_toc(regex_t *regex, regmatch_t *matches)
{
	char track_index[3];
	char track_start_address[10];
	int i_track;

	/* collect data */
	unsigned int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], len);
		submatch[len] = '\0';
		cdw_sdm ("matching submatch: \"%s\"\n", submatch);
		if (i == 1) {
			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);

			/* track index, 1-based */
			strncpy(track_index, submatch, len + 1);

			/* track number reported by dvd+rw-mediainfo can be
			   either a number or "AA" string */
			if (!strncmp(track_index, "AA", 2)) {
				i_track = 99;
			} else {
				i_track = atoi(track_index);
			}
		} else if (i == 5) {

			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);

			/* start of i-th track */
			strncpy(track_start_address, submatch, len + 1);

			/* track numbers reported by dvd+rw-mediainfo are
			   1-based, we will keep it that way; this will be also
			compatible with cdio */
			current_disc.dvd_rw_mediainfo_info.track_address[i_track] = atol(track_start_address);
		} else {
			;
		}
	}

	return;
}




/**
   \brief Handle dvd+rw-mediainfo strings with track start indexes

   dvd+rw-mediainfo output looks like this:
   \verbatim
   Track Start Address:   470672*2KB
   \endverbatim
   and it is repeated for every track. We use
   current_disc.dvd_rw_mediainfo_info.n_tracks to index
   current_disc.dvd_rw_mediainfo_info.track_address[] and point where to store
   successive track start addresses.

   The function has similar effect to handle_dvd_toc(), but it also captures
   starting sector of blank area, which is used as part of '-C' parameter
   of growisofs when writing multisession discs.

   handle_track_start_address() does not capture AA track.
   handle_dvd_toc() does it.
*/
void handle_track_start_address(regex_t *regex, regmatch_t *matches)
{
	unsigned int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		if (len < 0) {
			cdw_assert (len < 0, "len of subexpr is negative: %d\n", len);
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], (size_t) len);
		submatch[len] = '\0';

		cdw_sdm ("matching submatch: \"%s\"\n", submatch);
		if (i == 2) {
			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);
			int tsa = atoi(submatch); /* track start address */
			current_disc.dvd_rw_mediainfo_info.track_address[current_disc.dvd_rw_mediainfo_info.n_tracks++] = tsa;
		}
	}

	return;
}





/**
 * Capture address of first sector of next track/session
 *
 * dvd+rw-mediainfo prints following information about "empty" track, the
 * track that follows last track written to disc:
 * \verbatim
STDOUT READ TRACK INFORMATION[#4]:
 Track State:           blank
 Track Start Address:   470672*2KB
 Next Writable Address: 470672*2KB
 Free Blocks:           1824432*2KB
 Track Size:            1824432*2KB
\endverbatim
 *
 * The string that we must capture and process is
 * "Next Writable Address: 470672*2KB". This string is written after all data
 * is processed by handle_track_start_address().
 * Therefore we can use:
 *
 * (current_disc.dvd_rw_mediainfo_info.n_tracks - 1)
 * to get
 * last session start from current_disc.dvd_rw_mediainfo_info.track_address[]
 *
 * (current_disc.dvd_rw_mediainfo_info.n_tracks)
 * to get
 * next (blank) session start from current_disc.dvd_rw_mediainfo_info.track_address[]
 *
 * However "next (blank) session start" is not quite the same as "Next Writable
 * Address":
 * It may seem that values of "Track Start Address" and
 * "Next Writable Address" are the same, but dvd+rw-mediainfo uses two
 * different expressions to calculate the values. I use
 * "Next Writable Address" (nwa) instead of '"Track Start Address" of
 * blank track' as value of next_sess_start.
 */
void handle_next_writable_address(regex_t *regex, regmatch_t *matches)
{
	long int nwa = 0;

	/* collect data */
	unsigned int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		if (len < 0) {
			cdw_assert (len < 0, "len of subexpr is negative: %d\n", len);
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], (size_t) len);
		submatch[len] = '\0';

		cdw_sdm ("matching submatch: \"%s\"\n", submatch);
		if (i == 2) {
			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);

			/* next writable address */
			nwa = atol(submatch);
		} else {
			;
		}
	}


	if (current_disc.dvd_rw_mediainfo_info.n_tracks == 0) {
		/* no tracks, blank disc, so both last_sess_start
		   and next_sess_start should be 0 */
		current_disc.last_sess_start = 0;
		current_disc.next_sess_start = 0;
	} else {
		/* see comment at the beginning of file, starting with plus_16 */
		current_disc.last_sess_start = current_disc.dvd_rw_mediainfo_info.track_address[current_disc.dvd_rw_mediainfo_info.n_tracks - 1] + 16;
		current_disc.next_sess_start = nwa;
	}


	cdw_sdm ("current_disc.last_sess_start - 16 = %ld\n", current_disc.last_sess_start - 16);
	cdw_sdm ("current_disc.last_sess_start = %ld\n", current_disc.last_sess_start);
	cdw_sdm ("current_disc.next_sess_start = %ld\n", current_disc.next_sess_start);

	return;
}

#endif
