/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
   \file cdw_widgets.c
   \brief Set of ui controls built on top of ncurses

   Small cdw functions that can be used to implement widgets in other
   ncurses-based projects with small modifications or no modifications at all.
*/

#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* strndup(), and possibly strdup() too */

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>

#include <ncursesw/form.h>

#include "gettext.h"
#include "cdw_widgets.h"
#include "cdw_main_window.h" /* main_win_refresh_part() */
#include "cdw_string.h" /* trim() */
#include "cdw_debug.h"
#include "cdw_ncurses.h"
#include "cdw_form.h"


/* basic building block for creating more advanced dialogs */
typedef struct {
	/* type of dialog window: CDW_DIALOG_BUTTONS or
	   CDW_DIALOG_INPUT, but always with text on top */
	int dialog_type;
	cdw_colors_t colors;

	WINDOW *window;
	size_t n_rows;
	size_t n_cols;
	size_t begin_y;
	size_t begin_x;

	/* space between dialog window borders and content of the window:
	   the text area and buttons/input field; pad_x may be 0 if
	   more space is needed to display text without truncations */
	size_t pad_y;
	size_t pad_x;

	/* size of content (txt area + (buttons or input field)) of dialog
	   window, sum this with pad_x and pad_y to get total dialog window size */
	size_t n_rows_internal;
	size_t n_cols_internal;

	/* text area in upper part of dialog; it can be one-line prompt, but
	   it can be multi-line text too */
	struct {
		WINDOW *subwindow;
		size_t n_rows;
		size_t n_cols;
	} txt;

	/* area where buttons or input field is displayed */
	struct {
		size_t n_cols;
		size_t n_rows; /* will be always (?) equal one, see comment below for explanation */

		/* buttons dialog:
		   all buttons are in one row, the row's number is begin_y;

		   input line dialog:
		   input line occupies one row in dialog window, the row's number is begin_y */
		size_t begin_y;

		/* empty line between text area and entry (input) area below */
		size_t spacer_y;
	} input;

	/* if dialog->type is CDW_DIALOG_BUTTONS, then this struct
	   stores buttons visible below text area (in "input" area) */
	struct {
		/* 4 buttons should be enough for everyone */
		CDW_BUTTON *buttons[4];
		char *labels[4];
		size_t label_lens[4];

		/* currently selected button, used e.g. in driver */
		int current;
		/* how many buttons and of which kind */
		int type;
	} buttons;

} CDW_DIALOG_BASE;





/* symbolic names to control buttons in cdw_buttons_dialog()
   (e.g. to index buttons table) */
enum {
	BUTTON_OK = 0,
	BUTTON_YES = 1,
	BUTTON_NO = 2,
	BUTTON_CANCEL = 3
};



static CDW_DIALOG_BASE *cdw_dialog_base_new(int dialog_type);
static void cdw_dialog_base_delete(CDW_DIALOG_BASE **dialog);
static cdw_rv_t cdw_dialog_base_calculate_sizes(CDW_DIALOG_BASE *dialog, const char *message);
static cdw_rv_t cdw_dialog_base_calculate_input_sizes(CDW_DIALOG_BASE *dialog);
static cdw_rv_t cdw_dialog_base_calculate_subwindow_txt_sizes(CDW_DIALOG_BASE *dialog, const char *message);
static void cdw_dialog_base_display_basics(CDW_DIALOG_BASE *dialog, const char *title, const char *message, int align);

static void cdw_dialog_create_button_labels(CDW_DIALOG_BASE *dialog);
static cdw_rv_t cdw_dialog_driver(CDW_DIALOG_BASE *dialog);
static void cdw_dialog_display_buttons(CDW_DIALOG_BASE *dialog);

static void cdw_input_line_safe_driver_message(CDW_INPUT_LINE *input_line, char *insecure);

static size_t cdw_textarea_wrap_text(char *buffer, size_t width);
static size_t cdw_textarea_calculate_height(const char *message, size_t width);

static cdw_rv_t cdw_dropdown_expanded_driver(CDW_DROPDOWN *dropdown);
static cdw_rv_t cdw_dropdown_expand(CDW_DROPDOWN *dropdown);
static cdw_rv_t cdw_dropdown_collapse(CDW_DROPDOWN *dropdown);
static void cdw_dropdown_display_current_item_reverse(CDW_DROPDOWN *dropdown, bool reverse);
static int cdw_one_line_form_driver(FORM *form, int *validation_result, bool return_on_tab);




int cdw_one_line_form_driver(FORM *form, int *validation_result, bool return_on_tab)
{
	cdw_assert (form != (FORM *) NULL, "ERROR: can't operate on NULL form");
	WINDOW *window = form_win(form);
	cdw_assert (window != (WINDOW *) NULL, "ERROR: can't get window of a form with form_win()\n");

	int key = 0;
	do {
		int r = E_OK;
		key = wgetch(window);
		switch (key) {
			case KEY_LEFT:
				r = form_driver(form, REQ_PREV_CHAR);
				break;
			case KEY_RIGHT:
				r = form_driver(form, REQ_NEXT_CHAR);
				break;
			case KEY_BACKSPACE:
				r = form_driver(form, REQ_DEL_PREV);
				break;
			case KEY_DC:
				r = form_driver(form, REQ_DEL_CHAR);
				break;
		        case KEY_HOME:
				r = form_driver(form, REQ_BEG_LINE);
				break;
		        case KEY_END:
				r = form_driver(form, REQ_END_LINE);
				break;
			case CDW_KEY_ESCAPE:
				break;
			case CDW_KEY_ENTER:
				/* it seems that this line is a good idea,
				   but it won't work if user enters string
				   longer than limit - we will have to put
				   '\0' either way */
				/* form_driver(form, REQ_END_LINE);
				   form_driver(form, '\0'); */ /* push to form string's ending null char */

				/* this line is important, it's something like
				   saving entered value in field buffer or
				   flushing value to buffer */
				r = form_driver(form, REQ_VALIDATION);

				*validation_result = form_driver(form, REQ_VALIDATION);
				break;
		        case CDW_KEY_TAB:
				if (return_on_tab) {
					cdw_vdm ("INFO: captured TAB, will return\n");
					// r = E_OK;
					/* this line is important, it's something like
					   saving entered value in field buffer or
					   flushing value to buffer */
					r = form_driver(form, REQ_VALIDATION);
					*validation_result = form_driver(form, REQ_VALIDATION);
					break;
				}
			default:
				/* push user input to form */
				r = form_driver(form, key);
				wrefresh(window);
				break;
		} /* switch () */
		if (r != E_OK) {
			/* E_REQUEST_DENIED may be returned e.g. when user
			   enters n+1 char into field with limit of chars
			   set to n; not an error, just an information that
			   limit set in code was reached */
			cdw_vdm ("WARNING: form_driver() returns %s for key %s\n",
				 cdw_ncurses_error_string(r), cdw_ncurses_key_label(key));
		}

	} while ((key != CDW_KEY_ESCAPE) && (key != CDW_KEY_ENTER) && !(return_on_tab && key == CDW_KEY_TAB));

	return key;
}





cdw_rv_t cdw_safe_input_dialog(cdw_safe_input_data_t *data)
{
	CDW_DIALOG_BASE *dialog = cdw_dialog_base_new(CDW_DIALOG_INPUT);
	if (dialog == (CDW_DIALOG_BASE *) NULL) {
		cdw_vdm ("ERROR: failed to get new dialog base\n");
		return CDW_GEN_ERROR;
	}

	cdw_dialog_base_calculate_sizes(dialog, data->prompt_message);
	cdw_dialog_base_display_basics(dialog, data->window_title, data->prompt_message, CDW_ALIGN_LEFT);

	if (data->chars_max) {
		if (dialog->input.n_cols > data->chars_max) {
			/* input subwindow must be no wider than limit,
			   otherwise setting size limit for input buffer
			   will fail */
			dialog->input.n_cols = data->chars_max;
		}
	}

	int begin_x = (getmaxx(dialog->window) - (int) dialog->input.n_cols) / 2;
	CDW_INPUT_LINE *input_line = cdw_input_line_new(dialog->window,
							(int) dialog->input.begin_y, begin_x,
							(int) dialog->input.n_cols, *(data->buffer),
							data->input_type, (size_t) data->chars_max);
	if (input_line == (CDW_INPUT_LINE *) NULL) {
		cdw_vdm ("ERROR: failed to create input line\n");
		return CDW_GEN_ERROR;
	}

	cdw_rv_t retval = CDW_CANCEL;

	int key = cdw_input_line_safe_driver(input_line, data->buffer, data->attempts_max);
	if (key == CDW_KEY_ENTER || key == CDW_KEY_TAB) {
		/* user entered correct value, do noting */
		retval = CDW_OK;
	} else if (key == CDW_KEY_ESCAPE) {
		/* user cancelled entering string, do nothing */
		retval = CDW_CANCEL;
	} else {
		cdw_buttons_dialog(data->window_title,
				   data->error_message,
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		retval = CDW_GEN_ERROR;
	}

	cdw_input_line_delete(&input_line);
	cdw_dialog_base_delete(&dialog);

	return retval;
}





/**
   \brief Show dialog window with title, message and buttons

   Show dialog window with title and message. Depending on type of dialog
   user is presented with one, two or three buttons: value of
   parameter 'dialog_type' describes them.

   Function returns value of selected button. If user pressed Enter then it is:
   CDW_OK for 'Yes' or 'OK', CDW_CANCEL for 'Cancel', CDW_NO for 'No'.

   If user pressed Escape then function returns CDW_CANCEL.

   Background color depends on \p colors value.

   \param title - dialog window title
   \param message - message displayed in dialog window
   \param buttons_type - number and type of buttons, one of CDW_BUTTONS_* values
   \param colors - color scheme of dialog

   \return value of type cdw_rv_t, corresponding to selected button
   \return CDW_MEM_ERROR on malloc error
 */
cdw_rv_t cdw_buttons_dialog(const char *title, const char *message, int buttons_type, cdw_colors_t colors)
{
	cdw_assert (buttons_type == CDW_BUTTONS_OK
		    || buttons_type == CDW_BUTTONS_OK_CANCEL
		    || buttons_type == CDW_BUTTONS_YES_NO_CANCEL,

		    "ERROR: using wrong dialog buttons type %d\n", buttons_type);

	CDW_DIALOG_BASE *dialog = cdw_dialog_base_new(CDW_DIALOG_BUTTONS);
	if (dialog == (CDW_DIALOG_BASE *) NULL) {
		return CDW_MEM_ERROR;
	}

	dialog->colors = colors;
	dialog->buttons.type = buttons_type;

	cdw_dialog_create_button_labels(dialog);

	/* debug code */
	/* char *message = "\n  a  \n  b  \n  c  \n  d  \n  e  \n  f  \n  g  \n  h  \n  i  \n  j  \n  k  \n  l  \n  m  \n  n  \n  o  \n  p  \n  q  \n  r  \n  s  \n  t  \n  u  \n  w"; */
	/* char *message = "This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.\n\nThis program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA"; */

	cdw_dialog_base_calculate_sizes(dialog, message);
	cdw_dialog_base_display_basics(dialog, title, message, CDW_ALIGN_CENTER);

	cdw_dialog_display_buttons(dialog);

	redrawwin(dialog->window);
	wrefresh(dialog->window);

	cdw_rv_t retval = cdw_dialog_driver(dialog);

	cdw_dialog_base_delete(&dialog);

	return retval;
}





/**
   \brief Create new widget of type CDW_DIALOG_BASE

   \param dialog_type - CDW_DIALOG_BUTTONS or CDW_DIALOG_INPUT

   \return pointer to new widget on success
   \return NULL on error
*/
CDW_DIALOG_BASE *cdw_dialog_base_new(int dialog_type)
{
	CDW_DIALOG_BASE *base = (CDW_DIALOG_BASE *) malloc(sizeof (CDW_DIALOG_BASE));
	if (base == (CDW_DIALOG_BASE *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for dialog base\n");
		return (CDW_DIALOG_BASE *) NULL;
	}

	base->buttons.buttons[0] = (CDW_BUTTON *) NULL;
	base->buttons.buttons[1] = (CDW_BUTTON *) NULL;
	base->buttons.buttons[2] = (CDW_BUTTON *) NULL;
	base->buttons.buttons[3] = (CDW_BUTTON *) NULL;

	base->buttons.labels[0] = (char *) NULL;
	base->buttons.labels[1] = (char *) NULL;
	base->buttons.labels[2] = (char *) NULL;
	base->buttons.labels[3] = (char *) NULL;

	/* currently selected button, set by cdw_dialogbox_display_buttons(),
	   based on dialog_type */
	base->buttons.current = 0;

	base->window = (WINDOW *) NULL;
	base->n_rows = 0;
	base->n_cols = 0;
	base->begin_y = 0;
	base->begin_x = 0;
	/* keep some space between dialog window borders
	   and everything inside the window */
	base->pad_y = 1;
	base->pad_x = 1;

	/* keep space between text window and input area below the window */
	base->input.spacer_y = 1;

	base->txt.subwindow = (WINDOW *) NULL;
	/* initial text area size */
	base->txt.n_rows = 0;
	/* 40 is half of minimal supported terminal width, seems ok */
	base->txt.n_cols = 40;

	base->colors = CDW_COLORS_DIALOG;

	base->dialog_type = dialog_type;
	base->buttons.type = 0;

	return base;
}





/**
   \brief Deallocate resources allocated by cdw_dialogbox()

   Wrapper for few calls of free().
   Also frees variable pointed to by argument and sets it to NULL.

   \param dialog - pointer to widget created by cdw_dialog_base_new()
 */
void cdw_dialog_base_delete(CDW_DIALOG_BASE **dialog)
{
	for (int i = 0; i < 4; i++) {
		if ((*dialog)->buttons.buttons[i] != (CDW_BUTTON *) NULL) {
			cdw_assert ((*dialog)->dialog_type == CDW_DIALOG_BUTTONS,
				    "ERROR: button #%d is not null, but dialog type != CDW_DIALOG_BUTTONS\n", i);
			cdw_button_delete(&((*dialog)->buttons.buttons[i]));
			cdw_assert ((*dialog)->buttons.buttons[i] == (CDW_BUTTON *) NULL,
				    "ERROR: delete didn't set button pointer to NULL\n");
		}
	}

	if ((*dialog)->txt.subwindow != (WINDOW *) NULL) {
		delwin((*dialog)->txt.subwindow);
		(*dialog)->txt.subwindow = (WINDOW *) NULL;
	}

	if ((*dialog)->window != (WINDOW *) NULL) {
		delwin((*dialog)->window);
		(*dialog)->window = (WINDOW *) NULL;
	}

	free(*dialog);
	*dialog = (CDW_DIALOG_BASE *) NULL;

	return;
}





cdw_rv_t cdw_dialog_base_calculate_input_sizes(CDW_DIALOG_BASE *dialog)
{
	const size_t nice_input_width = 50;

	/* maximal width of input field or buttons area, -2 is for window borders */
	const size_t max_cols = (size_t) COLS - 2 - 2 * dialog->pad_x;

	if (dialog->dialog_type == CDW_DIALOG_BUTTONS) {
		if (dialog->buttons.type == CDW_BUTTONS_OK) {
			dialog->input.n_cols = dialog->buttons.label_lens[BUTTON_OK];
		} else if (dialog->buttons.type == CDW_BUTTONS_OK_CANCEL) {
			dialog->input.n_cols = dialog->buttons.label_lens[BUTTON_OK]
				+ dialog->buttons.label_lens[BUTTON_CANCEL];
		} else if (dialog->buttons.type == CDW_BUTTONS_YES_NO_CANCEL) {
			dialog->input.n_cols = dialog->buttons.label_lens[BUTTON_YES]
				+ dialog->buttons.label_lens[BUTTON_NO]
				+ dialog->buttons.label_lens[BUTTON_CANCEL];
		} else {
			cdw_assert (0, "ERROR: unknown buttons type %d\n", dialog->buttons.type);
		}
	} else if (dialog->dialog_type == CDW_DIALOG_INPUT) {
		/* input from user may have any length, but in order to not
		   to have input field as wide as main app window, or very
		   narrow in case of small limit, let's set it's width to
		   sth that will make the input field look "nicely" */
		dialog->input.n_cols = nice_input_width;
	} else {
		cdw_assert (0, "ERROR: unknown dialog type %d\n", dialog->dialog_type);
	}

	cdw_sdm ("INFO: initial required width of dialog base is %zd\n", dialog->input.n_cols);

	/* we have some expected, ideal number of columns to display given
	   set of buttons, or to display input field that would fit whole
	   input; now let's deal with limitations imposed by current screen */

	dialog->input.n_rows = 1; /* buttons have to fit in one row, input field always takes one row */

	bool buttons_crammed = false;
	if (dialog->input.n_cols > max_cols) {
		dialog->input.n_cols = max_cols;

		if (dialog->dialog_type == CDW_DIALOG_BUTTONS) {
			buttons_crammed = true;
			/* in case of buttons dialog the button labels may
			   overlap */
			cdw_vdm ("ERROR: buttons in a dialog will overlap\n");
		} else {
			/* in case of input dialog the input field
			   won't display whole content at once,
			   nothing wrong with that */
		}
	}

	cdw_sdm ("INFO: final required width of dialog input area is %zd\n", dialog->input.n_cols);

	if (buttons_crammed) {
		return CDW_GEN_ERROR;
	} else {
		return CDW_OK;
	}
}





cdw_rv_t cdw_dialog_base_calculate_sizes(CDW_DIALOG_BASE *dialog, const char *message)
{
	cdw_rv_t crv1 = cdw_dialog_base_calculate_input_sizes(dialog);
	if (crv1 != CDW_OK) {
		cdw_vdm ("ERROR: failed to calculate satisfactory size of input area, buttons may be crammed\n");
	}

	cdw_rv_t crv2 = cdw_dialog_base_calculate_subwindow_txt_sizes(dialog, message);
	if (crv2 != CDW_OK) {
		cdw_vdm ("ERROR: failed to calculate satisfactory sizes of txt subwindow, text may be clipped\n");
	}

	/* at this moment functions calculating sizes for input area and
	   textarea have specified their size requirements and have set
	   following fields in dialog data structure:

	   dialog->n_rows_input
	   dialog->n_cols_input
	   dialog->n_rows_txt
	   dialog->n_cols_txt

	   now we must resolve n_rows_internal and n_cols_internal */

	if (dialog->input.n_cols > dialog->txt.n_cols) {
		dialog->n_cols_internal = dialog->input.n_cols;
	} else {
		dialog->n_cols_internal = dialog->txt.n_cols;
	}

	size_t max = (size_t) LINES - 2   /* -2 for window borders */
		- 2 * dialog->pad_y       /* may be already set to 0 by calculate_subwindow_txt_size() */
		- dialog->input.n_rows    /* this is not to be changed anymore */
		- dialog->input.spacer_y; /* additional row above and below input area */

	if (max < dialog->txt.n_rows) {

		dialog->pad_y = 0;

		max = (size_t) LINES - 2          /* -2 for window borders */
			- 2 * dialog->pad_y       /* may be already set to 0 by calculate_subwindow_txt_size() */
			- dialog->input.n_rows    /* this is not to be changed anymore */
			- dialog->input.spacer_y; /* additional rows above and below input area */

		if (max < dialog->txt.n_rows) {

			dialog->input.spacer_y = 0;

			max = (size_t) LINES - 2          /* -2 for window borders */
				- 2 * dialog->pad_y       /* may be already set to 0 by calculate_subwindow_txt_size() */
				- dialog->input.n_rows    /* this is not to be changed anymore */
				- dialog->input.spacer_y; /* additional rows above and below input area */

			if (max < dialog->txt.n_rows) {
				dialog->txt.n_rows = max;
			}
		}
	}

	dialog->n_rows_internal = dialog->txt.n_rows + dialog->input.spacer_y + dialog->input.n_rows;
	dialog->input.begin_y = 1 + dialog->pad_y + dialog->txt.n_rows + dialog->input.spacer_y;

	if (crv1 != CDW_OK || crv2 != CDW_OK) {
		cdw_vdm ("ERROR: something went wrong when calculating sizes in dialogbox, expect layout problems\n");
		return CDW_GEN_ERROR;
	} else {
		return CDW_OK;
	}
}





cdw_rv_t cdw_dialog_base_calculate_subwindow_txt_sizes(CDW_DIALOG_BASE *dialog, const char *message)
{
	/* change value of height so that window will be high
	   enough to display whole message */
	dialog->txt.n_rows = cdw_textarea_calculate_height(message, dialog->txt.n_cols);

	size_t n_rows = dialog->txt.n_rows + 2 * dialog->pad_y + dialog->input.spacer_y + dialog->input.n_rows;
	if (n_rows > (size_t) LINES) {
		/* message won't fit into dialog window of current width,
		   stretch the width .. */
		/* first -2 is for window borders */
		dialog->txt.n_cols = (size_t) COLS - 2 - 2 * dialog->pad_x;
		/* ... and try again */
		dialog->txt.n_rows = cdw_textarea_calculate_height(message, dialog->txt.n_cols);

		n_rows = dialog->txt.n_rows + 2 * dialog->pad_y + dialog->input.spacer_y + dialog->input.n_rows;
		if (n_rows > (size_t) LINES) {

			/* last try, get rid of non-critical space... */
			dialog->pad_x = 0;
			dialog->txt.n_cols = (size_t) COLS - 2 - 2 * dialog->pad_x;
			/* ... and try again */
			dialog->txt.n_rows = cdw_textarea_calculate_height(message, dialog->txt.n_cols);

			n_rows = dialog->txt.n_rows + 2 * dialog->pad_y + dialog->input.spacer_y + dialog->input.n_rows;
		}
	}

	if (n_rows > (size_t) LINES) {
		cdw_vdm ("ERROR: failed to calculate satisfactory sizes of txt subwindow, text may be clipped\n");
		return CDW_GEN_ERROR;
	} else {
		return CDW_OK;
	}
}





void cdw_dialog_base_display_basics(CDW_DIALOG_BASE *dialog, const char *title, const char *message, int align)
{
	/* +2 is for borders */
	dialog->n_rows = dialog->n_rows_internal + 2 * dialog->pad_y + 2;
	dialog->n_cols = dialog->n_cols_internal + 2 * dialog->pad_x + 2;
	dialog->begin_y = ((size_t) LINES - dialog->n_rows) / 2;
	dialog->begin_x = ((size_t) COLS - dialog->n_cols) / 2;

	dialog->window = newwin((int) dialog->n_rows, (int) dialog->n_cols, (int) dialog->begin_y, (int) dialog->begin_x);

	wbkgd(dialog->window, COLOR_PAIR(dialog->colors));
	keypad(dialog->window, TRUE);

	werase(dialog->window);
	cdw_ncurses_nice_box(dialog->window, title, (char *) NULL);

	dialog->txt.subwindow = derwin(dialog->window,
				       (int) dialog->txt.n_rows, (int) dialog->txt.n_cols,
				       1 + (int) dialog->pad_y, 1 + (int) dialog->pad_x);
	wbkgd(dialog->txt.subwindow, COLOR_PAIR(dialog->colors));

	cdw_textarea_print_message(dialog->txt.subwindow, message, align);
	wrefresh(dialog->txt.subwindow);

	return;
}





void cdw_dialog_create_button_labels(CDW_DIALOG_BASE *dialog)
{
	/* 2TRANS: button label, please keep as short as possible */
	dialog->buttons.labels[BUTTON_OK] = _("OK");
	/* 2TRANS: button label, please keep as short as possible */
	dialog->buttons.labels[BUTTON_YES] = _("Yes");
	/* 2TRANS: button label, please keep as short as possible */
	dialog->buttons.labels[BUTTON_NO] = _("No");
	/* 2TRANS: button label, please keep as short as possible */
	dialog->buttons.labels[BUTTON_CANCEL] = _("Cancel");

	dialog->buttons.label_lens[BUTTON_OK] = strlen(dialog->buttons.labels[BUTTON_OK]);
	dialog->buttons.label_lens[BUTTON_YES] = strlen(dialog->buttons.labels[BUTTON_YES]);
	dialog->buttons.label_lens[BUTTON_NO] = strlen(dialog->buttons.labels[BUTTON_NO]);
	dialog->buttons.label_lens[BUTTON_CANCEL] = strlen(dialog->buttons.labels[BUTTON_CANCEL]);

	return;
}





/**
   \brief Function reacting to keyboard hits in dialogbox

   Function reads keys pressed by user when a dialogbox is displayed,
   and reacting appropriately: moving focus from one key to another, and
   returning appropriate value when user hits ENTER or ESCAPE key.

   \param window - dialog box window
   \param current_button - index of button that initially has focus
   \param dialog_type - type of dialog window

   \return CDW_OK if user selected 'OK' or 'Yes' button
   \return CDW_CANCEL if user selected 'Cancel' button or pressed ESCAPE key
   \return CDW_NO if user selected 'No' button
 */
cdw_rv_t cdw_dialog_driver(CDW_DIALOG_BASE *dialog)
{
	cdw_assert (dialog != (CDW_DIALOG_BASE *) NULL, "ERROR: cannot operate on NULL dialog\n");
	cdw_assert (dialog->window != (WINDOW *) NULL, "ERROR: cannot operate on NULL window\n");

	int c = 0;
	/* handle cursor keys - highlight selected button */
	while (((c = wgetch(dialog->window)) != CDW_KEY_ESCAPE) && (c != CDW_KEY_ENTER)) {
		/* select action according to dialog dialog_type,
		   key pressed and currently selected button */
		if (dialog->buttons.type == CDW_BUTTONS_OK_CANCEL) {
			switch(c){
				case KEY_RIGHT:
				case CDW_KEY_TAB:
				case KEY_LEFT:
					cdw_button_unfocus(dialog->buttons.buttons[dialog->buttons.current]);
					if (dialog->buttons.current == BUTTON_OK){
						dialog->buttons.current = BUTTON_CANCEL;
					} else if (dialog->buttons.current == BUTTON_CANCEL){
						dialog->buttons.current = BUTTON_OK;
					} else {
						;
					}
					cdw_button_focus(dialog->buttons.buttons[dialog->buttons.current]);
					break;
				default: /* do nothing - only arrows and tab move focus */
					break;
			}
		} else if (dialog->buttons.type == CDW_BUTTONS_YES_NO_CANCEL) {
			switch(c){
				case KEY_RIGHT:
				case CDW_KEY_TAB:
					cdw_button_unfocus(dialog->buttons.buttons[dialog->buttons.current]);
					if (dialog->buttons.current == BUTTON_YES){
						dialog->buttons.current = BUTTON_NO;
					} else if (dialog->buttons.current == BUTTON_NO){
						dialog->buttons.current = BUTTON_CANCEL;
					} else if (dialog->buttons.current == BUTTON_CANCEL){
						dialog->buttons.current = BUTTON_YES;
					}
					cdw_button_focus(dialog->buttons.buttons[dialog->buttons.current]);
					break;
				case KEY_LEFT:
					cdw_button_unfocus(dialog->buttons.buttons[dialog->buttons.current]);
					if (dialog->buttons.current == BUTTON_YES){
						dialog->buttons.current = BUTTON_CANCEL;
					} else if (dialog->buttons.current == BUTTON_NO){
						dialog->buttons.current = BUTTON_YES;
					} else if (dialog->buttons.current == BUTTON_CANCEL){
						dialog->buttons.current = BUTTON_NO;
					}
					cdw_button_focus(dialog->buttons.buttons[dialog->buttons.current]);
					break;
				default: /* do nothing - only arrows and tab move focus */
					break;
			} /* switch */
		} else if (dialog->buttons.type == CDW_BUTTONS_OK) {
			; /* nothing to do in one-button dialog */
		} else {
			cdw_assert (0, "ERROR: unknown buttons_type value: %d\n", dialog->buttons.type);
		}
		wrefresh(dialog->window);
	} /* while */

	if (c == CDW_KEY_ENTER) {
		if (dialog->buttons.current == BUTTON_OK || dialog->buttons.current == BUTTON_YES) {
			return CDW_OK;
		} else if (dialog->buttons.current == BUTTON_NO) {
			return CDW_NO;
		} else {
			return CDW_CANCEL;
		}
	} else { /* CDW_KEY_ESCAPE */
		return CDW_CANCEL;
	}
}





/**
   \brief Build buttons and display them in proper places of dialog window

   The function initializes button labels and buttons, calculates position
   of buttons and displays puts them in proper order and position - all
   that depending on type of dialog.

   \p dialog_type can be one of those: DIALOG_OK, DIALOG_OK_CANCEL,
   DIALOG_YES_NO_CANCEL

   \p colors is usually symbolic value passed to function by parent widget.

   \p current_button will be initialized by the function

   \param window - window in which you want to put buttons (dialog window)
   \param current_button - button that will have initial focus
   \param dialog_type - type of dialog in which caller wants to draw buttons
   \param colors - color scheme of the button
*/
void cdw_dialog_display_buttons(CDW_DIALOG_BASE *dialog)
{
	cdw_assert (dialog->window != (WINDOW *) NULL, "ERROR: cannot display buttons in NULL window\n");

	/* placing buttons is a bit tricky if you consider that translations
	   may have widths different than original strings; that is why the
	   code uses XXX_string_len to calculate placement of string. Width
	   of whole dialog window is divided into one, two or three equal
	   slots, depending on number of buttons. Then position of start of
	   button in each slot is calculated. I'm using XXX_string_len as
	   length of string, +2 for spaces around label to get better
	   placement of buttons. Maybe I should add 4 (spaces + brackets),
	   but dialog window might be too narrow for this (remember about
	   translations that may be longer than original strings). */

	size_t n_cols = (size_t) getmaxx(dialog->window) - 2;

	/* draw initial dialog box */
	if (dialog->buttons.type == CDW_BUTTONS_OK) {
		size_t ok_col = (n_cols - (dialog->buttons.label_lens[BUTTON_OK] + 2)) / 2;

		dialog->buttons.buttons[BUTTON_OK] = cdw_button_new(dialog->window,
								    dialog->input.begin_y, ok_col,
								    dialog->buttons.labels[BUTTON_OK],
								    dialog->colors);

		dialog->buttons.current = BUTTON_OK;

	} else if (dialog->buttons.type == CDW_BUTTONS_OK_CANCEL) {
		size_t ok_col = ((n_cols/2) - (dialog->buttons.label_lens[BUTTON_OK] + 2)) / 2;
		size_t cancel_col = (n_cols/2) + (((n_cols/2) - (dialog->buttons.label_lens[BUTTON_CANCEL] + 2)) / 2);

		dialog->buttons.buttons[BUTTON_OK] = cdw_button_new(dialog->window,
								    dialog->input.begin_y, ok_col,
								    dialog->buttons.labels[BUTTON_OK],
								    dialog->colors);
		dialog->buttons.buttons[BUTTON_CANCEL] = cdw_button_new(dialog->window,
									dialog->input.begin_y, cancel_col,
									dialog->buttons.labels[BUTTON_CANCEL],
									dialog->colors);

		dialog->buttons.current = BUTTON_OK;

	} else if (dialog->buttons.type == CDW_BUTTONS_YES_NO_CANCEL) {
		size_t blen = 0;
		blen = dialog->buttons.label_lens[BUTTON_YES] + 2;
		size_t yes_col = ((n_cols/3) - blen) / 2;
		blen = dialog->buttons.label_lens[BUTTON_NO] + 2;
		size_t no_col = (n_cols/3) + (((n_cols/3) - blen) / 2);
		blen = dialog->buttons.label_lens[BUTTON_CANCEL] + 2;
		size_t cancel_col = ((2 * n_cols)/3) + (((n_cols/3) - blen) / 2);

		dialog->buttons.buttons[BUTTON_YES] = cdw_button_new(dialog->window,
								     dialog->input.begin_y, yes_col,
								     dialog->buttons.labels[BUTTON_YES],
								     dialog->colors);
		dialog->buttons.buttons[BUTTON_NO] = cdw_button_new(dialog->window,
								    dialog->input.begin_y, no_col,
								    dialog->buttons.labels[BUTTON_NO],
								    dialog->colors);
		dialog->buttons.buttons[BUTTON_CANCEL] = cdw_button_new(dialog->window,
									dialog->input.begin_y, cancel_col,
									dialog->buttons.labels[BUTTON_CANCEL],
									dialog->colors);

		dialog->buttons.current = BUTTON_YES;
	} else {
		;
	}

	cdw_button_focus(dialog->buttons.buttons[dialog->buttons.current]);

	return;
}





/**
   \brief Print message in given window, wrap message to not to cross window border

   Print given text in a window. Test can be longer than window width.
   The function takes care of wrapping the text so that line breaks occur
   on white chars. Line breaks ('\n' chars) that exist in original message
   will be preserved.

   Make sure that window has correct height and width, e.g. by calling
   cdw_textarea_calculate_height().

   Caller must refresh \p window after calling this function.

   \param window - area in which you want to print the message
   \param message - message that has to be printed in a window
   \param align - alignment of message in window (CDW_ALIGN_CENTER, CDW_ALIGN_LEFT, CDW_ALIGN_RIGHT)

   \return 0 if \p window has just enough or more rows than needed to print a message without truncations
   \return number of lines that weren't displayed because \p window doesn't have enough rows
 */
size_t cdw_textarea_print_message(WINDOW *window, const char *message, int align)
{
	cdw_assert (window != (WINDOW *) NULL, "ERROR: cannot print in NULL window\n");
	cdw_assert (message != (char *) NULL, "ERROR: cannot print NULL message\n");

	char *w_message = strdup(message);
	cdw_assert (w_message != (char *) NULL, "ERROR: failed to strdup() message \"%s\"\n", message);

	size_t n_cols = (size_t) getmaxx(window);
	size_t n_rows = (size_t) getmaxy(window);

#if 0   /* debug code */
	cdw_vdm ("INFO: width = %d, control string is this:\n", (int) n_cols)
	for (size_t i = 0; i < n_cols; i++) {
		fprintf(stderr, "o");
	}
	fprintf(stderr, "\n");
#endif

	size_t n_lines = cdw_textarea_wrap_text(w_message, n_cols);
	if (n_rows < n_lines) {
		cdw_vdm ("ERROR: cdw_textarea_wrap_text() produces wrapped text with more lines (%zd) than rows in window (%zd)\n",
			 n_lines, n_rows);
	}
	size_t len = strlen(w_message);

	/* now display every line of wrapped message in given window,
	   row after row, nicely aligned */
	size_t col = 0;
	size_t row = 0;
	char *line = w_message;
	for (size_t i = 0; i < len; i++) {
		if (*(w_message + i) == '\n') {
			*(w_message + i) = '\0';

			if (align == CDW_ALIGN_LEFT) {
				col = 0;
			} else if (align == CDW_ALIGN_CENTER) {
				col = (n_cols - strlen(line)) / 2;
			} else { /* align == CDW_ALIGN_RIGHT */
				col = n_cols - strlen(line);
			}
			mvwprintw(window, (int) row, (int) col, line);

			line = w_message + i + 1;
			row++;
			if (row == n_rows) {
				/* TODO: this doesn't seem to be right,
				   inspect whole function and check
				   what "row == n_rows" means */
				cdw_vdm ("ERROR: failed to print %zd lines of text\n", n_lines - n_rows);
				break;
			}
		}
	}
	/* this displays tail of message - the part that is after last '\n' */
	if (align == CDW_ALIGN_LEFT) {
		col = 0;
	} else if (align == CDW_ALIGN_CENTER) {
		col = (n_cols - strlen(line)) / 2;
	} else { /* align == CDW_ALIGN_RIGHT */
		col = n_cols - strlen(line);
	}
	mvwprintw(window, (int) row, (int) col, line);

	/* debug message */
	cdw_sdm ("INFO: wrapped message is\n%s\n", w_message);

	free(w_message);
	w_message = (char *) NULL;

	if (n_rows < n_lines) {
		cdw_vdm ("ERROR: function returns !0, there were more lines of text than rows in window\n");
		return n_lines - n_rows;
	} else {
		return 0;
	}
}




/**
   \brief Calculate textarea height based on message length and textarea width

   Calculate height of textarea that would fit given message. There are
   two constraints: message length and textarea height. The function
   takes into consideration that the message text has to be wrapped so that
   line breaks occur on white chars.

   \param message - message that has to be printed in a window
   \param width - width of window (area) in which the message will be printed

   \return height of window that would fit given message
*/
size_t cdw_textarea_calculate_height(const char *message, size_t width)
{
	cdw_assert (message != (char *) NULL, "ERROR: cannot operate on NULL message\n");

	char *buffer = strdup(message);
	cdw_assert (buffer != (char *) NULL, "ERROR: failed to strdup message \"%s\"\n", message);

#if 0   /* debug code */
	cdw_vdm ("INFO: width = %d, control string is this:\n", (int) width);
	for (size_t k = 0; k < width; k++) {
		fprintf(stderr, "o");
	}
	fprintf(stderr, "\n");
#endif

	size_t height = cdw_textarea_wrap_text(buffer, width);

	free(buffer);
	buffer = (char *) NULL;

	return height;
}





/**
   \brief Wrap text from given text buffer

   Put line breaks in text stored in \p buffer so that the text
   can be displayed "correctly" in window that has given \p width.
   "Correctly" means that line breaks occur between words, not in
   the middle of words.

   \param buffer - buffer with text to wrap
   \param width - width of window, in which caller intends to print given text

   \return number of lines that the text will occupy in window
*/
size_t cdw_textarea_wrap_text(char *buffer, size_t width)
{
	cdw_assert (width > 5, "ERROR: textarea is far too narrow\n");
	cdw_assert (buffer != (char *) NULL, "ERROR: text to wrap is NULL\n");

	size_t len = strlen(buffer);

	/* The idea is to introduce line breaks ('\n') in proper places
	   of message so there is no substring longer than width of window.
	   Splitting message into substrings is done using substring_start
	   and width variables, and iterating through the substrings searching
	   for white chars where you can put line break. */
	for (size_t substring_start = 0; substring_start < len; ) {
		bool linebreak = false;

		if (len - substring_start <= width) {
			/* end of tmp string, no point in looking for
			   places where line can be broken */
			break;
		}

		/* Search for line break in substring, iterating from
		   start of substring. If you find one, no new line break
		   will have to be inserted in current substring and
		   substring_start will have to be moved forward. */
		for (size_t i = 0; i < width; i++) {
			if (*(buffer + substring_start + i) == '\n') {
				substring_start = substring_start + i + 1;
				linebreak = true;
				break;
			}
		}

		if (linebreak) {
			/* ok, got one line break in substring,
			   prepare new substring */
			linebreak = false;
		} else {
			/* search for last white char in substring - a good
			   place to insert line break; next char after
			   inserted line break will be start of next substring */

			/* did we manage to find white char in substring? */
			bool white_char = false;
			/* assertion on top of the function assures that
			   width > 5 so first statement in for () is correct;
			   we use 'j > 0' because putting '\n' in 0-th place of
			   string gains us nothing */
			for (size_t j = width - 1; j > 0; j--) {
				/* no need to look for '\n' here, we already
				   searched for it in this substring when
				   we iterated from beginning */
				if (*(buffer + substring_start + j) == ' '
				 || *(buffer + substring_start + j) == '\t') {

					*(buffer + substring_start + j) = '\n';
					substring_start = substring_start + j + 1;
					white_char = true;
					break;
				}
			}
			if (white_char) {
				/* all is well, we managed to produce
				   substring no longer than 'width' */
				;
			} else {
				/* not good! given substring is longer
				   than 'width'! string will be wrapped in
				   window in the middle of word - not an end
				   of the world, but won't look that well */
				substring_start = substring_start + width;
			}
		}
	}

	size_t height = 0;
	size_t i = 0;
	for (i = 0; i < len; i++) {
		if (*(buffer + i) == '\n') {
			height++;
		}
	}
	if (*(buffer + i - 1) != '\n') {
		/* add one more line for whatever text there is
		   after last '\n' in buffer */
		height++;
	}

	return height;
}





/**
   \brief Draw two square brackets around 1x1 input field

   Draw two square brackets around 1x1 input field. The field will behave
   as checkbox, and drawing brackets around will make it look like this:
                option_name [x]
   Second and third arguments are coordinates of input field, so brackets
   should be put before and after it - make sure that there is enough space
   for them on your window.

   \param window - window to draw on
   \param y - row in which input field is located
   \param x - column in which input field is located
*/
void cdw_put_ckbox(WINDOW *window, int y, int x)
{
	cdw_assert (window != (WINDOW *) NULL, "ERROR: cannot put checkbox in NULL window\n");

	mvwprintw(window, y, x - 1, "[");
	mvwprintw(window, y, x + 1, "]");

	return;
}





/**
   \brief Create new CDW_BUTTON widget

   \param parent - parent window, in which the button will be displayed
   \param begin_y - number of row of parent window, in which will be displayed upper border of button
   \param begin_x - number of column of parent window, in which will be displayed left border of button
   \param label - string that will be displayed as button label
   \param colors - color scheme of button

   \return pointer to new button widget on success
   \return NULL pointer if creating widget fails
*/
CDW_BUTTON *cdw_button_new(WINDOW *parent, size_t begin_y, size_t begin_x, const char *label, cdw_colors_t colors)
{
	cdw_assert (parent != (WINDOW *) NULL, "ERROR: parent window is null\n");

	CDW_BUTTON *button = (CDW_BUTTON *) malloc(sizeof(CDW_BUTTON));
	if (button == (CDW_BUTTON *) NULL) {
		return (CDW_BUTTON *) NULL;
	}

	button->label = strdup(label);
	if (button->label == (char *) NULL) {
		free(button);
		button = (CDW_BUTTON *) NULL;

		return (CDW_BUTTON *) NULL;
	}
	button->label_len = strlen(button->label);

	button->begin_x = begin_x;
	button->begin_y = begin_y;

	button->colors = colors;

	button->parent = parent;

	button->strip = derwin(parent, 1,  2 + (int) button->label_len + 2,
			       (int) button->begin_y, (int) button->begin_x);

	(void) wattrset(button->strip, COLOR_PAIR(button->colors));

	mvwprintw(button->strip, 0, 0, "[");
	mvwprintw(button->strip, 0, 2, "%s", button->label);
	mvwprintw(button->strip, 0, (int) button->label_len + 3, "]");

	wrefresh(button->strip);

	return button;
}





/**
   \brief Destroy CDW_BUTTON widget

   Deallocate all resources used by CDW_BUTTON widget. The function
   accepts NULL pointer.

   \param button - button to destroy
*/
void cdw_button_delete(CDW_BUTTON **button)
{
	if (*button != (CDW_BUTTON *) NULL) {
		delwin((*button)->strip);
		(*button)->strip = (WINDOW *) NULL;

		free((*button)->label);
		(*button)->label = (char *) NULL;

		free(*button);
		*button = (CDW_BUTTON *) NULL;
	}

	return;
}





/**
   \brief Highlight given button

   Redraw given button using its reversed 'colors' scheme. It will look as
   if the button has changed its state to 'has focus'.

   \param button - button to highlight
*/
void cdw_button_focus(CDW_BUTTON *button)
{
	cdw_assert (button != (CDW_BUTTON *) NULL, "ERROR: cannot focus NULL button\n");

	/* wattrset(button->strip, COLOR_PAIR(button->focus_color)); */
	(void) wattrset(button->strip, COLOR_PAIR(button->colors) | A_REVERSE);

	mvwprintw(button->strip, 0, 0, "[");
	mvwprintw(button->strip, 0, 1, " %s ", button->label);
	mvwprintw(button->strip, 0, (int) (button->label_len + 3), "]");

	wrefresh(button->strip);

	return;
}





/**
   \brief Display given button as it is in its normal state

   Redraw given button using its normal 'colors' color scheme. It will look
   as if the button has changed its state to 'has no focus'.

   \param button - button to redraw
*/
void cdw_button_unfocus(CDW_BUTTON *button)
{
	cdw_assert (button != (CDW_BUTTON *) NULL, "ERROR: cannot unfocus NULL button\n");

	(void) wattrset(button->strip, COLOR_PAIR(button->colors));

	mvwprintw(button->strip, 0, 0, "[");
	mvwprintw(button->strip, 0, 1, " %s ", button->label);
	mvwprintw(button->strip, 0, (int) (button->label_len + 3), "]");

	wrefresh(button->strip);

	return;
}





CDW_DROPDOWN *cdw_dropdown_new(WINDOW *parent, int begin_y, int begin_x, int width, int n_items_max, cdw_colors_t colors)
{
	cdw_assert (parent != (WINDOW *) NULL, "parent window is NULL\n");
	CDW_DROPDOWN *dropdown = (CDW_DROPDOWN *) malloc(sizeof(CDW_DROPDOWN));
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	dropdown->n_items_max = n_items_max;
	dropdown->n_items = 0;

	dropdown->items = (cdw_dropdown_item_t **) malloc((size_t) (dropdown->n_items_max + 1) * sizeof (cdw_dropdown_item_t *));

	if (dropdown->items == (cdw_dropdown_item_t **) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for dropdown items\n");
		free(dropdown);
		dropdown = (CDW_DROPDOWN *) NULL;
		return (CDW_DROPDOWN *) NULL;
	} else {
		for (int i = 0; i < dropdown->n_items_max; i++) {
			dropdown->items[i] = (cdw_dropdown_item_t *) NULL;
		}
	}

	dropdown->parent = parent;
	dropdown->begin_y = begin_y;
	dropdown->begin_x = begin_x;
	dropdown->width = width;
	dropdown->colors = colors;

	dropdown->visible = true;

	return dropdown;
}





/**
   \brief Add new item to a dropdown

   1. allocate space for new item and attach it to a dropdown, and
   2. copy label and id into new item, and
   3. increase number of items in dropdown

   \param dropdown - dropdown to which you want to add an item
   \param id - "id" field of a new item
   \param label - "label" field of a new item

   \return CDW_OK on success
   \return CDW_GEN_ERROR on error
*/
cdw_rv_t cdw_dropdown_add_item(CDW_DROPDOWN *dropdown, int id, const char *label)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: passing null dropdown to function\n");
	cdw_assert (dropdown->items != (cdw_dropdown_item_t **) NULL, "ERROR: dropdown items is null\n");
	cdw_assert (dropdown->n_items < dropdown->n_items_max, "ERROR: trying to create item #%d while maximum is #%d\n",
		    dropdown->n_items, dropdown->n_items_max);

	dropdown->items[dropdown->n_items] = (cdw_dropdown_item_t *) malloc(sizeof (cdw_dropdown_item_t));
	if (dropdown->items[dropdown->n_items] == (cdw_dropdown_item_t *) NULL) {
		cdw_vdm ("ERROR: failed to malloc memory for item #%d\n", dropdown->n_items);
		return CDW_GEN_ERROR;
	}

	dropdown->items[dropdown->n_items]->label = strndup(label, (size_t) dropdown->width);
	if (dropdown->items[dropdown->n_items]->label == (char *) NULL) {
		cdw_vdm ("ERROR: failed to strndup label #%d: \"%s\"\n", dropdown->n_items, label);
		free(dropdown->items[dropdown->n_items]);
		dropdown->items[dropdown->n_items] = (cdw_dropdown_item_t *) NULL;
		return CDW_GEN_ERROR;
	}
	dropdown->items[dropdown->n_items]->id = id;
	dropdown->n_items++;

	return CDW_OK;
}





/**
   \brief Do final calls to few functions to finalize creating a dropdown

   Call few ncurses functions to set/initialize some fields of a \p dropdown.
   This prepares the \p dropdown to be usable.
   This function works only for new style dropdowns.

   \param dropdown - dropdown to be finalized

   \return CDW_OK on success
   \return CDW_GEN_ERROR on failure
*/
cdw_rv_t cdw_dropdown_finalize(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: passing null dropdown to function\n");

	/* standard code creating ITEM elements */
	dropdown->menu_items = (ITEM **) calloc((size_t) dropdown->n_items + 1, sizeof(ITEM *));
	if (dropdown->menu_items == (ITEM **) NULL) {
		return CDW_GEN_ERROR;
	}

	for (int i = 0; i < dropdown->n_items; i++) {
		dropdown->menu_items[i] = new_item(dropdown->items[i]->label, "");
		if (dropdown->menu_items[i] == (ITEM *) NULL) {
			int e = errno;
			cdw_vdm ("ERROR: failed to create menu item for item #%d with label \"%s\", errno = \"%s\"\n",
				 i, dropdown->items[i]->label, strerror(e));
			return CDW_GEN_ERROR;
		}
	}
	/* obligatory ending element */
	dropdown->menu_items[dropdown->n_items] = (ITEM *) NULL;

	/* standard code creating MENU element */
	dropdown->menu = new_menu(dropdown->menu_items);
	if (dropdown->menu == (MENU *) NULL) {
		int e = errno;
		cdw_vdm ("ERROR: failed to create menu, errno = \"%s\"\n", strerror(e));
		return CDW_GEN_ERROR;
	}

	dropdown->parent_x = getbegx(dropdown->parent);
	dropdown->parent_y = getbegy(dropdown->parent);
	/* menu should be displayed between opening and closing bracket,
	   so it should be narrower than 'width' and moved a bit right */
	dropdown->menu_window = newwin(dropdown->n_items, dropdown->width - 2,
				       dropdown->parent_y + dropdown->begin_y,
				       dropdown->parent_x + dropdown->begin_x + 1);
	if (dropdown->menu_window == (WINDOW *) NULL) {
		int e = errno;
		cdw_vdm ("ERROR: failed to create dropdown window, errno = \"%s\"\n", cdw_ncurses_error_string(e));
		return CDW_GEN_ERROR;
	}
	int rv = set_menu_win(dropdown->menu, dropdown->menu_window);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to set menu window, errno = \"%s\"\n", cdw_ncurses_error_string(rv));
		return CDW_GEN_ERROR;
	}
	dropdown->menu_window_sub = derwin(dropdown->menu_window, (int) dropdown->n_items, (int)  (dropdown->width - 2), 0, 0);
	if (dropdown->menu_window_sub == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create menu subwindow\n");
		return CDW_GEN_ERROR;
	}
	rv = set_menu_sub(dropdown->menu, dropdown->menu_window_sub);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to set menu subwindow, errno = \"%s\"\n", cdw_ncurses_error_string(rv));
		return CDW_GEN_ERROR;
	}

	/* remember that post_menu() displays menu in its subwindow, not on the
	   screen - you have to call refresh() to make menu visible to user */
	rv = post_menu(dropdown->menu);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to post menu, errno = \"%s\"\n", cdw_ncurses_error_string(rv));
		return CDW_GEN_ERROR;
	}

	dropdown->visible = true;

	return CDW_OK;
}





/**
   \brief Redraw window in which a dropdown is displayed

   'Show' a dropdown simply by (re)drawing and refreshing window, in which
   dropdown is displayed. Such dropdown now displays all items available.
   TODO: this is not the best way to "show" dropdown.

   \param dropdown - dropdown to show

   \return CDW_OK
*/
cdw_rv_t cdw_dropdown_expand(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot show NULL dropdown\n");
	cdw_assert (dropdown->menu_window != (WINDOW *) NULL, "ERROR: cannot show dropdown with NULL window\n");
	cdw_assert (dropdown->menu_window_sub != (WINDOW *) NULL, "ERROR: cannot show dropdown with NULL subwindow\n");

	/* move highlight to current item */
	menu_driver(dropdown->menu, REQ_FIRST_ITEM);
	for (int j = 0; j < dropdown->current_item_ind && j < dropdown->n_items; j++) {
		menu_driver(dropdown->menu, REQ_DOWN_ITEM);
	}

	redrawwin(dropdown->menu_window);
	redrawwin(dropdown->menu_window_sub);
	wrefresh(dropdown->menu_window);
	wrefresh(dropdown->menu_window_sub);
	return CDW_OK;
}





/**
   \brief Collapse given dropdown

   Collapse expanded dropdown simply by (re)drawing and refreshing its parent
   window. "Minimized" dropdown with hidden list is still visible, but it
   only displays selected item.

   TODO: This may not be the best way to collapse a dropdown.

   This function does not display currently selected item in place where
   dropdown is displayed.

   \param dropdown - dropdown to collapse

   \return CDW_OK if no problems occurred
*/
cdw_rv_t cdw_dropdown_collapse(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot hide NULL dropdown\n");
	cdw_assert (dropdown->parent != (WINDOW *) NULL, "ERROR: cannot hide dropdown with NULL parent\n");

	redrawwin(dropdown->parent);
	wrefresh(dropdown->parent);

	return CDW_OK;
}





void cdw_dropdown_make_invisible(CDW_DROPDOWN *dropdown)
{
	mvwhline(dropdown->parent, (int) dropdown->begin_y, (int) dropdown->begin_x, ' ', (int) dropdown->width + 1);
	wrefresh(dropdown->parent);

	dropdown->visible = false;

	return;
}



void cdw_dropdown_make_visible(CDW_DROPDOWN *dropdown)
{
	dropdown->visible = true;
	cdw_dropdown_display_current_item_reverse(dropdown, false);

	return;
}




/**
   \brief Control keyboard input when focus is on dropdown - top level function

   This function calls cdw_dropdown_expanded_driver() to control keyboard input
   in expanded dropdown.

   \param dropdown - dropdown that has focus

   \return CDW_OK if user pressed ENTER in expanded dropdown
   \return CDW_CANCEL if user pressed ESCAPE in expanded dropdown
*/
cdw_rv_t cdw_dropdown_driver(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot control NULL dropdown\n");
	cdw_assert (dropdown->parent != (WINDOW *) NULL, "ERROR: cannot control dropdown with NULL parent\n");

	cdw_dropdown_expand(dropdown);
	cdw_rv_t crv = cdw_dropdown_expanded_driver(dropdown);
	cdw_dropdown_collapse(dropdown);

	if (crv == CDW_OK) {
		/* when dropdown is closed (hidden), it should remain
		   highlighted, so second parameter is true */
		cdw_dropdown_display_current_item_reverse(dropdown, true);
	}

	redrawwin(dropdown->parent);
	wrefresh(dropdown->parent);

	return crv;
}





/**
   \brief React to keys pressed by user when given dropdown is expanded (active)

   React to following keys: HOME, END, DOWN, UP, ENTER, ESCAPE, when
   given dropdown is active.

   Caller of this function must make sure that \p dropdown is valid.

   \param dropdown - active dropdown, on which user operates

   \return CDW_OK if user pressed ENTER,
   \return CDW_CANCEL if user pressed ESCAPE
*/
cdw_rv_t cdw_dropdown_expanded_driver(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot control NULL dropdown\n");

	/* a bit customized, but otherwise standard ncurses menu driver code */
	int key = 'a';
	while ((key = wgetch(dropdown->parent)) != CDW_KEY_ESCAPE) {
		switch (key) {
			case KEY_HOME:
				menu_driver(dropdown->menu, REQ_FIRST_ITEM);
				break;
			case KEY_END:
				menu_driver(dropdown->menu, REQ_LAST_ITEM);
				break;
			case KEY_DOWN:
				menu_driver(dropdown->menu, REQ_DOWN_ITEM);
				break;
			case KEY_UP:
				menu_driver(dropdown->menu, REQ_UP_ITEM);
				break;
			case CDW_KEY_ENTER:
			{
				ITEM *item = current_item(dropdown->menu);
				cdw_assert (item != (ITEM *) NULL, "ERROR: current item is NULL\n");
				/* remember index of selected item */
				dropdown->current_item_ind = item_index(item);

				return CDW_OK;
			}

			default: /* other (meaningless in this context) keys */
				break;
		}

		redrawwin(dropdown->menu_window_sub);
		wrefresh(dropdown->menu_window_sub);
	}

	return CDW_CANCEL;
}





/**
   \brief Destroy given dropdown, deallocate its resources

   Note that item_labels passed to constructor of dropdown is _not_ freed by
   destructor. The variable is owned by caller of cdw_dropdown_new().

   This destructor does not NULL pointer (non-initialized widget).

   \param dropdown - pointer to dropdown that you want to destroy

   \return CDW_OK
*/
cdw_rv_t cdw_dropdown_delete(CDW_DROPDOWN **dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN **) NULL,
		    "ERROR: you passed NULL pointer to function\n");

	if (*dropdown == (CDW_DROPDOWN *) NULL) {
		/* probably called the function when other function that
		   creates dropdown failed to allocate initial memory for
		   dropdown - hence null pointer */
		cdw_vdm ("WARNING: you passed NULL dropdown to function\n");
		return CDW_OK;
	}

	unpost_menu((*dropdown)->menu);

	if ((*dropdown)->menu != (MENU *) NULL) {
		unpost_menu((*dropdown)->menu);
		free_menu((*dropdown)->menu);
		(*dropdown)->menu = (MENU *) NULL;

		for (int j = 0; j < (*dropdown)->n_items; j++){
			free_item((*dropdown)->menu_items[j]);
			(*dropdown)->menu_items[j] = (ITEM *) NULL;
		}

		free((*dropdown)->menu_items);
		(*dropdown)->menu_items = (ITEM **) NULL;
	}

	for (int i = 0; i < (*dropdown)->n_items; i++) {
		if ((*dropdown)->items[i] != (cdw_dropdown_item_t *) NULL) {
			if ((*dropdown)->items[i]->label != (char *) NULL) {
				free((*dropdown)->items[i]->label);
				(*dropdown)->items[i]->label = (char *) NULL;
			} else {
				cdw_vdm ("WARNING: menu item label #%d is null\n", i);
			}

			free((*dropdown)->items[i]);
			(*dropdown)->items[i] = (cdw_dropdown_item_t *) NULL;
		} else {
			cdw_vdm ("WARNING: menu item #%d is null\n", i);
		}
	}
	free((*dropdown)->items);
	(*dropdown)->items = (cdw_dropdown_item_t **) NULL;

	delwin((*dropdown)->menu_window_sub);
	(*dropdown)->menu_window_sub = (WINDOW *) NULL;

	delwin((*dropdown)->menu_window);
	(*dropdown)->menu_window = (WINDOW *) NULL;

	free(*dropdown);
	*dropdown = (CDW_DROPDOWN *) NULL;

	return CDW_OK;
}





/**
   Return index of currently selected (active) dropdown item

   \param dropdown - dropdown from which caller wants to get index of current item

   \return index of currently selected item
*/
int cdw_dropdown_get_current_item_ind(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot get index of item in NULL dropdown\n");

	return dropdown->current_item_ind;
}





/**
   \brief Display in parent window item currently selected in dropdown

   \param dropdown - dropdown for which you want to display current item
*/
void cdw_dropdown_display_current_item(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot display current item in NULL dropdown\n");
	if (dropdown->visible) {
		cdw_dropdown_display_current_item_reverse(dropdown, false);
	}

	return;
}





/**
   \brief Display in parent window item currently selected in dropdown

   \param dropdown - dropdown for which you want to display current item
   \param reverse - mode of display
*/
void cdw_dropdown_display_current_item_reverse(CDW_DROPDOWN *dropdown, bool reverse)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot display item in NULL dropdown\n");
	if (!dropdown->visible) {
		return;
	}

	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot show current item of NULL dropdown\n");
	cdw_assert (dropdown->parent != (WINDOW *) NULL,  "ERROR: cannot show current item of dropdown with NULL parent\n");

	/* first erase old content */
	if (reverse) {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors) | A_REVERSE);
	} else {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors));
	}
	mvwhline(dropdown->parent, (int) dropdown->begin_y, (int) dropdown->begin_x, ' ', (int) dropdown->width);


	/* then opening and closing bracket */
	mvwprintw(dropdown->parent, (int) dropdown->begin_y, (int) dropdown->begin_x, "[");
	mvwprintw(dropdown->parent, (int) dropdown->begin_y, (int) (dropdown->begin_x + dropdown->width - 1), "]");

	/* 'real' content */
	mvwprintw(dropdown->parent, (int) dropdown->begin_y, (int) dropdown->begin_x + 1, dropdown->items[dropdown->current_item_ind]->label);

	/* and now fancy, bolded arrow by the dropdown */
	if (reverse) {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors) | A_REVERSE | A_BOLD);
	} else {
		(void) wattrset(dropdown->parent, COLOR_PAIR(dropdown->colors) | A_BOLD);
	}
	mvwprintw(dropdown->parent, (int) dropdown->begin_y, (int) (dropdown->begin_x + dropdown->width - 2), "v");

	(void) wattrset(dropdown->parent, A_NORMAL | COLOR_PAIR(dropdown->colors));

	wrefresh(dropdown->parent);

	return;
}





/**
   \brief Highlight given dropdown to mark that it has focus

   Use this function to change appearance of a non-expanded dropdown
   to make it look like it has now focus and may be expanded with
   keyboard stroke.

   \param dropdown - dropdown that you want to focus on
*/
void cdw_dropdown_focus(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot focus NULL dropdown\n");
	if (dropdown->visible) {
		cdw_dropdown_display_current_item_reverse(dropdown, true);
	} else {
		cdw_vdm ("ERROR: called the function for invisible dropdown\n");
	}

	return;
}





/**
   \brief Un-highlight given dropdown to mark that it has lost focus

   Use this function to undo changes of appearance of a non-expanded
   dropdown made by cdw_dropdown_focus() -  to make it look like it
   has lost focus and will not be expanded with keyboard stroke.

   \param dropdown - dropdown that you want to unfocus
*/
void cdw_dropdown_unfocus(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot unfocus NULL dropdown\n");
	if (dropdown->visible) {
		cdw_dropdown_display_current_item_reverse(dropdown, false);
	} else {
		cdw_vdm ("WARNING: called the function for invisible dropdown\n");
	}

	return;
}





/**
   \brief Set one of items of dropdown as selected value

   \param dropdown - dropdown from which you want to select item
   \param ind - index of item that you want to select
*/
cdw_rv_t cdw_dropdown_set_current_item_by_ind(CDW_DROPDOWN *dropdown, int ind)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot set current item for NULL dropdown\n");

	if (ind < dropdown->n_items) {
		dropdown->current_item_ind = ind;
		cdw_dropdown_display_current_item(dropdown);

		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: index (%d) is larger than number of items (%d)\n", ind, dropdown->n_items);
		return CDW_GEN_ERROR;
	}
}





cdw_rv_t cdw_dropdown_set_current_item_by_id(CDW_DROPDOWN *dropdown, int id)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: passing null dropdown to function\n");

	for (int i = 0; i < dropdown->n_items; i++) {
		if (dropdown->items[i]->id == id) {
			dropdown->current_item_ind = i;
			return CDW_OK;
		}
	}
	cdw_vdm ("ERROR: failed to set current item index by id, id = %d\n", id);
	return CDW_GEN_ERROR;
}





int cdw_dropdown_get_current_item_id(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot set current item id from NULL dropdown\n");

	return dropdown->items[dropdown->current_item_ind]->id;

}





const char *cdw_dropdown_get_current_item_label(CDW_DROPDOWN *dropdown)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot set current item id from NULL dropdown\n");

	const char *tmp = dropdown->items[dropdown->current_item_ind]->label;
	cdw_assert (tmp != (char *) NULL, "ERROR: current label is NULL\n");
	return tmp;
}





const char *cdw_dropdown_get_label_by_ind(CDW_DROPDOWN *dropdown, int ind)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot set current item id from NULL dropdown\n");
	cdw_assert (ind < dropdown->n_items, "ERROR: index (%d) larger than number of items (%d)\n",
		    ind, dropdown->n_items);

	char *tmp = dropdown->items[ind]->label;
	cdw_assert (tmp != (char *) NULL, "ERROR: label #%d is NULL\n", ind);

	return tmp;
}





const char *cdw_dropdown_get_label_by_id(CDW_DROPDOWN *dropdown, int id)
{
	cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: cannot set current item id from NULL dropdown\n");

	for (int i = 0; i < dropdown->n_items; i++) {
		if (dropdown->items[i]->id == id) {
			char *tmp = dropdown->items[i]->label;
			cdw_assert (tmp != (char *) NULL, "ERROR: item with id %d has null label\n", id);
			return tmp;
		}
	}

	cdw_vdm ("ERROR: can't find item for id %d\n", id);

	return (char *) NULL;
}





CDW_INPUT_LINE *cdw_input_line_new(WINDOW *parent, int begin_y, int begin_x, int n_cols, const char *initial_string, int input_type, size_t chars_max)
{
	CDW_INPUT_LINE *input_line = (CDW_INPUT_LINE *) malloc(sizeof(CDW_INPUT_LINE));
	if (input_line == (CDW_INPUT_LINE *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for input line widget\n");
		return (CDW_INPUT_LINE *) NULL;
	}

	int n_lines = 2; /* one line for input window, second line for message area */
	input_line->parent = parent;
	input_line->input_type = input_type;
	input_line->chars_max = chars_max;
	input_line->n_cols = n_cols;

	input_line->save_data_on_tab = false;

	input_line->win = derwin(input_line->parent, n_lines, n_cols, begin_y, begin_x);
	if (input_line->win == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create input line subwindow\n");
		cdw_input_line_delete(&input_line);
		return (CDW_INPUT_LINE *) NULL;
	}

	bool success = true;

	input_line->field[1] = (FIELD *) NULL; /* obligatory ending element */
	input_line->field[0] = cdw_ncurses_new_input_field(1, (size_t) input_line->n_cols, /* 1 - one line */
							   0, 0,                           /* begin_y, begin_x */
							   initial_string, (size_t) input_line->chars_max,
							   input_line->input_type, CDW_COLORS_INPUT);

	if (input_line->field[0] == (FIELD *) NULL) {
		cdw_vdm ("ERROR: failed to create form field\n");
		success = false;
	} else {
		input_line->form = cdw_form_new_form(input_line->parent, input_line->win, input_line->field);
		if (input_line->form == (FORM *) NULL) {
			cdw_vdm ("ERROR: failed to create form with new_form()\n");
			success = false;
		} else {
			int r = set_current_field(input_line->form, input_line->field[0]);
			if (r != E_OK) {
				cdw_vdm ("ERROR: failed to set_current_field with set_current_field(), error code is %s\n", cdw_ncurses_error_string(r));
				success = false;
			} else {
				r = form_driver(input_line->form, REQ_END_LINE);
				if (r != E_OK) {
					cdw_vdm ("ERROR: failed in form_driver(), the function returns %s\n", cdw_ncurses_error_string(r));
					success = false;
				} else {
					; /* input line ready, success is still true */
				}
			}
		}
	}

	if (! success) {
		cdw_input_line_delete(&input_line);
		return (CDW_INPUT_LINE *) NULL;
	} else {
		wrefresh(input_line->win);
		return input_line;
	}
}





void cdw_input_line_delete(CDW_INPUT_LINE **input_line)
{
	if (input_line == (CDW_INPUT_LINE **) NULL) {
		cdw_vdm ("ERROR: passing NULL pointer to the function\n");
		return;
	}
	if (*input_line == (CDW_INPUT_LINE *) NULL) {
		return;
	}

	/* IMPORTANT: for some reason you have first free fields
	   and then unpost and free form */
	if ((*input_line)->field[0] != (FIELD *) NULL) {
		free_field((*input_line)->field[0]);
		unpost_form((*input_line)->form);
		(*input_line)->field[0] = (FIELD *) NULL;
	}

	if ((*input_line)->form != (FORM *) NULL) {
		int e = free_form((*input_line)->form);
		(*input_line)->form = (FORM *) NULL;
		if (e != E_OK) {
			cdw_vdm ("ERROR: free_form() returns \"%s\"\n", cdw_ncurses_error_string(e));
		}
	}

	if ((*input_line)->win != (WINDOW *) NULL) {
		delwin(((*input_line)->win));
		(*input_line)->win = (WINDOW *) NULL;
	}

	free(*input_line);
	*input_line = (CDW_INPUT_LINE *) NULL;

	return;
}





/**
   Pressing both Enter and Tab keys is treated as:
   \li confirmation of entered data, and
   \li attempt to leave input line, with
   \li intent to save entered data in \p buffer.

   \return CDW_KEY_ENTER or CDW_KEY_TAB if user pressed Enter or Tab key
   \return CDW_KEY_ESCAPE if user pressed Escape key
   \return KEY_EXIT on errors
*/
int cdw_input_line_safe_driver(CDW_INPUT_LINE *input_line, char **buffer, int attempts_max)
{
	char *local_buffer = (char *) NULL;
	int key = 'a';
	for (int i = 0; i < attempts_max; i++) {
		curs_set(1);

		int validation_result = E_OK;
		key = cdw_one_line_form_driver(input_line->form, &validation_result, true);

		if (input_line->input_type == CDW_NCURSES_INPUT_NONE) {
			validation_result = E_OK;
		}

		curs_set(0);

		/* TODO: save_data_on_tab is hackish, fix it somehow;
		   perhaps add return_chars[] to the widget */
		if (key == CDW_KEY_TAB && !input_line->save_data_on_tab) {
			;
		} else if (key == CDW_KEY_ENTER ||
		    (key == CDW_KEY_TAB && input_line->save_data_on_tab)) {

			if (validation_result == E_OK) {

				/* all that can go wrong at this point is malloc()
				   failure - return value will be CDW_GEN_ERROR */
				cdw_rv_t crv = cdw_ncurses_get_field_buffer(input_line->field[0], &local_buffer, (size_t) input_line->chars_max);
				if (crv != CDW_OK) {
					cdw_vdm ("ERROR: failed to get field buffer with cdw_ncurses_get_field_buffer()\n");
					key = KEY_EXIT;
					break;
				}
				/* TODO:
				  placing security check here makes input line
				  less versatile, as I can imagine places
				  where security checking may be unwanted, or
				  may need different implementation; I have to
				  update implementation of input line so that
				  it is more configurable and universal; */
				char insecure[2];
				cdw_rv_t sec = cdw_string_security_parser(local_buffer, insecure);
				if (sec == CDW_OK) {
					cdw_vdm ("INFO: \"%s\" is secure\n", local_buffer);
					crv = cdw_string_set(buffer, local_buffer);
					cdw_assert (crv == CDW_OK, "ERROR: failed to save final string \"%s\"\n", local_buffer);
					/* key = key */
					break;
				} else {
					cdw_vdm ("ERROR: insecure char in \"%s\"\n", local_buffer);
					cdw_input_line_safe_driver_message(input_line, insecure);
					if (i == attempts_max - 1) {
						/* too many attempts to enter correct string;
						   break loop to avoid printing "try again"
						   error message and to go straight to final
						   error message */
						key  = CDW_KEY_ESCAPE;
						break;
					} else {
						; /* key = key */
					}
				}
			} else if (validation_result == E_INVALID_FIELD) {
				/* invalid form content */
				cdw_vdm ("ERROR: failed to get field buffer (arg error)\n");
				key = KEY_EXIT;
			} else {
				/* some other error, but we won't inspect it */
				cdw_vdm ("ERROR: failed to get field buffer (gen error)\n");
				key = KEY_EXIT;
			}

		} else if (key == CDW_KEY_ESCAPE) {
			break;
		} else {
			key = KEY_EXIT;
			cdw_assert (0, "ERROR: incorrect key from one line form driver: %d\n", key);
			break;
		}
	}
	if (local_buffer != (char *) NULL) {
		free(local_buffer);
		local_buffer = (char *) NULL;
	}
	if (key == CDW_KEY_TAB && !input_line->save_data_on_tab) {
		key = CDW_KEY_ESCAPE;
	}
	return key;
}





void cdw_input_line_safe_driver_message(CDW_INPUT_LINE *input_line, char *insecure)
{
	/* error message can't be longer than input line's width */
	int len_max = input_line->n_cols;
	char *message = (char *) malloc((size_t) len_max + 1);
	if (message == (char *) NULL) {
		cdw_vdm ("ERROR: failed to malloc() message buffer\n");
		return;
	}
#define N_SAFETY_MESSAGES 3
	char *messages[N_SAFETY_MESSAGES] = {
		/* 2TRANS: this is error message in dialog window, "%s" is
		   string with single insecure character */
		_("ERROR: insecure char %s "),
		/* 2TRANS: this is error message in dialog window, "%s" is
		   string with single insecure character */
		_("Insecure char %s "),
		/* 2TRANS: this is error message in dialog window, "%s" is
		   string with single insecure character */
		_("Insec. char %s")};

	for (int i = 0; i < N_SAFETY_MESSAGES; i++) {
		/* +1 for ending '\0'; no format string, so will result in COMPILER WARNING */
		int n = snprintf(message, (size_t) len_max + 1, messages[i], insecure);
		if (n < 0) {
			cdw_vdm ("ERROR: snprintf() returns negative value for string #%d = \"%s\"\n", i, messages[i]);
		} else if (n >= 0 && n <= len_max) {
			/* snprintf() printed at most len_max chars
			   (excluding ending '\0'); message #i was
			   short enough, accept it */
			break;
		} else {
			cdw_vdm ("ERROR: snprintf() truncates message; limit of printable chars = %d, would print %d chars\n",
				 len_max, n);
			cdw_vdm ("ERROR: attempted to print string \"%s\"\n",
				 messages[i]);
			cdw_vdm ("ERROR: printed string            \"%s\"\n",
				 message);
		}
	}

	/* at this point we have in "message" one of three messages[]
	   string without truncations, or - as last resort - last of
	   messages[] string with truncation */

	(void) wattrset(input_line->win, COLOR_PAIR(CDW_COLORS_WARNING));
	mvwprintw(input_line->win, 1, 0, message);
	free(message);
	message = (char *) NULL;
	wrefresh(input_line->win);
	/* error message about insecure character is displayed under
	   input field only for X seconds; during this time input from
	   user is blocked */
	sleep(2);

	/* TODO: it would seem that COLOR_PAIR(input_line->colors) would work,
	   but it doesn't */
        (void) wattrset(input_line->win, COLOR_PAIR(CDW_COLORS_DIALOG));
	mvwhline(input_line->win, 1, 0, ' ', len_max);
	wrefresh(input_line->win);
	return;
}





void cdw_input_line_refresh_with_string(CDW_INPUT_LINE *input_line, const char *string)
{
	cdw_assert (string != (char *) NULL, "ERROR: string passed to function is NULL\n");
	int rv = set_field_buffer(input_line->field[0], 0, string);
#ifndef NDEBUG
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed to set buffer of input line, error = %s\n", cdw_ncurses_error_string(rv));
		assert (0);
	} else {
		char *buffer_string = cdw_string_rtrim(field_buffer(input_line->field[0], 0));
		if (strcmp(buffer_string, string)) {
			cdw_vdm ("ERROR: buffer string and input string are different:\nbuffer string: \"%s\"\ninput string: \"%s\"\n",
				 buffer_string, string);
		}
	}
#endif
	rv = form_driver(input_line->form, REQ_END_LINE);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed call to form_driver(), error = %s\n", cdw_ncurses_error_string(rv));
		assert (0);
	}
	wrefresh(input_line->win);

	return;
}



