/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE /* asprintf() */
#include <string.h>
#include <stdlib.h>
#include <math.h> /* floor(), log10() */
#include <stdbool.h>

#include "cdw_config.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_debug.h"
#include "cdw_ncurses.h"
#include "cdw_string.h"
#include "cdw_tabs_window.h"
#include "cdw_ext_tools.h"
#include "cdw_file_picker.h"
#include "cdw_drive.h"




/**
   There are currently four pages, grouping related options:
   \li options related to audio CDs: page titled "Audio"
   \li options related to hardware: page titled "Hardware"
   \li options related to external tools: page titled "Tools"
   \li options related to logging facility and other options: page titled "Log and misc"

   One important note:
   tabbed window tabs in c_twindow must be referred to by index:
   c_twindow->tabs[i];
   config pages in c_pages must be referred to by id of a page:
   c_pages[id];

   id is taken from c_twindow, like this:
   int i = c_twindow->current_tab;
   cdw_id_t id = c_twindow->tabs[i].id;
   FORM *form = c_pages[id].cdw_form->form;

   This is because tabs in tabbed window are always a non-sparse table,
   and pages table (c_pages) may be sparse (some of pages in the table may
   be unused, i.e. may be uninitialized).
*/



extern cdw_config_t global_config;

/* is set to true if something went wrong during module setup -
   cdw works without reading/writing configuration to disk file */
extern bool failsafe_mode;


static cdw_rv_t cdw_config_window_create(cdw_config_t *config);
static cdw_rv_t cdw_config_window_driver(cdw_config_t *config);
static void     cdw_config_window_destroy(void);





/**
 * \brief Top level function for configuration window
 *
 * \return CDW_GEN_ERROR
 * \return CDW_OK
 */
cdw_rv_t cdw_config_window(void)
{
	cdw_config_t tmp_config;
	cdw_config_var_init_fields(&tmp_config);
	/* DON'T read configuration from disk file to config variable:
	   this is done already in cdw_config_module_init(); and between
	   calling cdw_config_module_init() and opening configuration
	   window user could modify configuration variable in other
	   places; reading config from file again could overwrite user's
	   changes  */
	/**** read_conf(); ****/

	/* copy current option values from main config variable to
	   temporary config variable and use the temporary variable to
	   initialize option fields. */
	cdw_config_var_copy(&tmp_config, &global_config);
	cdw_config_debug_print_config(&tmp_config);

	cdw_rv_t crv = cdw_config_window_create(&tmp_config);
	if (crv != CDW_OK) {
		/* no need to call cdw_config_ui_window_destroy(),
		   create() function should do necessary clean up */
		cdw_vdm ("ERROR: failed to create config ui window\n");
		cdw_config_var_free_fields(&tmp_config);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Failed to create configuration window"),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}
	cdw_main_ui_main_window_wrefresh();
	crv = cdw_config_window_driver(&tmp_config);
	if (crv == CDW_OK) { /* "save changes and exit from cdw config window" */
		cdw_config_var_copy(&global_config, &tmp_config);
		cdw_config_debug_print_config(&global_config);

		if (failsafe_mode) {
			/* cannot work with file system because of previous
			   errors detected in cdw_conifg.c */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot write to config file. Configuration will not be saved to file."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

			/* cannot save changed values to file, but they were
			   saved to config variable and will be used only in
			   this run of cdw */
		} else {
			/* write current configuration (from config) to disk file */
			cdw_config_write_to_file();
		}
	} else { /* crv == CDW_CANCEL */
		; /* current values in config are without changes, they aren't
		     replaced by those from tmp_config */
	}

	cdw_config_var_free_fields(&tmp_config);
	cdw_config_window_destroy();
	/* user may have changed "ISO volume size", make this change
	   visible in main window */
	cdw_main_window_volume_info_view_update(false, global_config.follow_symlinks);

	cdw_main_ui_main_window_wrefresh();

	return CDW_OK;
}






/* 74min CD, 80min CD, DVD, DVD+R DL, custom, auto */
#define CDW_CONFIG_VOLUME_ITEMS_MAX 6


/* some size and layout constants */
/* width and height of main options window can't be smaller
   than size of minimal supported terminal */

/** \brief Width of configuration window */
#define CONF_WIDTH 80
/** \brief Height of configuration window */
#define CONF_HEIGHT 24
/** \brief Width of right-hand area with tabs */
#define TABS_WIDTH 23
/** \brief Width of (most) labels in configuration window */
#define LABEL_WIDTH 30
/** \brief Width of some fields that should be as wide as possible: input
    fields storing file path, some dropdowns, "other options" fields,
    txt subwins, long labels */
#define WIDE_FIELD_WIDTH (CONF_WIDTH - TABS_WIDTH - 4)
/** \brief Window column of first column of items in a configuration page  */
#define FIRST_COL 1
/** \brief Window column of second column of items in a configuration page  */
#define SECOND_COL (FIRST_COL + LABEL_WIDTH + 2)


/* Number of form fields on given page of options; ending field == NULL not included */
#define PAGE_HW_N_FIELDS         7  /* hardware */
#define PAGE_AUDIO_N_FIELDS      3
#define PAGE_TOOLS_N_FIELDS     15
#define PAGE_OTHER_N_FIELDS      9  /* log and other */


/* number of pages in configuration window */
#define N_CONFIG_PAGES_ALL       4



static cdw_rv_t page_audio_build_fields(cdw_form_t *cdw_form, void *cdw_config, int dummy1, int dummy2, int dummy3, int dummy4);
static cdw_rv_t page_hw_build_fields(cdw_form_t *cdw_form, void *cdw_config, int dummy1, int dummy2, int dummy3, int dummy4);
static cdw_rv_t page_tools_build_fields(cdw_form_t *cdw_form, void *cdw_config, int dummy1, int dummy2, int dummy3, int dummy4);
static cdw_rv_t page_other_build_fields(cdw_form_t *cdw_form, void *cdw_config, int dummy1, int dummy2, int dummy3, int dummy4);

static int page_tools_toggle_manual_selection(cdw_form_t *cdw_form, void *state);
static int page_audio_handle_enter_on_audiodir_button(cdw_form_t *cdw_form, void *dummy);
static int page_other_handle_enter_on_log_path_button(cdw_form_t *cdw_form, void *dummy);

static void cdw_config_window_save_config_fields(cdw_config_t *config);
static void cdw_config_window_save_non_config_fields(void);
static void cdw_config_window_refresh_page(cdw_id_t id);

static char *cdw_config_window_page_tools_message(int n_cols);
static char *cdw_config_window_page_hw_message(int n_cols);
static cdw_rv_t cdw_config_window_draw(void);

static CDW_DROPDOWN *cdw_config_window_make_volume_size_dropdown(WINDOW *window, int begin_y, int begin_x, int width);


static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_config_window_make_volume_size_dropdown,
	cdw_drive_make_drives_dropdown,
	cdw_ext_tools_make_iso9660_dropdown,
	cdw_ext_tools_make_cd_handlers_dropdown,
	cdw_ext_tools_make_dvd_handlers_dropdown,
	cdw_ext_tools_make_mkisofs_dropdown,
	cdw_ext_tools_make_cdrecord_dropdown,
       	cdw_ext_tools_make_digest_handlers_dropdown
};

static cdw_form_text_maker_t text_makers[] = {
	cdw_config_window_page_hw_message,
	cdw_config_window_page_tools_message
};


/* used by functions that don't accept cdw_config_t argument */
static cdw_config_t *local_config = (cdw_config_t *) NULL;


static FIELD *page_hw_fields[PAGE_HW_N_FIELDS + 1]; /* +1 for last field == NULL */
static FIELD *page_audio_fields[PAGE_AUDIO_N_FIELDS + 1]; /* +1 for last field == NULL */
static FIELD *page_tools_fields[PAGE_TOOLS_N_FIELDS + 1]; /* +1 for last field == NULL */
static FIELD *page_other_fields[PAGE_OTHER_N_FIELDS + 1]; /* +1 for last field == NULL */


/* text displayed at the bottom of configuration window */
static char *cdw_config_window_tip = (char *) NULL;



/* this is first main component of configuration window: a big table
   aggregating data and functions that prepare the data; the data
   represents widgets displayed in pages of configuration window */
static cdw_config_page_t c_pages[] = {
	/* 2TRANS: this is name of tab of notepad-style widget:
	   "Audio" means 'audio cd ripping configuration' */
	{ false, PAGE_AUDIO_N_FIELDS,   page_audio_build_fields,   gettext_noop("Audio"),          (cdw_form_t *) NULL },
	/* 2TRANS: this is name of tab of notepad-style widget:
	   "Hardware" refers to CD/DVD drives */
	{ false, PAGE_HW_N_FIELDS,      page_hw_build_fields,      gettext_noop("Hardware"),       (cdw_form_t *) NULL },
	/* 2TRANS: this is name of tab of notepad-style widget: "tools" are
	   external programs (like cdrecord or dvd+rw-mediainfo) used by cdw */
	{ false, PAGE_TOOLS_N_FIELDS,   page_tools_build_fields,   gettext_noop("Tools"),          (cdw_form_t *) NULL },
	/* 2TRANS: this is name of tab of notepad-style widget: "log" is a
	   file where the program writes various stuff that can be later
	   examined by user; "misc" means "miscellaneous settings" */
	{ false, PAGE_OTHER_N_FIELDS,   page_other_build_fields,   gettext_noop("Log and misc"),  (cdw_form_t *) NULL }};


/* this is a pointer to second main component of configuration window:
   tabbed window, which enables switching between pages */
static cdw_tabs_window_t *c_twindow = (cdw_tabs_window_t *) NULL;





/**
   \brief Create configuration window

   The function performs full initialization and creation of configuration
   window, so that next call can be a call to configuration window driver.

   \param config - configuration variable, used to initialize fields in configuration window

   \return CDW_ERROR if some resource was not initialized properly
   \return CDW_OK on success
*/
cdw_rv_t cdw_config_window_create(cdw_config_t *config)
{
	local_config = config;

	/* 2TRANS: this is message at the bottom of configuration window;
	   cancel means: quit without saving; '%d' is an integer used to
	   create label of function key, e.g. "F10" */
	int rv = asprintf(&cdw_config_window_tip, _("Press F%d key to save changes or ESC to cancel"), CDW_CONFIG_UI_SnC_KEY);
	if (cdw_config_window_tip == (char *) NULL || rv == -1) {
		cdw_vdm ("ERROR: failed to create window tip\n");
		return CDW_ERROR;
	}
	/* 2TRANS: this is title of cdw configuration main window */
	c_twindow = cdw_tabs_window_init(N_CONFIG_PAGES_ALL, _("Configuration"), cdw_config_window_tip);
	cdw_tabs_window_set_geometry(c_twindow, CONF_HEIGHT, CONF_WIDTH, 4, 4, TABS_WIDTH);


	/* prepare tabbed window */
	int ind = 0;
	cdw_id_t id = 0;

	id = CONFIG_PAGE_ID_OTHER;
	cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label); /* log and misc */
	ind++;

	id = CONFIG_PAGE_ID_TOOLS;
	cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label); /* tools */
	ind++;

	id = CONFIG_PAGE_ID_AUDIO;
	cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label); /* audio */
	ind++;

	id = CONFIG_PAGE_ID_HW;
	cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label); /* hardware */
	ind++;

	cdw_tabs_window_add_return_key(c_twindow, KEY_F(CDW_CONFIG_UI_SnC_KEY));
	cdw_tabs_window_add_return_key(c_twindow, CDW_KEY_ESCAPE);
	cdw_tabs_window_add_return_key(c_twindow, 'q');
	cdw_tabs_window_add_return_key(c_twindow, 'Q');
	c_twindow->driver_reader = cdw_config_ui_tabs_window_form_reader;
	cdw_tabs_window_finalize(c_twindow);


	/* prepare forms in the tabbed window */

	for (int i = 0; i < N_CONFIG_PAGES_ALL; i++) {
		id = c_twindow->tabs[i].id;

		c_pages[id].cdw_form = cdw_form_new(c_pages[id].n_fields);
		c_pages[id].cdw_form->form_id = id;
		c_pages[id].cdw_form->window = c_twindow->tabs[i].window;
		c_pages[id].cdw_form->subwindow = c_twindow->tabs[i].subwindow;

		c_pages[id].visible = true;

		cdw_rv_t crv = c_pages[id].fields_builder(c_pages[id].cdw_form, config, FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to build fields in page #%d\n", i);
			break;
		}

		c_pages[id].cdw_form->form = cdw_ncurses_new_form(c_pages[id].cdw_form->window,
								  c_pages[id].cdw_form->subwindow,
								  c_pages[id].cdw_form->fields);

		for (int j = 0; j < N_CONFIG_PAGES_ALL; j++) {
			cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(j + CDW_CONFIG_UI_FX_START_KEY));
		}
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY));
		cdw_form_add_return_char(c_pages[id].cdw_form, 'q'); /* will work only for non-text fields */
	}
	c_twindow->user_data = (void *) c_pages;

	cdw_tabs_window_draw_tabs(c_twindow);
	cdw_config_window_draw();

	return CDW_OK;
}





/**
   \brief Destructor function for cdw configuration window
*/
void cdw_config_window_destroy(void)
{
	if (cdw_config_window_tip != (char *) NULL) {
		free(cdw_config_window_tip);
		cdw_config_window_tip = (char *) NULL;
	}

	for (int i = 0; i < c_twindow->n_tabs; i++) {
		cdw_id_t id = c_twindow->tabs[i].id;
		if (c_pages[id].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete_form_objects(c_pages[id].cdw_form);
		}
		if (c_pages[id].cdw_form != (cdw_form_t *) NULL) {
			cdw_form_delete(&(c_pages[id].cdw_form));
		}
	}

	cdw_tabs_window_delete(&c_twindow);

	return;
}





/**
   \brief Function reacting to keys pressed when cdw config window is displayed

   Function catches all keys pressed when cdw configuration window is displayed,
   and reacts to them either by switching pages, or by passing the keys to
   page driver(s).

   Keys configured as hotkeys for "tabbed window" tabs are used to switch
   between pages. F(CDW_CONFIG_SnC_KEY) key is interpreted as "save and close"
   key - values of all option fields are stored in \p config and validated.
   Function then returns CDW_OK if validation is successful, or displays
   incorrect field to user.
   ESCAPE key causes the function to return with CDW_CANCEL.

   \param config - variable in which function saves values of all option fields

   \return CDW_CANCEL if user pressed ESCAPE key in cdw config window
   \return CDW_OK if user pressed F10 and validation of \p tmp config was successful
*/
cdw_rv_t cdw_config_window_driver(cdw_config_t *config)
{
	while (1) {
		int key = cdw_tabs_window_driver(c_twindow);
		if (key == CDW_KEY_ESCAPE) {
			break;
		} else if (key == KEY_F(CDW_CONFIG_UI_SnC_KEY)) { /* SnC = Save and Close */

			/* flush */
			for (int i = 0; i < c_twindow->n_tabs; i++) {
				cdw_id_t id = c_twindow->tabs[i].id;
				form_driver(c_pages[id].cdw_form->form, REQ_VALIDATION);
			}

			/* this function also saves indexes of selected
			   tool paths into cdw_tools[] */
			cdw_config_window_save_config_fields(config);
#ifndef NDEBUG
			cdw_config_debug_print_config(config);
#endif
			/* we are in driver and in theory we shouldn't
			   validate config here, but depending on result
			   of validation we may have to jump to one of pages;
			   moving validation outside of driver may be too
			   complicated */
			cdw_id_t page_id = -1; /* -1 = validate fields from all pages */
			int fi = 0;
			cdw_rv_t valid = cdw_config_var_validate(config, &page_id, &fi);

			if (valid == CDW_NO) { /* some option field is invalid */
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_tabs_window_show_tab_by_id(c_twindow, page_id);
				cdw_form_driver_go_to_field(c_pages[page_id].cdw_form, fi);

				/* loop */
			} else { /* valid == CDW_OK */

				/* we are ready to leave configuration window;
				   values of most of fields from the config
				   window are already saved in config var, but
				   there are still some values that need to
				   be saved in other places */
				cdw_config_window_save_non_config_fields();

				/* CDW_OK means: values from "config" function
				   argument can be saved to global config variable */
				return CDW_OK;
			}
		} else {
			/* loop */
		}
	} /* while (1) */

	return CDW_CANCEL;
}





/**
   \brief Create fields that are shown on "hardware" page configuration window

   Create fields that are shown on "hardware" page configuration window,
   including initialization of size and position, set their types, options
   and appearance. Fields will be initialized with values from \p config.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_hw_build_fields(cdw_form_t *cdw_form, void *cdw_config, int first_col, __attribute__((unused)) int second_col, int width_wide, int width_narrow)
{
	cdw_config_t *config = cdw_config;
	cdw_form->fields = page_hw_fields;
	int help_lines = CONF_HEIGHT - 2 - 9;
	cdw_form_descr_t descr[] = {
		/*     type      begin_y   begin_x     n_cols     n_lines   field enum           data1                           data2 */

		/* 2TRANS: this is label in configuration window, "drive" is an optical drive */
		{ CDW_WIDGET_LABEL,    1,  first_col,  width_narrow,    1,  f_selected_drive_l,  _("cdw should use this drive:"),    0 },
		{ CDW_WIDGET_DROPDOWN, 2,  first_col,  width_wide - 3,  1,  f_selected_drive_dd, dropdown_makers,                    1 },
		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds path to cd reader/writer device */
		{ CDW_WIDGET_LABEL,    4,  first_col,  width_narrow,    1,  f_custom_drive_l,    _("Custom path to drive:"),         0 },
		{ CDW_WIDGET_INPUT,    5,  first_col,  width_wide,      1,  f_custom_drive_i,    config->custom_drive,               0 },
		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds address of scsi writing device */
		{ CDW_WIDGET_LABEL,    6,  first_col,  width_narrow,    1,  f_scsi_l,            _("SCSI device (for cdrecord):"),   0 },
		{ CDW_WIDGET_INPUT,    7,  first_col,  width_wide,      1,  f_scsi_i,            config->scsi,                       0 },
		{ CDW_WIDGET_TEXT,     9,  1,          width_wide,      help_lines, f_hw_txt_t,  text_makers,                        0 },
		/* guard */
		{ -1,                  0,  0,          0,               0,  0,                   (void *) NULL,                      0 }};

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		/* set_field_type(fields[f_scsi_i], TYPE_REGEXP, "^(ATAPI:|REMOTE:){0,1}[0-9],[0-9],[0-9]"); */
		return CDW_OK;
	}
}





/**
   \brief Create fields that are shown on "audio" page configuration window

   Create fields that are shown on "audio" page configuration window,
   including initialization of size and position, set their types, options
   and appearance. Fields will be initialized with values from \p config.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_audio_build_fields(cdw_form_t *cdw_form, void *cdw_config, int first_col, __attribute__((unused)) int second_col, int width_wide, int width_narrow)
{
	cdw_config_t *config = cdw_config;
	cdw_form->fields = page_audio_fields;
	int audio_dir_button_col =  first_col + width_wide - 3;
	cdw_form_descr_t descr[] = {
		/*      type    begin_y   begin_x     n_cols     n_lines   field enum       data1               data2 */

		/* 2TRANS: this is input field label: keep no longer than original;  input field
		   holds path to directory, into which audio tracks will be ripped */
		{ CDW_WIDGET_LABEL,   1,  first_col,  width_narrow,    1,  f_audiodir_l,  _("Audio output dir"),  0 },
		{ CDW_WIDGET_INPUT,   2,  first_col,  width_wide - 3,  1,  f_audiodir_i,  config->audiodir,       0 },
		/* this fields is created only because there will be a button
		   here, that I need to be able to navigate to, so string can be empty */
		{ CDW_WIDGET_BUTTON,  2,  audio_dir_button_col,  1,    1,  f_audiodir_b,  _(">"), CDW_COLORS_DIALOG },
		/* guard */
		{ -1,                 0,  0,          0,               0,  0,             (void *) NULL,          0 } };

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_set_function(cdw_form, f_audiodir_b, page_audio_handle_enter_on_audiodir_button);
		return CDW_OK;
	}
}





/**
   \brief Create fields that are shown on "external tools" page configuration window

   Create fields that are shown on "external tools" page configuration window,
   including initialization of size and position, set their types, options
   and appearance. Fields will be initialized with values from \p config.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration, unused

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_tools_build_fields(cdw_form_t *cdw_form, __attribute__((unused)) void *cdw_config, int first_col, __attribute__((unused)) int second_col, int width_wide, int width_narrow)
{
	cdw_form->fields = page_tools_fields;
	bool ms = cdw_ext_tools_config_is_manual();
	cdw_form_descr_t descr[] = {
		/*     type        begin_y   begin_x         n_cols     n_lines   field enum                           data1                       data2 */

		{ CDW_WIDGET_TEXT,       1,  first_col,      width_wide,      2,  f_ext_tools_no_save_t,               text_makers,                    1 },
		{ CDW_WIDGET_CHECKBOX,   3,  first_col + 1,  1,               1,  f_ext_tools_manual_selection_cb,     (void *) NULL,         ms ? 1 : 0 },
		/* 2TRANS: this is checkbox label, it means that user will perform manual
		   selection of values in dropdowns representing available tools */
		{ CDW_WIDGET_LABEL,      3,  first_col + 4,  width_wide - 3,  1,  f_ext_tools_manual_selection_l,      _("Configure tools manually"),  0 },

		/* 2TRANS: this is label of input field */
		{ CDW_WIDGET_LABEL,      5,  first_col,      width_wide,      1,  f_ext_tools_iso9660_l,               _("Tool for creating stand alone ISO9660 file:"),  0 },
		{ CDW_WIDGET_DROPDOWN,   6,  first_col + 2,  width_wide - 3,  1,  f_ext_tools_iso9660_dd,              dropdown_makers,                2 },

		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,      7,  first_col,      width_narrow,           1,  f_ext_tools_cd_handlers_l,    _("Tools for handling CDs:"),   0 },
		{ CDW_WIDGET_DROPDOWN,   8,  first_col + 2,  width_wide - 3,  1,  f_ext_tools_cd_handlers_dd,          dropdown_makers,                3 },

		/* 2TRANS: this is label of input field; */
		{ CDW_WIDGET_LABEL,      9,  first_col,      width_wide,      1,  f_ext_tools_dvd_handlers_l,          _("Tools for handling DVDs:"),  0 },
		{ CDW_WIDGET_DROPDOWN,  10,  first_col + 2,  width_wide - 3,  1,  f_ext_tools_dvd_handlers_dd,         dropdown_makers,                4 },

		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,     11,  first_col,      width_narrow,           1,  f_ext_tools_mkisofs_path_l,   _("Path to mkisofs:"),          0 },
		{ CDW_WIDGET_DROPDOWN,  12,  first_col + 2,  width_wide - 3,  1,  f_ext_tools_mkisofs_path_dd,         dropdown_makers,                5 },
		/* 2TRANS: this is label of input field; "path" is path in filesystem */
		{ CDW_WIDGET_LABEL,     13,  first_col,      width_narrow,           1,  f_ext_tools_cdrecord_path_l,  _("Path to cdrecord:"),         0 },
		{ CDW_WIDGET_DROPDOWN,  14,  first_col + 2,  width_wide - 3,  1,  f_ext_tools_cdrecord_path_dd,        dropdown_makers,                6 },

		/* 2TRANS: this is label of input field */
		{ CDW_WIDGET_LABEL,     15,  first_col,      width_wide,           1,  f_ext_tools_digest_handlers_l,  _("Digest tool:"),              0 },
		{ CDW_WIDGET_DROPDOWN,  16,  first_col + 2,  width_wide - 3,  1,  f_ext_tools_digest_handlers_dd,      dropdown_makers,                7 },

		/* guard */
		{ -1,                    0,  0,              0,                     0,  0,                             (void *) NULL,                  0 }};

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_set_function(cdw_form, f_ext_tools_manual_selection_cb, page_tools_toggle_manual_selection);
		return CDW_OK;
	}
}





/**
   \brief Create fields that are shown on "log and other" page configuration window

   Create fields that are shown on "log and other" page configuration window,
   including initialization of size and position, set their types, options
   and appearance. Fields will be initialized with values from \p config.

   \param cdw_form - form page for which the fields will be built
   \param config - variable holding current configuration

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_ERROR if one of fields was created incorrectly
*/
cdw_rv_t page_other_build_fields(cdw_form_t *cdw_form, void *cdw_config, int first_col, int second_col, int width_wide, int width_narrow)
{
	cdw_config_t *config = cdw_config;
	cdw_form->fields = page_other_fields;
#define CDW_CVF_WIDTH 6 /* CVF = custom value field */

	char cvf_buffer[CDW_CVF_WIDTH + 1];
	if (config->volume_size_custom_value <= 0) {
		config->volume_size_custom_value = 0;
	} else {
		if ((floor(log10((double) config->volume_size_custom_value)) + 1) > CDW_CVF_WIDTH) {
			config->volume_size_custom_value = 0;
		}
	}
	snprintf(cvf_buffer, CDW_CVF_WIDTH + 1, "%ld", config->volume_size_custom_value);

	int log_path_button_col = first_col + width_wide - 3;
	cdw_form_descr_t descr[] = {
		/*    type        begin_y   begin_x         n_cols      n_lines  field enum              data1                             data2 */

		/* 2TRANS: this is input field label: keep no longer than original;
		   input field holds full path to log file */
		{ CDW_WIDGET_LABEL,     1,  first_col,       width_wide,      1,  f_log_fp_l,            _("Log file path (obligatory):"),     0 },
		{ CDW_WIDGET_INPUT,     2,  first_col,       width_wide - 3,  1,  f_log_fp_i,            config->log_full_path,                0 },
		/* this fields is created only because there will be a button
		   here, that I need to be able to navigate to, so string can be empty */
		{ CDW_WIDGET_BUTTON,    2,  log_path_button_col,  3,          1,  f_log_fp_b,            _(">"),               CDW_COLORS_DIALOG },
		{ CDW_WIDGET_CHECKBOX,  3,  first_col + 1,   1,               1,  f_showlog_cb,          (void *) NULL,  config->showlog ? 1 : 0 },
		/* 2TRANS: this is checkbox label: keep no longer than original */
		{ CDW_WIDGET_LABEL,     3,  first_col + 4,   width_narrow,    1,  f_showlog_l,           _("Show log after actions"),          0 },

		/* 2TRANS: this is label above a dropdown, a maximal size of created/update ISO9660 volume */
		{ CDW_WIDGET_LABEL,     6,  first_col,       width_narrow,    1,  f_volume_size_l,       _("ISO volume size:"),                0 },
		{ CDW_WIDGET_DROPDOWN,  7,  first_col + 2,   width_wide - 3,  1,  f_volume_size_dd,      dropdown_makers,                      0 },
		/* 2TRANS: this is input field label, user can enter into the field a custom value of max. size of ISO volume; "MB" is megabytes */
		{ CDW_WIDGET_LABEL,     8,  first_col,       width_narrow,    1,  f_cust_volume_size_l,  _("Custom ISO volume size (MB):"),    0 },
		{ CDW_WIDGET_INPUT,     8,  second_col + 5,  CDW_CVF_WIDTH,   1,  f_cust_volume_size_i,  cvf_buffer,               CDW_CVF_WIDTH },
		/* guard */
		{ -1,                   0,  0,             0,                 0,  0,                     (void *) NULL,                        0 } };

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_set_function(cdw_form, f_log_fp_b, page_other_handle_enter_on_log_path_button);
		return CDW_OK;
	}
}





/**
   \brief Draw widgets in configuration window

   \return CDW_OK if there were no problems
*/
cdw_rv_t cdw_config_window_draw(void)
{
	cdw_id_t id = CONFIG_PAGE_ID_TOOLS;
	if (c_pages[id].visible) {
		CDW_CHECKBOX *checkbox = cdw_form_get_checkbox(c_pages[id].cdw_form, f_ext_tools_manual_selection_cb);
		bool state = cdw_checkbox_get_state(checkbox);
		checkbox->on_toggle(c_pages[id].cdw_form, &state);
	}

	cdw_config_ui_tabs_window_form_draw(c_twindow, c_pages);

	return CDW_OK;
}





void cdw_config_window_save_non_config_fields(void)
{
	/* *** HARDWARE *** */
	CDW_DROPDOWN *dropdown = cdw_form_get_dropdown(c_pages[CONFIG_PAGE_ID_HW].cdw_form, f_selected_drive_dd);
	cdw_id_t id = cdw_dropdown_get_current_item_id(dropdown);
	cdw_drive_save_configuration(id);

	/* *** AUDIO *** */

	/* *** TOOLS *** */
	cdw_form_t *cdw_form = c_pages[CONFIG_PAGE_ID_TOOLS].cdw_form;
	cdw_ext_tools_configuration_t etc;

	etc.manual_selection = cdw_form_get_checkbox_state(cdw_form, f_ext_tools_manual_selection_cb);

	etc.iso9660 = cdw_form_get_dropdown(cdw_form, f_ext_tools_iso9660_dd);
	etc.cd_handlers = cdw_form_get_dropdown(cdw_form, f_ext_tools_cd_handlers_dd);
	etc.dvd_handlers = cdw_form_get_dropdown(cdw_form, f_ext_tools_dvd_handlers_dd);
	etc.mkisofs = cdw_form_get_dropdown(cdw_form, f_ext_tools_mkisofs_path_dd);
	etc.cdrecord = cdw_form_get_dropdown(cdw_form, f_ext_tools_cdrecord_path_dd);
	etc.digest_handlers = cdw_form_get_dropdown(cdw_form, f_ext_tools_digest_handlers_dd);

	cdw_ext_tools_save_configuration(&etc);

	/* *** LOGGING AND OTHER *** */

	return;
}





/**
   \brief Save options in given config variable

   \param config - variable to which all field values will be saved
*/
void cdw_config_window_save_config_fields(cdw_config_t *config)
{
	cdw_rv_t ss;

	/* *** HARDWARE *** */
	cdw_form_t *cdw_form = c_pages[CONFIG_PAGE_ID_HW].cdw_form;
	snprintf(config->custom_drive, OPTION_FIELD_LEN_MAX + 1, "%s", cdw_ncurses_get_field_string(*(cdw_form->fields + f_custom_drive_i)));
	snprintf(config->scsi, OPTION_FIELD_LEN_MAX + 1, "%s", cdw_ncurses_get_field_string(*(cdw_form->fields + f_scsi_i)));

	CDW_DROPDOWN *dropdown = cdw_form_get_dropdown(cdw_form, f_selected_drive_dd);
	cdw_id_t id = cdw_dropdown_get_current_item_id(dropdown);
	cdw_drive_set_config_fields(config, id);


	/* *** AUDIO *** */
	cdw_form = c_pages[CONFIG_PAGE_ID_AUDIO].cdw_form;
	cdw_string_set((&config->audiodir), cdw_ncurses_get_field_string(*(cdw_form->fields + f_audiodir_i)));


	/* *** TOOLS *** */
	/* no values to be saved in config variable */


	/* *** LOGGING AND OTHER *** */
	cdw_form = c_pages[CONFIG_PAGE_ID_OTHER].cdw_form;
	ss = cdw_string_set(&(config->log_full_path), cdw_ncurses_get_field_string(*(cdw_form->fields + f_log_fp_i)));
	config->showlog = cdw_form_get_checkbox_state(cdw_form, f_showlog_cb);
	dropdown = cdw_form_get_dropdown(cdw_form, f_volume_size_dd);
	config->volume_size_id = cdw_dropdown_get_current_item_id(dropdown);

	char *cvs_buffer = cdw_ncurses_get_field_string(*(cdw_form->fields + f_cust_volume_size_i));
	char *cvs_buffer_rest = (char *) NULL;
	config->volume_size_custom_value = strtol(cvs_buffer, &cvs_buffer_rest, 10);
	cdw_vdm ("INFO: volume size: buf = \"%s\", rest = \"%s\"\n", cvs_buffer, cvs_buffer_rest);
	if (strlen(cvs_buffer_rest)) {
		/* "-1" will be recognized as indicator of invalid buffer
		   content at time of validation of config variable */
		config->volume_size_custom_value = -1;
		cdw_vdm ("WARNING: invalid substring \"%s\" in custom volume size string \"%s\"\n", cvs_buffer_rest, cvs_buffer);
	} else {
		cdw_vdm ("INFO: volume size string \"%s\" converted to int %ld\n", cvs_buffer, config->volume_size_custom_value);
	}
#if 0

	/* TODO: not a best place to set config->volume_size_value,
	   because if volume_size_id == AUTO then it will be hard to
	   do disc check here */
	if (config->volume_size_id == CDW_CONFIG_VOLUME_SIZE_CUSTOM) {
		config->volume_size_value = config->volume_size_custom_value;
	} else {
		config->volume_size_value = cdw_config_volume_size_by_id(config->volume_size_id);
	}
#endif
	return;
}





int page_tools_toggle_manual_selection(cdw_form_t *cdw_form, void *state)
{
	bool ms = *((bool *) state);

	int (* field_opts_toggle)(FIELD *, Field_Options) = ms ? field_opts_on : field_opts_off;
	void (* dd_toggle)(CDW_DROPDOWN *) = ms ? cdw_dropdown_make_visible : cdw_dropdown_make_invisible;

	/* "manual selection" label field is the last field that should
	   be always visible, so don't toggle it here;
	   hence "i = f_manual_selection_l + 1" in for () loop */
	for (int i = f_ext_tools_manual_selection_l + 1; i < cdw_form->n_fields; i++) {
		field_opts_toggle(*(cdw_form->fields + i), O_VISIBLE);

		CDW_DROPDOWN *dd = cdw_form_get_dropdown(cdw_form, i);
		if (dd != (CDW_DROPDOWN *) NULL) {
			/* the field really is a dropdown (and not a label) */
			dd_toggle(dd);
		}
	}

	cdw_config_window_refresh_page(CONFIG_PAGE_ID_TOOLS);
	form_driver(cdw_form->form, REQ_FIRST_FIELD);

	return 'a';
}





int page_other_handle_enter_on_log_path_button(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	/* 2TRANS: this is title of window with file picker */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Log file"),
					     /* 2TRANS: this is message in file picker
						window, explaining what file should be
						selected */
					     _("Select new or existing log file"),
					     &(local_config->log_full_path),
					     CDW_FS_FILE,
					     R_OK | W_OK,
					     CDW_FS_NEW | CDW_FS_EXISTING);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(cdw_form->fields[f_log_fp_i], 0, local_config->log_full_path);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", local_config->log_full_path);
		} else {
			form_driver(cdw_form->form, REQ_END_LINE);
		}
	}

	cdw_main_ui_main_window_wrefresh();
	cdw_config_window_refresh_page(cdw_form->form_id);

	return 'a';
}





int page_audio_handle_enter_on_audiodir_button(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	/* 2TRANS: this is title of window with file picker */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Audio output dir"),
					     /* 2TRANS: this is message in file picker
						window, explaining what file should be
						selected */
					     _("Select existing output dir"),
					     &(local_config->audiodir),
					     CDW_FS_DIR,
					     R_OK | W_OK | X_OK,
					     CDW_FS_EXISTING);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(cdw_form->fields[f_audiodir_i], 0, local_config->audiodir);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", local_config->audiodir);
		} else {
			form_driver(cdw_form->form, REQ_END_LINE);
		}
	}

	cdw_main_ui_main_window_wrefresh();
	cdw_config_window_refresh_page(cdw_form->form_id);

	return 'a';
}





void cdw_config_window_refresh_page(cdw_id_t id)
{
	redrawwin(c_pages[id].cdw_form->subwindow);
	wrefresh(c_pages[id].cdw_form->subwindow);
	redrawwin(c_pages[id].cdw_form->window);
	wrefresh(c_pages[id].cdw_form->window);

	return;
}





static cdw_id_clabel_t cdw_config_volume_size_items[CDW_CONFIG_VOLUME_ITEMS_MAX] = {
	/* 2TRANS: this is dropdown item label: 650MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,               gettext_noop("74 min CD (650 MB)") },
	/* 2TRANS: this is dropdown item label: 700MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD80,               gettext_noop("80 min CD (700 MB)") },
	/* 2TRANS: this is dropdown item label: DVD */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC,        gettext_noop("Generic DVD (4.7 GB)") },
	/* 2TRANS: this is dropdown item label: DVD+R DL */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,          gettext_noop("DVD+R DL (8.5 GB)") },
	/* 2TRANS: this is dropdown item label: custom value of
	   ISO volume; user can enter arbitrary natural number */
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,             gettext_noop("Custom value") },
	/* 2TRANS: this is dropdown item label:
	   "automatic" = automatic detection/resolution of size of ISO volume */
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,               gettext_noop("Get sizes from disc") }
};


CDW_DROPDOWN *cdw_config_window_make_volume_size_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_clabel_t items[CDW_CONFIG_VOLUME_ITEMS_MAX];
	int n_items = 0;
	for (int i = 0; i < CDW_CONFIG_VOLUME_ITEMS_MAX; i++) {
		if (cdw_config_volume_size_items[i].id == CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL
		    && !local_config->support_dvd_rp_dl) {
			/* skipping this item won't introduce empty
			   item in dropdown */
			continue;
		}
		items[n_items].id = cdw_config_volume_size_items[i].id;
		items[n_items].label = cdw_config_volume_size_items[i].label;
		n_items++;
	}

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, n_items, items);

	if (local_config->volume_size_id == CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL
	    && !local_config->support_dvd_rp_dl) {
		/* this may happen when value saved in config file is
		   "DVD+R DL" (because previously cdw was called with
		   enabled DVD+R DL support, and user has selected DVD+R DL
		   in this dropdown, but this time cdw is called with support
		   for DVD+R DL disabled; in such case try with CD-R */
		local_config->volume_size_id = CDW_CONFIG_VOLUME_SIZE_CD74;
	}
	cdw_dropdown_set_current_item_by_id(dd, local_config->volume_size_id);

	return dd;
}





char *cdw_config_window_page_tools_message(int n_cols)
{
	/* 2TRANS: this is a description in one of
	   configuration window panels; a series of
	   dropdown widgets is displayed below the label */
	char *message = _("Changes made here won't be saved after closing cdw.");

        int n_lines = cdw_string_wrap_for_form_get_n_lines(message, n_cols);
	char *new_message = cdw_string_wrap_for_form(message, n_cols, n_lines);

	return new_message;
}





char *cdw_config_window_page_hw_message(int n_cols)
{
	char *message = cdw_drive_get_help_message();
	int n_lines = cdw_string_wrap_for_form_get_n_lines(message, n_cols);
	char *new_message = cdw_string_wrap_for_form(message, n_cols, n_lines);

	free(message);
	message = (char *) NULL;

	return new_message;
}







/**
   \brief Draw widgets in configuration window

   \return CDW_OK if there were no problems
*/
cdw_rv_t cdw_config_ui_tabs_window_form_draw(cdw_tabs_window_t *twindow, cdw_config_page_t *config_pages)
{
	for (int i = 0; i < twindow->n_tabs; i++) {
		cdw_id_t id = twindow->tabs[i].id;
		cdw_assert (config_pages[id].visible, "ERROR: page #%d (id = %lld) is not visible\n", i, id);
		cdw_form_redraw_widgets(config_pages[id].cdw_form);
		form_driver(config_pages[id].cdw_form->form, REQ_FIRST_FIELD);
	}

	return CDW_OK;
}





/**
   \brief Reader function for tabbed window

   Configuration window is built on tabbed window widget. Since the widget
   requires a reader function (cdw_tabs_window_driver_reader_t driver_reader),
   configuration window module must provide one. This is it.

   The function operates on a form (of cdw_form_t type) that is coupled
   with tabbed windows window. It returns a key that has been pressed in the
   form/window, provided that the key has been configured as "return key" for
   this specific form.

   \param dummy - the function does not make any use of the argument

   \return function returns key pressed in a page of configuration window
*/
int cdw_config_ui_tabs_window_form_reader(void *twindow)
{
	cdw_tabs_window_t *t_window = (cdw_tabs_window_t *) twindow;
	int ind = t_window->current_tab;
	cdw_config_page_t *config_pages = t_window->user_data;
	cdw_id_t id = t_window->tabs[ind].id;
	int fi = field_index(current_field(config_pages[id].cdw_form->form));
	return cdw_form_driver(config_pages[id].cdw_form, fi);
}


