/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>

#include "cdw_processwin.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_drive.h"
#include "cdw_thread.h"
#include "cdw_dvd_rw_format.h"
#include "cdw_debug.h"
#include "cdw_erase_disc.h"


/* was blanking of DVD+RW really performed? if yes, then it will be correct
   to put '100% done' message in processwin; otherwise it would be misleading */
extern bool dvd_rw_format_pipe_regexp_performed;

cdw_rv_t cdw_dvd_rw_format_run_task(cdw_task_t *task, cdw_disc_t *disc)
{
	char *command = (char *) NULL;

	char speed_string[9 + 1];
	cdw_assert (task->erase.speed_id != 0, "ERROR: task->erase.speed was not set\n");
	snprintf(speed_string, 9 + 1, "%lld", task->erase.speed_id);

	const char *drive = cdw_drive_get_drive_fullpath();

	if (disc->type == CDW_DVD_RW
	    || disc->type == CDW_DVD_RW_SEQ
	    || disc->type == CDW_DVD_RW_RES) {

		/* in case of blanking such kind of discs (DVD-RW) it is
		   important to know not only if this should be 'fast' or
		   'all', but also mode in which formatted disc should be:
		   'Restricted Overwrite' or 'Incremental Sequential' */
		if (task->erase.format_mode == CDW_ERASE_DVD_RW_FORMAT_SEQ) {
			cdw_sdm ("INFO: task->erase.format_mode == DVD_RW_FORMAT_SEQ\n");
			cdw_assert (task->erase.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
			command = cdw_string_concat(task->erase.tool.label,
						    " -speed=", speed_string, " ",
						    " -blank", task->erase.erase_mode == CDW_ERASE_MODE_FAST ? " " : "=full ",
						    drive,
						    (char *) NULL);
		} else if (task->erase.format_mode == CDW_ERASE_DVD_RW_FORMAT_RES) {
			cdw_sdm ("INFO: task->erase.format_mode == DVD_RW_FORMAT_RES\n");
			cdw_assert (task->erase.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
			command = cdw_string_concat(task->erase.tool.label,
						    " -speed=", speed_string, " ",
						    " -force", task->erase.erase_mode == CDW_ERASE_MODE_FAST ? " " : "=full ",
						    drive,
						    (char *) NULL);
		} else {
			cdw_assert (0, "ERROR: task->erase.format_mode is incorrect: %lld\n",
				    task->erase.format_mode);
		}

	} else if (disc->type == CDW_DVD_RWP) {
		cdw_assert (task->erase.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
		command = cdw_string_concat(task->erase.tool.label,
					    " -speed=", speed_string, " ",
					    /* "-blank" doesn't work for DVD+RW, and "=full"
					       is rejected for both "-blank" and "-force";
					       the only thing that works is plain "-force" */
					    " -force ",
					    drive,
					    (char *) NULL);
	} else {
		cdw_assert (0, "ERROR: called this function for incorrect disc type %d\n",
			    disc->type);
	}


	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat() command for formatting\n");
		return CDW_ERROR;
	}

	/* 2TRANS: this is message in process window - there is some
	   preparation to erasing, but the erasing hasn't started yet */
	cdw_processwin_display_sub_info(_("Preparing to erase. Please wait."));

	cdw_sdm ("INFO: command is \"%s\"\n", command);
	int rv = run_command(command, task);
	free(command);
	command = (char *) NULL;
	if (rv != 0) {
		return CDW_ERROR;
	}

	if (disc->type == CDW_DVD_RWP
	    && dvd_rw_format_pipe_regexp_performed) {
		/* blanking dvd+rw is so fast that there are only a few values
		   of percentage displayed. 100% is not one of them. let's
		   fake it here */

		cdw_processwin_display_progress(100, 100, (char *) NULL);
	}

	return CDW_OK;
}

