#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#if HAVE_STDIO_H
#include <stdio.h> 
#endif /* HAVE_STDIO_H */

#if HAVE_STDARG_H 
#include <stdarg.h>
#endif /* HAVE_STDARG_H */

#if HAVE_STDLIB_H 
#include <stdlib.h>
#endif  /* HAVE_STDLIB_H */

#if HAVE_ERRNO_H
#include <errno.h>
#endif /* HAVE_ERRNO_H */

#if HAVE_UNISTD_H
#include <unistd.h>
#endif  /* HAVE_UNISTD_H */

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_STRING_H
#include <string.h>
#endif /* HAVE_STRING_H */

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif /* HAVE_FCNTL_H */

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif  /* HAVE_SYS_SELECT_H */

#if HAVE_ERRNO_H
#include <errno.h>
#endif /* HAVE_ERRNO_H */

#include "io.h"
#include "error.h"

#define MAXLINE 4096

static FILE *dfd = NULL;
static int debug_on = 0;

static const char *debug_begin = "";
static const char *debug_end   = "";

static void process_error(void) {
   if(errno == EINTR)
      err_ret("%s:%d -> ERRNO = EINTR", __FILE__, __LINE__);
   else if(errno == EAGAIN)
      err_ret("%s:%d -> ERRNO = EAGAIN", __FILE__, __LINE__);
   else if(errno == EIO)
      err_ret("%s:%d -> ERRNO = EIO", __FILE__, __LINE__);
   else if(errno == EISDIR)
      err_ret("%s:%d -> ERRNO = EISDIR", __FILE__, __LINE__);
   else if(errno == EBADF)
      err_ret("%s:%d -> ERRNO = EBADF", __FILE__, __LINE__);
   else if(errno == EINVAL)
      err_ret("%s:%d -> ERRNO = EINVAL", __FILE__, __LINE__);
   else if(errno == EFAULT)
      err_ret("%s:%d -> ERRNO = EFAULT", __FILE__, __LINE__);
}

int io_debug_init(const char *filename){
   char config_dir[MAXLINE];
   if ( filename == NULL )
       return -1;
   
   strcpy( config_dir, filename );
   
   if( (dfd = fopen(config_dir, "w")) == NULL) {
      err_msg("%s:%d -> could not open debug file", __FILE__, __LINE__);
      return -1;
   }

   /* A nifty trick to have debug dump to the terminal
    * dfd = stderr;*/

   debug_on = 1;

   return 0;
}

void io_debug_write(const char *write){
   fprintf(dfd, "%s%s%s", debug_begin, write, debug_end);
   fflush(dfd);
} 

void io_debug_write_fmt(const char *fmt, ...) {
    va_list ap;
    char va_buf[MAXLINE];

    va_start(ap, fmt);
#ifdef   HAVE_VSNPRINTF
    vsnprintf(va_buf, sizeof(va_buf), fmt, ap);  /* this is safe */
#else
    vsprintf(va_buf, fmt, ap);             /* this is not safe */
#endif
    va_end(ap);

    fprintf(dfd, "%s", va_buf);
    fflush(dfd);
}

int io_read_byte(char *c, int source){
   int ret_val = 0;
   
   if( ( ret_val = read(source, c, 1)) == 0) {
      return -1;
   } else if ( ret_val == -1 ) {
      err_msg("%s:%d -> I/O error", __FILE__, __LINE__); process_error();
      return -1;
   } 

   return 0;
}

int io_write_byte(int dest, char c){
   if(write(dest, &c, 1) != 1)
      return -1;

   return 0;   
}

int io_rw_byte(int source, int dest){
   char c;
   
   if(read(source, &c, 1) != 1){
      err_msg("%s:%d -> I/O error", __FILE__, __LINE__);
      process_error();
      err_ret("source bad\n");
      return -1;
   }
      
   if(write(dest, &c, 1) != 1){
      err_msg("dest bad\n");
      return -1;
   }

   return 0;      
}

ssize_t io_read(int fd, void *buf, size_t count){
   ssize_t amountRead;
   tgdb_read:
   
   if( (amountRead = read(fd, buf, count)) == -1){ /* error */
      if(errno == EINTR)
         goto tgdb_read;
      else if ( errno != EIO ) {
         err_ret("%s:%d -> error reading from fd", __FILE__, __LINE__);
         return -1;
      } else {
         return 0; /* Happens on EOF for some reason */
      }

   } else if(amountRead == 0){ /* EOF */
      return 0;
   } else {
      char *tmp = (char *)buf;
      tmp[amountRead] = '\0';
      if(debug_on == 1){
         int i;
         fprintf(dfd, "%s", debug_begin);
         for(i =0 ; i < amountRead; ++i) {
            if(((char *)buf)[i] == '\r')
               fprintf(dfd, "(%s)", "\\r");
            else if(((char *)buf)[i] == '\n')
               fprintf(dfd, "(%s)\n", "\\n");
            else if(((char *)buf)[i] == '\032')
               fprintf(dfd, "(%s)", "\\032");
            else if(((char *)buf)[i] == '\b')
               fprintf(dfd, "(%s)", "\\b");
            else
               fprintf(dfd, "%c", ((char *)buf)[i]);
         }
         fprintf(dfd, "%s", debug_end);
         fflush(dfd);
      }
      return amountRead;

   }
}

ssize_t io_writen(int fd, const void *vptr, size_t n) {
   ssize_t nwritten;
   size_t nleft = n;
   const char *ptr = (const char *)vptr;

   while(nleft > 0) {
      if( (nwritten = write( fd, ptr, nleft)) <= 0) {
         if(errno == EINTR)
            nwritten = 0;
         else
            return -1;
      } // end if
      nleft -= nwritten;
      ptr += nwritten;
   } // end while

   return (n);
} // end writen

void io_display_char(FILE *fd, char c){
   if(c == '\r')
      fprintf(fd, "(%s)", "\\r");
   else if(c == '\n')
      fprintf(fd, "(%s)\n", "\\n");
   else if(c == '\032')
      fprintf(fd, "(%s)", "\\032");
   else if(c == '\b')
      fprintf(fd, "(%s)", "\\b");
   else
      fprintf(fd, "(%c)", c);

   fflush(fd);
}

int io_data_ready ( int fd, int ms ) {
	int ret;

#if defined(HAVE_SELECT)
    fd_set readfds, exceptfds;
    struct timeval timeout;
    
	FD_ZERO(&readfds);
    FD_ZERO (&exceptfds);
    FD_SET (fd, &readfds);
    FD_SET (fd, &exceptfds);
    
    timeout.tv_sec = 0;
    timeout.tv_usec = 1000*ms;

    ret = select (fd + 1, &readfds, (fd_set *)NULL, &exceptfds, &timeout);

    if (ret <= 0)
        return 0;   /* Nothing to read. */
	else
		return 1;
#endif
}

int io_getchar ( int fd, unsigned int ms ) {
    char c;
    int ret;
    int flag = 0;

    /* This is the timeout that readline uses, so it must be good.
     * However, when I hit ESC and then o from the gdb window, the input lib 
     * returns the key binding Alt-o which doesn't cause the filedlg to be 
     * opened. This is very annoying.
     */
    /*timeout.tv_usec = 100000; */  /* 0.1 seconds; it's in usec */

    /* This is a good value that causes Alt-o to be returned when it should
     * be and ESC, o when it should be.
     */
	if ( io_data_ready ( fd, ms ) == 0 )
        return 0;   /* Nothing to read. */

    /* Set nonblocking */
    flag = fcntl(fd, F_GETFL, 0);
    fcntl(fd, F_SETFL, flag | O_NONBLOCK);

read_again:

    /* Read a char */
    ret = read(fd, &c, 1);

    if ( ret == -1 && errno == EAGAIN )
        c = 0;  /* No data available */
    else if ( ret == -1 && errno == EINTR )
        goto read_again;
    else if ( ret == -1 ) {
        c = 0; 
        err_msg("Errno(%d)\n", errno);
    } else if ( ret == 0 ) {
        c = 0; 
        ret = -1;
        err_msg("Read returned nothing\n"); 
    }

    /* Set to original state */
    fcntl(fd, F_SETFL, flag);

    if ( ret == -1 )
        return -1;

    return c;
}
