//
// C++ Implementation: chorus_effect
//
// Description:
//
//
// Author: Juan Linietsky <coding@reduz.com.ar>, (C) 2003
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "chorus_effect.h"


#define FRACTIONAL_BITS 13
#define MIX_CALCULATE_INCREMENT_INTERVAL 8


inline static double get_msecs_from_samples(double p_samples,double p_mixfreq) {

	return (p_samples/p_mixfreq)*1000.0;


}

inline static double get_samples_from_msecs(double p_msecs,double p_mixfreq) {

	return (p_msecs*p_mixfreq)/1000.0;

}

void Chorus_Effect::reset() {


	memset(&ringbuffer[0],0,ringbuffer.size()*sizeof(Sint32));
}


void Chorus_Effect::ensure_ringbuffer_size() {

	int max_delay=real_parm.voices * real_parm.delay;
	max_delay+=real_parm.lfo_depth*2;

	int needed_samples=get_samples_from_msecs(max_delay,mix_freq);
	needed_samples+=BUFFER_EXTRA_ALLOCATON;

	int i;
	for (i=0;i<31;i++) {

	 	int maskd=(1<<i)-1;
		maskd=~maskd; //invert bits!
		if ( (needed_samples&maskd) ==0 ) { //found a buffer that contains all

			break;
		}
	}

	buffer_bits=i;
	buffer_mask=(1<<buffer_bits)-1;

	if (((1<<buffer_bits)*2)!=ringbuffer.size()) {

		ringbuffer.resize((1<<buffer_bits)*2);
		memset(&ringbuffer[0],0,ringbuffer.size()*sizeof(Sint32));
		buffer_index=0;
	}


}


void Chorus_Effect::process_buffer(sample_32s_t* p_buffer,int p_length,bool p_stereo_samples,int p_buffer_freq) {

	mix_freq=p_buffer_freq;
	real_parm=bridge_parm;

	ensure_ringbuffer_size();

        int aux_buffer_index=buffer_index;
	double aux_counter=counter;

	sample_32s_t dry_mult=real_parm.dry * 65535/127;

	Sint32* ringbuffer_ptr=&ringbuffer[0];

	for (int j=0;j<(p_length*2);j+=2) {

		aux_buffer_index=(aux_buffer_index-2) & (buffer_mask<<1);

		ringbuffer_ptr[aux_buffer_index]=p_buffer[j];
		ringbuffer_ptr[aux_buffer_index+1]=p_buffer[j+1];

		p_buffer[j]>>=16;
		p_buffer[j]*=dry_mult;
		p_buffer[j+1]>>=16;
		p_buffer[j+1]*=dry_mult;
	}

	for (int i=0;i<real_parm.voices;i++) {

		sample_32s_t src_l,src_r,aux,next_l,next_r;
		int todo=p_length;

		double divisor=(M_PI*2.0) / (1000.0/(double)real_parm.lfo_speed);
		double increment=get_msecs_from_samples(1.0,mix_freq);

		aux_counter=counter;
		double depth_multipler=get_samples_from_msecs((double)real_parm.lfo_depth/100.0,mix_freq);
		double delay =get_samples_from_msecs(real_parm.delay,mix_freq);

		int voice_width=real_parm.width;
		if (i%2)
			voice_width=-voice_width;

		sample_32s_t width=voice_width*65535/64;
		sample_32s_t width_inv=(64-voice_width)*65535/64;
		width=real_parm.wet*width/127;
		width_inv=real_parm.wet*width_inv/127;

		aux_buffer_index=buffer_index;


		for (int j=0;j<(p_length*2);j+=2) {

                        double src_idx=((double)i+1.0) * delay;
			src_idx+=depth_multipler+sin( aux_counter * divisor ) * depth_multipler;

			aux_counter+=increment;
			unsigned int src_idx_int=lrint(src_idx);
			int src_fixed=lrintf((double)(src_idx-(double)src_idx_int)*(double)(1<<FRACTIONAL_BITS));

   			aux_buffer_index=(aux_buffer_index-2) & (buffer_mask<<1);

			src_idx_int=(aux_buffer_index + src_idx_int*2) & (buffer_mask<<1);

			src_l=ringbuffer_ptr[src_idx_int];
			src_r=ringbuffer_ptr[src_idx_int+1];

			src_idx_int=(src_idx_int+2)&(buffer_mask<<1);

			next_l=ringbuffer_ptr[src_idx_int];
			next_r=ringbuffer_ptr[src_idx_int+1];
			next_l>>=16;
			next_r>>=16;
                        src_l>>=16;
                        src_r>>=16;

			src_l=src_l+((next_l-src_l)*src_fixed >> FRACTIONAL_BITS);
			src_r=src_r+((next_r-src_r)*src_fixed >> FRACTIONAL_BITS);

			aux=src_r;

			src_r=src_r*width_inv +src_l*width;
			src_l=aux*width +src_l*width_inv;

			p_buffer[j]+=src_l;
			p_buffer[j+1]+=src_r;
		}

	}

	counter=aux_counter; //wont count if no voices.. that wont hurt anyway
	buffer_index=aux_buffer_index;

}

string Chorus_Effect::get_name() {

	return "chorus";
}

string Chorus_Effect::get_display_name() {

	return "Chorus";
}

list<Property_Bridge*> Chorus_Effect::get_properties() {

	list<Property_Bridge*> plushie;
	plushie.push_back(&bridge_voices);
	plushie.push_back(&bridge_delay);
	plushie.push_back(&bridge_lfo_speed);
	plushie.push_back(&bridge_lfo_depth);
	plushie.push_back(&bridge_wet);
	plushie.push_back(&bridge_dry);
	plushie.push_back(&bridge_width);
	return plushie;
}


Chorus_Effect::Chorus_Effect() :
	bridge_voices("Voices",&bridge_parm.voices,1,5,"voices"),
	bridge_delay("Delay(1/10 ms)",&bridge_parm.delay,1,127,"delay"),
	bridge_lfo_speed("Speed(hz)",&bridge_parm.lfo_speed,1,50,"lfo_speed"),
	bridge_lfo_depth("Depth(ms)",&bridge_parm.lfo_depth,1,127,"lfo_depth"),
	bridge_wet("Wet%",&bridge_parm.wet,0,127,"lfo_feedback"),
	bridge_dry("Dry%",&bridge_parm.dry,0,127,"lfo_feedback"),
	bridge_width("Width%",&bridge_parm.width,-64,64,"width") {

	bridge_parm.voices=1;
	bridge_parm.delay=20;
	bridge_parm.lfo_speed=3;
	bridge_parm.lfo_depth=20;
	bridge_parm.wet=80;
	bridge_parm.dry=80;
	bridge_parm.width=20;
	mix_freq=44100;
	real_parm=bridge_parm;
	counter=0;
	buffer_index=0;

	ensure_ringbuffer_size();
}

Chorus_Effect::~Chorus_Effect() {

}
