/***************************************************************************
    This file is part of the CheeseTronic Music Tools
    url                  : http://reduz.com.ar/cheesetronic
    copyright            : (C) 2003 by Juan Linietsky
    email                : coding@reduz.com.ar
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "tracker_voice.h"


int Tracker_Voice::get_current_freq() {

	return info.current_frequency;
}
int Tracker_Voice::get_current_sample_pos() {

	return info.current_index >> FRACTIONAL;
}

void Tracker_Voice::set_priority(int p_prio) {

	current_prio=p_prio;
}

//virtuals
Voice::Priority Tracker_Voice::get_priority() {


	return (Voice::Priority)current_prio;
}

int Tracker_Voice::set_channel(int p_channel) {

	channel_index=p_channel;
}

int Tracker_Voice::get_channel() {

	return channel_index;
}

void Tracker_Voice::set_mix_frequency(float p_mixfreq) {

	mixfreq=(int)p_mixfreq;
}
void Tracker_Voice::set_preamp(float p_preamp) {

	preamp=p_preamp;
}

bool Tracker_Voice::can_mix() {

	if ( info.current_frequency==0 )
		return false;
	if ( (info.sample_data_ptr==NULL) || (info.sample_data_ptr->get_data_16()==NULL) )
		return false;
	if (!info.volume && (!info.oldlvol) && (!info.oldrvol))
		return false;

	return true;

}



void Tracker_Voice::mix(int p_amount,sample_32s_t* p_where) {

	if ( info.current_frequency==0 ) info.active=false;
	if ( (info.sample_data_ptr==NULL) || (info.sample_data_ptr->get_data_16()==NULL) ) info.active=false;

	if ( !info.active ) {

		return;
	}



	int i, pan, vol;
		// Calculate increment index depending on frequency difference
	info.increment_index=((Sint64)(info.current_frequency<<FRACTIONAL))/mixfreq;

	if (info.playing_backwards) info.increment_index=-info.increment_index;

	vol = info.volume;
	pan = info.panning;

	info.oldlvol=info.lvolsel;
	info.oldrvol=info.rvolsel;




	if(pan != PAN_SURROUND) {

		info.lvolsel=(vol*(PAN_RIGHT-pan))>>8;
		info.rvolsel=(vol*pan)>>8;

	} else	{
		info.lvolsel=info.rvolsel=vol/2;
	}

	if (info.first_mix) {
		info.oldlvol=info.lvolsel;
		info.oldrvol=info.rvolsel;
		info.first_mix=false;
	}

	idxsize = (info.sample_data_ptr->get_size())? ((Sint64)info.sample_data_ptr->get_size() << FRACTIONAL)-1 : 0;
	idxlend = (info.sample_data_ptr->get_loop_end())? ((Sint64)info.sample_data_ptr->get_loop_end() << FRACTIONAL)-1 : 0;
	idxlpos = (Sint64)info.sample_data_ptr->get_loop_begin() << FRACTIONAL;

	if (info.increment_index==0) {

		info.active=false;
		return;
	}
	add_to_mix_buffer(p_amount,p_where);
}

void Tracker_Voice::add_to_mix_buffer(int p_amount,sample_32s_t *p_buffer) {


	Sint64 end,done;
	sample_32s_t *mixing_buffer_index;

	mixing_buffer_index=p_buffer;

	int total=p_amount;
	int todo=total;

	float ramp_tangent_l;
	float ramp_tangent_r;

        if ( info.sample_data_ptr == NULL ) {

		info.current_index=0;
		info.active=false;
		return;
	}


	/* precalculate ramp */


	/* update the 'current' index so the sample loops, or stops playing if it
	   reached the end of the sample */

	bool loop_active=info.sample_data_ptr->is_loop_enabled() && (idxlend>idxlpos);

	if (total<=0) return;

	ramp_tangent_l=(float)(info.lvolsel-info.oldlvol)/(float)total;
	ramp_tangent_r=(float)(info.rvolsel-info.oldrvol)/(float)total;

	//                  2

	Resampler::Mix_Data &mixdata=resampler.get_mixdata();

	Sint64 lend=loop_active?idxlpos:0;
	Sint64 rend=loop_active?idxlend:idxsize;

	while(todo>0) {

		Sint64 endpos;

		if ( info.playing_backwards ) {
			/* The sample is playing in reverse */
			if( ( loop_active )&&(info.current_index<idxlpos) ) {
				/* the sample is looping and has reached the loopstart index */
				if ( info.sample_data_ptr->is_loop_ping_pong() ) {
					/* sample is doing bidirectional loops, so 'bounce' the
					   current index against the idxlpos */
					info.current_index = idxlpos+(idxlpos-info.current_index);
					info.playing_backwards=false;
					info.increment_index = -info.increment_index;
				} else
					/* normal backwards looping, so set the current position to
					   loopend index */
					info.current_index=idxlend-(idxlpos-info.current_index);
			} else {
				/* the sample is not looping, so check if it reached index 0 */
				if(info.current_index < 0) {
					/* playing index reached 0, so stop playing this sample */
					info.current_index=0;
					info.active=false;

					break;
				}
			}
		} else {
			/* The sample is playing forward */
			if ( (loop_active) && (info.current_index >= idxlend)) {
				/* the sample is looping, check the loopend index */
				if( info.sample_data_ptr->is_loop_ping_pong() ) {
					/* sample is doing bidirectional loops, so 'bounce' the
					   current index against the idxlend */
					info.playing_backwards=true;
					info.increment_index = -info.increment_index;
					info.current_index = idxlend-(info.current_index-idxlend);
				} else
					/* normal backwards looping, so set the current position
					   to loopend index */
					info.current_index=idxlpos+(info.current_index-idxlend);
			} else {
				/* sample is not looping, so check if it reached the last
				   position */
				if(info.current_index >= idxsize) {
					/* yes, so stop playing this sample */
					info.current_index=0;
					info.active=false;

					break;
				}
			}
		}

		end=info.playing_backwards?lend:rend;

		done=MIN((end-info.current_index)/info.increment_index+1,todo);

		if ( done<=0 ) {
			info.active = 0;
			break;
		}

		endpos=info.current_index+done*info.increment_index;

//		if (info.volume || (info.oldlvol) || (info.oldrvol)) {

		if((info.current_index<0x7fffffff)&&(endpos<0x7fffffff)) {

			int l_vol_old,l_vol_new;
			int r_vol_old,r_vol_new;
			int written=total-todo;

			l_vol_old=info.oldlvol+lrintf((float)written*ramp_tangent_l);
			r_vol_old=info.oldrvol+lrintf((float)written*ramp_tangent_r);
			l_vol_new=info.oldlvol+lrintf((float)(written+done)*ramp_tangent_l);
			r_vol_new=info.oldrvol+lrintf((float)(written+done)*ramp_tangent_r);

			mixdata.sample=info.sample_data_ptr;
			mixdata.dst_buff=mixing_buffer_index;

			mixdata.sample_offset=info.current_index;
			mixdata.increment=info.increment_index;
			mixdata.samples_to_mix=done;
			mixdata.l_volume_prev=l_vol_old;
			mixdata.r_volume_prev=r_vol_old;
			mixdata.l_volume=l_vol_new;
			mixdata.r_volume=r_vol_new;

			mixdata.fractional_size=FRACTIONAL;


			resampler.set_filter(info.filter_enabled,info.filter_coef1,info.filter_coef2,info.filter_coef3);

			resampler.mix();

		}

                info.current_index=endpos;
		todo-=done;

		mixing_buffer_index += done*2;
	}

}


/*********************************/


void Tracker_Voice::setup_sample(Sample_Data *p_sample_data,int p_offset) {


	if (p_sample_data==NULL) {

		ERROR("p_sample_data is null!!");
		return;
	}

	if (p_offset<0) {
         	ERROR("start index < 0???");
		return;
	}

	if (p_offset>=p_sample_data->get_size()) {

		if (p_sample_data->is_loop_enabled()) {

			ERROR("Sample offset in loop not implemented!");

		} else {

			return;
		}
	}


	info.sample_data_ptr=p_sample_data;
	info.start_index=p_offset;
	info.restart();

}
bool Tracker_Voice::is_active() {

	return info.active;
}

void Tracker_Voice::set_frequency(Uint32 p_freq) {

	info.current_frequency=p_freq;
}
void Tracker_Voice::set_panning(int p_pan) {

	info.panning=p_pan;
}
void Tracker_Voice::set_volume(int p_vol) {

	info.volume=p_vol;
}
void Tracker_Voice::set_filter(Sint32 p_coef1, Sint32 p_coef2,Sint32 p_coef3,bool p_enabled) {


	info.filter_coef1=p_coef1;
	info.filter_coef2=p_coef2;
	info.filter_coef3=p_coef3;
	info.filter_enabled=p_enabled;

	if (!p_enabled) {

		info.filter_history1=0;
		info.filter_history2=0;
	}

}


bool Tracker_Voice::has_stopped() {


 	return was_removed_from_mixer();
}

void Tracker_Voice::reset() {

	preamp=1;
	mixfreq=1;
	current_prio=0;
	info.clear();
	clear_status();
	channel_index=0;

}


Tracker_Voice::Tracker_Voice()
  : Voice()
{

	preamp=1;
	mixfreq=1;
	current_prio=0;
	channel_index=0;
}


Tracker_Voice::~Tracker_Voice()
{
}
