/*-----------------------------------------------------------------------------

    File        : greycstoration.cpp

    Description : Implementation of several diffusion tensor-directed
                  diffusion PDE's for image regularization and interpolation,
		  including :

                  "Fast Anisotropic Smoothing of Multi-Valued Images 
		  using Curvature-Preserving PDE's"
		  (D. Tschumperl)
		  Research Report "Cahiers du GREYC 05/01", February 2005),
		  GREYC, CNRS UMR 6072
		  (see also http://www.greyc.ensicaen.fr/~dtschump/greycstoration)
		  
		  "Vector-Valued Image Regularization with PDE's : A Common Framework
		  for Different Applications"
		  (D. Tschumperl, R. Deriche).
		  IEEE Transactions on Pattern Analysis and Machine Intelligence,
		  Vol 27, No 4, pp 506-517, April 2005.

		  "Coherence-enhancing diffusion of colour images"
		  (J. Weickert)
		  Image and Vision Computing, Vol. 17, 201-212, 1999.		  

    Copyright  : David Tschumperle - http://www.greyc.ensicaen.fr/~dtschump/

  This software is governed by the CeCILL  license under French law and
  abiding by the rules of distribution of free software.  You can  use, 
  modify and/ or redistribute the software under the terms of the CeCILL
  license as circulated by CEA, CNRS and INRIA at the following URL
  "http://www.cecill.info". 
  
  As a counterpart to the access to the source code and  rights to copy,
  modify and redistribute granted by the license, users are provided only
  with a limited warranty  and the software's author,  the holder of the
  economic rights,  and the successive licensors  have only  limited
  liability. 
  
  In this respect, the user's attention is drawn to the risks associated
  with loading,  using,  modifying and/or developing or reproducing the
  software by the user in light of its specific status of free software,
  that may mean  that it is complicated to manipulate,  and  that  also
  therefore means  that it is reserved for developers  and  experienced
  professionals having in-depth computer knowledge. Users are therefore
  encouraged to load and test the software's suitability as regards their
  requirements in conditions enabling the security of their systems and/or 
  data to be ensured and,  more generally, to use and operate it in the 
  same conditions as regards security. 
  
  The fact that you are presently reading this means that you have had
  knowledge of the CeCILL license and that you accept its terms.

  ------------------------------------------------------------------------------*/

#include "../CImg.h"
// The line below is necessary for compilation with non-standart C++ compilers.
#if (( defined(_MSC_VER) && _MSC_VER<=1200 ) || defined(__DMC__))
#define std
#endif

using namespace cimg_library;
typedef unsigned char uchar;
unsigned int iter = 0, nb_iter = 0;

// get_geom() : read geometry from a string ('320x256' or '200%x200%' are valid geometries).
//------------
void get_geom(const char *geom, int &geom_w, int &geom_h) {
  char tmp[16];
  std::sscanf(geom,"%d%7[^0-9]%d%7[^0-9]",&geom_w,tmp,&geom_h,tmp+1);
  if (tmp[0]=='%') geom_w=-geom_w;
  if (tmp[1]=='%') geom_h=-geom_h;
}

// Blur an image with Curvature-preserving PDE's.
//-----------------------------------------------
template<typename T,typename t> 
CImg<T>& blur_curvaturePDE(CImg<T>& img, const CImg<t>& G, const float strength=10.0f,
			   const float dl=0.8f,const float da=45.0f,
			   const float gauss_prec=2.0f, const bool linear=false) {
  
  const int dx1 = img.dimx()-1, dy1 = img.dimy()-1;
  CImg<t> dest(img.width,img.height,img.depth,img.dim,0), tmp(img.dim), W(img.width,img.height,img.depth,2);
  int N = 0, counter = 0, countermax = (int)(img.width*img.height*(360/da));
  
  for (float theta=(360%(int)da)/2.0f; theta<360; (theta+=da),N++) {
    const float thetar = (float)(theta*cimg::PI/180), vx = (float)std::cos(thetar), vy = (float)std::sin(thetar);
    
    const t *pa = G.ptr(0,0,0,0), *pb = G.ptr(0,0,0,1), *pc = G.ptr(0,0,0,2);
    t *pd0 = W.ptr(0,0,0,0), *pd1 = W.ptr(0,0,0,1);
    cimg_mapXY(G,xg,yg) {
      const t
	a = *(pa++), b = *(pb++), c = *(pc++), 
	u = a*vx + b*vy, v = b*vx + c*vy,
	n = (float)(dl/std::sqrt(1e-5+u*u+v*v));
      *(pd0++) = u*n;
      *(pd1++) = v*n;
    }
    
    cimg_mapXY(img,x,y) {
      if (!((counter++)%20000))
	std::fprintf(stderr,"\r- Processing : %u %%    \t\t",
		     (int)(100.0*(counter+iter*countermax)/(nb_iter*countermax)));
      
      tmp.fill(0);
      const t cu = W(x,y,0,0), cv = W(x,y,0,1);
      const float
	fsigma = (float)(2*std::sqrt((cu*cu+cv*cv)*strength)),
	length = gauss_prec*fsigma,
	fsigma2 = 2*fsigma*fsigma;
      float l, S=0, pu=cu, pv=cv, X=(float)x, Y=(float)y;
      if (linear) for (l=0; l<length; l+=dl) {
	const float coef = (float)std::exp(-l*l/fsigma2);
	t u = (t)(W.linear_pix2d(X,Y,0,0)), v = (t)(W.linear_pix2d(X,Y,0,1));
	if ((pu*u+pv*v)<0) { u=-u; v=-v; }
	cimg_mapV(img,k) tmp[k]+=(t)(coef*img.linear_pix2d(X,Y,0,k));
	X+=(pu=u); Y+=(pv=v); S+=coef;
      } else for (l=0; l<length; l+=dl) {
	const float 
	  coef = (float)std::exp(-l*l/fsigma2),
	  Xn = X<0?0:(X>=dx1?dx1:X),
	  Yn = Y<0?0:(Y>=dy1?dy1:Y);	  
	const int xi = (int)(Xn+0.5f), yi = (int)(Yn+0.5f);
	t u = W(xi,yi,0,0), v = W(xi,yi,0,1);
	if ((pu*u+pv*v)<0) { u=-u; v=-v; }
	cimg_mapV(img,k) tmp[k]+=(t)(coef*(img)(xi,yi,0,k));
	X+=(pu=u); Y+=(pv=v); S+=coef;
      }
      if (S>0) cimg_mapV(dest,k) dest(x,y,0,k)+=tmp[k]/S;
      else cimg_mapV(dest,k) dest(x,y,0,k)+=(t)((img)(x,y,0,k));
    }
  }
  const float *ptrs = dest.data+dest.size(); cimg_map(img,ptrd,T) *ptrd = (T)(*(--ptrs)/N);
  return img;
}

// Blur an image with Trace-based PDE's.
//--------------------------------------
template<typename T,typename t> 
CImg<T>& blur_tracePDE(CImg<T>& img, const CImg<t>& G, const float dt=10.0f) {
  CImg<T> veloc(img);
  CImg_3x3(I,float);
  int counter = 0, countermax = (int)(img.width*img.height);
  
  cimg_mapV(img,k) cimg_map3x3(img,x,y,0,k,I) {
    if (!((counter++)%20000))
      std::fprintf(stderr,"\r- Processing : %u %%    \t\t",
		   (int)(100.0*(counter+iter*countermax)/(nb_iter*countermax)));
    const float
      ixx = Inc+Ipc-2*Icc,
      iyy = Icn+Icp-2*Icc,
      ixy = 0.25f*(Inn+Ipp-Inp-Ipn);
    veloc(x,y,0,k) = G(x,y,0)*ixx + 2*G(x,y,1)*ixy + G(x,y,2)*iyy;
  }
  const CImgStats stats(veloc,false);
  const float xdt = dt/cimg::max(cimg::abs(stats.min),cimg::abs(stats.max));
  veloc*=xdt;
  veloc*=dt;
  img+=veloc;
  return img;
}

// Blur an image with divergence-based PDE's.
//-------------------------------------------
template<typename T,typename t> 
CImg<T>& blur_divergencePDE(CImg<T>& img, const CImg<t>& G, const float dt=10.0f) {
  CImg<T> veloc(img);
  CImg_3x3(I,float);
  int counter = 0, countermax = (int)(img.width*img.height);
  
  cimg_mapV(img,k) cimg_map3x3(img,x,y,0,k,I) {
    if (!((counter++)%20000))
      std::fprintf(stderr,"\r- Processing : %u %%    \t\t",
		   (int)(100.0*(counter+iter*countermax)/(nb_iter*countermax)));
    const float
      a = G(x,y,0), b = G(x,y,1), c = G(x,y,2),
      anc = 0.5f*(a + G(_nx,y,0)),
      apc = 0.5f*(a + G(_px,y,0)),
      bnc = 0.5f*(b + G(_nx,y,1)),
      bpc = 0.5f*(b + G(_px,y,1)),
      bcn = 0.5f*(b + G(x,_ny,1)),
      bcp = 0.5f*(b + G(x,_py,1)),
      ccn = 0.5f*(c + G(x,_ny,2)),
      ccp = 0.5f*(c + G(x,_py,2)),
      unc = anc*(Inc-Icc) + bnc*0.5f*(Icn+Inn-Icp-Inp),
      upc = apc*(Icc-Ipc) + bpc*0.5f*(Icn+Ipc-Icp-Ipp),
      vcn = bcn*0.5f*(Inc+Inn-Ipc-Ipn) + ccn*(Icn-Icc),
      vcp = bcp*0.5f*(Inc+Inp-Ipc-Ipp) + ccp*(Icc-Icp);
    veloc(x,y,0,k) = unc-upc + vcn-vcp;
  }
  const CImgStats stats(veloc,false);
  const float xdt = dt/cimg::max(cimg::abs(stats.min),cimg::abs(stats.max));
  veloc*=xdt;
  veloc*=dt;
  img+=veloc;
  return img;
}

/*-----------------
  Main procedure
  ----------------*/
int main(int argc,char **argv) {
  cimg_usage("Fast Anisotropic Smoothing of Multi-valued Images with Curvature-Preserving PDE's");
  std::fprintf(stderr,
	       "---------------------------------------------------------------------------------------------\n"
	       " Note on using GREYCstoration :\n"
	       " ------------------------------\n"
	       " The GREYCstoration algorithm is the result of a research work done in the\n"
	       " IMAGE group of the GREYC Lab (CNRS, UMR 6072)/France, by David Tschumperle\n"
	       " ( http://www.greyc.ensicaen.fr/EquipeImage/ and http://www.greyc.ensicaen.fr/~dtschump/ ).\n"
	       " It can be used to regularize, inpaint or resize 2D color images.\n"
	       " This algorithm is distributed to help people for processing image data.\n"
	       " This is an open source software, distributed within the CImg Library package\n"
	       " (http://cimg.sourceforge.net), and submitted to the CeCiLL License (view file LICENSE.txt).\n"
	       " If you are interested to distribute this algorithm in a closed-source product,\n"
	       " you are invited to contact David Tschumperle (mail available on his web page).\n"
	       " Please also cite the related paper when using results of the GREYCstoration\n"
	       " algorithm in your own publications :\n\n"
	       "    \"Fast Anisotropic Smoothing of Multi-Valued Images using Curvature-Preserving PDE's\"\n"
	       "    (D. Tschumperle)\n"
	       "    GREYC, CNRS UMR 6072, Equipe Image, 6 Bd du Marechal Juin, 14050 Caen Cedex.\n\n"
	       " You can also take a look at the official GREYCstoration web page :\n"
	       "    http://www.greyc.ensicaen.fr/~dtschump/greycstoration\n"
	       "---------------------------------------------------------------------------------------------\n");

  // Read command line parameters
  const char *restore  = cimg_option("-restore",(const char*)NULL,"Restore the image specified after '-restore'");
  const char *inpaint  = cimg_option("-inpaint",(const char*)NULL,"Inpaint the image specified after '-inpaint'");
  const char *resize   = cimg_option("-resize",(const char*)NULL, "Resize the image specified after '-resize'");

  if (argc==1) {
    cimg::dialog(cimg::basename(argv[0]),"Try option '-h' for the list of available options.");
    std::exit(0);
  }
  if (!restore && !inpaint && !resize) {
    cimg::dialog(cimg::basename(argv[0]),
		 "You must specify one of the '-restore','-inpaint',or '-resize' flags.\n"
		 "(try option '-h')\n");
    std::exit(0);
  }
  
  // Init variables
  //----------------
  CImg<> img;
  CImg<uchar> mask;
  float dt=0,p1=0,anisotropy=0,alpha=0,sigma=0;
   
  // Specific parameters for image restoration
  if (restore) { 
    dt         = (float)cimg_option("-dt",20.0f,"Smoothing strength");
    p1         = (float)cimg_option("-p",2.0f,"Contour preservation");
    anisotropy = (float)cimg_option("-a",0.7f,"Smoothing anisotropy");
    alpha      = (float)cimg_option("-alpha",0.3f,"Noise scale");
    sigma      = (float)cimg_option("-sigma",0.8f,"Geometry regularity");
    nb_iter    = cimg_option("-iter",1,"Number of iterations");
    const float ng = (float)cimg_option("-ng",0.0f,"Add gaussian noise before denoising");
    const float nu = (float)cimg_option("-nu",0.0f,"Add uniform noise before denoising");
    const float ns = (float)cimg_option("-ns",0.0f,"Add Salt&Pepper noise before denoising");

    img = CImg<>(restore);
    CImgStats stats(img,false);
    img.noise(ng,0).noise(nu,1).noise(ns,2).cut((float)stats.min,(float)stats.max);
  }
  
  // Specific parameters for image inpainting
  if (inpaint) {
    dt         = (float)cimg_option("-dt",30.0f,"Smoothing strength");
    p1         = (float)cimg_option("-p",0.1f,"Contour preservation");
    anisotropy = (float)cimg_option("-a",0.8f,"Smoothing anisotropy");
    alpha      = (float)cimg_option("-alpha",0.1f,"Noise scale");
    sigma      = (float)cimg_option("-sigma",2.0f,"Geometry regularity");
    nb_iter    = cimg_option("-iter",100,"Number of iterations");
    const char *file_m         = cimg_option("-m",(const char*)NULL,"Input inpainting mask");
    const unsigned int dilate  = cimg_option("-dilate",0,"Inpainting mask dilatation");
    const unsigned int ip_init = cimg_option("-init",2,"Inpainting init (0=black, 1=white, 2=noise, 3=unchanged, 4=interpol)");

    img = CImg<>(inpaint);   
    if (!file_m) { cimg::dialog("GREYCstoration","You need to specify an inpainting mask (option '-m')","OK"); std::exit(0); }
    if (cimg::strncasecmp("block",file_m,5)) mask = CImg<uchar>(file_m);
    else {
      int l=16; std::sscanf(file_m,"block%d",&l);
      mask = CImg<uchar>(img.dimx()/l,img.dimy()/l);
      cimg_mapXY(mask,x,y) mask(x,y)=(x+y)%2;
    }
    mask.resize(img.dimx(),img.dimy(),1,1);
    if (dilate) mask.dilate(dilate);
    switch (ip_init) {
    case 0 : { cimg_mapXYV(img,x,y,k) if (mask(x,y)) img(x,y,k) = 0; } break;
    case 1 : { cimg_mapXYV(img,x,y,k) if (mask(x,y)) img(x,y,k) = 255; } break;
    case 2 : { cimg_mapXYV(img,x,y,k) if (mask(x,y)) img(x,y,k) = (float)(255*cimg::rand()); } break;
    case 3 : break;
    default: {
      CImg<uchar> tmask(mask),ntmask(tmask);
      CImg_3x3(M,uchar);
      CImg_3x3(I,float);
      while (CImgStats(ntmask,false).max>0) {
	cimg_map3x3(tmask,x,y,0,0,M) if (Mcc && (!Mpc || !Mnc || !Mcp || !Mcn)) {
	  const float ccp = Mcp?0.0f:1.0f, cpc = Mpc?0.0f:1.0f,
	    cnc = Mnc?0.0f:1.0f, ccn = Mcn?0.0f:1.0f, csum = ccp + cpc + cnc + ccn;
	  cimg_mapV(img,k) {
	    cimg_get3x3(img,x,y,0,k,I);
	    img(x,y,k) = (ccp*Icp + cpc*Ipc + cnc*Inc + ccn*Icn)/csum;
	  }
	  ntmask(x,y) = 0;
	}
	tmask = ntmask;
      }
    } break;    
    }
  }
  
  // specific parameters for image resizing
  if (resize) {
    dt         = (float)cimg_option("-dt",3.0f,"Smoothing strength");
    p1         = (float)cimg_option("-p",0.1f,"Contour preservation");
    anisotropy = (float)cimg_option("-a",1.0f,"Smoothing anisotropy");
    alpha      = (float)cimg_option("-alpha",0.1f,"Noise scale");
    sigma      = (float)cimg_option("-sigma",1.0f,"Geometry regularity");
    nb_iter    = cimg_option("-iter",3,"Number of iterations");
    const char *geom  = cimg_option("-g",(const char*)NULL,"Output image geometry");
    const bool anchor = cimg_option("-anchor",true,"Anchor original pixels");
    if (!geom) { cimg::dialog("GREYCstoration","You need to specify an output geomety (option -g)","OK"); std::exit(0); }
    int w,h; get_geom(geom,w,h);
    img = CImg<>(resize);
    mask = CImg<uchar>(img.dimx(),img.dimy(),1,1,255);
    if (!anchor) mask.resize(w,h,1,1,1); else mask = ~mask.resize(w,h,1,1,4);
    img.resize(w,h,1,-100,5);
  }

  // Read implementation-specific parameters
  //-----------------------------------------
  float dl,da,gauss_prec;
  bool linear;
  const bool improved_st   = cimg_option("-improve",true,"Improve structure tensor computation");
  const unsigned int save  = cimg_option("-save",0,"Iteration saving step");
  const bool visu          = cimg_option("-visu",true,"Enable/Disable visualization");
  const char *file_o       = cimg_option("-o",(const char*)NULL,"Output image");
  const bool append_result = cimg_option("-append",false,NULL);
  const float p2           = (float)((1+anisotropy)/(1e-10+1-anisotropy)*p1); 
  const int scheme         = cimg_option("-scheme",0,"Smoothing scheme (0=Curvature-preserving PDE, "
					 "1=Trace-based PDE, or 2=Divergence-based PDE)");
  switch(scheme) {
  case 0:    
    dl           = (float)cimg_option("-dl",0.8f,"Spatial integration step");
    da           = (float)cimg_option("-da",45.0f,"Angular integration step (in degrees)");
    gauss_prec   = (float)cimg_option("-prec",2.0f,"Precision of the gaussian function");
    linear       = cimg_option("-linear",false,"Use linear interpolation for integration");
    break;
  default:
    break;
  }

  // Init images and display  
  //-------------------------
  CImg<> dest(img),G(dest.dimx(),dest.dimy(),1,3),val,vec;
  CImgDisplay *disp=NULL;
  if (visu) disp = new CImgDisplay(dest,"Processing");
  std::fprintf(stderr,"\n");

  //-------------------------------------
  // Begin regularization PDE iterations
  //-------------------------------------
  for (iter=0; iter<nb_iter && (!disp || !disp->closed); iter++) {
    
    // Compute smoothed structure tensor field G
    //-------------------------------------------
    CImg_3x3(I,float);
    G.fill(0);
    CImg<> blur = dest.get_blur(alpha);
  
    if (improved_st) cimg_mapV(img,k) cimg_map3x3(blur,x,y,0,k,I) {
      const float ixf = Inc-Icc, iyf = Icn-Icc, ixb = Icc-Ipc, iyb = Icc-Icp;
      G(x,y,0) += 0.5f*(ixf*ixf+ixb*ixb);
      G(x,y,1) += 0.25f*(ixf*iyf+ixf*iyb+ixb*iyf+ixb*iyb);
      G(x,y,2) += 0.5f*(iyf*iyf+iyb*iyb);
    } else cimg_mapV(img,k) cimg_map3x3(blur,x,y,0,k,I) {
      const float ix = 0.5f*(Inc-Ipc), iy = 0.5f*(Icn-Icp);
      G(x,y,0) += ix*ix;
      G(x,y,1) += ix*iy;
      G(x,y,2) += iy*iy;
    }
    G.blur(sigma);

    // Compute tensor field sqrt(T) in G
    //----------------------------------
    float lmax = 0;  
    cimg_mapXY(G,x,y) {
      G.get_tensor(x,y).symeigen(val,vec);
      const float
	l1 = val[1],
	l2 = val[0],
	u = vec(1,0),
	v = vec(1,1),      
	n1 = (float)(1.0/std::pow(1.0f+l1+l2,0.5f*p1)),
	n2 = (float)(1.0/std::pow(1.0f+l1+l2,0.5f*p2));     
      G(x,y,0) = n1*u*u + n2*v*v;
      G(x,y,1) = u*v*(n1-n2);
      G(x,y,2) = n1*v*v + n2*u*u;
      if (n1>lmax) lmax=n1;
    }    
    G/=lmax;   
    if (mask.data) cimg_mapXY(G,x,y) if (!mask(x,y)) G(x,y,0)=G(x,y,1)=G(x,y,2)=0;

    // Apply an anisotropic smoothing following G
    //-------------------------------------------
    switch (scheme) {
    case 0: // Curvature-preserving PDE
      blur_curvaturePDE(dest,G,dt,dl,da,gauss_prec,linear);
      break;
    case 1: // Trace-based PDE
      blur_tracePDE(dest,G,dt);
      break;
    case 2: // Divergence-based PDE
      blur_divergencePDE(dest,G,dt);
      break;
    }
          
    // Prepare for next iteration
    //---------------------------
    if (visu) dest.display(*disp);
    if (file_o && save && !(iter%save)) dest.save(file_o,iter);
  }
  
  // Save result and end program
  //-----------------------------
  std::fprintf(stderr,"\r- Processing : Done !\n");
  if (file_o) {
    std::fprintf(stderr,"- Saving output image '%s'\n",file_o);
    if (!append_result) dest.save(file_o);
    else CImgl<>(img,dest).get_append('x').save(file_o);
  }
  if (disp) {
    disp->resize(-200,-100);
    CImgl<>(img,dest).get_append('x').feature_selection(NULL,1,*disp);
  }
    
  return 0;
}
