/*-------------------------------------------------------------------------------

  File        : CImg_demo.cpp

  Description : A multi-part demo demonstrating some of the CImg capabilities

  Copyright  : David Tschumperle - http://www.greyc.ensicaen.fr/~dtschump/

  This software is governed by the CeCILL-C license under French law and
  abiding by the rules of distribution of free software.  You can  use,
  modify and/ or redistribute the software under the terms of the CeCILL-C
  license as circulated by CEA, CNRS and INRIA at the following URL
  "http://www.cecill.info".

  As a counterpart to the access to the source code and  rights to copy,
  modify and redistribute granted by the license, users are provided only
  with a limited warranty  and the software's author,  the holder of the
  economic rights,  and the successive licensors  have only  limited
  liability.

  In this respect, the user's attention is drawn to the risks associated
  with loading,  using,  modifying and/or developing or reproducing the
  software by the user in light of its specific status of free software,
  that may mean  that it is complicated to manipulate,  and  that  also
  therefore means  that it is reserved for developers  and  experienced
  professionals having in-depth computer knowledge. Users are therefore
  encouraged to load and test the software's suitability as regards their
  requirements in conditions enabling the security of their systems and/or
  data to be ensured and,  more generally, to use and operate it in the
  same conditions as regards security.

  The fact that you are presently reading this means that you have had
  knowledge of the CeCILL-C license and that you accept its terms.

  ------------------------------------------------------------------------------*/

#include "../CImg.h"
using namespace cimg_library;

// Routine used to compute point depth in Mandelbrot fractals
inline int mandelbrot(const double x, const double y, const int maxiter) {
  static double old_x, fx, fy;
  static int m;
  fx = fy = m = 0;
  do { old_x = fx; fx = fx*fx-fy*fy+x; fy = 2*old_x*fy+y; m++; }
  while (((fx*fx+fy*fy)<4) && (m<maxiter));
  return m;
}

// Class used by the metaballs demo
struct metaballs3d {
  float cx1,cy1,cz1,cx2,cy2,cz2,cx3,cy3,cz3;
  inline float operator()(const float x,const float y,const float z) const {
    const float
      x1 = x-cx1, y1 = y-cy1, z1 = z-cz1,
      x2 = x-cx2, y2 = y-cy2, z2 = z-cz2,
      x3 = x-cx3, y3 = y-cy3, z3 = z-cz3,
      r1 = 0.3f*(x1*x1 + y1*y1 + z1*z1),
      r2 = 0.4f*(x2*x2 + y2*y2 + z2*z2),
      r3 = 0.5f*(x3*x3 + y3*y3 + z3*z3);
    float potential = 0;
    if (r1<1.3) potential+= 1.0f - r1*(r1*(4*r1+17)-22)/9;
    if (r2<1.3) potential+= 1.0f - r2*(r2*(4*r2+17)-22)/9;
    if (r3<1.3) potential+= 1.0f - r3*(r3*(4*r3+17)-22)/9;
    return potential;
  }
};

/*---------------------------

  Main procedure

  --------------------------*/
int main() {

  // Display info about the CImg Library configuration
  //--------------------------------------------------
  cimg::info();

  // Demo selection menu
  //---------------------
  const unsigned char white[3] = {255,255,255}, red[3]    = {120,50,80},
		     yellow[3] = {200,155,0},   green[3]  = {30,200,70};
  float t=0, rx=(float)(2*cimg::crand()), ry=(float)(2*cimg::crand());
  int demo_number,y0=2*13;
  bool stopflag = false;

  CImg<unsigned char> back(1,2,1,3,10), img;
  back(0,1,2) = 235;
  back.resize(320,390,1,3,3).get_shared_channel(2).noise(10,1).draw_plasma();
  back.draw_rectangle(0,y0,back.dimx()-1,y0+12,red).
    draw_text(20,y0,white,0,13,1,
	      "=> CImg %g Demos : \n\n"
	      "- Blurring Gradient\n"
	      "- Rotozoom\n"
	      "- Image Denoising\n"
	      "- Fractal Animation\n"
	      "- Gamma Correction and Histograms\n"
	      "- Filled Triangles\n"
	      "- Mandelbrot explorer\n"
	      "- Mini-Paint\n"
	      "- Soccer Bobs\n"
	      "- Bump Effect\n"
	      "- Bouncing Bubble\n"
	      "- Virtual Landscape\n"
	      "- Plasma & Sinus Scroll\n"
	      "- Oriented Convolutions\n"
	      "- Shade Bobs\n"
	      "- Fourier Filtering\n"
	      "- Image Zoomer\n"
	      "- Blobs Editor\n"
	      "- Double Torus\n"
	      "- 3D Metaballs\n"
	      "- Fireworks\n"
	      "- Rubber Logo\n"
	      "- Image Waves\n"
              "- Breakout\n",
	      cimg_version);

  CImgDisplay disp(back,"CImg Demo Menu",0,3,false,true);
  disp.move((CImgDisplay::screen_dimx()-disp.window_width)/2,
	    (CImgDisplay::screen_dimy()-disp.window_height)/2);
  img=back; back*=0.1;
  for(y0+=2*13; !stopflag; ) {
    while(!(disp.button&1) && !stopflag) {
      if (disp.is_closed || disp.key == cimg::keyQ || disp.key==cimg::keyESC) stopflag=true;
      img*=0.9; img+=back;
      int y = disp.mouse_y;
      if (y>=y0 && y<y0+24*13) {
        y = (y/13)*13+7;
        for (int yy=y-7; yy<=y+6; yy++) img.draw_rectangle(0,yy,0,1,img.dimx()-1,yy,0,1,(unsigned char)(130-15*cimg::abs(yy-y)));
        img.draw_triangle(2,y-4,2,y+4,8,y,yellow).draw_triangle(img.dimx()-2,y-4,img.dimx()-2,y+4,img.dimx()-8,y,yellow);
      }
      for (int i=0; i<20; i++) {
        const int
          mx = (int)(img.dimx()/2+(img.dimx()/2-30)*cos(3*t+rx*i*36*cimg::PI/180)),
          my = (int)(img.dimy()/2+(img.dimy()/2-30)*sin(4*t+ry*i*36*cimg::PI/180));
	const float
	  mz = (float)(1.0f+0.4f*sin(2*t+(rx+ry)*i*20*cimg::PI/180));
	img.draw_circle(mx,my,10*mz,i%3?(i%2?green:red):yellow,0L,0.2f).draw_circle((int)(mx+3*mz),my-3,3*mz,white,0L,0.1f);
      }
      if ((t+=0.007f)>1) { rx=(float)(2*cimg::crand()); ry=(float)(2*cimg::crand()); t=0; }
      disp.resize(disp,false).display(img).wait(25);
    }
    disp.button=0;
    if (!stopflag) demo_number = (disp.mouse_y-y0)/13; else break;

    switch(demo_number) {
    case 0: {

      // Blurring gradient demo
      //------------------------
      const CImg<> src("img/milla.ppm");
      CImgList<> grad = src.get_gradientXY(3);
      CImgList<unsigned char> visu = src<<((grad[0].pow(2))+(grad[1].pow(2))).sqrt().normalize(0,255)<<src;
      CImgDisplay disp(visu,"Image, Gradient Norm and Blurring Gradient",0,1);
      for (double sigma=0; !disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC; sigma+=0.05) {
	visu[2] = visu[1].get_blur((float)cimg::abs(30*cos(sigma))).normalize(0,255);
	disp.resize(false).display(visu).wait(20);
      }
    } break;

    case 1: {

      // Rotozoom demo
      //---------------
      CImg<unsigned char> src = CImg<unsigned char>("img/milla.ppm").resize(400,300,1,3,3), img(src);
      CImgDisplay disp(img,"Rotozoom Demo");
      float alpha=0, t=0, angle=0, zoom0=-0.5f;
      unsigned char color[3]={0,0,0};
      while(!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	cimg_forYV(src,y,k) {
	  const int xc = 4*src.dimx()+(int)(60*sin((float)y*3/src.dimy()+10*t));
	  cimg_forX(src,x) {
	    const float val = (float)(src((xc+x)%src.dimx(),y,0,k)*
				      (1.3f+0.20*sin(alpha+k*k*((float)src.dimx()/2-x)*
						     ((float)src.dimy()/2-y)*cos(t)/300.0)));
	    img(x,y,0,k) = (unsigned char)(val>255.0f?255:val);
	  }
	}
	const float zoom=(float)(zoom0+0.3*(1+cos(3*t)));
	img.get_rotate(angle,0.5f*img.dimx(),0.5f*img.dimy(),1+zoom,1).
	  draw_text("Mouse buttons\nto zoom in/out",3,3,color,0,24,1.0f).display(disp);
	alpha+=0.7f;
	t+=0.01f;
	angle+=0.8f;
	if (disp.button&1) zoom0+=0.10f;
	if (disp.button&2) zoom0-=0.10f;
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(), disp.screen_dimy(), false);
	  else disp.resize(img,false);
	}
	disp.resize(false).wait(20);
      }
    } break;

    case 2: {

      // Image denoising (Total variation PDE)
      //---------------------------------------
      const CImg<> src = CImg<>("img/milla.ppm").noise(-3,2).noise(-10,1);
      CImgList<> images(src,src);
      CImgDisplay disp(images,"Image denoising",1,1);
      float white[3]={255,255,255}, black[3]={0,0,0};
      CImg_3x3(I,float);
      for (unsigned int iter=0; !disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC; iter++) {
	CImg<> veloc(src);
	cimg_forV(src,k) cimg_for3x3(images[1],x,y,0,k,I) {
	  const float ix = (Inc-Ipc)/2, iy = (Icn-Icp)/2, ng = ix*ix + iy*iy;
	  veloc(x,y,k) = (ng>1e-14)?( ix*ix * ( Icn + Icp - 2*Icc ) + iy*iy * ( Inc + Ipc - 2*Icc ) - ix*iy * ( Ipp + Inn - Ipn - Inp )/2 )/ng:0;
	}
	images[1]+=veloc*0.02;
	images[0].draw_text(0,0,white,black,11,1,"iter %u",iter++);
	disp.resize(false).display(images);
      }
    } break;

    case 3: {

      // Fractal-like animation
      //------------------------
      CImg<unsigned char> img(256,256,1,3,0),noise(3,2,1,3);
      CImgDisplay disp(img,"Super Zoomator",0,3);
      double zoom=0;
      for (unsigned int iter=0; !disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC; iter++,zoom+=0.2) {
	img.draw_image(noise.fill(0).noise(255,1),(img.dimx()-noise.dimx())/2,(img.dimy()-noise.dimy())/2);
	img.rotate((float)(10*sin(iter/25.0)),0.5f*img.dimx(),0.5f*img.dimy(),(float)(1.04+0.02*sin(zoom/10)),0);
	disp.resize(img.resize(disp.window_dimx(),disp.window_dimy()),false).display(img).wait(25);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(),disp.screen_dimy(),false);
	  else disp.resize(img,false);
	}
      }
    } break;

    case 4: {

      // Gamma correction and histogram visualization
      //----------------------------------------------
      CImg<> img = CImg<>("img/milla.ppm").normalize(0,1);
      CImgList<unsigned char> visu(img*255, CImg<unsigned char>(512,300,1,3,0));
      const unsigned char yellow[3] = {255,255,0}, blue[3]={0,155,255}, blue2[3]={0,0,255}, blue3[3]={0,0,155}, white[3]={255,255,255};
      CImgDisplay disp(visu,"Image and Histogram (Mouse click to set the Gamma correction)");
      for (double gamma=1;!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC; ) {
	cimg_forXYZV(visu[0],x,y,z,k) visu[0](x,y,z,k) = (unsigned char)(pow((double)img(x,y,z,k),1.0/gamma)*256);
	const CImg<> hist = visu[0].get_histogram(50,0,255);
	visu[1].fill(0).draw_text(50,5,white,0,24,1,"Gamma = %g",gamma).
	  draw_graph(hist,yellow,1,20000,0).draw_graph(hist,white,2,20000,0).
	  draw_axis(0,256,20000,0,white,-60,-60,0,0,0.7f);
	const int xb = (int)(50+gamma*150);
        visu[1].draw_grid(20,20,0,0,white,0xCCCCCCCC,0xCCCCCCCC,false,false,0.3f);
	visu[1].draw_rectangle(51,31,xb-1,39,blue2).draw_rectangle(50,30,xb,30,blue).draw_rectangle(xb,30,xb,40,blue);
	visu[1].draw_rectangle(xb,40,50,39,blue3).draw_rectangle(50,30,51,40,blue3);
	if (disp.button && disp.mouse_x>=img.dimx()+50 && disp.mouse_x<=img.dimx()+450) gamma = (disp.mouse_x-img.dimx()-50)/150.0;
	disp.resize(disp,false).display(visu).wait();
      }
    } break;

    case 5: {

      // Filled triangle demos
      //-----------------------
      CImg<> background(640,480,1,3);
      cimg_forXY(background,x,y) {
	background(x,y,0) = (float)(x*cos(6.0*y/background.dimy())+y*sin(9.0*x/background.dimx()));
	background(x,y,1) = (float)(x*sin(8.0*y/background.dimy())-y*cos(11.0*x/background.dimx()));
	background(x,y,2) = (float)(x*cos(13.0*y/background.dimy())-y*sin(8.0*x/background.dimx()));
      }
      background.normalize(0,180);
      CImg<unsigned char> img0(background), img;
      unsigned char white[3] = { 255,255,255 }, color[100][3];
      CImgDisplay disp(img0,"Filled Triangle Demo",0,3);

      float posx[100],posy[100],rayon[100],angle[100],veloc[100],opacity[100];
      int num=1;
      srand((unsigned int)time(NULL));
      for (int k=0; k<100; k++) {
	posx[k]  = (float)(cimg::rand()*img0.dimx());
	posy[k]  = (float)(cimg::rand()*img0.dimy());
	rayon[k] = (float)(10+cimg::rand()*50);
	angle[k] = (float)(cimg::rand()*360);
	veloc[k] = (float)(cimg::rand()*20-10);
	color[k][0] = (unsigned char)(cimg::rand()*255);
	color[k][1] = (unsigned char)(cimg::rand()*255);
	color[k][2] = (unsigned char)(cimg::rand()*255);
	opacity[k] = (float)(0.3+1.5*cimg::rand());
      }
      while(!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	img = img0;
	for (int k=0; k<num; k++) {
	  const int
	    x0 = (int)(posx[k]+rayon[k]*cos(angle[k]*cimg::PI/180)),
	    y0 = (int)(posy[k]+rayon[k]*sin(angle[k]*cimg::PI/180)),
	    x1 = (int)(posx[k]+rayon[k]*cos((angle[k]+120)*cimg::PI/180)),
	    y1 = (int)(posy[k]+rayon[k]*sin((angle[k]+120)*cimg::PI/180)),
	    x2 = (int)(posx[k]+rayon[k]*cos((angle[k]+240)*cimg::PI/180)),
	    y2 = (int)(posy[k]+rayon[k]*sin((angle[k]+240)*cimg::PI/180));
	  if (k%10) img.draw_triangle(x0,y0,x1,y1,x2,y2,color[k],opacity[k]);
	  else img.draw_triangle(x0,y0,x1,y1,x2,y2,img0,0,0,img0.dimx()-1,0,0,img.dimy()-1,opacity[k]);
	  img.draw_line(x0,y0,x1,y1,white,(unsigned long)~0,opacity[k]).
	    draw_line(x1,y1,x2,y2,white,(unsigned long)~0,opacity[k]).
	    draw_line(x2,y2,x0,y0,white,(unsigned long)~0,opacity[k]);
	  angle[k]+=veloc[k];
	  if (disp.button && disp.mouse_x>0 && disp.mouse_y>0) {
	    float u = disp.mouse_x-posx[k], v = disp.mouse_y-posy[k];
	    if (disp.button&2) { u=-u; v=-v; }
	    posx[k]-=0.1f*u, posy[k]-=0.1f*v;
	    if (posx[k]<0 || posx[k]>=img.dimx()) posx[k]=(float)(cimg::rand()*img.dimx());
	    if (posy[k]<0 || posy[k]>=img.dimy()) posy[k]=(float)(cimg::rand()*img.dimy());
	  }
	}
	img.draw_text(5,5,white,0,11,0.5f,"%u frames/s",(unsigned int)disp.frames_per_second());
	disp.display(img).resize(false).wait(20);
	img0.resize(disp);
	num++; if (num>100) num=100;
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(),disp.screen_dimy(),false);
	  else disp.resize(img,false);
	}
      }
    } break;

    case 6: {

      // Mandelbrot explorer
      //----------------------
      CImg<unsigned char> img(800,600,1,3,0);
      CImg<unsigned int> iterimg(img.dimx(),img.dimy());
      CImgDisplay disp(img,"Mandelbrot Explorer");
      unsigned char color[3] = { 255,255,255 };
      for (bool endflag=false; !endflag;) {
	bool stopflag = false;
	double xmin=-2.25, xmax=1, ymin=-1.5, ymax=1.5;
	const unsigned int cr = (2+(int)(8*cimg::rand()))*4;
	const unsigned int cg = (2+(int)(8*cimg::rand()))*4;
	const unsigned int cb = (2+(int)(8*cimg::rand()))*4;
	unsigned int maxiter=64;

	while(!stopflag) {
	  if (disp.is_resized) { disp.resize(); iterimg.resize(disp,0); img.resize(disp,0); }
	  const double dx=(xmax-xmin)/img.dimx();
	  const double dy=(ymax-ymin)/img.dimy();
	  double y = ymin;
	  cimg_forY(img,py) {
	    double x = xmin;
	    cimg_forX(img,px) {
	      const unsigned int m = mandelbrot(x,y,maxiter);
	      iterimg(px,py) = m;
	      img(px,py,0)=(unsigned char)(cr*m);
	      img(px,py,1)=(unsigned char)(cg*m);
	      img(px,py,2)=(unsigned char)(cb*m);
	      x+=dx;
	    }
	    y+=dy;
	  }
	  int selection[6];
	  img.feature_selection(selection,2,disp,NULL,color);
	  if (selection[0]>=0 && selection[1]>=0 && selection[3]>=0 && selection[4]>=0) {
	    if (selection[3]-selection[0]<5 || selection[4]-selection[1]<5) stopflag=true;
	    xmax = xmin + selection[3]*dx; ymax = ymin + selection[4]*dy;
	    xmin = xmin + selection[0]*dx; ymin = ymin + selection[1]*dy;
	  }
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF))
	    disp.resize(disp.is_fullscreen?800:CImgDisplay::screen_dimx(),
			disp.is_fullscreen?600:CImgDisplay::screen_dimy()).
	      toggle_fullscreen().is_resized = true;
	  CImgStats stats(iterimg,0);
	  if (stats.mean>0.7*maxiter) maxiter+=64;
	  if (maxiter>1024) maxiter=1024;
	  if (disp.is_closed || disp.key==cimg::keyQ || disp.key==cimg::keyESC) stopflag=endflag=true;
	}
      }
    } break;

    case 7: {

      // Mini-Paint demo
      //-----------------
      int xo=-1,yo=-1,x=-1,y=-1;
      bool redraw = true;
      CImg<unsigned char> img(256,256+64,1,3,0);
      unsigned char color[3]={255,255,255};
      cimg_forXY(img,xx,yy) if (yy>=256) {
	img(xx,yy,0) = (unsigned char)xx;
	img(xx,yy,1) = (unsigned char)((yy-256)*4);
	img(xx,yy,2) = (unsigned char)((3*xx)%256);
      }
      CImgDisplay disp(img.draw_text("   ",5,5,color,color),"Mini-Paint");
      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	const unsigned int but = disp.button;
	redraw = false;
	xo = x; yo = y; x = disp.mouse_x; y = disp.mouse_y;
	if (xo>=0 && yo>=0 && x>=0 && y>=0) {
	  if (but&1 || but&4) {
	    if (y<253) {
	      const float
		tmax = (float)cimg::max(cimg::abs(xo-x),cimg::abs(yo-y))+0.1f,
		radius = (but&1?3.0f:0.0f) + (but&4?6.0f:0.0f);
	      for (float t=0; t<=tmax; t++) img.draw_circle((int)(x+t*(xo-x)/tmax),(int)(y+t*(yo-y)/tmax),radius,color);
	    }
	    if (y>=256) { color[0]=img(x,y,0); color[1]=img(x,y,1); color[2]=img(x,y,2); img.draw_text("   ",5,5,color,color); }
	    redraw = true;
	  }
	  if (y>=253) y=252;
	  if (disp.button&2) { img.draw_fill(x,y,color); redraw = true; }
	}
	if (redraw) disp.display(img);
	disp.resize(disp).wait();
	if (disp.key) cimg_forV(img,k) { img.get_shared_lines(0,255,0,k).fill(0); img.display(disp); }
      }
    } break;

    case 8: {

      // Soccer Bobs Demo
      //------------------
      const unsigned int nb_canvas=16;
      unsigned int curr_canvas=0;
      float zoom=0.2f;
      unsigned char color[3]={255,255,0};
      CImg<unsigned char> foot = "img/foot.ppm", canvas0(640,480,1,3,0);
      cimg_forXY(canvas0,x,y) canvas0(x,y,1)=(unsigned char)(20+(y*215/canvas0.dimy())+cimg::crand()*19);
      canvas0.draw_text("Left/Right Mouse Button = Zoom In/Out\nMiddle Button = Reset Screen",1,1,color);
      CImgList<unsigned char> canvas(nb_canvas, canvas0);
      CImg<float> mask(foot.dimx(),foot.dimy());
      { cimg_forXY(mask,x,y) mask(x,y) = (foot(x,y,0)==255 && !foot(x,y,1) && !foot(x,y,2))?0:0.5f; }
      CImgDisplay disp(canvas0,"Unlimited Soccer Bobs");
      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	if (disp.mouse_x>=0 && disp.mouse_y>=0)
	  canvas[curr_canvas].draw_image(foot.get_resize((int)(foot.dimx()*zoom),(int)(foot.dimy()*zoom)),
					 mask.get_resize((int)(foot.dimx()*zoom),(int)(foot.dimy()*zoom)),
					 (int)(disp.mouse_x-0.5*zoom*foot.dimx()),(int)(disp.mouse_y-0.5*zoom*foot.dimy()),0,0);
	if (disp.button&1) zoom+=0.03f;
	if (disp.button&2) zoom-=0.03f;
	if (disp.button&4) cimglist_for(canvas,l) canvas[l] = canvas0;
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) disp.toggle_fullscreen();
	if (zoom>1) zoom=1;
	if (zoom<0.1) zoom=0.1f;
	disp.display(canvas[curr_canvas++]).resize(disp,false).wait(20);
	curr_canvas%=nb_canvas;
      }
    } break;

    case 9: {

      // Bump Effect
      //-------------
      float t=0,color = 255;
      CImg<> logo = CImg<>(56,32,1,1,0).draw_text("I Love\n CImg!",9,5,&color).resize(-800,-800,1,1,3).blur(6).normalize(0,255);
      logo += CImg<>(logo.dimx(),logo.dimy(),1,1,0).noise(80,1).deriche(2,0,'y',0).deriche(10,0,'x',0);
      CImgList<> grad = logo.get_gradientXY();
      cimglist_apply(grad,normalize)(-140,140);
      logo.normalize(0,255);
      CImg<> light(300+2*logo.dimx(),300+2*logo.dimy());
      light.draw_gaussian(0.5f*light.dimx(),0.5f*light.dimy(),80,&color);
      CImg<unsigned char> img(logo.dimx(),logo.dimy(),1,3,0);
      CImgDisplay disp(img,"Bump Effect");
      while(!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	const int
	  mouse_x = (disp.mouse_x>=0 && disp.button)?disp.mouse_x*img.dimx()/disp.dimx():(int)(img.dimx()/2 + img.dimx()*cos(1*t)/2),
	  mouse_y = (disp.mouse_y>=0 && disp.button)?disp.mouse_y*img.dimy()/disp.dimy():(int)(img.dimy()/2 + img.dimy()*sin(3*(t+=0.03f))/2);
	cimg_forXY(img,x,y) {
	  const int gx = (int)grad[0](x,y), gy = (int)grad[1](x,y);
	  const float val = 40+(gx+gy)/2+light(light.dimx()/2+mouse_x-x+gx,light.dimy()/2+mouse_y-y+gy);
	  img(x,y,0) = img(x,y,1) = img(x,y,2) = (unsigned char)(val>255?255:(val<0?0:val));
	}
	disp.resize(false).display(img.draw_image(logo,0,0,0,1,0.1f)).wait(25);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(),disp.screen_dimy(),false);
	  else disp.resize(img,false);
	}
      }
    } break;

    case 10: {

      // Bouncing bubble
      //------------------
      CImg<unsigned char> back(320,256,1,3,0),img;
      cimg_forXY(back,x,y) back(x,y,2) = (unsigned char)((y<2*back.dimy()/3)?30:(255-2*(y+back.dimy()/2)));
      CImgDisplay disp(back,"Bouncing bubble",0,1);
      const unsigned char col1[3]={40,100,10}, col2[3]={20,70,0}, col3[3]={40,150,10}, col4[3]={200,255,100}, white[3]={255,255,255};
      double u = sqrt(2.0),  cx = back.dimx()/2, t = 0, vt=0.05, vx = 2;
      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	img = back;
	int xm =(int)cx, ym = (int)(img.dimy()/2-70 + (img.dimy()/2+10)* (1-cimg::abs(cos((t+=vt)))));
	float r1 = 50, r2 = 50;
	vt=0.05;
	if (xm+r1>=img.dimx())    { const float delta = (xm+r1)-img.dimx(); r1-=delta; r2+=delta; }
	if (xm-r1<0)              { const float delta = -(xm-r1); r1-=delta; r2+=delta; }
	if (ym+r2>=img.dimy()-40) { const float delta = (ym+r2)-img.dimy()+40; r2-=delta; r1+=delta; vt=0.05-0.0015*(50-r2); }
	if (ym-r2<0)              { const float delta = -(ym-r2); r2-=delta; r1+=delta; }
	img.draw_ellipse(xm,ym,r1,r2,1,0,col1);
	img.draw_ellipse((int)(xm+0.03*r1*u),(int)(ym-0.03*r2*u),0.85f*r1,0.85f*r2,1,0,col2);
	img.draw_ellipse((int)(xm+0.1*r1*u),(int)(ym-0.1*r2*u),0.8f*r1,0.8f*r2,1,0,col1);
	img.draw_ellipse((int)(xm+0.2*r1*u),(int)(ym-0.2*r2*u),r1/2,r2/2,1,0,col3);
	img.draw_ellipse((int)(xm+0.3*r1*u),(int)(ym-0.3*r2*u),r1/4,r2/4,1,0,col4);
	img.draw_image(img.get_crop(0,img.dimy()-80,img.dimx()-1,img.dimy()-40).mirror('y'),0,img.dimy()-40,0,0,0.45f);
	img.draw_text(xm-60,(int)(ym-r2-25),white,0,17,1,"Bubble (%d,%d)",xm,ym);
	if ((cx+=20*vt*vx)>=img.dimx()-30 || cx<30) vx=-vx;
	disp.display(img).wait(20);
	if (disp.is_resized) {
	  disp.resize(disp.window_dimx()>200?disp.window_dimx():200,disp.dimy(),false);
	  back.resize(disp);
	  cx = back.dimx()/2;
	}
      }
    } break;

    case 11: {

      // Virtual landscape
      //-------------------
      CImg<int> background(400,300,1,3,0), visu(background);
      cimg_forXY(background,x,y) {
	if (y>background.dimy()/2) { background(x,y,2)=255; background(x,y,0)=(y-background.dimy()/2)*512/background.dimy(); }
	else background(x,y,2)=y*512/background.dimy();
      }
      const int white[3]={255,255,255};
      CImgDisplay disp(visu.draw_text("Please wait, generating landscape....",10,10,white,0,19).
		       normalize(0,255),"Virtual Landscape",0);
      const CImg<> map = 5*(CImg<>(700,700,1,1,300).noise(300).draw_plasma(0.2,300).normalize(-140,150).blur(5).cut(0,150));
      CImg<> cmap(map.dimx(),map.dimy());
      CImg_3x3(I,float);
      { cimg_for3x3(map,x,y,0,0,I) { const float nox=0.5f*(Inc-Ipc), noy=0.5f*(Icn-Icp); cmap(x,y)=cimg::max(0.0f,0.5f*nox+noy); }}
      cmap.normalize(0,255);

      for (float t=0; !disp.is_closed && disp.key!=cimg::keyESC && disp.key!=cimg::keyESC; t+=0.0025f) {
	visu = background;
	int xm = (int)(map.dimx()/2 + (map.dimx()/3)*cos(4.2*t)), ym = (int)(map.dimy()/2 + (map.dimy()/3)*sin(5.6*t));
	const CImg<> smap = map.get_crop(xm,ym,xm+100,ym+90), scmap = cmap.get_crop(xm,ym,xm+100,ym+90);
	CImg<int> ymin(visu.dimx(),1,1,1,visu.dimy()), ymax(ymin.dimx(),1,1,1,0);
	cimg_forY(smap,z) {
	  const int y0 = (int)(visu.dimy()-1-10*pow((double)z,0.63) + 80);
	  cimg_forX(visu,x) {
	    const int nz = smap.dimy()-z;
	    float mx = x*(smap.dimx()-2.0f*nz*0.2f)/visu.dimx() + nz*0.2f;
	    const int y = (int)(y0-smap.linear_pix1d(mx,z)/(1+0.02*z));
	    const float cc = (float)scmap.linear_pix1d(mx,z);
	    if (y<visu.dimy() && y<ymin(x)) {
	      const float cz = (smap.dimy()-(float)z)/smap.dimy(), czz = cz>0.25?1:4*cz;
	      if (y!=y0) for (int l=y>0?y:0; l<ymin(x); l++) {
		visu(x,l,0) = (int)((1-czz)*visu(x,l,0)+4*cc*czz);
		visu(x,l,1) = (int)((1-czz)*visu(x,l,1)+3*cc*czz);
		visu(x,l,2) = (int)((1-czz)*visu(x,l,2)+  cc*czz);
	      } else for (int l=y>0?y:0; l<ymin(x); l++) { int cl = l-visu.dimy()/2;
	      visu(x,l,0) = 10; visu(x,l,1) = 200-cl; visu(x,l,2) = 255-cl;
	      }
	    }
	    ymin(x)=cimg::min(ymin(x),y);
	    ymax(x)=cimg::max(ymax(x),y);
	  }
	}
	visu.draw_text(5,5,white,0,11,0.5f,"%u frames/s",(unsigned int)disp.frames_per_second());
	disp.resize(false).display(visu.cut(0,255)).wait(25);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) disp.toggle_fullscreen();
      }
    } break;

    case 12: {

      // Plasma effect with sinus scrolling.
      //------------------------------------
      CImg<> plasma,camp(3),cfreq(3),namp(3),nfreq(3);
      CImgList<unsigned char> font = CImgList<unsigned char>::get_font(57);
      CImg<unsigned char> visu(400,300,1,3,0), letter,
	scroll(visu.dimx()+2*font['W'].dimx(),font['W'].dimy(),1,1,0);
      const char *text = "   This is a nice plasma effect, isn't it ?";
      CImgDisplay disp(visu,"Plasma Effect");
      const unsigned char white[3]={ 255, 255, 255 };
      unsigned int cplasma = 0, pos = 0, tpos = 0, lwidth = 0;
      float tx=0, ts=0, alpha=2.0f, beta=0;
      namp.fill(0).noise(visu.dimy()/4,0);
      nfreq.fill(0).noise(0.1);

      visu.draw_text("Please wait, the plasma is generating...",10,10,white).display(disp);
      const unsigned int nb_plasmas = 5;
      plasma = CImg<>(5*visu.dimx()/3,visu.dimy(),1,nb_plasmas,0);
      plasma.noise(100).draw_plasma();
      cimg_forV(plasma,k) plasma.get_shared_channel(k).blur((float)(cimg::rand()*6)).normalize(0,255);

      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	if (alpha>1) {
	  alpha-=1;
	  cplasma=(cplasma+1)%plasma.dimv();
	  camp = namp;
	  cfreq = nfreq;
	  namp.fill(0).noise(100).normalize(0,visu.dimy()/4.0f);
	  nfreq.fill(0).noise(0.2);
	}

	const unsigned int v0=cplasma, v1=(cplasma+1)%plasma.dimv(), v2=(cplasma+2)%plasma.dimv(), v3=(cplasma+3)%plasma.dimv();
	const float umalpha = 1-alpha;
	unsigned char
	  *pR = visu.ptr(0,0,0,0),
	  *pG = visu.ptr(0,0,0,1),
	  *pB = visu.ptr(0,0,0,2);
	cimg_forY(visu,y) {
	  const float
	    *pR1 = plasma.ptr((unsigned int)(camp(0)*(1+sin(tx+cfreq(0)*y))),y,v0),
	    *pG1 = plasma.ptr((unsigned int)(camp(1)*(1+sin(tx+cfreq(1)*y))),y,v1),
	    *pB1 = plasma.ptr((unsigned int)(camp(2)*(2+sin(tx+cfreq(2)*y))),y,v2),
	    *pR2 = plasma.ptr((unsigned int)(namp(0)*(1+sin(tx+nfreq(0)*y))),y,v1),
	    *pG2 = plasma.ptr((unsigned int)(namp(1)*(1+sin(tx+nfreq(1)*y))),y,v2),
	    *pB2 = plasma.ptr((unsigned int)(namp(2)*(2+sin(tx+nfreq(2)*y))),y,v3);
	  cimg_forX(visu,x) {
	    *(pR++) = (unsigned char)(umalpha*(*(pR1++))+alpha*(*(pR2++)));
	    *(pG++) = (unsigned char)(umalpha*(*(pG1++))+alpha*(*(pG2++)));
	    *(pB++) = (unsigned char)(umalpha*(*(pB1++))+alpha*(*(pB2++)));
	  }
	}

	if (!pos) {
	  const CImg<unsigned char>& letter = font(text[tpos]);
	  lwidth = (unsigned int)letter.dimx();
	  scroll.draw_image(letter,visu.dimx(),0);
	  tpos++; tpos%=strlen(text);
	}
	scroll.translate(2);
	pos+=2; if (pos>lwidth+2) pos=0;

	cimg_forX(visu,x) {
	  const int y0 = (int)(visu.dimy()/2+visu.dimy()/4*sin(ts+x/(70+30*cos(beta))));
	  cimg_forY(scroll,y) {
	    if (scroll(x,y)) {
	      const unsigned int y1 = y0+y+2; visu(x,y1,0)/=2; visu(x,y1,1)/=2; visu(x,y1,2)/=2;
	      const unsigned int y2 = y1-6;   visu(x,y2,0)=visu(x,y2,1)=visu(x,y2,2)=255;
	    }
	  }
	}
	alpha+=0.007f;
	beta+=0.04f;
	tx+=0.09f;
	ts+=0.04f;
	disp.resize(false).display(visu).wait(20);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(),disp.screen_dimy(),false);
	  else disp.resize(visu,false);
	}
      }
    } break;

    case 13: {

      // Oriented convolutions
      //-----------------------
      const CImg<unsigned char> img = CImg<unsigned char>("img/sh0r.pgm").noise(50,2);
      CImg<float> mask(16,16);
      CImgList<unsigned char> visu = img<<img<<img;
      const float value = 255;
      CImgDisplay disp(visu,"Original image, Oriented kernel and Convolved image",0,1);
      for (float angle=0; !disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC; angle+=0.1f) {
	const float ca = (float)cos(angle), sa = (float)sin(angle);
	const CImg<> u = CImg<>::vector(ca,sa), v = CImg<>::vector(-sa,ca),
	  tensor = 30*u*u.get_transpose() + 2*v*v.get_transpose();
	mask.draw_gaussian(0.5f*mask.dimx(),0.5f*mask.dimy(),tensor,&value);
	mask/=mask.sum();
	visu[1] = mask.get_resize(img).normalize(0,255).
	  draw_text(2,2,&value,0,11,1,"Angle = %d deg",cimg::mod((int)(angle*180/cimg::PI),360));
	visu[2] = img.get_convolve(mask);
	disp.resize(disp.window_dimx(),(int)(disp.dimy()*disp.window_dimx()/disp.dimx()),false).
	  display(visu).wait(25);
      }
    } break;

    case 14: {

      // Shade bobs
      //------------
      CImg<unsigned char> img(300,300,1,1,0), palette;
      CImgDisplay disp(img,"Shade Bobs");
      float t=100, rx=0, ry=0, rz=0, rt=0, rcx=0;
      const unsigned char one = 1;
      int nbbobs = 0, rybobs = 0;

      while (!disp.is_closed && disp.key!=cimg::keyESC && disp.key!=cimg::keyQ) {
	if ((t+=0.015f)>4*cimg::PI) {
	  img.fill(0);
	  rx=(float)(cimg::crand());
	  ry=(float)(cimg::crand());
	  rz=(float)(cimg::crand());
	  rt=(float)(cimg::crand());
	  rcx = 0.6f*(float)(cimg::crand());
	  t=0;
	  palette = CImg<unsigned char>(3,4+(int)(12*cimg::rand()),1,1,0).noise(255,2).resize(3,256,1,1,3);
	  palette(0)=palette(1)=palette(2)=0;
	  nbbobs = 20+(int)(cimg::rand()*80);
	  rybobs = (10+(int)(cimg::rand()*50))*cimg::min(img.dimx(),img.dimy())/300;
	  disp.key = disp.button = 0;
	}

	for (int i=0; i<nbbobs; i++) {
	  const float
	    r = (float)(ry + rx*cos(6*rz*t) + (1-rx)*sin(6*rt*t)),
	    a = (float)((360*sin(rz*t)+30*ry*i)*cimg::PI/180),
	    ax = (float)(i*2*cimg::PI/nbbobs+t);
	  const int
	    cx = (int)((1+rcx*cos(ax)+r*cos(a))*img.dimx()/2),
	    cy = (int)((1+rcx*sin(ax)+r*sin(a))*img.dimy()/2);
	  img.draw_circle(cx,cy,(float)rybobs,&one,0L,-1.0f);
	}
	CImg_3x3(I,unsigned char);
	CImg<unsigned char> tmp(img);
	cimg_for3x3(tmp,x,y,0,0,I) img(x,y) = (Inc+Ipc+Icn+Icp+(Icc<<2))>>3;
	CImg<unsigned char> visu(img.dimx(),img.dimy(),1,3);
	cimg_forXY(visu,xx,yy) {
	  const unsigned char *col = palette.ptr(0,img(xx,yy));
	  visu(xx,yy,0) = *(col++);
	  visu(xx,yy,1) = *(col++);
	  visu(xx,yy,2) = *(col++);
	}
	disp.display(visu).wait(25);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF))
          disp.resize(disp.is_fullscreen?300:640,disp.is_fullscreen?300:480).toggle_fullscreen();
	if (disp.is_resized) img.resize(disp.resize(false),3);
	if ((disp.key && disp.key!=cimg::keyCTRLLEFT) || disp.button) t=70;
      }
    } break;

    case 15: {

      // Fourier Filtering
      //-------------------
      const CImg<unsigned char> img = CImg<unsigned char>("img/lena.pgm").resize(256,256);
      CImgList<> F = img.get_FFT();
      cimglist_apply(F,translate)(img.dimx()/2,img.dimy()/2,0,0,2);
      const CImg<unsigned char> mag = ((F[0].get_pow(2) + F[1].get_pow(2)).sqrt()+1.0f).log().normalize(0,255);
      CImgList<unsigned char> visu(img,mag);
      CImgDisplay disp(visu,"Fourier Filtering");
      CImg<unsigned char> mask(img.dimx(),img.dimy(),1,1,1);
      unsigned char one[1] = { 1 }, zero[1] = { 0 }, white[1]={255};
      float rmin = 0, rmax = 256;
      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	disp.wait();
	const int xm = disp.mouse_x-img.dimx(), ym = disp.mouse_y, x = xm-img.dimx()/2, y = ym-img.dimy()/2;
	if (disp.button && xm>=0 && ym>=0) {
	  const float r = cimg::max(0.0f,(float)sqrt((float)x*x+y*y)-3.0f);
	  if (disp.button&1) rmax = r;
	  if (disp.button&2) rmin = r;
	  if ((int)rmin>=(int)rmax) rmin = cimg::max(rmax-1.0f,0.0f);
	  mask.fill(0).draw_circle(mag.dimx()/2,mag.dimy()/2,(float)rmax,one).
	    draw_circle(mag.dimx()/2,mag.dimy()/2,(float)rmin,zero);
	  CImgList<> nF(F);
	  cimglist_for(F,l) nF[l].mul(mask).translate(-img.dimx()/2,-img.dimy()/2,0,0,2);
	  visu[0] = nF.FFT(true)[0].normalize(0,255);
	}
	if (disp.is_resized) disp.resize(disp);
	visu[1] = mag.get_mul(mask).draw_text(5,5,white,zero,11,0.6f,"Freq Min/Max = %d / %d",(int)rmin,(int)rmax);
	visu.display(disp);
      }
    } break;

    case 16: {

      // Image Zoomer
      //--------------
      const CImg<unsigned char> img("img/logo.ppm");
      CImgDisplay disp(img,"Original Image",0,3), dispz(300,300,"Zoomed Image",0,3);
      disp.move((CImgDisplay::screen_dimx()-dispz.dimx())/2,(CImgDisplay::screen_dimy()-dispz.dimy()-disp.dimy())/2);
      dispz.move(disp.window_x,disp.window_y + disp.window_height + 40);

      int factor = 20,x=0,y=0;
      bool grid = false, redraw = false;
      while (!disp.is_closed && !dispz.is_closed && disp.key!=cimg::keyQ && dispz.key!=cimg::keyQ &&
	     disp.key!=cimg::keyESC && dispz.key!=cimg::keyESC ) {
	if (disp.mouse_x>=0) { x = disp.mouse_x; y = disp.mouse_y; redraw = true; }
	if (redraw) {
	  const int
	    x0 = x-factor, y0 = y-factor,
	    x1 = x+factor, y1 = y+factor;
	  const unsigned char red[3] = { 255,0,0 }, black[3] = { 0,0,0 }, white[3] = { 255,255,255 };
	  (+img).draw_line(x0,y0,x1,y0,red).draw_line(x1,y0,x1,y1,red).
	    draw_line(x1,y1,x0,y1,red).draw_line(x0,y1,x0,y0,red).display(disp);
	  CImg<unsigned char> visu = img.get_crop(x0,y0,x1,y1).draw_point(x-x0,y-y0,red,0.2f).resize(dispz);
	  if (grid) {
	    const int bfac = 2*factor+1;
	    for (int i=0; i<bfac; i++) {
	      const int X = i*dispz.dimx()/bfac, Y = i*dispz.dimy()/bfac;
	      visu.draw_line(X,0,X,dispz.dimy()-1,black).draw_line(0,Y,dispz.dimx()-1,Y,black);
	    }
	  }
	  visu.draw_text(2,2,white,0,11,1.0f,"Coords (%d,%d)",x,y).display(dispz);
	}
	if (disp.button&1) { factor=(int)(factor/1.5f); if (factor<5) factor = 5; disp.button=0; redraw = true; }
	if (disp.button&2) { factor=(int)(factor*1.5f); if (factor>40) factor = 40; disp.button=0; redraw = true; }
	if (disp.button&4 || dispz.button) { grid = !grid; disp.button = dispz.button = 0; redraw = true; }
	if (disp.is_resized) disp.resize(disp);
	if (dispz.is_resized) { dispz.resize(); redraw = true; }
	CImgDisplay::wait(disp,dispz);
      }
    } break;

    case 17: {

      // Blobs Editor
      //--------------
      CImg<unsigned int> img(300,300,1,3);
      CImgList<unsigned int> colors;
      CImgList<> blobs;
      CImgDisplay disp(img,"Blobs Editor",0);
      bool moving = false;
      unsigned int white[3] = { 255,255,255 };

      for (float alpha=0; !disp.is_closed && disp.key!=cimg::keyESC && disp.key!=cimg::keyQ; alpha+=0.1f) {
	const int xm = disp.mouse_x*img.dimx()/disp.dimx(), ym = disp.mouse_y*img.dimy()/disp.dimy();
	int selected = -1;
	img.fill(0);

	if (!blobs.is_empty()) {
	  float dist=0, dist_min = (float)img.dimx()*img.dimx() + img.dimy()*img.dimy();
	  cimglist_for(blobs,l) {
	    const CImg<>& blob = blobs[l];
	    const float xb = blob[0], yb = blob[1], rb = blob[2],
	      sigma = (float)(rb*(1+0.05f*cos(blob[3]*alpha))),
	      sigma2 = 2*sigma*sigma, precision = 4.5f*sigma2;
	    const int tx0 = (int)(xb-3*sigma), ty0 = (int)(yb-3*sigma), tx1 = (int)(xb+3*sigma), ty1 = (int)(yb+3*sigma);
	    const unsigned int
	      col1 = colors[l](0), col2 = colors[l](1), col3 = colors[l](2), wh = img.dimx()*img.dimy(),
	      x0 = tx0<0?0:tx0, y0 = ty0<0?0:ty0,
	      x1 = tx1>=img.dimx()?(img.dimx()-1):tx1, y1 = ty1>=img.dimy()?(img.dimy()-1):ty1;
	    float dy = y0-yb;
	    unsigned int *ptr = img.ptr(x0,y0);
	    for (unsigned int y=y0; y<=y1; y++) {
	      float dx = x0-xb;
	      for (unsigned int x=x0; x<=x1; x++) {
		float dist = dx*dx + dy*dy;
		if (dist<precision) {
		  const float val = (float)exp(-dist/sigma2);
		  *ptr += (unsigned int)(val*col1);
		  *(ptr+wh) += (unsigned int)(val*col2);
		  *(ptr+2*wh) += (unsigned int)(val*col3);
		}
		++dx; ++ptr;
	      }
	      ptr+=img.dimx()-(x1-x0)-1;
	      ++dy;
	    }
	    if ((dist=(xb-xm)*(xb-xm)+(yb-ym)*(yb-ym))<dist_min) { dist_min = dist; selected = l; }
	  }

	  for (unsigned int *ptr1 = img.ptr(0,0,0,1), *ptr2 = img.ptr(0,0,0,2),
		 *ptr3 = img.ptr(img.size()-1)+1,
		 off=0, wh=img.dimx()*img.dimy(); ptr1>img.data; off++) {
	    unsigned int val1 = *(--ptr1), val2 = *(--ptr2), val3 = *(--ptr3);
	    const unsigned int pot = val1*val1 + val2*val2 + val3*val3;
	    if (pot<128*128) { *ptr1=*ptr3=255*off/wh; *ptr2=180*off/wh; }
	    else {
	      if (pot<140*140) { *ptr1>>=1; *ptr2>>=1; *ptr3>>=1; }
	      else {
		*ptr1 = val1<255?val1:255;
		*ptr2 = val2<255?val2:255;
		*ptr3 = val3<255?val3:255;
	      }
	    }
	  }
	  cimglist_for(blobs,ll) {
	    const CImg<>& blob = blobs[ll];
	    const int rb = (int)(blob[2]*(1+0.05f*cos(blob[3]*alpha))),
	      xb = (int)(blob[0]+rb/2.5f), yb = (int)(blob[1]-rb/2.5f);
	    img.draw_circle(xb,yb,rb/2.0f,white,0,0.2f).draw_circle(xb,yb,rb/3.0f,white,0,0.2f).
	      draw_circle(xb,yb,rb/5.0f,white,0,0.2f);
	  }
	} else {
	  CImg<unsigned int> text;
	  text.draw_text("CImg Blobs Editor\n"
			 "-----------------\n\n"
			 "* Left mouse button :\n   Create and Move Blobs.\n\n"
			 "* Right mouse button :\n  Remove nearest Blobs.\n\n"
			 "* Colors and size of Appearing Blobs\n"
			 "  are randomly chosen.\n\n\n"
			 " >> Press mouse button to start ! <<",
			 0,0,white);
	  img.fill(100).draw_image(text,text,(img.dimx()-text.dimx())/2,(img.dimy()-text.dimy())/2,0,0,255U);
	}

	if (disp.mouse_x>=0 && disp.mouse_y>=0) {
	  if (disp.button&1) {
	    float dist_selected = 0;
	    if (selected>=0) {
	      const float a = xm-blobs[selected](0), b = ym-blobs[selected](1), c = blobs[selected](2);
	      dist_selected = a*a+b*b-c*c;
	    }
	    if (moving || dist_selected<0) { blobs[selected](0) = (float)xm; blobs[selected](1) = (float)ym; }
	    else {
	      blobs.insert(CImg<>::vector((float)xm,(float)ym,(float)(10+30*cimg::rand()),(float)(3*cimg::rand())));
	      colors.insert(CImg<>(3).fill(0).noise(255,1).normalize(0,255));
	    }
	    moving = true;
	  } else moving = false;
	  if (selected>=0 && disp.button&2) { blobs.remove(selected); colors.remove(selected); disp.button = 0; }
	}

	img.display(disp.wait(25));
	if (disp.is_resized) {
	  img.resize(disp.resize(false));
	  cimglist_for(blobs,l) if (blobs[l](0)>=img.dimx() || blobs[l](1)>=img.dimy()) { blobs.remove(l); colors.remove(l--); }
	}
      }
    } break;

    case 18: {

      // Double Torus
      //-------------
      CImg<unsigned char> visu(300,256,1,3,0);
      CImgDisplay disp(visu,"Double 3D Torus");
      CImgList<> points, opacities;
      CImgList<unsigned int> primitives;
      CImgList<unsigned char> colors;

      const float a = 60, b = 20;
      const unsigned int na = 20, nb = 10;
      for (unsigned int v=0; v<na; v++)
	for (unsigned int u=0; u<nb; u++) {
	  const float
	    alpha = (float)(u*2*cimg::PI/nb),
	    beta = (float)(v*2*cimg::PI/na),
	    x = (float)((a+b*cos(alpha))*cos(beta)),
	    y = (float)((a+b*cos(alpha))*sin(beta)),
	    z = (float)(b*sin(alpha));
	  points.insert(CImg<>::vector(x,y,z));
	}
      CImgList<> points2(points);
      const CImg<> rot0 = CImg<>::rotation_matrix(1,0,0,(float)cimg::PI/2), dx = CImg<>::vector(30,0,0);
      cimglist_for(points,l) {
	points2[l] = rot0*points[l] + dx;
	points[l]-=dx;
      }
      const unsigned int N = points.size;
      points.insert(points2);

      for (unsigned int vv=0; vv<na; vv++)
	for (unsigned int uu=0; uu<nb; uu++) {
	  const int nv = (vv+1)%na, nu = (uu+1)%nb;
	  primitives.insert(CImg<unsigned int>::vector(nb*vv+nu,nb*vv+uu,nb*nv+uu));
	  primitives.insert(CImg<unsigned int>::vector(nb*vv+nu,nb*nv+uu,nb*nv+nu));
	  primitives.insert(CImg<unsigned int>::vector(N+nb*vv+nu,N+nb*vv+uu,N+nb*nv+uu));
	  primitives.insert(CImg<unsigned int>::vector(N+nb*vv+nu,N+nb*nv+uu,N+nb*nv+nu));
	  colors.insert(CImg<>::vector(255,255,0));
	  colors.insert(CImg<>::vector(255,200,0));
	  colors.insert(CImg<>::vector(100,255,100));
	  colors.insert(CImg<>::vector(200,255,200));
	  opacities.insert(2,CImg<>(1,1,1,1,1.0f));
	  opacities.insert(2,CImg<>(1,1,1,1,0.3f));
	}

      float alpha=0, beta=0, gamma=0, theta=0;
      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	visu.get_shared_channels(1,2).fill(0);

	visu.get_shared_line(visu.dimy()-1,0,0).noise(200,1);
	CImg_3x3(I,unsigned char);
	cimg_for3x3(visu,x,y,0,0,I) visu(x,y,0) = (Icc+Ipn+Icn+Inn)>>2;
	{ for (unsigned int y=0; y<100; y++) memset(visu.ptr(0,y,0,2),255-y*255/100,visu.dimx()); }

	const CImg<>
	  rot = CImg<>::rotation_matrix(1,1,0,(alpha+=0.01f))*CImg<>::rotation_matrix(1,0,1,(beta-=0.02f))*
	  CImg<>::rotation_matrix(0,1,1,(gamma+=0.03f));
	CImgList<> rpoints(points);
	cimglist_for(points,l) rpoints[l]=rot*points[l];
	if (disp.is_resized) disp.resize(false);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(), disp.screen_dimy(),false);
	  else disp.resize(visu,false);
	}
	visu.draw_object3d(visu.dimx()/2.0f,visu.dimy()/2.0f,0.0f,
			   rpoints,primitives,colors,opacities,4,
			   true,500.0f,(float)(cos(theta+=0.01f)+1)*visu.dimx()/2.0f,(float)visu.dimy(),-100.0f,0.2f).
	  display(disp.wait(25));
      }
    } break;

    case 19: {

      // 3D Metaballs
      //--------------
      CImg<unsigned char> img(512,320,1,3);
      img = CImg<unsigned char>(100,100,1,3,0).noise(100,2).draw_plasma(0,0,99,99).resize(img,3).blur(4);
      img.get_shared_channel(2)/=4; img.get_shared_channel(1)/=2;
      metaballs3d met;
      CImgList<float> points;
      CImgList<unsigned int> primitives;
      CImgList<unsigned char> colors(8000,3,1,1,1,255);
      unsigned char white[3] = {255,255,255};

      float alpha=0, beta=0, delta=0, theta=0, gamma=0;
      CImgDisplay disp(img,"3D Metaballs");
      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	met.cx2=1.5f*(float)cos(theta);     met.cy2=2.5f*(float)sin(3*(theta+=0.04f));
	met.cy1=2.0f*(float)sin(4*gamma);   met.cz1=1.2f*(float)cos(2*(gamma-=0.02f));
	met.cx3=2.5f*(float)cos(2.5*delta); met.cz3=1.5f*(float)sin(2*(delta+=0.03f));
	points.assign(); primitives.assign();
	cimg::marching_cubes(met,0.8f,-4.5f,-4.5f,-3.5f,4.5f,4.5f,3.5f,0.25f,0.25f,0.25f,points,primitives,true);
	CImgList<> rpoints(points);
	const CImg<> rot = 50*CImg<>::rotation_matrix(0,0,1,(alpha+=0.04f))*CImg<>::rotation_matrix(1,1,0,(beta+=0.023f));
	cimglist_for(rpoints,l) rpoints(l) = rot*points(l);
	cimglist_for(primitives,ll) {
	  colors(ll,0) = 191+64*ll/primitives.size;
	  colors(ll,1) = 191+64*ll/primitives.size;
	  colors(ll,2) = 255*ll/primitives.size;
	}

	if (primitives.size) {
	  (+img).draw_object3d(img.dimx()/2.0f,img.dimy()/2.0f,0.0f,
			       rpoints,primitives,
			       colors.get_crop(0,primitives.size-1,true),
			       4,false,500,0,0,-500,0,0.9f).
	    draw_text(5,5,white,0,11,0.5f,"%u frames/s",(unsigned int)disp.frames_per_second()).display(disp);
	}
	if (disp.is_resized) disp.resize(false);
	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) disp.toggle_fullscreen();
      }
    } break;

    case 20: {

      // Fireworks
      //----------
      CImg<unsigned char> img(640,480,1,3,0);
      CImgDisplay disp(img,"Fireworks");
      CImgList<unsigned char> colors;
      unsigned char white[3] = {255,255,255}, black[3] = {128,0,0};
      CImgList<> particles;
      float time=0, speed=100.0f;

      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {

	if (disp.button&1 || !particles.size || (--time)<0) {
	  particles.insert(CImg<>::vector((float)cimg::rand()*img.dimx(),(float)img.dimy(),
					  (float)cimg::crand()*4,-6-(float)cimg::rand()*3,
					  30+60*(float)cimg::rand(),3));
	  colors.insert(CImg<unsigned char>::vector(255,255,255));
	  time = (float)(cimg::rand()*speed);
	}
	img*=0.92f;

	cimglist_for(particles,l) {
	  bool remove_particle = false;
	  float &x = particles(l,0), &y = particles(l,1), &vx = particles(l,2), &vy = particles(l,3),
	    &t = particles(l,4), &r = particles(l,5);
	  const float n = (float)sqrt(1e-5f+vx*vx+vy*vy), nvx = vx/n, nvy = vy/n,
	    r2 = (t>0 || t<-42)?r/3:r*(1-2*(-(t+2)/40.0f)/3);
	  img.draw_ellipse((int)x,(int)y,r,r2,nvx,nvy,colors[l].ptr(),0U,0.6f);
	  x+=vx; y+=vy; vy+=0.09f; t--;
	  if (y>img.dimy()+10 || x<0 || x>=img.dimx()+10) remove_particle = true;

	  if (t<0 && t>=-1) {
	    if ((speed*=0.9f)<10) speed=10.0f;
	    const unsigned char
	      r = cimg::min(50+3*(unsigned char)(100*cimg::rand()), 255),
	      g = cimg::min(50+3*(unsigned char)(100*cimg::rand()), 255),
	      b = cimg::min(50+3*(unsigned char)(100*cimg::rand()), 255);
	    const float di = 10+(float)cimg::rand()*60, nr = (float)cimg::rand()*30;
	    for (float i=0; i<360; i+=di) {
	      const float rad = i*(float)cimg::PI/180, c = (float)cos(rad), s = (float)sin(rad);
	      particles.insert(CImg<>::vector(x,y,2*c+vx/1.5f,2*s+vy/1.5f,-2.0f,nr));
	      colors.insert(CImg<unsigned char>::vector(r,g,b));
	    }
	    remove_particle = true;
	  } else if (t<-1) { r*=0.95f; if (r<0.5f) remove_particle=true; }
	  if (remove_particle) { particles.remove(l); colors.remove(l); l--; }
	}
	if (disp.button&2) cimglist_for(particles,l) if (particles(l,4)>0) particles(l,4)=0.5f;
	img.draw_text(5,5,white,black,11,0.5f,"%u frames/s",(unsigned int)disp.frames_per_second());
	disp.display(img).wait(25);

	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) {
	  disp.toggle_fullscreen();
	  if (disp.is_fullscreen) disp.resize(disp.screen_dimx(),disp.screen_dimy(),false);
	  else disp.resize(img,false);
	}
	if (disp.is_resized) disp.resize(disp,false);
      }
    } break;

    case 21: {

      // Rubber Logo
      //-------------
      const unsigned char white[3] = {255,255,255};
      CImg<unsigned char> background = CImg<unsigned char>(300,300).noise(100,2);
      background(0,0) = background(299,0) = background(299,299) = background(0,299) = 0;
      background.draw_plasma(0,0,299,299).blur(1.0f,14.0f,0.0f,0).resize(-100,-100,1,3);
      CImgDisplay disp(CImg<unsigned char>(background).
		       draw_text(10,10,white,0,11,1.0f,"Generating the rubber object..."),"3D Rubber Logo");

      CImg<unsigned char> vol = CImg<unsigned char>().draw_text("CImg",30,30,white,0,48).resize(-100,-100,15,1);
      for (unsigned int k=0; k<5; k++) { vol.get_shared_plane(k).fill(0); vol.get_shared_plane(vol.dimz()-1-k).fill(0); }
      vol.resize(vol.dimx()+30,vol.dimy()+30,-100,1,0).blur(2).resize(-50,-50);
      CImgList<> points;
      CImgList<unsigned int> faces;
      CImgList<unsigned char> colors;
      vol.marching_cubes(45,points,faces,true);
      colors.insert(faces.size,CImg<unsigned char>::vector(100,100,255));
      cimglist_for(colors,l) {
	const float x = (points(faces(l,0),0) + points(faces(l,1),0) + points(faces(l,2),0))/3;
	if (x<27) colors[l] = CImg<unsigned char>::vector(255,100,100);
	else { if (x<38) colors[l] = CImg<unsigned char>::vector(200,155,100);
	else { if (x<53) colors[l] = CImg<unsigned char>::vector(100,255,155);
	}}}
      {cimglist_for(points,l) { points(l,0)-=vol.dimx()/2; points(l,1)-=vol.dimy()/2; points(l,2)-=vol.dimz()/2; }}
      points*=5.5;

      CImgList<unsigned char> frames(100,background);
      bool ok_visu = false;
      unsigned int nb_frame = 0;
      float alpha=0, beta=0, gamma=0;

      while (!disp.is_closed && disp.key!=cimg::keyQ && disp.key!=cimg::keyESC) {
	CImg<unsigned char>& frame = frames[nb_frame++];
	if (nb_frame>=frames.size) { ok_visu = true; nb_frame = 0; }
	const CImg<>
	  rot = CImg<>::rotation_matrix(0,1,0.2f,alpha+=0.011f)*
	  CImg<>::rotation_matrix(1,0.4f,1,beta+=0.015f)*
	  (1+0.1f*cos((double)(gamma+=0.1f)));
	(frame=background).draw_object3d(frame.dimx()/2.0f,frame.dimy()/2.0f,frame.dimz()/2.0f,rot*points,faces,colors,5);

	if (ok_visu) {
	  CImg<unsigned char> visu(frame);
	  cimglist_for(frames,l) {
	    const unsigned int
	      y0 = l*visu.dimy()/frames.size,
	      y1 = (l+1)*visu.dimy()/frames.size-1;
	    cimg_forV(visu,k) visu.get_shared_lines(y0,y1,0,k) = frames[(nb_frame+l)%frames.size].get_shared_lines(y0,y1,0,k);
	  }
	  visu.draw_text(5,5,white,0,11,0.5f,"%u frames/s",(unsigned int)disp.frames_per_second());
	  disp.display(visu).wait(20);
	}

	if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) disp.toggle_fullscreen();
	if (disp.is_resized) disp.resize();
      }

    } break;

    case 22: {

      // Image Waves
      //-------------
      const CImg<unsigned char> img = CImg<unsigned char>("img/milla.ppm").get_resize(128,128,1,3);
      const unsigned int w = img.dimx()+1, h = img.dimy()+1;
      CImgList<> points0;
      CImgList<unsigned int> faces0;
      CImgList<unsigned char> colors0;
      { for (unsigned int y=0; y<h; y++) for (unsigned int x=0; x<w; x++)
		  points0.insert(CImg<>::vector(3*(x-w/2.0f),3*(y-w/2.0f),0)); }
      cimg_forXY(img,x,y) {
	faces0.insert(CImg<unsigned int>::vector(x+y*w,x+(y+1)*w,x+1+(y+1)*w,x+1+y*w));
	colors0.insert(CImg<unsigned char>::vector(img(x,y,0),img(x,y,1),img(x,y,2)));
      }
      CImgList<> opacities0(faces0.size,CImg<>::vector(1.0f));

      CImg<unsigned char>
	back = CImg<unsigned char>(400,300,1,3).sequence(0,130),
	ball = CImg<unsigned char>(12,12,1,3,0).draw_circle(6,6,5,CImg<unsigned char>::vector(0,128,64).ptr());
      const CImg<> mball = ball.get_norm_pointwise(2).normalize(0,1.0f).dilate(3);
      ball.draw_circle(7,5,4.5f,CImg<unsigned char>::vector(16,96,52).ptr()).
	draw_circle(8,4,2.5f,CImg<unsigned char>::vector(0,128,64).ptr()).
	draw_circle(8,4,1.5f,CImg<unsigned char>::vector(64,196,128).ptr());

      CImg<> uc(img.dimx()/2,img.dimy()/2,1,1,0), up(uc), upp(uc);
      CImgDisplay disp(back,"Image Waves");
      CImgList<int> particles;

      for (float alpha=0.0f, count=10.0f; !disp.is_closed && disp.key!=cimg::keyESC && disp.key!=cimg::keyQ; ) {
	if ((disp.button&1 && disp.mouse_x>=0) || --count<0) {
	  particles.insert(CImg<int>::vector((int)(cimg::rand()*(img.dimx()-1)),(int)(cimg::rand()*(img.dimy()-1)),-200,0));
	  count = (float)(cimg::rand()*15);
	}
	alpha = (disp.mouse_x>=0 && disp.button&2)?(float)(disp.mouse_x*2*cimg::PI/disp.dimx()):(alpha+0.04f);
        if (disp.is_typed(cimg::keyCTRLLEFT,cimg::keyF)) disp.toggle_fullscreen();

	cimglist_for(particles,l) {	// Handle particles
	  float& z = up(particles(l,0)>>1,particles(l,1)>>1);
	  if ((particles(l,2)+=(particles(l,3)++))>z-10) { z = 250.0f; particles.remove(l--); }
	}

	CImg_3x3(U,float);  // Apply wave effect
	cimg_for3x3(up,x,y,0,0,U) uc(x,y) = (Unc+Upc+Ucn+Ucp)/2 - upp(x,y);
	(uc-=(float)CImgStats(uc.blur(0.7f),false).mean).swap(upp).swap(up);

	CImgList<> points(points0);
	CImgList<unsigned int> faces(faces0);
	CImgList<unsigned char> colors(colors0);
	CImgList<> opacities(opacities0);
	cimglist_for(points,p) points(p,2) = cimg::min(30 + uc.linear_pix2d((p%w)/2.0f,(p/w)/2.0f),70.0f);
	{ cimglist_for(particles,l) {
	  points.insert(CImg<>::vector(3*(particles(l,0)-w/2.0f),3*(particles(l,1)-h/2.0f),30.0f+particles(l,2)));
	  faces.insert(CImg<unsigned int>::vector(points.size-1));
	  colors.insert(ball);
	  opacities.insert(mball);
	}}
	const CImg<> rot = CImg<>::rotation_matrix(1.0f,0,0,(float)(cimg::PI/3.0f))*CImg<>::rotation_matrix(0,0,1.0f,alpha);
	(+back).draw_object3d(back.dimx()/2.0f,back.dimy()/2.0f,0,rot*points,faces,colors,opacities,4,false,500.0f,0,0,0,0.3f).
	  display(disp.resize(false).wait(20));
      }

    } break;

    case 23: {

      // Breakout
      //----------

      // Init graphics
      CImg<unsigned char> board(8,10,1,1,0),
        background = CImg<unsigned char>(board.dimx()*32,board.dimy()*16+200,1,3,0).noise(20,1).draw_plasma().blur(1,8,0),
        visu0(background/2), visu(visu0), brick(16,16,1,1,200), racket(64,8,1,3,0), ball(8,8,1,3,0);
      const unsigned char white[3] = { 255,255,255 }, green1[3] = { 60,150,30 }, green2[3] = { 130,255,130 };
      { cimg_for_borderXY(brick,x,y,1) brick(x,y) = x>y?255:128; }
      { cimg_for_insideXY(brick,x,y,1) brick(x,y) = cimg::min(255,64+8*(x+y)); }
      brick.resize(31,15,1,1,1).resize(32,16,1,1,0);
      ball.draw_circle(4,4,3,white); ball-=ball.get_erode(3)/1.5f;
      racket.draw_circle(4,3,3.7f,green1).draw_circle(3,2,1.8f,green2);
      { cimg_forY(racket,y) racket.draw_rectangle(4,y,racket.dimx()-7,y,CImg<unsigned char>::vector(y*4,255-y*32,255-y*25).ptr()); }
      racket.draw_image(racket.get_crop(0,0,racket.dimx()/2-1,racket.dimy()-1).mirror('x'),racket.dimx()/2,0);
      const int
        w = visu.dimx(), h = visu.dimy(), w2 = w/2, h2 = h/2,
        bw = ball.dimx(), bh = ball.dimy(), bw2 = bw/2, bh2 = bh/2,
        rw = racket.dimx(), rh = racket.dimy(), rw2 = rw/2;
      float xr = (float)(w-rw2), oxr = (float)xr, xb = 0, yb = 0, oxb = 0, oyb = 0, vxb = 0, vyb = 0;

      // Begin game loop
      CImgDisplay disp(visu,"CImg Breakout");
      disp.move((CImgDisplay::screen_dimx()-w)/2,(CImgDisplay::screen_dimy()-h)/2);
      for (unsigned int N=0, N0=0; !disp.is_closed && disp.key!=cimg::keyESC && disp.key!=cimg::keyQ; ) {
        if (N0) {
          int X = (int)xr;
          if (disp.mouse_x>=0) X = (int)(w2+((disp.mouse_x<0?w2:disp.mouse_x)-w2)*2);
          else disp.set_mouse(xr>w2?w-81:80,h2);
          if (X<rw2) { X = rw2; disp.set_mouse(80,h2); }
          if (X>=w-rw2) { X = w-rw2-1; disp.set_mouse(w-81,h2); }
          oxr = xr; xr = (float)X; oxb = xb; oyb = yb; xb+=vxb; yb+=vyb;
          if ((xb>=w-bw2) || (xb<bw2)) { xb-=vxb; yb-=vyb; vxb=-vxb; }
          if (yb<bh2) { yb = (float)bh2; vyb=-vyb; }
          if (yb>=h-rh-8-bh2 && yb<h-8-bh2 && xr-rw2<=xb && xr+rw2>=xb) {
            xb = oxb; yb = h-rh-8.0f-bh2; vyb=-vyb; vxb+=(xr-oxr)/4;
            if (cimg::abs(vxb)>8) vxb*=8/cimg::abs(vxb);
          }
          if (yb<board.dimy()*16) {
            const int X = (int)xb/32, Y = (int)yb/16;
            if (board(X,Y)) {
              board(X,Y) = 0;
              N++;
              const unsigned int x0 = X*brick.dimx(), y0 = Y*brick.dimy(), x1 = (X+1)*brick.dimx()-1, y1 = (Y+1)*brick.dimy()-1;
              visu0.draw_image(background.get_crop(x0,y0,x1,y1),x0,y0);
              if (oxb<(X<<5) || oxb>=((X+1)<<5)) vxb=-vxb;
              else if (oyb<(Y<<4) || oyb>=((Y+1)<<4)) vyb=-vyb;
            }
          }
          disp.set_title("Score : %u/%u",N,N0);
        }
        if (yb>h || N==N0) {
          disp.show_mouse();
          while (!disp.key && !disp.is_closed && !disp.button) {
            ((visu=visu0)/=2).draw_text(N0?"Game Over !":"Get Ready ?",50,visu.dimy()/2-10,white,0,25).
              display(disp);
            disp.wait();
            if (disp.is_resized) disp.resize(disp);
          }
          board.fill(0); visu0 = background;
          cimg_forXY(board,x,y) if (0.2f+cimg::crand()>=0) {
            CImg<> cbrick = CImg<double>::vector(100+cimg::rand()*155,100+cimg::rand()*155,100+cimg::rand()*155).
              unroll('v').resize(brick.dimx(),brick.dimy());
            cimg_forV(cbrick,k) (cbrick.get_shared_channel(k).mul(brick))/=255;
            visu0.draw_image(cbrick,x*32,y*16);
            board(x,y) = 1;
          }
          N0 = (int)board.sum(); N = 0;
          oxb = xb = (float)w2; oyb = yb = board.dimy()*16.0f+bh; vxb = 2.0f; vyb = 3.0f;
          disp.hide_mouse();
        } else disp.display((visu=visu0).draw_image(racket,(int)(xr-rw2),h-rh-8).draw_image(ball,(int)(xb-bw2),(int)(yb-bh2)));
        if (disp.is_resized) disp.resize(disp);
        disp.wait(20);
      }
    } break;
    }
  }

  // Exit demo menu
  //----------------
  fprintf(stderr,"Exit CImg Demo\n");
  return 0;
}
