/*
 * FollowCurveWindow.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  FollowCurveWindow.cpp 
  
  TODO:
   o Fuer das Tabs-Panel ein Fl_Cartesius ohne Coord-picker anbieten, damit
     Platz besser ausgenutzt werden kann. Obendrein ist zur Zeit der 
     Coord-picker aus unbekanntem Grunde nicht zu sehen.
*/

#include <cassert>

#include "FL/Fl.H"
#include "FL/Fl_Box.H"

#include "Br.hpp"                   // `Br', SPUR_BR_EVENT()
#include "BrGUI.hpp"                // `BrGUI'
#include "FollowCurveWindow.hpp"
#include "fl_print_event.hpp"       // Fltk debug utilities


typedef struct { const char* text; FollowCurveMethod method; } MenuMap_method;
MenuMap_method menuMap_method[] = {
  {"z-average",        Z_AVERAGE},
  {"z-average-smooth", Z_AVERAGE_SMOOTH}
};

Fl_Menu_Item  menu_choice_method[] = {
  {menuMap_method[0].text},                          // {"z-average"},
  {menuMap_method[1].text, 0,0,0, FL_MENU_INACTIVE}, // {"z-average-smooth"},
  {0}
};

typedef struct { const char* text; char channel; } MenuMap_channel;
MenuMap_channel menuMap_channel[] = {
  {"R", 'R'},
  {"G", 'G'},
  {"B", 'B'}
};

Fl_Menu_Item  menu_choice_channel[] = {
  {menuMap_channel[0].text},    // {"R"},
  {menuMap_channel[1].text},    // {"G"},
  {menuMap_channel[2].text},    // {"B"},
  {0}
};


/**=================================================================
 *
 * FollowCurvePanel  --  class 
 *
 * Die Sache ist nicht ganz ausgekungelt in der Hinsicht, dass dies hier
 *  nicht nur der passive "Blick" auf schon existiernde Kurvendaten ist,
 *  sondern deren Berechnung (auch im Ctor) hier auch veranlasst wird.
 *  Wenn man das trennen wollte, muesste man in BrGUI analog zu compute_CCD()
 *  Funktionen zum Aendern von `refpic_' einfuehren, die die Neuberechnung
 *  veranlassen und dann ein FOLLOW_UPDATED senden, und auf dieses hin 
 *  stellt sich dann ein "Blick" neu dar.
 *
 *==================================================================*/
/**--------------
 * Constructor...
 *
 * Note: Ctor end()ed!
 *---------------*/
FollowCurvePanel::FollowCurvePanel(int X, int Y, int W, int H, const char* la)
  : Fl_Group (X,Y,W,H, la),
    choice_refpic_ (0)
{ 
  CTOR(label())
  // in Distributor(en) einloggen...
  Br.distrib_event.login((Callback*)event_msg_, this);
  Br.distrib_refpic.login((Callback*)refpic_msg_, this);
  
  channel_  = 'R';
  method_   = Z_AVERAGE;
  
  begin();
    plot_ = new CurvePlotClass (X, Y, W, H-36); // 40
    //plot_ -> box(FL_FLAT_BOX);
    plot_ -> end();   // otherwise the following included in plot_!
   
    { Fl_Box* o = new Fl_Box(X, Y+H-34, W, 2);  // Trennstrich
      o->box (FL_ENGRAVED_BOX);
    }
    { Fl_Choice* ch = new Fl_Choice(X+60, Y+H-29, 100, 25, "Method"); 
      //choice_method_ = ch;
      ch -> menu(menu_choice_method);
      ch -> align(FL_ALIGN_LEFT);
      ch -> callback((Fl_Callback*)cb_choice_method_, this);
      //ch -> value(i_method_default);             
      //ch -> tooltip(tooltip_method);
    }
    { Fl_Choice* ch = new Fl_Choice(X+220, Y+H-29, 50, 25, "Ref-Img"); 
      choice_refpic_ = ch;
      ch -> callback((Fl_Callback*)cb_choice_refpic_, this);
      build_choice_refpic();
    }
    { Fl_Choice* ch = new Fl_Choice(X+330, Y+H-29, 50, 25, "Channel"); 
      ch -> menu(menu_choice_channel);
      ch -> align(FL_ALIGN_LEFT);
      ch -> callback((Fl_Callback*)cb_choice_channel_, this);
    }
#if 0
    { Fl_Button* b = new Fl_Button(X+W-120, Y+H-29, 100, 25, "Update");
      b -> callback((Fl_Callback*)cb_update_, this);
    }
#endif
  end();
  
  resizable(plot_);
  
  BrGUI.compute_FollowCurves(); 
  build_plot();    
}

/**--------------
 * Destructor... 
 *---------------*/
FollowCurvePanel::~FollowCurvePanel()
{
  DTOR(label()) 
  // aus Distributor(en) ausloggen...
  Br.distrib_event.logout(this);
  Br.distrib_refpic.logout(this);    
}


/**---------------------------------------
 * Callback for the refpic-choice menu...
 *
 * @param w: choice menu (calling widget)
 *----------------------------------------*/
void FollowCurvePanel::cb_choice_refpic (Fl_Menu_* w) 
{
  const Fl_Menu_Item* m = w->mvalue();      // BTW: mvalue() != value()!
  if (!m)                                   
    printf("[%s]: Menu not initialized --> Programmer!\n",__func__);
  else {
    // Set the global variable and broadcast it...
    Br.refpic (w->value());                 
    printf("\"%s\", %d, refpic = %d\n", m->label(), w->value(), Br.refpic());
  }
}

/**---------------------------------------
 * Callback for the method-choice menu...
 *
 * @param w: choice menu (calling widget)
 *----------------------------------------*/
void FollowCurvePanel::cb_choice_method (Fl_Menu_* w) 
{
  const Fl_Menu_Item* m = w->mvalue();      // BTW: mvalue() != value()!
  if (!m)                                   
    printf("[%s]: Menu not initialized --> Programmer!\n",__func__);
  else {
    method_ = menuMap_method[ w->value() ].method; 
    printf("\"%s\", %d, method=%d\n", m->label(), w->value(), method_);
  }
}

/**---------------------------------------
 * Callback for the channel-choice menu...
 *
 * @param w: choice menu (calling widget)
 *----------------------------------------*/
void FollowCurvePanel::cb_choice_channel (Fl_Menu_* w) 
{ DB_
  const Fl_Menu_Item* m = w->mvalue();      // BTW: mvalue() != value()!
  if (!m)                                   
    printf("[%s]: Menu not initialized --> Programmer!\n",__func__);
  else {
    //printf("\"%s\", %d\n", m->label(), w->value());
    channel_ = menuMap_channel[ w->value() ].channel; 
    //printf("\"%s\", %d, channel=%c\n", m->label(), w->value(), channel_);
    
    update_plot();
  }
}

/**-----------------------------------
 * Callback for the "Update" button...                  UNUSED
 *------------------------------------*/
void FollowCurvePanel::cb_update (Fl_Button* b) 
{
  printf("FollowCurvePanel::cb_update()...\n");
  update(); 
  
  return;
  
  // debug output...
  switch (channel_)     // list the current channel...
  {     
    case 'R': Br.camera()-> list(Br.camera()->follow_R); break;
    case 'G': Br.camera()-> list(Br.camera()->follow_G); break;
    case 'B': Br.camera()-> list(Br.camera()->follow_B); break;
  }
}

/**------------------------------------------------------------------
 * build_plot()
 *
 * Baut Kurven-Widget aus den aktuell in Camera vorhandene Folgekurven 
 *   fuer Kanal `channel_'. OHNE plot_->redraw()!
 *
 * NOTE: Zwar jetzt (camera==0)-sicher, aber bei existierender Camera wird
 *   ein vorgaengiges Br->compute_FollowCurves() vorausgesetzt!! Insofern
 *   compute_FollowCurves() intelligenterweise nur rechnet, wenn notwendig,
 *   koennte es auch hier eingefuegt werden und diesbzgl. waere's sicher. Gut?
 *   Naja, build_plot() auch nach channel-Wahl benutzt, und da keine
 *   Kalkulation erwuenscht.
 * Die Abfrage `if (!Br.camera())' sollte praezisiert werden zu einem 
 *   `if (!Br.followup_curves_exist())'. 
 *-------------------------------------------------------------------*/
void FollowCurvePanel::build_plot ()
{
  plot_ -> clear();                 // --> empty_look(true)
  if (!Br.camera()) return;
  
  plot_ -> make_current_cartes();   // wichtig fuer add()
  plot_ -> start_insert();          // init_minmax() + empty_look(false)
 
  // the common X-values of the curves...
  Array1D<PlotDataT> X(Br.camera()->n_zvals());    
  for (int i=0; i < Br.camera()->n_zvals(); i++) 
    X[i] = i;
  
  // the Y-values of the curves...
  for (int p=0; p < Br.camera()->nImages(); p++)  
  {
    Array1D<PlotDataT> Y,Y0,Y1;
    if (p==Br.refpic()) {       // refpic curve always a straight line
      Y = Br.camera()-> get_FollowCurve(p, channel_);
      plot_ -> add(X,Y);
    } 
    else {
      Br.camera()-> get_FollowCurve(Y,Y0,Y1, p, channel_);
      plot_ -> add(X,Y,Y0,Y1);
    }
  }
  plot_ -> style(STYLE_LINES);
  
  // Behelfs-Histogramm...  (interessant waere mal ein logarithmisches)
  const ExpectationFollowValue* follow;        
  switch (channel_) 
  {
    case 'R': follow = Br.camera()->follow_R[0]; break;
    case 'G': follow = Br.camera()->follow_G[0]; break;
    default : follow = Br.camera()->follow_B[0]; break;
  }
  size_t nMax = follow[0].n;
  for (int i=1; i < Br.camera()->n_zvals(); i++)
    if (follow[i].n > nMax) nMax = follow[i].n;
  
  // so skalieren, dass y-Wertebereich ausgefuellt...
  PlotDataT fac = (plot_->ymax() - plot_->ymin()) / nMax;
  
  printf("nPoints=%d, nMax=%d, ymin=%f ymax=%f, fac = %f\n", 
    Br.camera()->nPoints(), nMax, plot_->ymin(), plot_->ymax(), fac);
  
  Array1D<PlotDataT> H(Br.camera()->n_zvals()); 
  for (int i=0; i < Br.camera()->n_zvals(); i++)
    H[i] = fac * follow[i].n; // + plot_->ymin()
  
  plot_ -> add(X,H);  
  plot_ -> curve( plot_->nCurves()-1 ) -> style(STYLE_LINES);
  plot_ -> curve( plot_->nCurves()-1 ) -> color(FL_BLACK); //FL_LIGHT3);
  //plot_ -> reorder(plot_->nCurves()-1, 0); // Histogramm "nach unten"; 
  // veraendert leider auch UI-Kurvennummern, daher erstmal raus
}

/**----------------------------------------------------------------
 * build_choice_refpic()    
 *
 * (Re)Builds choice menu for the current image number in camera!
 *-----------------------------------------------------------------*/
void FollowCurvePanel::build_choice_refpic ()
{
  printf("FollowCurvePanel::%s()...\n",__func__);
  
  choice_refpic_ -> clear();
  if (Br.camera()) {
    for (int i=0; i < Br.camera()->nImages(); i++) 
    {
      char s[4];
      snprintf(s, 4, "%d", Br.imgVec_index(i)+1); // begins with "1"
      choice_refpic_ -> add(s,0,0);
    }
    choice_refpic_ -> value(Br.refpic());  // implies a redraw()
  }
  else
    choice_refpic_ -> redraw();         // therefore here too 
}

/**----------------------------------------------------------------
 * update()
 *
 * Rebuild all what depends on Br.camera for the current Br.camera
 *   including(??) re-calc of follow-up curves. INCLUDES redraw()!
 *-----------------------------------------------------------------*/
void FollowCurvePanel::update ()
{ 
  printf("FollowCurvePanel::%s()...\n",__func__);
  
  // Rebuild choice menu for the current number of images in camera...
  build_choice_refpic();
  // Compute follow-up curves for `Br.refpic_' (only if needed)...
  BrGUI.compute_FollowCurves();
  // Plot the follow-up curves existing in Br.camera...  
  update_plot();    
}


/**--------------------------------------------------
 * event_msg()  --  Distributor callback
 *---------------------------------------------------*/
void FollowCurvePanel::event_msg (BracketingCore::Event e)
{
  SPUR_BR_EVENT(("FollowCurvePanel::%s(%d) [visible=%d]: ",
                 __func__, e, visible()));
  switch (e)
  {
  case BracketingCore::CAMERA_INIT:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      // Rebuild all what depends on camera...
      update();       
      break;
      
  case BracketingCore::CAMERA_DELETED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      // Clear the widget...
      update();
      break;    
  
  default:
      SPUR_BR_EVENT(("not handled\n"));
  }
}

/**---------------------------------------------------------------------
 * refpic_msg()  --  Distributor Callback 
 *
 * Called, if someone calls Br.distrib_refpic.value(pic), i.e. a new
 *   reference picture `pic' (==`Br.refpic()') was choosen: Update 
 *   local output accordingly! 
 *----------------------------------------------------------------------*/
void FollowCurvePanel::refpic_msg (int pic) 
{
  //printf("%s(): pic=%d  Br.refpic=%d\n", __func__, pic, Br.refpic);
  
  // Output the new value...
  if (choice_refpic_) 
    choice_refpic_ -> value(Br.refpic());
  
  // Compute new curves and plot it...
  BrGUI.compute_FollowCurves();  // (computes only if required)
  update_plot();    
}


/**=================================================================
 *
 * FollowCurveTabPanel  --  class 
 * 
 * Fl_Tabs-Problem: Ein Widget kann und sollte nur fuer sich selber sorgen;
 *   wenn es fuer (X,Y,W,H) sich konstruieren soll, sollte es dieses Gebiet
 *   auch vollstaendig nutzen duerfen. Ein Fl_Tabs-Kind jedoch muss oben
 *   Platz fuer den *Eltern*-Rand lassen. Fl_Cartesius muesste dann bei
 *   Y+2 (oder so) beginnen. Weil hier Cartesius bei Y beginnt, daher fuer
 *   die Tabs-Anwendung die Einhuellung durch ein followPanelFrame.
 *   Aber machen die zwei Pixel den Kohl wirklich fett? Und muss nicht auch
 *   bei einem Hauptfenster dessen Rand beachtet werden? Schon richtig,
 *   aber das obliegt nicht Cartesius, sondern dem Anwender, d.h. der
 *   Hauptfensteranwendung, Cartesius entsprechend einzusetzen. 
 *
 *   Es bleibt das Problem, dass bei einem Tabs-Kind nicht dessen *eigener*
 *   Rand von seinen x-y-w-h-Werten abgezogen werden muss, sondern der
 *   Eltern-Rand! Ebenso kann natuerlich auch Cartesius keine Ruecksicht
 *   auf Raender von eventl. Eltern nehmen, und ebenso sollte auch 
 *   FollowCurveTabPanel keine solche Ruecksicht nehmen muessen. Man koennte also
 *   zwar hier 'mal das Cartesius-Widget erst bei Y+2 beginnen lassen 
 *   (trotz eigener Randlosigkeit), als Prinzip taugt das aber nicht!
 *
 * We assume, we can use the widget region completely. But, if the widget
 *   shall be used as content of a Fl_Tabs-child, X,Y,W,H has to be choosen
 *   considering the Fl_tabs border. As the border is always taken from the
 *   y-pos of a child (if the tabs are above), a FollowCurvePanel can not
 *   directly used as a child (because the top border would extent into our
 *   widget), but is to wrap by a Fl_Group with a suitabel y-pos (see
 *   `followPanelFrame_' and `followPanel_' in class `PrimeWindow').
 *
 *==================================================================*/

/**-------------------------------------------------------------
 * event_msg()  --  Distributor callback
 *
 * Hack: We know, that `FollowCurveTabPanel' is wrapped in the main window
 *   (class PrimeWindow in "PrimeWindow.cpp") by a Fl_Group, and this Group is
 *   a tab tab entry of an Fl_Tabs. This tab shall be inactive, if camera==0,
 *   and active, if camera!=0. The CAMERA_INIT event comes only for camera!=0,
 *   so we should activate the tab here, if it was inactive before. The 
 *   address of the tab (== the wrapping Group) we get here by `parent()'!
 * A better solution would be wrapping FollowCurveTabPanel by a class
 *   FollowPanelFrame, and FollowPanelFrame's event_msg() should handle
 *   the events. (The intrinsic reason of the wrapping is an unlovely
 *   feature of Fl_Tabs, see remarks above).
 * Jedenfalls laufen an dieser Stelle FollowCurvePanel und FollowCurveTabPanel
     auseinander, was bei einer Ableitung zu beachten.
 *--------------------------------------------------------------*/
void FollowCurveTabPanel::event_msg (BracketingCore::Event e)
{
  SPUR_BR_EVENT(("FollowTabPanel::%s(%d) [visible=%d]: ", 
                __func__, e, visible()));
  switch (e)
  {
  case BracketingCore::CAMERA_INIT:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      if (!parent()->active()) parent()-> activate(); // The Hack!
      update();  // rebuilds all what depends on camera
      break;
      
  case BracketingCore::CAMERA_DELETED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      update();
      parent()-> deactivate(); // Hack: deacitvate my parent()!
      break;    
      
  default:
      SPUR_BR_EVENT(("not handled\n"));
  }
}


/**=================================================================
 *
 * FollowCurveWindow  --  class 
 *
 * Gedacht als eigenstaendiges Hauptwindow. Wie verhindern, dass aus 
 *   Versehen irgendwo eingereiht? Na z.B. durch Fl_Group::current(0) 
 *   VOR der Konstruktion.
 *
 *==================================================================*/
/**--------------
 * Constructor...
 *---------------*/
FollowCurveWindow::FollowCurveWindow(int W, int H, const char* la)
  : Fl_Window (W,H, la) 
{
  panel_ = new FollowCurvePanel (0,0,W,H);
  callback((Fl_Callback*)cb_window_, this);
  resizable(this);
  end();
}

FollowCurveWindow::FollowCurveWindow(int X, int Y, int W, int H, const char* la)
  : Fl_Window (X,Y,W,H, la)
{ 
  panel_ = new FollowCurvePanel (0,0,W,H);
  callback((Fl_Callback*)cb_window_, this);
  resizable(this);
  end();
}

/*---------------------------------
 * Calling when window is closed...
 *---------------------------------*/
void FollowCurveWindow::cb_window()   
{
  WINCALL(label())
  hide();               // BTW: this is the default
}


  
//-------------------
// static elements...  
//-------------------
const char* FollowCurveWindow::default_title_ = "FollowUpCurves (Bracketing to HDR)";


// END OF FILE
