/*
 * PrimeWindow.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  PrimeWindow.cpp  --  primary window of the "Bracketing to HDR" plug-in
*/    

//#define BR_WITH_SHIFT         // with shift correction? (rudiment)

#include <cstdio>               
#include <cmath>                // pow()
#include <cstdlib>              // strtof()
#include <cstring>              // strcmp()

#include <FL/Fl.H>
#include <FL/Fl_Box.H>          
#include <FL/Fl_Window.H>
#include <FL/Fl_Menu_Bar.H>
#include <FL/Fl_Tabs.H>
#include <FL/Fl_Choice.H>       
#include <FL/Fl_File_Chooser.H>
#include <FL/fl_draw.H>         // fl_shortcut_label()

#include "br_defs.hpp"          // BR_WITH_SHIFT
#include "Br.hpp"               // `Br', SPUR_BR_EVENT()
#include "BrGUI.hpp"            // `BrGUI'
#include "plugin_main.hpp"      // cpaint_load_image()
#include "ImagePanel.hpp"       // class ImagePanel
#include "ShiftPanel.hpp"       // class ShiftPanel
#include "ShiftMenu.hpp"        // class ShiftMenu
#include "CCDCurveMenu.hpp"     // class CCDCurveWindow
#include "StatusLine.hpp"       // class StatusLine, __statusline
#include "PrimeWindow.hpp"      // class PrimeWindow
#include "fl_print_event.hpp"   // debug utilities for Fltk

#define YS_DEBUG
#undef YS_DEBUG
#include "YS/ys_dbg.hpp"        // debug macros CTOR(), DTOR()


/**
 * Structures for the stops menus (once twofold used in class `PrimeWindow') 
 *   (Could be included in class `PrimeWindow'.)
 */
struct { const char* text; float value; }
menuMap_stops[] = {
  {"0.33...", 1./3.   }, 
  {"0.5"    , 0.5     },
  {"0.66...", 2./3.   },
  {"1"      , 1.      },
  {"1.33...", 1.+1./3.},
  {"1.5"    , 1.5     },
  {"1.66...", 1.+2./3.},
  {"2"      , 2.      }
};  
Fl_Menu_Item  menu_choice_stops[] = {
  {menuMap_stops[0].text},
  {menuMap_stops[1].text},
  {menuMap_stops[2].text},
  {menuMap_stops[3].text},
  {menuMap_stops[4].text},
  {menuMap_stops[5].text},
  {menuMap_stops[6].text},
  {menuMap_stops[7].text},
  {0}
};
//const int n_items_stops = sizeof(menuMap_stops) / sizeof(menuMap_stops[0]);


//-----------------------------
// Global callback functions...
//-----------------------------

// No more used, causes the console message "plugin crashed", what's uneloquent
void _cb_quit(Fl_Widget*, void*) 
{ 
  exit(0); 
}  

// For testing of menus...
void _cb_test(Fl_Widget* w, void* arg2) 
{
  Fl_Menu_* mw = (Fl_Menu_*)w;
  const Fl_Menu_Item* m = mw->mvalue();
  
  if (!m)
    printf("NULL\n");
  else if (m->shortcut())
    printf("\"%s\" - %s\n", m->label(), fl_shortcut_label(m->shortcut()));
  else
    printf("\"%s\", arg2 = %d\n", m->label(), (int)arg2);
}



/**=========================================================
 *
 * PrimeWindow  -  class
 *
 *==========================================================*/
 
/**----------------
 * Constructor...
 *-----------------*/
PrimeWindow::PrimeWindow ()
  : Fl_Window (0,0, 480,400, "Bracketing to HDR"),
    file_chooser_ (0)
{
  CTOR(label())
  Br.imgVec.list();
  
  Fl::background(220,220,220);
  //Fl::background(230,230,230);
  
  // in Distributor(en) einloggen... 
  Br.distrib_event.login ((Callback*)event_msg_, this);
  
  // defaults for `stops' value...
  int i_stops_default = 3;      // 0 -> first item, 3 -> "1.0"
  val_stops_ = menuMap_stops [i_stops_default].value;
  
  // defaults for some numeric parameters... (temporary here)
  shift_nx = shift_ny = 10;
  shift_mx = shift_my = 40;
  shift_xc = shift_yc = 0;  // later width/2 and height/2 as default?
  
  // some dimensions and positions...
  int dy_statusline = 25, //28
      y_statusline  = h() - dy_statusline,  
      statusl_extra = 0,    // distance between statusline and rest
      // dimension of the Fl_Tabs...
      x_tabs  = 0,         
      y_tabs  = 34,        
      dx_tabs = w(),
      dy_tabs = h() - dy_statusline - y_tabs - statusl_extra,
      // dimension of Tabs-children (consider border of Fl_Tabs)...
      x_tab   = x_tabs,    dx_tab  = dx_tabs,
      y_tab   = y_tabs+25, dy_tab  = dy_tabs-25;
 
  begin();
    statusline_ = new StatusLine (0, y_statusline, w(), dy_statusline);
    statusline_ -> color (color());  // bg color of the window
    statusline_ -> box(FL_DOWN_BOX);
    statusline_ -> default_text(0);
    statusline_ -> out_default();
    __statusline = statusline_;   // set the global pointer
   
    // Progress bar: Where and how shall it appear?
    //  - As a small line in the status line? (then Progress::text() should
    //      change to `Progress::text(s) { __statusline->out(s);}'! ) 
    //  - Fullfilling the status line?
    //  - Separately from the status line?  
#if 1   
    // Progress bar within the status line...
    progress_ = new Progress (0, y_statusline, w(), dy_statusline);
    //progress_ = new Progress (0, y_statusline+22, w(), 6);
    progress_ -> color (color());   // bg color of the window
    progress_ -> box (FL_DOWN_BOX); // same as statusline!
    progress_ -> hide();
    __progressinfo = progress_;     // set the global pointer
#else
    // Progress bar seperately...
    progress_ = new Progress (10, h()/2, w()-20, 30);
    progress_ -> hide();
    __progressinfo = progress_;     // set the global pointer
#endif  
  
    menubar_ = new Fl_Menu_Bar (0,0, w(), 30); 
    menubar_ -> menu(menu_menubar_);
    menubar_ -> callback(_cb_test);
    menubar_ -> user_data(this);        // this-Zeiger deponieren
  
    tabs_ = new Fl_Tabs (x_tabs, y_tabs, dx_tabs, dy_tabs);
    tabs_ -> selection_color(FL_LIGHT2);   // FLTK 1.1.7
    tabs_ -> box(FL_UP_BOX);
    tabs_ -> visible_focus(0);
    tabs_ -> callback((Fl_Callback*)cb_tabs_, this);
  
      imgPanel_ = new ImagePanel (x_tab, y_tab, dx_tab, dy_tab, "Images");
      imgPanel_ -> selection_color(FL_DARK1);
      imgPanel_ -> end();

#ifdef BR_WITH_SHIFT    
      shiftPanel_ = new ShiftPanel (x_tab, y_tab, dx_tab, dy_tab, "Shifts");
      shiftPanel_ -> selection_color(FL_DARK1);
      shiftPanel_ -> hide();
      shiftPanel_ -> deactivate();
      shiftPanel_ -> end();
#endif

      followPanelFrame_ = new Fl_Group (x_tab, y_tab, dx_tab, dy_tab, "FollowUpCurves");
        followPanel_ = new FollowCurveTabPanel (x_tab, y_tab+2, dx_tab, dy_tab-2);
        followPanel_ -> end();
      followPanelFrame_ -> selection_color(FL_DARK1);
      followPanelFrame_ -> hide();
      //followPanelFrame_ -> resizable(followPanel_);
      followPanelFrame_ -> deactivate();
      followPanelFrame_ -> end();
    
    tabs_ -> end();
    
    // Erase the here unlovely below "UP_BOX" border of `tabs' by overwriting
    //   it with a flat box...
    {Fl_Box* o = new Fl_Box(0, y_statusline-statusl_extra-2, w(), 2);
     o -> box(FL_FLAT_BOX); 
    }
    
    // Resizable region... (Exclude also the tabs! But why this don't work?)
    Fl_Box* resizebox = new Fl_Box(x_tabs, y_tabs+25, dx_tabs, dy_tabs-25);
    //resizebox->box(FL_EMBOSSED_FRAME); // make it visible for checks
  end();  
  
  //resizable (imgPanel_->table()); // Seit Tabs dazwischen, unwirksam, why?    
  resizable (resizebox);
  callback((Fl_Callback*)cb_window_, this);
  
  //show();  // shows primary window before the file open dialog
  
  // Open file chooser dialog self-acting
  cb_open_file (menubar_);
    // Problem: Usualy an user driven opened window seems to be positioned by
    //   the window manger under the mouse pointer. If opened via the "Open"
    //   item of br_to_hdr, the file_chooser window was thus well positioned:
    //   lapped with the PrimeWindow. But now, opened self-acting, the
    //   PrimeWindow appears "anywhere", and the file_chooser window appears
    //   under the mouse pointer of CinePaint's <toolmenu>->"File"->"New
    //   from..." item. Unlovely! Explicite positioning by hand not supported
    //   by the Ctor of Fl_File_Chooser (Btw: File_Chooser not derived from an
    //   Fl_Widget). Or we open file_chooser window before we show PrimeWindow,
    //   but this I find misleading too. 

  //new CurveWindow (760,400);
  BrGUI.pFollowCurveWindow = new FollowCurveWindow (x()+w()+6,0,380,400);
  BrGUI.pFollowCurveWindow -> show();
  BrGUI.pCCDCurveWindow = new CCDCurveWindow (x()+w()+11+381,0,380,400);
  BrGUI.pCCDCurveWindow -> show();
  
  //printf("take_focus = %d\n", take_focus());
  show();    
  
  printf("fontsize = %d\n", fl_size());
  //fl_print_children(this);  // debug output
}


/**------------------------------------------------------
 * Destructor...
 *   But don't free what was freed already in cb_window()!
 *-------------------------------------------------------*/
PrimeWindow::~PrimeWindow()
{ 
  DTOR(label()) 
  
  // Logout from Distributor(s)...
  Br.distrib_event.logout (this);
  
  // Destroy top window(s) owned by "me"...
  delete file_chooser_; 
}


/**------------------------------------------------------------
 * cb_window()  --  Calling when window closed;
 *                  used in cb_window_() *and* cb_quit_()
 *-------------------------------------------------------------*/
void PrimeWindow::cb_window (Fl_Widget* w)  
{
  WINCALL(w->label())           // w == window_
  ((Fl_Window*)w) -> hide();    // BTW: this is the default
}

/**-----------------------------------------------------------------
 * cb_clear()  --  Callback for the "Clear" item in the menubar
 * 
 * Free camera and clear all container. Whether secondary windows will
 *   be closed, lies in the responsibitly now of those windows itself.
 * 
 * Updating of GUI is asked by the messages broadcasted by Br.clear()
 *   and is done locally in the event_msg()'s of the affected GUI-objects.
 *------------------------------------------------------------------*/
void PrimeWindow::cb_clear(Fl_Menu_*)
{ 
  // Delete secondary top windows if any open...
  //BrGUI.clear();   
  
  // Free Camera instance and clear global data container...
  Br.clear();
}

/**----------------------------------------------
 * Callback for the "stops" menu in the menubar   [at present hide()d]
 *
 * @param 1: not used
 * @param v: read as index to `menuMap_stops'
 *----------------------------------------------*/
void PrimeWindow::cb_stops_menubar(Fl_Menu_*, void* v) 
{
  // Calc. new times accordingly to stops value...
  val_stops_ = menuMap_stops[ int(v) ].value;
  Br.imgVec.set_times_by_stops(val_stops_);
  
  // update time column in imgPanel...
  imgPanel_->table()-> redraw();   
  
  // update the stops choice menu in imgPanel_ (output the new value)...
  imgPanel_->choice_stops()-> value(int(v));  
}

/**-----------------------------------------------------
 * Callback for the "Response Curves" item in the menubar...
 *-----------------------------------------------------*/
void PrimeWindow::cb_CCDCurveWindow(Fl_Menu_* w) 
{ 
  BrGUI.do_CCDCurveWindow();
}

/**------------------------------------------------------
 * Callback for the "Shifts" item in the menubar...
 *------------------------------------------------------*/
void PrimeWindow::cb_ShiftMenu(Fl_Menu_* w) 
{
  BrGUI.do_ShiftMenu();
}

/**--------------------------------------------------------
 * Callback for the "FollowUpCurves" item in the menubar...
 *---------------------------------------------------------*/
void PrimeWindow::cb_FollowCurveWindow()
{
  BrGUI.do_FollowCurveWindow();
}
 

/**--------------------------------------------------
 * cb_tabs()  --  Callback for the Fl_Tabs.                 EXAMPLE
 *
 * @param w: Fl_Tabs, w->value() = the selected child
 *
 * Called, if someone selects a tab.
 *---------------------------------------------------*/
void PrimeWindow::cb_tabs(Fl_Tabs* w)    // w == tabs_
{
  if (w->value() == followPanelFrame_) {
    do_FollowPanel();
  }
} 
/**--------------------------------------------------
 * do_FollowPanel()  --  helper for cb_tabs().              EXAMPLE
 *---------------------------------------------------*/
void PrimeWindow::do_FollowPanel() 
{
  //printf("PrimeWindow::%s()... \n",__func__);
}


/**----------------------------------------------------------
 * do_file_chooser()
 *
 * Taken over from "flphoto/album.cxx" (Author Michael Sweet).
 *   Argument order changed.
 *-----------------------------------------------------------*/
int PrimeWindow::do_file_chooser(   // O - Number of selected files
        const char *value,          // I - Initial value (path)
        const char *pattern,        // I - Filename patterns
        int        type,            // I - Type of file to select
        const char *title)          // I - Window title string
{
  // Create the file chooser if it doesn't exist...
  if (!file_chooser_)
  {
    if (!value)
      value = ".";

    Fl_File_Icon::load_system_icons();
    
    file_chooser_ = new Fl_File_Chooser(value, pattern, type, title);
  }
  else
  {
    file_chooser_->type(type);
    file_chooser_->filter(pattern);
    file_chooser_->label(title);

    if (value && *value)
      file_chooser_->value(value);
    else
      file_chooser_->rescan();
  }

  // Show the chooser and wait for something to happen...
  file_chooser_->show();

  while (file_chooser_->shown())
    Fl::wait();

  return (file_chooser_->count());
}


void PrimeWindow::cb_open_file(Fl_Menu_*)
{
  int count = do_file_chooser (0, 
        "Image Files (*.{tif,jpg,png})\tHeader-Files (*.{h,H,hpp})\tC/C++-Files (*.{c,C,cpp,cxx})",  
        Fl_File_Chooser::MULTI, "Open File(s) for Bracketing to HDR");
        
  printf ("Number of selected files = %d\n", count);
  
  for (int i=1; i <= count; i++)
  {
    printf ("\t%s\n", file_chooser_->value(i));
    
    // Bild laden...
    load_file (file_chooser_->value(i));
  }  
}

/**-----------------------------------------------------------
 * load_file()
 *
 * @return: True, if image loaded, False else
 *   
 * Loading a file changes the image base Br.imgVec and outdated thereby
 *   (an existing) Br.camera. All nessecary GUI updating is driven by 
 *   broadcast messages via Br.distrib_event(..) and is done locally in the
 *   event_msg() callbacks of the classes plugged-in into Br.distrib_event().
 *   Here we have to do not much more than to send the right messages.
 *------------------------------------------------------------*/
bool PrimeWindow::load_file(const char* fname)
{
  // Bild laden...
  if (! cpaint_load_image(fname))   // fills the global `Br'
    return false;

  // Automatic exposure time generation...
  Br.imgVec.set_times_by_stops(val_stops_);   // Immer sinnvoll?
  
  // Set global status variable and broadcast IMAGE_LOADED...
  Br.camera_uptodate (false);
  Br.distrib_event.value (BracketingCore::IMAGE_LOADED);
   
  // After image loading we must either init camera (-> CAMERA_INIT)
  //  or broadcast CAMERA_OUTDATED (entire alternative!), else GUI could
  //  get confused. But what, if no camera exists till now and only one image 
  //  is loaded - the camera can't initalized for one image. Makes it sense
  //  to say then, the camera is "outdated" if it doesn't exist?! See the
  //  comment in ["ImagePanel.cpp"]:ImagePanel::event_msg().
  //  What would happen, if we wouldn't send any CAMERA-Msg then?
  
  // (Re)init camera if possible...
  if (Br.imgVec.size_active() > 1) {
    Br.init_camera();       // sends CAMERA_INIT
    // Choice a probably good ref. picture... (provisionally)
    Br.set_refpic(Br.camera()->nImages() / 2);
  }
  else
    Br.distrib_event.value (BracketingCore::CAMERA_OUTDATED);
 
  return true;
}


/**-----------------------------------------------------------
 * event_msg()  --  Distributor Callback
 *
 * Obacht: Fuer ausgeschaltetes BR_WITH_SHIFT faellt der Menuepunkt "Shift"
 *   weg, alle Positionen ab 14 vermindern sich um 1, 14 wird ein '\0'-Item.
 *   Dessen De/Aktivierung stoert nicht, deshalb hier erstmal so gelassen.
 *------------------------------------------------------------*/
void PrimeWindow::event_msg (BracketingCore::Event e)
{
  SPUR_BR_EVENT(("PrimeWindow::%s(%d): %s\n", __func__, e, br_eventnames[e]));

  Fl_Menu_Item* menu = (Fl_Menu_Item*)menubar_->menu();
  
  // De|activate the items/buttons depending on imgVec.size_active()...
  if (Br.imgVec.size_active() > 1) {
    if (!menu[6].active()) menu[6].activate();    // "init camera"
    if (!menu[9].active()) menu[9].activate();    // "Make HDR image"
  }
  else {
    if (menu[6].active()) menu[6].deactivate();   // "init camera"
    if (menu[9].active()) menu[9].deactivate();   // "Make HDR image"
  }

  // De|activate the items/buttons depending on camera...
  if (Br.camera() && Br.camera()->nImages() > 1) {
    if (!menu[ 7].active()) menu[ 7].activate();  // "Compute CCD curve"
    if (!menu[ 8].active()) menu[ 8].activate();  // "Compute HDR image"
    //if (!menu[12].active()) menu[12].activate();  // "FollowUpCurves"
    //if (!menu[13].active()) menu[13].activate();  // "Shifts" or "Response"
    //if (!menu[14].active()) menu[14].activate();  // "CCD" or \0
  } 
  else {
    if (menu[ 7].active()) menu[ 7].deactivate();  // "Compute CCD curve"
    if (menu[ 8].active()) menu[ 8].deactivate();  // "Compute HDR image"
    //if (menu[12].active()) menu[12].deactivate();  // "FollowUpCurves"
    //if (menu[13].active()) menu[13].deactivate();  // "Shifts" or "Response"
    //if (menu[14].active()) menu[14].deactivate();  // "CCD" or \0
  }

  if (Br.size() < 2) 
    statusline_ -> out ("Please load at least two images first.");
  else  
    statusline_ -> out_default();
}


void PrimeWindow::cb_help_about ()
{
  if (!BrGUI.pHelpAbout) 
    BrGUI.pHelpAbout = new HelpAbout(this);
  
  BrGUI.pHelpAbout -> show();
}


void PrimeWindow::cb_help_tutorial ()
{
  if (!BrGUI.pHelpTutorial)
    BrGUI.pHelpTutorial = new HelpTutorial;
  
  BrGUI.pHelpTutorial -> show();
}


//=================================
// Definitions of static members...
//=================================

/**
 * menu_menubar
 */
Fl_Menu_Item PrimeWindow::menu_menubar_[] = {
  {"&File",0,0,0,FL_SUBMENU},
    {"&Open", FL_ALT+'o', (Fl_Callback*)cb_open_file_},
    {"Clear", 0,          (Fl_Callback*)cb_clear_    },
    //{"Load response fnct", 0,0,0, FL_MENU_INACTIVE},
    {"&Quit", FL_ALT+'q', (Fl_Callback*)cb_quit_},
    {0},
  
  {"Make",0,0,0,FL_SUBMENU},
    {"&Init Calculator", FL_ALT+'i', (Fl_Callback*)cb_init_camera_, 0, FL_MENU_INACTIVE},
    
    {"Compute &Response functions", FL_ALT+'r', (Fl_Callback*)cb_compute_CCD_, 0, FL_MENU_INACTIVE},

    {"Compute &HDR image", FL_ALT+'h', (Fl_Callback*)cb_compute_HDR_, 0, FL_MENU_INACTIVE},
    
    {"Make HDR image", 0,              (Fl_Callback*)cb_make_HDR_, 0, FL_MENU_INACTIVE},
    
    {0},

#if 0  
  {"Analyze",0,0,0,FL_SUBMENU},  
    {"FollowUpCurves",0, (Fl_Callback*)cb_FollowCurveWindow_, 0, FL_MENU_INACTIVE}, 
#ifdef BR_WITH_SHIFT  
    {"Shifts",0, (Fl_Callback*)cb_ShiftMenu_, 0, FL_MENU_INACTIVE},
#endif    
    {"Response functions",  0, (Fl_Callback*)cb_CCDCurveWindow_, 0, FL_MENU_INACTIVE},
    {0},
#endif
  
  {"Analyze",0,0,0,FL_SUBMENU},  
    {"FollowUpCurves",0, (Fl_Callback*)cb_FollowCurveWindow_},
#ifdef BR_WITH_SHIFT  
    {"Shifts",0, (Fl_Callback*)cb_ShiftMenu_},
#endif    
    {"Response functions",  0, (Fl_Callback*)cb_CCDCurveWindow_},
    {0},
/*
  {"Stops",0,0,0,FL_SUBMENU},
    {menuMap_stops[0].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)0},
    {menuMap_stops[1].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)1},  
    {menuMap_stops[2].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)2},  
    {menuMap_stops[3].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)3},
    {menuMap_stops[4].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)4},
    {menuMap_stops[5].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)5},
    {menuMap_stops[6].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)6},
    {menuMap_stops[7].text, 0, (Fl_Callback*)cb_stops_menubar_, (void*)7},
    {0},
*/
  {"&Help",0,0,0,FL_SUBMENU},
    {"About...", 0, (Fl_Callback*)cb_help_about_},
    {"HDR-Tutorial...", 0, (Fl_Callback*)cb_help_tutorial_},
    {0},
  {0}
};

// END of FILE
