/*
 * br_Image.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file br_Image.cpp  
   
  Referenzaehlende Image-Klasse.
*/
#include <iostream>
#include <cmath>                    // pow()

#include "br_types.hpp"             // ImageID
#include "br_enums.hpp"             // DataType, ReportWhat
#include "br_enums_strings.hpp"     // DATA_TYPE_STR[] etc.
#include "br_messages.hpp"          // ERROR_MSG()
#include "br_Image.hpp"             // Image, ExposureData, BrImgVector
#include "br_macros.hpp"            // CTOR(), DTOR()


namespace br {

/*=============================================================================
// 
//  class  Image
// 
//===========================================================================*/

/**+*************************************************************************\n
  Static function, usable in the `:-scope of a Ctor.
******************************************************************************/
int Image::byte_depth (DataType dtype)
{
     switch (dtype) 
     {
       case DATA_U8:  return 1;
       case DATA_U16: return 2;
       case DATA_F32: return 4;
       case DATA_F64: return 8;
       default: ERROR_MSG(("Not handled data type: %d - '%s'", dtype, data_type_str(dtype)));
     }
     return 0;
}

/**+*************************************************************************\n
  Default Ctor
******************************************************************************/
Image::Image()      
  : 
    id_         (0),
    width_      (0),
    height_     (0),
    channels_   (0),
    byte_depth_ (0),
    bit_depth_  (0),
    data_type_     (DATA_NONE),
    storing_scheme_(STORING_NONE),
    image_type_    (IMAGE_NONE)
{
    CTOR("Default");
  
    // ID-Vergabe wie im Init-Ctor sinnvoll? Ich denke, nein. Wir erlauben
    //  keine Moeglichkeit, ein solches Leer-Objekt zu fuellen, Objekte
    //  koennen nur kopiert werden, wobei die ImageID mitkopiert wird! 
}                  

/**+*************************************************************************\n
  Ctor with ImageType argument
******************************************************************************/
Image::Image (int W, int H, ImageType iType, const char* name)
{
    CTOR(name);
    
    image_type_ = iType;
    
    switch (iType)
    {
      case IMAGE_RGB_U8:
        _Image (W,H, 3, DATA_U8, STORING_INTERLEAVE, name); break;
      
      case IMAGE_RGB_U16:
        _Image (W,H, 3, DATA_U16, STORING_INTERLEAVE, name); break;
      
      case IMAGE_RGB_F32:
        _Image (W,H, 3, DATA_F32, STORING_INTERLEAVE, name); break;
        
      default:
        //image_type_ = IMAGE_UNKNOWN;
        ERROR_MSG(("Unhandled ImageType: %d - '%s'\n", iType, image_type_str(iType)));  
        exit(1);                            /// @TODO besser dann werfen!
    }
}                  

/**+*************************************************************************\n
  Ctor for INTERLEAVE storing schemes 
******************************************************************************/
Image::Image (int W, int H, int channels, DataType dtype, const char* name)
{
    CTOR(name);
    _Image (W,H, channels, dtype, STORING_INTERLEAVE, name);
    eval_image_type();
}                  

/**+*************************************************************************\n
  Private helper for Ctors.   No setting of `image_type_!!
******************************************************************************/
void Image::_Image (int W, int H, int channels, DataType dtype, 
                        StoringScheme store, const char* name)
{
    id_counter_++;  
    if (id_counter_==0) id_counter_= 1;     // counter overflow: start again with `1
    id_ = id_counter_;
    
    width_          = W;
    height_         = H;
    channels_       = channels;
    byte_depth_     = byte_depth (dtype);
    bit_depth_      = 8 * byte_depth_;      // default: all bits
    data_type_      = dtype;
    storing_scheme_ = store;

    if (name) {
      name_ = TNT::i_refvec<char> (strlen(name) + 1);
      strcpy (name_.begin(), name);
    }

    int buffer_size = W * H * channels_ * byte_depth_;
    *(TNT::Array1D<unsigned char>*)this = TNT::Array1D<unsigned char> (buffer_size);
}

/**+*************************************************************************\n
  Private helper for Ctors: Determine image_type from data_type, channels and 
   storing_scheme.
******************************************************************************/
void Image::eval_image_type()
{
    if ((channels_ == 3) && (storing_scheme_ == STORING_INTERLEAVE))
      switch (data_type_) 
      {
        case DATA_U8:  image_type_ = IMAGE_RGB_U8;  break;
        case DATA_U16: image_type_ = IMAGE_RGB_U16; break;
        case DATA_F32: image_type_ = IMAGE_RGB_F32; break;
        case DATA_F64: image_type_ = IMAGE_RGB_F64; break;
        default:       image_type_ = IMAGE_UNKNOWN;
      }
    else
      image_type_ = IMAGE_UNKNOWN; 
}

/**+*************************************************************************\n
  Dtor
******************************************************************************/
Image::~Image()
{   
    DTOR(name());
    
    //std::cout << "\tref counter: " << ref_count() << '\n';
    //std::cout << "\tref counter name: " << name_.ref_count() << '\n';
}

void Image::name (const char* s)
{
    if (s) {
      name_ = TNT::i_refvec<char> (strlen(s) + 1);
      strcpy (name_.begin(), s);
    }
    else
      name_ = TNT::i_refvec<char> ();  // Null-vector
}

void Image::report (ReportWhat what)  const
{
    std::cout << "Report for Image \"" << (name() ? name() : "(unnamed)")
              << "\" (Id=" << imageID() << "):"
              << "\n  [ " << width() << " x " << height() << " ]";
    if (what & REPORT_BASE) 
    {
      std::cout << "\n  [Base Data...]"
                << "\n    channels   : " << channels()
                << "\n    DataType   : " << DATA_TYPE_STR[data_type()]
                << "\n    byte_depth : " << byte_depth()
                << "\n    bit_depth  : " << bit_depth()
                << "\n    buffer_size: " << buffer_size() 
                << "\n    storing scheme: " << STORING_SCHEME_STR[storing()];
    }
    std::cout << '\n';
}


/*=============================================================================
//
//  struct  ExposureData
//
//===========================================================================*/

ExposureData::ExposureData (float tm)
{
    time     = tm;
    aperture = 0.0;
    ISO      = 0.0;
    active   = true;      // eigentlich kein ExposureData
}

void ExposureData::report() const
{  
    std::cout << "  [Exposure Data...]"
              << "\n    Time    : " << time
              << "\n    Aperture: " << aperture
              << "\n    ISO     : " << ISO
              << '\n';
}


/*=============================================================================
//
//  struct  ImgStatisticsData
//
//===========================================================================*/

void ImgStatisticsData::report() const
{  
    std::cout << "  [Image Statistics...]"
              << "\n    average brightn. per pixel: " << brightness
              << "\n    pixel at lower bound      : " << n_low
              << "\n    pixel at upper bound      : " << n_high
              << "\n    pixel in working range    : " << r_wk_range << " %"
              << '\n';
}


/*=============================================================================
//
//  class  BrImage  -  sub-class of Image
// 
//===========================================================================*/

void BrImage::report (ReportWhat what) const
{
    Image::report (what);
    if (what & REPORT_EXPOSE_INFO) exposure.report();
    if (what & REPORT_STATISTICS)  statistics.report();
}



/*=============================================================================
//
//  class  BrImgVector  -  image container
// 
//===========================================================================*/

/**+*************************************************************************\n
  Dtor
******************************************************************************/
BrImgVector::~BrImgVector()
{
    DTOR("");
    vec_.clear();
}

/**+*************************************************************************\n
  DataType of the images (all images should have the same)
******************************************************************************/
DataType BrImgVector::data_type() const
{
    if (vec_.size())  return vec_[0].data_type(); 
    else              return DATA_NONE;
}

/**+*************************************************************************\n
  ImageType of the images (all images should have the same)
******************************************************************************/
ImageType BrImgVector::image_type() const
{
    if (vec_.size())  return vec_[0].image_type(); 
    else              return IMAGE_NONE;
}

/**+*************************************************************************\n
  Insert `img' so, that order in brightness increases. I.e. brightness
   computation had to be done before!
******************************************************************************/
void BrImgVector::add (BrImage & img)
{
    for (int i=0; i < size(); i++)
    {  
      if (img.brightness() < vec_[i].brightness())
      { 
        vec_.insert(vec_.begin()+i, img);
        return;
      }
    } 
    //  No image had smaller brightness than `img': add it back...
    vec_.push_back(img);
}

/**+*************************************************************************\n
  Number of activated Images...
******************************************************************************/
int BrImgVector::size_active() const
{ 
    int s = 0;
  
    for (int i=0; i < size(); i++) 
      if (vec_[i].active())  s++;
  
    return s;
}

/**+*************************************************************************\n
  Set time(i) = time(i-1) * 2^stop
******************************************************************************/
void BrImgVector::set_TimesByStop (double stop)
{
    for (int i=1; i < size(); i++) 
    { 
      vec_[i].exposure.time = vec_[i-1].exposure.time * pow(2.0, stop);
    }
}


/**+*************************************************************************\n
  Report meta data of vector elements...
******************************************************************************/
void BrImgVector::report (ReportWhat what) const
{ 
    printf ("=== BrImgVector: meta data === [%d]\n", size());  
    for (int i=0; i < size(); i++)
    {
      printf("[%2d]: ", i);  vec_[i].report(what);
    }
}



//=============================================================================
//  Helper for the macro NOT_IMPL_IMAGE_CASE()
//=============================================================================
void not_impl_image_case (const char* file, int line, const char* func, const Image& img)
{
    fprintf (stderr, "\nBr2Hdr-WARNING **: [%s:%d] %s():\n\tNot implemented for images of: Type='%s', Data='%s', Storing='%s', channels=%d.\n", 
        file, line, func, 
        IMAGE_TYPE_SHORT_STR [img.image_type()], 
        DATA_TYPE_SHORT_STR  [img.data_type()],
        STORING_SCHEME_STR   [img.storing()],
        img.channels() );
}


//=============================================================================
//
//  Definition of the STATIC elements
//
//=============================================================================
ImageID Image::id_counter_ = 0;


}  // namespace "br"

// END OF FILE
