/* cp_screen.c = Screen procedures and graphic ftns for circle packing */

#include "cp_head.h"

extern void textsw_possibly_normalize();

/* Drawing flag scheme: flags indicate drawing instructions
   for objects -- circles/lines/faces. Bits set as follows:
      1  ->  draw object?
      2  ->  fill? (4 and 16 imply 2, also)
      4  ->  off = foreground, on = background 
             (applies to interior, not border, overriden by bit 16)
      8  ->  border color? (default=foreground, else recorded color)
      16 ->  interior color? (default set by bit 4, on --> recorded color)
      32 ->  display label?

Eg.  flag=3: filled object, in foreground
     flag=9: open object, border in (recorded) color
             (for 'edge', this gives colored edge)
     flag=19: filled in color, border in foreground
     flag=27: filled, border and interior in color
     flag=15: filled with background, border in color
     flag=32: label only (face or circle)

Normally, flag for each type of object; often passed on to
subroutine, so may need color code with it:
Eg. (cflag, ccol, ecol) for circle flag, int color, border color.
*/

int dum_int;

int display_call(struct p_data *p,char *datastr)
{
	int count=0,vert;
	int eflag,fflag,dflag,nflag,cflag,ccol;
	int i,j,col,ecol,fcol,hes,length,hits,pnum;
	int dspeed,v,w,indx;
	double dotsize,r;
	char *nextpoint,*endptr,next[64];
	struct Vertlist *vertlist=NULL,*trace,*facelist=NULL;
	struct Edgelist *edgelist=NULL,*track,*eelist=NULL,*eetrack;
	struct RedList *rtrace;
	complex ctr;
	XPoint *Xptr=NULL;
	struct s_data *q;

	if (cmd_mode) return 1; /* don't try to draw in cmd-only mode */
	pnum=pack_num(p);
	q=p->screen;
	dspeed=draw_speed; 
	if (!(p->status)) return 1;
	if (!xv_get(canvas_frame[pnum],WIN_SHOW,0))
	  {
	    xv_set(canvas_frame[pnum],WIN_SHOW,TRUE,
		   XV_LABEL,packlabels[pnum],0);
	    Flush();
	  }
	nextpoint=datastr;
	if (!grab_next(&nextpoint,next) || next[0]!='-')
	 {
		disp_screen(p,q->cflag_opt,q->fflag_opt,0);
		return 1;
	 }
	hes=p->hes;
	do
	 {
for (i=strlen(next);i<10;i++) next[i]='\0'; /* clear old info */
if (next[0]!='-') return count;
if (next[1]=='w') 
  {
    clear_canvas(p->screen);
    if (hes>0) unitcircle(p->screen); 
    count++;
  }
else if (next[1]=='c') /* draw circles */
  {
    ecol=col=FG_COLOR;
    cflag=1;
    if (strstr(next+2,"c")) cflag += 8; /* color object borders */
    if (strstr(next,"n")) cflag += 32; /* labels */
    if (strstr(next,"f")) cflag += 2; /* want filled circles */
    if (strstr(next,"bg")) cflag += 6;  /* fill with background */
    else if (strstr(next,"fg"));
    else if (cflag & 2) cflag += 16; /* default, use recorded color */
    if (cflag & 4) col=BG_COLOR;

    if ((vertlist=node_link_parse(p,nextpoint,
       &endptr,&hits))!=NULL)
      {
	nextpoint=endptr;
	trace=vertlist;
	do
	  {
	    vert=trace->v;
	    if (cflag & 16) col=p->packK_ptr[vert].color;
	    if (cflag & 8)
	      {
		ecol=p->packK_ptr[vert].color;
		if (ecol>=bc_min && ecol<=bc_max) ecol+=bc_offset;
	      }
	    draw_any_circle(p,vert,cflag,ecol,col,dspeed);
	    count++;
	    trace=trace->next;
	  } while (trace!=NULL);
	vert_free(&vertlist);
	if (!dspeed) refresh_canvas(q);
      }
    else
      {
	q->cflag_opt=cflag;
	disp_screen(p,q->cflag_opt,0,0);
	count++;
      }
  }
else if (next[1]=='d' && p->hes<=0) /* post small dot instead of circle.*/
 {
   dotsize=(0.005)*(p->screen->box.ry-p->screen->box.ly); 
   if (next[2]=='c') dflag=1;  /* color? */
   else dflag=0;
   if ((vertlist=node_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
     {
       nextpoint=endptr;
       trace=vertlist;
       do 
	 {
	   vert=trace->v;
	   count += draw_any_dot(p,vert,dotsize,dflag,1,dspeed);
	   trace=trace->next;
	 } 
       while (trace!=NULL);
       vert_free(&vertlist);
     }
   else for (i=1;i<=p->nodecount;i++) /* default to all */
     count += draw_any_dot(p,i,dotsize,dflag,1,dspeed);
    if (!dspeed) refresh_canvas(p->screen); 
 }
else if (next[1]=='d' && p->hes>0) /* show only small dot
instead of circle, sphere only.*/
  {
    for (i=1;i<=p->nodecount;i++)
      if (p->packK_ptr[i].plot_flag)
	{
	  ccol=p->packK_ptr[i].color;
	  ctr=ss_view(p->screen,p->packR_ptr[i].center,1,&dum_int);
	  if (cos(ctr.re)>0) /* on front */
	    {
	      ctr.re=sin(ctr.im)*sin(ctr.re);
	      ctr.im=cos(ctr.im);
	      circle(p->screen,ctr,.005,7,ccol,ccol,dspeed);
	    }
	}
    if (!dspeed) refresh_canvas(p->screen); 
  }
else if (next[1]=='C' || next[1]=='B')
/* draw circles or circles/faces, recomputed in drawing order */
  {
    if (strstr(next,"c")) cflag=fflag=8; /* color object borders */
    else cflag=fflag=0;
    if (strstr(next,"f")) {cflag += 2;fflag += 2;} /* filled */
    if (strstr(next,"bg")) {cflag += 6;fflag += 6;} /* fill with background */
    else if (strstr(next,"fg"));
    else if (cflag & 2) {cflag += 16;fflag += 16;}
    cflag += 1; /* always want circles */
    if (next[1]=='B') fflag += 1; /* also faces */
    if (strstr(next,"n")) {cflag += 32;fflag += 32;} /* labels */
    stripsp(nextpoint);
    if (*nextpoint=='r') /* want just red circles/faces */
      {
	/* if B, then red faces first */
	if ((fflag & 1)
	    && (facelist=face_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	  {
	    nextpoint=endptr;
	    count += layout_facelist(p,fflag,&facelist,1,0,1);
	  }
	else nextpoint++;

	/* now the circles: here use data in redfaces by segment */
	vert=0;
	for (i=1;p->edge_pair[i].edge;i++) 
	  {
	    /* first (corner) circle black */
	    rtrace=p->edge_pair[i].edge;
	    fcol=ecol=FG_COLOR; /* corner circles black */
	    if (rtrace->next->face==rtrace->prev->face
		&& p->edge_pair[i].edge_indx!=rtrace->v_flag)
	      /* blue face, want end of second edge, so get circle 
		 center from prev face */
	      {
		ctr=rtrace->prev->center;
		r=rtrace->prev->rad;
		if (cflag & 32) vert=p->faces[rtrace->prev->face].
		  vert[rtrace->prev->v_flag];
		draw_any_circle_data(p->screen,p->hes,r,ctr,
		  cflag,ecol,fcol,vert,1);
	      }
	    else
	      {
		ctr=rtrace->center;
		r=rtrace->rad;
		if (cflag & 32) 
		  vert=p->faces[rtrace->face].vert[rtrace->v_flag];
		draw_any_circle_data(p->screen,p->hes,r,ctr,
		  cflag,ecol,fcol,vert,1);
	      }
	    /* now keep going until end */
	    fcol=ecol=p->edge_pair[i].color;
	    if (ecol>=bc_min && ecol<=bc_max) ecol+=bc_offset;
	    if (rtrace->next->face==rtrace->prev->face
		&& (indx=p->edge_pair[i].edge_indx)==rtrace->v_flag)
	      /* blue, just did first edge; increment indx */
	      indx=(indx+1) % 3;
	    else
	      {
		rtrace=rtrace->next_edge;
		indx=(nghb_tri(p,rtrace->prev->face,rtrace->face)+2) % 3;
	      }
	    while (!(rtrace->corner_flag[indx] & 2))
	      {
		 if (rtrace->next->face==rtrace->prev->face
		     && indx!=rtrace->v_flag)
		   /* blue face, second edge; get data from prev edge */
		   {
		     ctr=rtrace->prev->center;
		     r=rtrace->rad;
		     if (cflag & 32) vert=p->faces[rtrace->prev->face].
		       vert[rtrace->prev->v_flag];
		   }
		 else
		   {
		     ctr=rtrace->center;
		     r=rtrace->rad;
		     if (cflag & 32) 
		       vert=p->faces[rtrace->face].vert[rtrace->v_flag];
		   }
		 draw_any_circle_data(p->screen,p->hes,r,ctr,
		     cflag,ecol,fcol,vert,1);
		 if (rtrace->next->face==rtrace->prev->face
		     && indx==rtrace->v_flag)
		   /* blue, just did first edge; just increment indx */
		   indx=(indx+1) % 3;
		 else
		   {
		     rtrace=rtrace->next_edge;
		     indx=(nghb_tri(p,rtrace->prev->face,rtrace->face)+2) % 3;
		   }	 
	      } /* end of while */
	    /* last (corner) circle black */
	    fcol=ecol=FG_COLOR; /* corner circles black */
	    if (rtrace->next->face==rtrace->prev->face
		&& indx!=rtrace->v_flag)
	      /* blue face, second edge; get data from prev face */
	      {
		ctr=rtrace->prev->center;
		r=rtrace->prev->rad;
		if (cflag & 32) vert=p->faces[rtrace->prev->face].
		  vert[rtrace->prev->v_flag];
	      }
	    else
	      {
		ctr=rtrace->center;
		r=rtrace->rad;
		if (cflag & 32) 
		  vert=p->faces[rtrace->face].vert[rtrace->v_flag];
	      }
	    draw_any_circle_data(p->screen,p->hes,r,ctr,
		  cflag,ecol,fcol,vert,1);
	  } /* end of i for-loop through segments */
      } /* end of drawing just red */
    else count += draw_in_order(p,cflag,fflag,0,1);
  } 
else if (next[1]=='e') /* draw edges */
  {
    if (next[2]=='e' && (vertlist=node_link_parse(p,nextpoint,
       &endptr,&hits))!=NULL) /* hex extended edges */
      {
	nextpoint=endptr;
	trace=vertlist;
	while (trace && (v=trace->v)
	       && trace->next && (w=trace->next->v)) /* get pairs */
	  {
	    trace=trace->next;
	    if ((eelist=get_extended_edge(p,v,w,256))!=NULL)
	      {
		eetrack=eelist;
		do 
		  {
		    count+=draw_edge(p,eetrack->v,
				     eetrack->w,FG_COLOR,dspeed);
		    eetrack=eetrack->next;
		  } while (eetrack!=NULL);
		edge_free(&eelist);
	      }
	  }
	vert_free(&vertlist);
	if (!dspeed) refresh_canvas(p->screen);
      }
    else if ((edgelist=node_pair_link(p,nextpoint,&endptr,&hits))
	     !=NULL)
      {
	nextpoint=endptr;
	track=edgelist;
	do {
	  count+=draw_edge(p,track->v,track->w,FG_COLOR,dspeed);
	  track=track->next;
	} while (track!=NULL);
	edge_free(&edgelist);
	if (!dspeed) refresh_canvas(q);
      }
    else 
      {
	q->fflag_opt=1;
	disp_screen(p,0,q->fflag_opt,0);
	count++;
      }
  } /* finished with edges */
else if (next[1]=='f' || next[1]=='F') /* draw faces */
  {
    ecol=col=FG_COLOR;
    if (strstr(next,"c")) fflag=9; /* color object borders */
    else fflag=1;
    if (strstr(next,"n")) fflag += 32; /* labels */
    if (strstr(next+2,"f")) /* filled faces */
      {
	fflag += 2; /* fill (note: fg/bg imply fill also) */
	if (strstr(next,"bg")) fflag += 4;  /* fill with background */
	else if (strstr(next,"fg"));
	else fflag += 16; /* use recorded color */
      }
   if ((facelist=face_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
     {
       nextpoint=endptr;
       if (next[1]=='F')  /* locate faces as they're drawn */ 
	 count += layout_facelist(p,fflag,&facelist,1,0,1);
       else 
	 {
	   count += layout_facelist(p,fflag,&facelist,0,0,dspeed);
	   if (!dspeed) refresh_canvas(p->screen);
	 }
     }
   else if (next[1]=='F') /* use draworder, but recompute centers */
     count += draw_in_order(p,0,fflag,0,1);
   else if ((fflag & 2) && !(fflag & 16)) /* filled, but not color */
     {
       if (p->hes<0) unitcircle(q);
       ecol=col=FG_COLOR;
       for (i=1;i<=p->facecount;i++)
	 {
	   if (fflag & 8)
	     {
	       ecol=p->faces[i].color;
	       if (ecol>=bc_min && ecol<=bc_max) ecol+=bc_offset;
	     }
	   if (fflag & 4) col=BG_COLOR;
	   count += draw_any_face(p,i,fflag,ecol,col,dspeed);
	 }
       if (!dspeed) refresh_canvas(q);
    }
   else 
     {
       q->fflag_opt=fflag;
       disp_screen(p,0,q->fflag_opt,0);
       count++;
     }
  } /* finished with faces */
else if (next[1]=='R' 
  && p->edge_pair[2].edge) /* specified paired edges in non-simply 
			       connected cases; draw thick & in color */
  {
    if (strstr(next,"p")) dflag=1; /* draw mated pairs */
    else dflag=0;
    nflag=8; /* flag of options for draw_bdry_edge; 8=color */
    if (strstr(next,"n")) nflag |= 4; /* label segments */
    if (strstr(next,"c")) nflag |= 16; /* draw numbered circles also */
    length=0;
    while (p->edge_pair[length+1].edge) length++;
    /* fixup: should we know number of segments from euler/etc? */
    XSetLineAttributes(display,gc,4, 
       LineSolid,CapRound,JoinRound); /* set thicker line */
    if ((vertlist=node_link_parse(p,nextpoint,
       &endptr,&hits))!=NULL)
      {
	nextpoint=endptr;
	trace=vertlist;
	while (trace && (j=trace->v)>0 && j <= length)
	  {
	    count += draw_bdry_seg(p,j,nflag,p->edge_pair[j].color);
	    if (dflag && (i=index_of_paired_edge(p,j)))
	      draw_bdry_seg(p,i,nflag,p->edge_pair[i].color);
	    trace=trace->next;
	  }
      }
    else for (i=1;p->edge_pair[i].edge;i++)
      draw_bdry_seg(p,i,nflag,p->edge_pair[i].color);
    vert_free(&vertlist);
    XSetLineAttributes(display,gc,1, 
       LineSolid,CapRound,JoinRound); /* reset line thickness */
    /* fixup: should restore original thickness */
  }
else if (next[1]=='s') /* polygon through vertlist */
  {
    col=FG_COLOR;
    eflag=1;
    if (strstr(next,"bg")) {col=BG_COLOR;eflag += 6;}
    else if (strstr(next,"fg")) eflag +=2;
    else if (strstr(next,"f")) eflag += 18;

    if ((vertlist=node_link_parse(p,nextpoint,
       &endptr,&hits))!=NULL)
      {
	trace=vertlist;
	while(trace && trace->next) {trace=trace->next;} /* find end? */
	if (trace && trace->v!=vertlist->v) /* repeat first vert at end */
	  {
	    trace->next=(struct Vertlist *)
	      calloc((size_t)1,sizeof(struct Vertlist));
	    trace->next->v=vertlist->v;
	  }	    
	nextpoint=endptr;
	trace=vertlist;
	if (p->hes<0)
	  count += h_polygon(p,vertlist,eflag,FG_COLOR,col,dspeed);
	else if (p->hes==0)
	  count += e_polygon(p,vertlist,eflag,FG_COLOR,col,dspeed);
	else count += s_polygon(p,vertlist,eflag,FG_COLOR,col,dspeed);
	vert_free(&vertlist);
	refresh_canvas(p->screen);
      }
  }
else if (next[1]=='n') /* label */
  {
    if (next[2]=='f') /* faces */
      {
	if ((facelist=face_link_parse(p,
	   nextpoint,&endptr,&hits))!=NULL)
	  {
	    nextpoint=endptr;
	    trace=facelist;
	    do {
	      count += 
		draw_face_number(p,trace->v,trace->v,dspeed);
	      trace=trace->next;
	    } while (trace!=NULL);
	    vert_free(&facelist);
	    refresh_canvas(p->screen);
	  }
	else 
	  {
	    disp_screen(p,0,32,0);
	    count++;
	  }
      }
    else if (next[2]=='c') /* circles */
      {
	if ((vertlist=node_link_parse(p,
	   nextpoint,&endptr,&hits))!=NULL)
	  {
	    nextpoint=endptr;
	    trace=vertlist;
	    do {
	      count +=
		draw_cir_number(p,trace->v,trace->v,dspeed);
	      trace=trace->next;
	    } while (trace!=NULL);
	    vert_free(&vertlist);
	    refresh_canvas(p->screen);
	  }
	else
	  {
	    disp_screen(p,32,0,0);
	    count++;
	  }
      }
    else if ( 			/* 'l' or 'z' */
      (
        (
	  next[2]=='z'  		/* given point */
		&& grab_next(&nextpoint,next)
		   && sscanf(next,"%lf",&(ctr.re))
		&& grab_next(&nextpoint,next)
		   && sscanf(next,"%lf",&(ctr.im))
		&& !(i=0)
	)
	||
	(
	  next[2]=='l'  		/* or center of circle */
	  && (i=grab_one_vert(p,&nextpoint))
	)
      )
      && grab_next(&nextpoint,next) 
      && strlen(strncpy(buf,next,32))!=0 
      )
		/* put string either at cent of i or at given ctr. */
      {
	strcat(buf,"\0");
	if (i>0) ctr=p->packR_ptr[i].center;
	count += write_at_pt(p,ctr,buf,1); /* write string at pt. */
      }
  } /* finished with labeling */
else if (next[1]=='g') /* display a closed path */
  {
    if ((length=pathlength)>3)
      {
	Xptr=path_XPoints(p->screen,pathlist,&length);
	DrawLines(p->screen->xpm,Xptr,length,fgcolor);
	count++;
      }
    if (next[2]=='i')  /* find and draw circles inside path. */
      {
	count+=find_path_int(p); 
	for (i=1;i<=p->nodecount;i++)
	  if (p->packK_ptr[i].util_flag)
	    draw_any_circle(p,i,7,FG_COLOR,FG_COLOR,0);
      }
    free(Xptr);
    refresh_canvas(p->screen);
  }
else if (next[1]=='r' && next[2]=='c' && (rtrace=p->redfaces)) 
  /* test: draw circles from redlist data */
  {
    ecol=col=FG_COLOR;
    cflag=1;
    if (strstr(next+3,"c")) cflag += 8; /* color object borders */
    if (strstr(next,"n")) cflag += 32; /* labels */
    if (strstr(next,"f")) cflag += 2; /* want filled circles */
    if (strstr(next,"bg")) cflag += 6;  /* fill with background */
    else if (strstr(next,"fg"));
    else if (cflag & 2) cflag += 16; /* default, use recorded color */
    if (cflag & 4) col=BG_COLOR;

    i=0;
    while (rtrace!=p->redfaces || !(i++))
      {
	vert=p->faces[rtrace->face].vert[rtrace->v_flag];
	if (cflag & 16) col=p->packK_ptr[vert].color;
	if (cflag & 8)
	  {
	    ecol=p->packK_ptr[vert].color;
	    if (ecol>=bc_min && ecol<=bc_max) ecol+=bc_offset;
	  }
	count += draw_any_circle_data(p->screen,p->hes,rtrace->rad,
	  rtrace->center,cflag,ecol,col,vert,1);
	rtrace=rtrace->next;
      }
  }
    
else if (next[1]=='r') /* reset */
  {
    reset_screen(p->screen);
    count++;
  }
else if (next[1]=='u') /* unit circle */
  {
    if (p->hes>0) equator(p->screen);
    else unitcircle(p->screen);
  }
else if (next[1]=='x') /* display coord axes */
  {
    coord_axes(p);
  }
else if (next[1]=='t') /* display caption (filename=default); must
	be last action requested in this disp call. */
  {
    if (strlen(strcpy(buf,nextpoint))==0) strcpy(buf,p->file_name);
    count += caption(p->screen,buf);
    return count;
  } 
else if (next[1]=='z') /* set draw_speed */
  {
    if (next[2]=='f') dspeed = 0; /* fast (i.e. accumulate, display once*/
    else if (next[2]=='s') dspeed =1; /* slow */
    else dspeed=(dspeed) ? 0 : 1; /* toggle */
  }
	 } /* end of do */
	while (nextpoint!=NULL && grab_next(&nextpoint,next) );
	return count;		
} /* display_call */

int disp_screen(struct p_data *p,int cflag,int fflag,int show)
/* display pack p, standard flag options. */
{
  if (!p->status)
    {sprintf(msgbuf,"pack %d is empty",pack_num(p));msg();return 1;}
  if (p->hes>0) unitcircle(p->screen);
  if (p->screen->unitcircle)
    {
      if (p->hes>0) equator(p->screen);
      else unitcircle(p->screen);
    }
  if (p->screen->coord_flag) coord_axes(p);
  if (cflag) draw_circles(p,cflag,show);
  if (fflag) draw_complex(p,fflag,show);
  refresh_canvas(p->screen); 
  return 1;
} /* disp_screen */

int coord_axes(struct p_data *p)
/* display coord axes on screen */
{
  int front,i;
  complex pt,end1,end2;
  double V[3];

/* new (possibly temporary), May 2000: in hyp case, put spoke/circle
   grid on unit disc, 8 lines each. */

  if (cmd_mode) return 1;
  if (p->hes<0)
    {
      end1.re=end1.im=0.0;
      for (i=0;i<6;i++)
	{
	  end2.re=cos(M_PI*i/3);
	  end2.im=sin(M_PI*i/3);
	  kline(p->screen,end1,end2,FG_COLOR,1);
	  circle(p->screen,end1,(double)i/6,1,FG_COLOR,FG_COLOR,1);
	}
      return 1;
    }
	if (p->hes<=0)
	 {
		end1.re=-1.1;end2.re=1.1;
		end1.im=end2.im=0.0;
		kline(p->screen,end1,end2,FG_COLOR,1);
		end1.re=end2.re=0.0;
		end1.im=-1.1;end2.im=1.1;
		kline(p->screen,end1,end2,FG_COLOR,1);
		return 1;
	 }
/* x-axis */
	end1.re=end1.im=0.0;
	end2.re=0.0;end2.im=M_PI_2;
	pt=ss_view(p->screen,end2,1,&front);
	s_pt_to_vec(pt,V);
	end2.re=V[1]*1.1;end2.im=V[2]*1.1;
	kline(p->screen,end1,end2,FG_COLOR,1);
/* y-axis */
	end2.re=M_PI_2;end2.im=M_PI_2;
	pt=ss_view(p->screen,end2,1,&front);
	s_pt_to_vec(pt,V);
	end2.re=V[1]*1.1;end2.im=V[2]*1.1;
	kline(p->screen,end1,end2,FG_COLOR,1);
/* z-axis */
	end2.re=0.0;end2.im=0.0;
	pt=ss_view(p->screen,end2,1,&front);
	s_pt_to_vec(pt,V);
	end2.re=V[1]*1.1;end2.im=V[2]*1.1;
	kline(p->screen,end1,end2,FG_COLOR,1);
	return 1;
} /* coord_axes */

int caption(struct s_data *q,char *datastr)
/* writes buffer contents as title on screen */
{
	int x,y;

	if (cmd_mode) return 1;
	x=40; /* offset from edge */
	y=q->pix_box.ry-q->pix_box.ly-x;
	DrawString(q->xpm,x,y,buf,strlen(datastr));
	refresh_canvas(q);
	return 1;
} /* caption */

int clear_canvas(struct s_data *q)
/* clear canvas q */
{
	if (cmd_mode) return 1;
	SetForeground(bgcolor,0);
	FillRectangle(q->xpm,0,0,MAX_PIXEL,MAX_PIXEL);
	SetForeground(fgcolor,0);
	refresh_canvas(q);
	return 1;
} /* clear_canvas */

int draw_in_order(struct p_data *p,int cflag,int fflag,int pl,int show)
/* draw all faces and/or all circles in drawingorder, but draw and 
recompute as you go along. E.g., will give 'ghosts'. Standard flag 
arrangement. If pl is set, then place first face; else, leave in 
current location. */
{
	int nf,ind,a,b,c,ck=0,col,ccol,v,n0,indx;

   /* handle first face */
	nf=p->first_face;
	ccol=col=FG_COLOR;
	if (pl) place_face(p,nf,p->faces[nf].index_flag);
	if (cflag & 1) /* circles? */
	 {
	   ind=p->faces[nf].index_flag;
	   a=p->faces[nf].vert[ind];
	   b=p->faces[nf].vert[(ind+1) % 3];
	   c=p->faces[nf].vert[(ind+2) % 3];
	   if (cflag & 16) col=p->packK_ptr[a].color;
	   if (cflag & 8)
	     {
	       ccol=p->packK_ptr[a].color;
	       if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
	     }
	   draw_any_circle(p,a,cflag,ccol,col,show);
	   if (cflag & 16) col=p->packK_ptr[b].color;
	   if (cflag & 8)
	     {
	       ccol=p->packK_ptr[b].color;
	       if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
	     }
	   draw_any_circle(p,b,cflag,ccol,col,show);
	   if (cflag & 16) col=p->packK_ptr[c].color;
	   if (cflag & 8)
	     {
	       ccol=p->packK_ptr[c].color;
	       if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
	     }
	   draw_any_circle(p,c,cflag,ccol,col,show);
	 }
	ccol=col=FG_COLOR;
	if (fflag & 1) /* faces? */
	  {
	    if (fflag & 16) col=p->faces[nf].color;
	    if (fflag & 8)
	      {
		ccol=p->faces[nf].color;
		if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
	      }
	    draw_any_face(p,nf,fflag,ccol,col,show);
	  }
   /* now the rest */
	while ( nf>0 && nf <= p->facecount 
		&& (nf=p->faces[nf].next_face)!=p->first_face
		&& ck<2*p->facecount )
	 {
	   v=p->faces[nf].vert[(indx=(p->faces[nf].index_flag+2)%3)];
	   n0=nghb(p,v,p->faces[nf].vert[(indx+1)%3]);
	   fancy_comp_center(p,v,n0,n0,1,0,1,1,toler);
	   col=ccol=FG_COLOR;
	   if (cflag & 1) /* circle? */
	     {
	       c=p->faces[nf].vert[(p->faces[nf].index_flag + 2) % 3];
	       if (cflag & 16) col=p->packK_ptr[c].color;
	       if (cflag & 8)
		 {
		   ccol=p->packK_ptr[c].color;
		   if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
		 }
	       draw_any_circle(p,c,cflag,ccol,col,show); 
				/* new circle */
	     }
	   ccol=col=FG_COLOR;
	   if (fflag & 1) /* faces? */
	     {
	       if (fflag & 16) col=p->faces[nf].color;
	       if (fflag & 8)
		 {
		   ccol=p->faces[nf].color;
		   if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
		 }
	       draw_any_face(p,nf,fflag,ccol,col,show);
	     }
	   ck++;
	 }
	if (nf!=p->first_face) /* some error */
		return 0;
	return 1;
} /* draw_in_order */

struct Vertlist *parse_facepath(struct p_data *p,struct Pathlist *path,
				int *livenode)
/* list faces along given path. Pass (possible) start and ending node. */
{
	int flag,k;
	complex ctr[3],pt,normpt;
	double dum,det;
	struct Vertlist *thelist=NULL,*trace=NULL,*nodelist;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if(path==NULL) return NULL;
	pt.re=path->x;pt.im=path->y;
	if (*livenode<1 || *livenode>p->facecount 
	   || !pt_in_tri(p,*livenode,pt.re,pt.im)) 
		*livenode=0;
	if (*livenode)
	 {
		trace=thelist=(struct Vertlist *)
			calloc(1,sizeof(struct Vertlist));
		trace->v=*livenode;
	 }
	else {while (path!=NULL && thelist==NULL)
	 {
		r_to_pix(pt,&normpt,p->screen->pix_box,p->screen->box);
		if ( (nodelist=tri_search(p,
	   	   (int)(normpt.re),(int)(normpt.im)))!=NULL)
		 {
			trace=thelist=(struct Vertlist *)
			   calloc(1,sizeof(struct Vertlist));
			trace->v=*livenode=nodelist->v;
			vert_free(&nodelist);
		 }
		path=path->next;
	 }}
	if (path==NULL) return thelist;
	while (path!=NULL)
	 {
		for (k=0;k<=2;k++)
		 {
		   if (p->hes<0) 
			h_to_e_data(pR_ptr[p->faces[*livenode].vert[k]].
	   	           center,pR_ptr[p->faces[*livenode].vert[k]].rad,
		           &ctr[k],&dum);
		   else ctr[k]=pR_ptr[p->faces[*livenode].vert[k]].center;
		 }
		if ((ctr[0].re*(ctr[1].im-ctr[2].im)
		   -ctr[1].re*(ctr[0].im-ctr[2].im)
		   +ctr[2].re*(ctr[0].im-ctr[1].im))>0) det=1.0;
		else det=-1.0;
		while (path!=NULL 
		   && (det*row_col(path->x,path->y,
			ctr[1].re,ctr[2].re,ctr[1].im,ctr[2].im))>0
	   	   && (det*row_col(path->x,path->y,
			ctr[2].re,ctr[0].re,ctr[2].im,ctr[0].im))>0
	   	   && (det*row_col(path->x,path->y,
			ctr[0].re,ctr[1].re,ctr[0].im,ctr[1].im))>0 )
			path=path->next;
		flag=0;
		while (path!=NULL && !flag)
		 {
			pt.re=path->x;pt.im=path->y;
			r_to_pix(pt,&normpt,p->screen->pix_box,
				p->screen->box);
			if ( ((nodelist=tri_search(p,
	   	  	 (int)(normpt.re),(int)(normpt.im)))!=NULL) && 
			 nodelist->v!=*livenode )
			 {
				trace->next=(struct Vertlist *)
				   calloc(1,sizeof(struct Vertlist));
				trace=trace->next;
				trace->v=*livenode=nodelist->v;
				vert_free(&nodelist);
				flag=1;
			 }
			path=path->next;
		 }
	 }
	return thelist;
} /* parse_facepath */

int refresh_canvas(struct s_data *q)
/* puts pixmap onto canvas */
{
	int snum=screen_num(q);
	
	if (cmd_mode) return 1;
	if (canvas[snum])
	 {
	   XCopyArea(display,q->xpm,q->xid,gc,0,0,
		q->pix_box.rx,q->pix_box.ry,0,0);
	   Flush(); /* ?? put this after next? */
	   if ((int) xv_get(manip_frame[snum],WIN_SHOW))
		xv_set(manip_frame[snum],WIN_SHOW,TRUE,0);
	 }
	return 1;
} /* refresh_canvas */

int set_cursor(struct s_data *q,int m)
/* set ms_flag, set item, set cursor.
   (old "enfold" cursor no longer used, it's middle button on add_cir. */
{
	int code=0,snum=screen_num(q);
	Canvas canvas_q;

	if (cmd_mode) return 1;
	canvas_q=canvas[snum];
	if (packdata[snum].locks && m>1 && m <8 && m!=5) m=0;
	  /* certain actions inapprop for locked pack */
	q->ms_flag=m;
	switch(m)
	 {
case 0: /* desensitized mouse */
 {
	if (packdata[snum].locks) xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, lock_cursor,0);
	else if (current_p==snum) xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, act_can_cursor,0);
	else xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR,inact_can_cursor,0);	
	code=0;
	break;
 }
case 2: /* add_cir and enfold (on left and middle buttons, resp.) */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, add_cursor,0);
	code=1;
	break;
 }
case 3: /* rm_cir */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, delete_cursor,0);
	code=3;
	break;
 }
case 4: /* enfold */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, enfold_cursor,0);
	code=2;
	break;
 }
case 5: /* coords */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, axes_cursor,0);
	code=9;
	break;
 }
case 6: /* inc_rad */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR,incr_cursor,0);
	code=4;
	break;
 }
case 7: /* dec_rad */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR,decr_cursor,0);
	code=5;
	break;
 }
case 8: /* path */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR,path_cursor,0);
	code=7;
	break;
 }
case 9: /* project */
 {
	xv_set(canvas_paint_window(canvas_q),
		WIN_CURSOR, proj_cursor,0);
	code=10;
	break;
 }
	 } /* end of switch */
 	panel_set(manip_item[snum],PANEL_VALUE,code,0);
	return 1;
} /* set_cursor */

/* ==================== various message printers ===================== */

	/* message in panel */

int msg()
/* send msgbuf contents to message subwindow */
{
	int textlength;

	if (cmd_mode)
	 {
	 	fprintf(stdout,msgbuf);
		if (log_fp) fprintf(log_fp,"MSG: %s\n",msgbuf);
	 	return 1;
	 }
	xv_set(msg_sw,TEXTSW_INSERTION_POINT,TEXTSW_INFINITY,0);
	textlength=strlen(msgbuf)-1;
	if (textlength<0) return 1;
	if (textlength>254) msgbuf[256]='\0';
	if (msgbuf[textlength]!='\n') strcat(msgbuf,"\n");
	textsw_insert(msg_sw,msgbuf,strlen(msgbuf));
	if ( (textlength=((int)xv_get(msg_sw,TEXTSW_LENGTH)-MSG_LENGTH))>0 )
	 {
		textsw_erase(msg_sw,0,textlength+1);
	 }
	textsw_possibly_normalize(msg_sw,
		(Textsw_index)xv_get(msg_sw,TEXTSW_INSERTION_POINT));
	Flush();
	return 1;
} /* msg */

int emsg()
/* send emsgbuf contents to message subwindow */
{
	int textlength;

	if (cmd_mode)
	 {
	 	fprintf(stdout,emsgbuf);
		if (log_fp) fprintf(log_fp,"ERROR: %s\n",emsgbuf);
	 	return 1;
	 }
	xv_set(msg_sw,TEXTSW_INSERTION_POINT,TEXTSW_INFINITY,0);
	textlength=strlen(msgbuf)-1;
	if (textlength<0) return 1;
	if (textlength>264) emsgbuf[266]='\0';
	if (msgbuf[textlength]!='\n') strcat(emsgbuf,"\n");
	textsw_insert(msg_sw,emsgbuf,strlen(emsgbuf));
	if ( (textlength=((int)xv_get(msg_sw,TEXTSW_LENGTH)-MSG_LENGTH))>0 )
	 {
		textsw_erase(msg_sw,0,textlength+1);
	 }
	textsw_possibly_normalize(msg_sw,
		(Textsw_index)xv_get(msg_sw,TEXTSW_INSERTION_POINT));
	window_bell(msg_sw);window_bell(msg_sw);
	Flush();
	return 1;
} /* emsg */

int smsg()
/* send msgbuf contents to end of scratch subwindow, reset current
   to original position. */
{
	int textlength;
	Textsw_index point;

	if (cmd_mode)
	 {
	 	fprintf(stdout,msgbuf);
		if (log_fp) fprintf(log_fp,"SCRATCH: %s\n",msgbuf);
	 	return 1;
	 }
	point=(Textsw_index) 
	  xv_get(scratch_sw,TEXTSW_INSERTION_POINT);
	xv_set(scratch_sw,TEXTSW_INSERTION_POINT,TEXTSW_INFINITY,0);
	textlength=strlen(msgbuf)-1;
	if (textlength<0) return 1;
	if (textlength>254) msgbuf[256]='\0';
	textsw_insert(scratch_sw,msgbuf,strlen(msgbuf));
	if ( (textlength=((int)xv_get(scratch_sw,
				      TEXTSW_LENGTH)-MSG_LENGTH))>0 )
	 {
		textsw_erase(scratch_sw,0,textlength+1);
	 }
	textsw_possibly_normalize(scratch_sw,
		(Textsw_index)xv_get(scratch_sw,TEXTSW_INSERTION_POINT));
	xv_set(scratch_sw,TEXTSW_INSERTION_POINT,
	       point,NULL);
	Flush();
	return 1;
} /* smsg */

	/* in history subwindow */
int hmsg(char *msg)
{
  if (cmd_mode)  return 1;
  xv_set(log_sw,TEXTSW_INSERTION_POINT,TEXTSW_INFINITY,NULL);
  textsw_insert(log_sw,msg,strlen(msg));
  return 1;
} /* hmsg */

	/* set labels of packs */
int pmsg(int i,char *msg)
{
  if (cmd_mode)  return 1;
  if (strlen(msg)>45) msg[45]='\0';
  strcpy(packlabels[i]+8,msg);
  set_pack_labels();
  return 1;
}

	/* message out to remote procedure */

int msg_remote(int i)
{
	if (i<0 || i > NUM_PROC || !remote[i].pid) return 1;
	if (strlen(msgbuf)>255) msgbuf[256]='\0';
	if (msgbuf[strlen(msgbuf)]!='\n') strcat(msgbuf,"\n");
	fprintf(remote[i].fp_to_remote,"%s",msgbuf);
	fflush(remote[i].fp_to_remote);
	return 1;
}

int canvmsg(int i)
{
  if (cmd_mode)  return 1;
  xv_set(canvas_frame[i],XV_LABEL,packlabels[i],0);
  return 1;
}

int set_pack_labels()
{
  if (cmd_mode)  return 1;
  switch (NUM_PACKS)
    {
case 4:
 {
	xv_set(active_pack_item,PANEL_CHOICE_STRINGS,
		packlabels[0],
		packlabels[1],
		packlabels[2],
		packlabels[3],0,
		0);
	break;
 }
case 3:
 {
	xv_set(active_pack_item,PANEL_CHOICE_STRINGS,
		packlabels[0],
		packlabels[1],
		packlabels[2],0,
		0);
	break;
 }
case 2:
 {
	xv_set(active_pack_item,PANEL_CHOICE_STRINGS,
		packlabels[0],
		packlabels[1],0,
		0);
	break;
 }
case 1:
 {
	xv_set(active_pack_item,PANEL_CHOICE_STRINGS,
		packlabels[0],0,
		0);
 }
    } /* end of switch */
  return 1;
} /* set_pack_labels */

int repack_activity_msg(char *datastr)
/* report packing activity */
{
  if (cmd_mode)  return 1;
  textsw_insert(msg_sw,datastr,1); 
  Flush();	
  return 1;
} /* repack_activity_msg */


