#include "cp_types.h"
#include "cp_proto.h"

/* ------------------- Mobius matrix manipulations ------------ */

Mobius ident_mob()
     /* returns identity Mobius transform. */
{
  Mobius identM;

  identM.a.re=identM.d.re=1.0;
  identM.b.re=identM.c.re=0.0;
  identM.a.im=identM.b.im=identM.c.im=identM.d.im=0.0;
  identM.flip=0;
  return identM;
} /* ident_mob */

int matrix_conj(Mobius *M)
{
  M->a=cconj(M->a);
  M->b=cconj(M->b);
  M->c=cconj(M->c);
  M->d=cconj(M->d);
  return 1;
} /* matrix_conj */

int matrix_transpose(Mobius *M)
{
  complex tmp;

  tmp=M->b;
  M->b=M->c;
  M->c=tmp;
  return 1;
} /* matrix_transpose */

  
Mobius matrix_product(Mobius A,Mobius B)
     /* returns AB.  fixup: what to do about flip? */
{
  Mobius C;

  C.a=cadd(cmult(A.a,B.a),cmult(A.b,B.c));
  C.b=cadd(cmult(A.a,B.b),cmult(A.b,B.d));
  C.c=cadd(cmult(A.c,B.a),cmult(A.d,B.c));
  C.d=cadd(cmult(A.c,B.b),cmult(A.d,B.d));
  if ((A.flip && B.flip) || (!A.flip && !B.flip)) C.flip=0;
  else C.flip=1;
  return (C);
} /* matrix_product */

complex det(Mobius M)
     /* returns the determinant of M */
{
  complex answer;
	
  answer=csub(cmult(M.a,M.d),cmult(M.c,M.b));

  return(answer);
} /* det */

Mobius inv_mob(Mobius M)
     /* Returns inverse of Mobius M; returns identity if
|det(M)| is too small. */
{
  complex d,NEG_ONE;
  Mobius newMob;

  NEG_ONE.re=-1.0;NEG_ONE.im=0.0;
  d=det(M);
  if ( cAbs(d) < .00001) return (ident_mob()); /* almost singular */
  newMob.a=cdiv(M.d,d);
  newMob.d=cdiv(M.a,d);
  newMob.c=cmult(NEG_ONE,cdiv(M.c,d));
  newMob.b=cmult(NEG_ONE,cdiv(M.b,d));
  newMob.flip=M.flip;
  return newMob;
} /* inv_mob */

/* ----------------- creating Mobius transforms ------- */

Mobius trans_abAB(complex a,complex b,complex A,complex B,
		  int *err_flag)
     /* return conformal automorphism of unit 
disc mapping a->A, b->B; a,b,A,B must all be on unit circle. */
{
  complex boaa,Boa,boa,Two,One;
  Mobius M;

  M=ident_mob();
  if (cAbs(a)< 1.0-m_toler || cAbs(b)< 1.0-m_toler ||
      cAbs(A)< 1.0-m_toler || cAbs(B)< 1.0-m_toler )
    /* not close enough to being on circle */
    {
      (*err_flag)++;
      return M;
    }
  if (cAbs(csub(a,b))<m_toler || cAbs(csub(A,B))<m_toler) 
    /* too close to work with, just rotate a to A */
    {
      M.a=A;
      return M;
    }
  boaa=cdiv(b,cmult(a,a));
  Boa=cdiv(B,a);
  boa=cdiv(b,a);
  One.re=1.0;Two.re=2.0;
  One.im=Two.im=0.0;
  M.a=cadd(cmult(boaa,B),cdiv(csub(A,cmult(Two,B)),a));
  M.b=csub(B,cmult(boa,A));
  M.c=csub(boaa,cdiv(B,cmult(a,A)));
  M.d=cadd(One,cmult(boa,csub(cdiv(B,A),Two)));

  return M;
} /* trans_abAB */

Mobius auto_abAB(complex a,complex b,complex A,complex B,
		 double *err,int *err_flag)
     /* automorphism of unit disc carrying a->A and b->B, where a,b,A,B 
are interior points. Technically, must have hyp dist (a,b) = hyp 
dist (A,B). But this routine carries geodesic through
a,b to that through A,B, with midpoint going to midpoint. */
{
  int err_hit=0;
  Mobius M1,M2,M,tmp;

  M=ident_mob();
  M1=standard_mob(a,b,&err_hit);
  tmp=standard_mob(A,B,&err_hit);
  if (err_hit)  
    {
      (*err_flag)++;
      return M;
    }
  M2=inv_mob(tmp);
  M=matrix_product(M2,M1);
  *err=cAbs(csub(B,mobius(M,b,1)));
  return M;
} /* auto_abAB */

Mobius standard_mob(complex a,complex b,int *err_flag)
     /* Find automorphism of unit disc carrying a to origin, b to 
positive x-axis. Has form e^{it}(z-a)/(1-a~z), where 
t=-arg((b-a)/(1-a~b)). (~=conjugate) */
{
  double arg,x,sp,sm,tp,tm;
  complex w,ex,exa,ac;
  Mobius M;

  ac=cconj(a);
  w=cmult(ac,b);
  w.re=1.0-w.re;
  w.im=-w.im;
  if (cAbs(w)< 0.0000000000000001) goto A_ERR;
  w=cdiv(csub(b,a),w);
  arg=(-1.0)*Arg(w);
  ex=cexp(1.0,arg);
  x=cAbs(w);
  sp=sqrt(1+x);
  if ((tm=(1-x))<0.0) goto A_ERR;
  sm=sqrt(1-x);
  tp=sp+sm;
  tm=sp-sm;
  M.a.re=ex.re*tp+ac.re*tm;
  M.a.im=ex.im*tp+ac.im*tm;
  exa=cmult(ex,a);
  M.b.re=-exa.re*tp-tm;
  M.b.im=-exa.im*tp;
  M.c.re=-ex.re*tm-ac.re*tp;
  M.c.im=-ex.im*tm-ac.im*tp;
  M.d.re=exa.re*tm+tp;
  M.d.im=exa.im*tm;
  M.flip=0;
  return M;

 A_ERR:
  (*err_flag)++;
  return (ident_mob());
} /* standard_mob */

Mobius affine_mob(complex a,complex b,complex A,complex B,int *err_flag)
     /* Define automorphism of eucl plane carrying a->A, b->B. Return 
identity on error. */
{
  Mobius M;

  if (cAbs(csub(a,b))<100*m_toler || cAbs(csub(A,B))<100*m_toler)
    {
      (*err_flag)++;
      return (ident_mob());
    }
  M.a=cdiv(csub(A,B),csub(a,b));
  M.b=csub(A,cmult(M.a,a));
  M.c.re=M.c.im=M.d.im=0.0;
  M.d.re=1.0;
  M.flip=0;
  return M;
} /* affine_mob */

Mobius recip_mob()
     /* z-->1/z */
{
  Mobius C;

  C.a.re=C.d.re=C.a.im=C.d.im=0.0;
  C.b.re=C.c.re=1.0;C.b.im=C.c.re=0.0;
  C.flip=1;
  return (C);
} /* recip_mob */
	
int set_reflection(Mobius *M,complex ctr,double r)
     /* set coeff's of Mobius giving reflection in euclidean circle */
{
  if (r<.000000000000001) /* r too small */
    {
      *M=ident_mob();
      return 0;
    }
      
  M->a=ctr;
  M->b.re=r*r-cAbs(ctr);M->b.im=0.0;
  M->c.re=1.0;M->c.im=0.0;
  M->d.re=(-1.0)*ctr.re;M->d.im=ctr.im;
  M->flip=1;
  return 1;
} /* set_reflection */

Mobius cir_invert(complex ctr1,double rad1,complex ctr2,double rad2,
		  int *err_flag)
     /* Given eucl circles c1 and c2, set up an orientation
preserving Mobius which maps outside of c1 onto inside of c2. 
Return identity on error and increment err_flag. */
{
  complex a,b,w;
  Mobius mob1,mob2;

  a=ctr1;
  a.re += rad1;
  b=ctr2;
  b.re += rad2;
  mob1=affine_mob(ctr1,a,ctr2,b,err_flag);
  if (*err_flag)
    return (ident_mob());
  /* inversion in c2 with center/rad (c.r) is given by 
     z--> (c*z + r*r - c*c)/(z-c) */
  mob2=ident_mob();
  mob2.a=ctr2;
  w=cmult(ctr2,ctr2);
  mob2.b.re=rad2*rad2-w.re;
  mob2.b.im=-w.im;
  mob2.c.re=1.0;mob2.c.im=0.0;
  mob2.d.re=-ctr2.re;mob2.d.im=-ctr2.im;
  return (matrix_product(mob2,mob1));
} /* cir_invert */
  
/* ----------------- evaluating Mobius transforms ------- */
/* fixup: check how we should handle operations with infinity */

complex mobius(Mobius M,complex z,int flag)
     /* return M(z) (flag>=0) or M^{-1} (flag==-1). 
fixup: what to do for infinity? */
{
  complex a,b,c,d,ans;
  Mobius MM;

  if (flag>=0) /* apply M */
    {
      MM=M;
      if (M.flip<0) z.im *= -1.0;
    }
  else MM=inv_mob(M);

  a=cmult(MM.c,z);
  b=cadd(MM.d,a);
  c=cmult(MM.a,z);
  d=cadd(MM.b,c);
  ans=cdiv(d,b);
  if (flag<0 && M.flip<0) ans.im *= -1.0;
  return (ans);
} /* mobius */

complex mob_trans(complex z,complex a)
     /* return value for (a-z)/(1-z*conj(a)) */
{
  complex w,COMP_UNIT;

  COMP_UNIT.re=1.0;COMP_UNIT.im=0.0;
  w=cdiv(csub(a,z),csub(COMP_UNIT,cmult(z,cconj(a)) ) );
  return (w);
} /* mob_trans */

complex mob_rotate(complex z,double a)
     /* return value of z rotated by angle */
{
  complex ea,w;

  ea.re=cos(a);ea.im=sin(a);
  w=cmult(z,ea);
  return (w);
} /* mob_rotate */

complex mob_norm_inv(complex w,complex a,complex b)
     /* returns preimage of w under mobius of disc which maps a to 
zero and b to positive x-axis */
{
  complex z;
  double c;

  if (cAbs(csub(a,b)) < m_toler) return (w);
  z=mob_trans(b,a);
  c=cAbs(z);
  z.re/=c;
  z.im/=c;
  z=mob_trans(cmult(w,z),a);
  return (z);
} /* mob_norm_inv */

complex mob_norm(complex z,complex a,complex b)
     /* returns value at z of mobius transform of unit
disc which maps a to zero and b to positive x-axis. */
{
  complex w,v;
  double c;

  v=mob_trans(b,a);
  c=cAbs(v);
  if (c<m_toler) return (mob_trans(z,a));
  w=cdiv(mob_trans(z,a),v);
  w.re*=c;
  w.im*=c;
  return (w);
} /* mob_norm */

Mobius mult_mobius(Mobius M,Mobius C,int flag)
     /* C is circle in 2x2 matrix form. This returns 
the image of C under M (flag==1) or M^{-1} (flag==-1) */
{
  Mobius G,image,image1,MM;
  complex NEG_ONE;

  if (flag==(-1)) MM=inv_mob(M);
  else MM=M;

  NEG_ONE.re=-1.0;NEG_ONE.im=0.0;
  G.a=MM.d;		/* G=M^(-1) * det(M) */
  G.b=cmult(NEG_ONE,MM.b);
  G.c=cmult(NEG_ONE,MM.c);
  G.d=MM.a;

  matrix_transpose(&G);
  image1=matrix_product(G,C);
  matrix_transpose(&G);
  matrix_conj(&G);
  image=matrix_product(image1,G);
  if (MM.flip<0) /* orientation reversing */
    {
      image.b=cconj(image.b);
      image.c=cconj(image.c);
    }

  return (image);
} /* mult_mobius */


/* ----------------- applying Mobius to single circles ------- */

int mobius_of_circle(Mobius Mob,int hes,complex z,double r,
		     complex *newz,double *newr,int flag)
     /* apply mobius (flag 1) or inverse (flag -1) to a single circle.
Note that in eucl case, negative newr means use outside of circle;
user will have to handle this in calling routine. */
{
  double tmpr,dist;
  complex tmpz,z1,z2,z3,tmp1,tmp2,tmp3,nd,new_inf;
  Mobius C,CC;

  if (hes<0) /* hyperbolic */
    {
      h_to_e_data(z,r,&tmpz,&tmpr); 
      /* fixup? Compare procedure here to center_point.
	 which is superior? */
      /* fixup? Don't know what will happen in hyp case if mobius
	 maps circle outside unit disc. */
      z1=tmpz;z1.re += tmpr;
      z2=tmpz;z2.im += tmpr;
      z3=tmpz;z3.re -= tmpr;
      tmp1=mobius(Mob,z1,flag);
      tmp2=mobius(Mob,z2,flag);
      tmp3=mobius(Mob,z3,flag);
      circle_3(tmp1,tmp2,tmp3,&tmpz,&tmpr);
      e_to_h_data(tmpz,tmpr,newz,newr);
      return 1;
    }
  if (hes>0) /* sph */
    {
      s_to_matrix_data(z,r,&C);
      CC=mult_mobius(Mob,C,flag);
      matrix_to_s_data(CC,newz,newr);
      return 1;
    }
  else
    {
      z1=z;z1.re += r;
      z2=z;z2.im += r;
      z3=z;z3.re -= r;
      tmp1=mobius(Mob,z1,flag);
      tmp2=mobius(Mob,z2,flag);
      tmp3=mobius(Mob,z3,flag);
      circle_3(tmp1,tmp2,tmp3,newz,newr);

      /* find out if point mapped to infinity is inside
	 the circle */
      if (cAbs(Mob.c)<m_toler) /* infinity is fixed */
	{
	  if (r<0) (*newr) *= -1;
	  return 1;
	}
      nd.re=-Mob.d.re;nd.im=-Mob.d.im;
      new_inf=cdiv(nd,Mob.c);
      dist=cAbs(csub(z,new_inf));
      if ((r<0 && (dist+r)>0.0) /* it is, so we want outside of circle */
	  || (r>0 && dist<r))
	(*newr) *= -1;
      return 1;
    }
} /* mobius_of_circle */
