#!/usr/bin/env python


################################################################################
# This is the dialog creator module for the CLAW3 file browser
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
 #   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##################################################################

import os, os.path, sys, shutil
import gtk, pango, gobject
import thread, commands, string


class file_mover:
	def __init__(self, move_type, x, y, moving_message, start_location, end_location):
		'''This is the file mover dialog. It copies a file from one location to another.
		Pass (move_type, x, y, moving_message,  start_location, end_location), where the
		move_type is "copy" or "move", x and y are the location you want the dialog to appear, 
		the moving message is the message to display (moving file to ...), the start location 
		is where the file is now, the end location is where it is to be moved to i.e.
		('/home/be/text.txt', '/mnt/fedora/') '''

		# Create the main window and title it
		self.progressWindow = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.progressWindow.connect('destroy', self.kill)
		self.progressWindow.set_title("Claw3 File Mover")

		# Place the progress window at the desired location
		self.progressWindow.move(x, y)     

		# Create a box for the dialog (in case I want to add more stuff later)
		self.mainTable = gtk.Table(1, 2, gtk.FALSE)
		self.mainTable.show()
		self.progressWindow.add(self.mainTable)

		# Create a Label for the Dialog (to hold the file name)
		self.mainLabel = gtk.Label(moving_message)
		self.mainLabel.modify_font(pango.FontDescription("sans bold 12"))
		self.mainTable.attach(self.mainLabel, 0,1,0,1)

		# Create the progress bar and add it to the box
		self.progressBar = gtk.ProgressBar()
		self.progressBar.show()
		self.mainTable.attach(self.progressBar, 0,1,1,2)

		# Set the window size and show all
		self.progressWindow.resize(self.progressWindow.allocation.width, 50)
		self.progressWindow.show_all()

		# Find out if this is a directory or a file 
		if os.path.isdir(start_location):
			self._moveDirectory(move_type, start_location, end_location, moving_message)
		else:
			# Start the file moving
			self._moveFile(move_type, start_location, end_location)
			
		self.kill()

	###################################################################
	# This function opens the file and moves it piece by piece, showing the progress
	# in the progress bar
	###################################################################
	def _moveFile(self, move_type, start_location, end_location):

		# Resize the window so the name fits
		width, height = self.progressBar.size_request()
		self.progressWindow.set_size_request(width, 25)	

		# Get the size of the file to move
		filesize = os.path.getsize(start_location)
		
		# If the file size is zero
		if filesize == 0:
			return "file_is_empty"
			
		if move_type == "copy":	
			thread.start_new_thread(shutil.copy, (start_location, end_location))
		else:
			thread.start_new_thread(shutil.move, (start_location, end_location + "/" + os.path.basename(start_location)))		

		while not os.path.exists(end_location + "/" + os.path.basename(start_location)):
			pass

		while filesize > os.path.getsize(end_location + "/" + os.path.basename(start_location)):
			new_filesize = float(os.path.getsize(end_location + "/" + os.path.basename(start_location)))
			adjustment =  float(new_filesize/filesize)
			
			# This function allows the gui to update in case the machine is too busy
			self.updategui()
				
			# Set the progress bar progress amount	
 			self.progressBar.set_fraction(adjustment)	
 		
			if new_filesize  < 1000:
				self.progressBar.set_text(str(int(new_filesize)) +" bytes of " + str(filesize) + "bytes")
			else:
				self.progressBar.set_text(str(int(new_filesize/1024)) +" Kbs of " + str(filesize/1024) + "Kbs")


	###################################################################
	# This function moves the contents of directories
	###################################################################
	def _moveDirectory(self, move_type, start_location, end_location, moving_message):

		# Resize the window so the name fits
		width, height = self.progressBar.size_request()
		self.progressWindow.set_size_request(width, 25)	

		# If the directory is empty, then just move it and return
		if os.listdir(start_location) == []:
			shutil.copytree(start_location, end_location + "/" + os.path.basename(start_location))
			return True

		# Get the original file size 
		tree_size = commands.getoutput("du -bs " + self.returnClean(start_location))
		tree_size = float(string.split(tree_size)[0])
		
		
		# Start the move or copy (threaded, of course)
		if move_type == "copy":
			try:
				thread.start_new_thread(shutil.copytree, (start_location, end_location + "/" + os.path.basename(start_location)))			
			except:
				print "COPY FAILED"
				return "cannot_copy"				
		elif move_type == "move":
			try:
				thread.start_new_thread(shutil.move, (start_location, end_location + "/" + os.path.basename(start_location)))
			except:
				print "MOVE FAILED"
				return "cannot_move"		
		
		while not os.path.exists(end_location + "/" + os.path.basename(start_location)):
			pass
		
		# Use the new and old directory tree sizes to calculate the progress bar
		new_tree_size = 0
		while tree_size > new_tree_size:
			new_tree_size = commands.getoutput("du -bs " + self.returnClean(end_location + "/" + os.path.basename(start_location)))
			new_tree_size = float(string.split(new_tree_size)[0])		
			adjustment = float(new_tree_size/tree_size)
				
			# This function allows the gui to update in case the machine is too busy
			self.updategui()
				
			# Set the progress bar progress amount	
 			self.progressBar.set_fraction(adjustment)	
 		
			if new_tree_size  < 1000:
				self.progressBar.set_text(str(int(new_tree_size)) +" bytes of " + str(int(tree_size)) + "bytes")
			else:
				self.progressBar.set_text(str(int(new_tree_size/1024)) +" Kbs of " + str(int(tree_size/1024)) + "Kbs")

		print tree_size, new_tree_size
						
	###################################################################
	# This function makes all current processes end before allowing it to continue
	# This allows the GUI to update even when the processor is peaked
	###################################################################
	def updategui(self):
		while  gtk.events_pending() :
			gtk.main_iteration()
		return


	###################################################################
	# This function destroys the widget when we are finished
	###################################################################
	def kill(self, button=None):
		self.progressWindow.destroy()
		return True


	########################################################################
	# This function returns BASH friendly text
	########################################################################		
	def returnClean(self, text):
	
		# Here are the bad characters
		badChars = [" ","\'","&","(",")"]
		
		# Here are the replacement characters
		goodChars=["\ ","\\'","\&","\(","\)"]
		
		# Replace the bad with the good
		for x in range(len(badChars)):
			text = string.replace(text, badChars[x], goodChars[x])		
		
		# Return the "cleaned", BASH friendly text
		return text


###################################################################
# This dialog is used when listing the directory
###################################################################
class waiting_dialog:
	''' This dialog produces a popup window with an animated
	gif image followed by a message. When calling, send the x
	 and y positions of where the window should be and the
	 location of the animated gif or other image to show. Call
	 the .start(message_text) method to add a message and
	 spinner. It will return a gtk.Label. Use the .end(label) method
	 to make the returned label turn red (i.e. it's done). Call the
	 .quit() method to destroy the widget'''

	def __init__(self, x, y, progress_image):

		global __PROGRESS_IMAGE__
		__PROGRESS_IMAGE__= progress_image

		# Create a borderless window (they look cooler)
		self.waitingWindow = gtk.Window(gtk.WINDOW_POPUP)
		self.waitingWindow.resize(100, 10)
	
		# Set the position of the window
		self.waitingWindow.move(x, y)	

		# Create a box to hold the items of the widget
		self.waitingBox = gtk.VBox()
	
		# Add the progressbar to the window
		self.waitingWindow.add(self.waitingBox)
		
		# Show everything
		self.waitingWindow.show_all()
		
		while  gtk.events_pending() :
			gtk.main_iteration()	
			
	##############################################################
	# This adds a start box
	##############################################################
	def start(self, text):
		box = gtk.HBox()
		
		spinner = gtk.Image()
		spinner.set_from_file(__PROGRESS_IMAGE__)
		box.pack_start(spinner, gtk.FALSE, gtk.TRUE, 5)
		
		label = gtk.Label(text)
		label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(0, 30000, 0, 0))
		box.pack_start(label, gtk.FALSE, gtk.TRUE, 5)
		box.show_all()
		
		self.waitingBox.pack_start(box)

		while  gtk.events_pending() :
			gtk.main_iteration()	

		return label
	
	##############################################################
	# This ends the start box
	##############################################################
	def end(self, label):
		label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(65000, 0, 0, 0))
		
		
	##############################################################
	# This function handles destroying the waitingDialog widget
	##############################################################	
	def quit(self):
		self.waitingWindow.destroy()

########################################################################
#						CHOICE DIALOG
########################################################################
def choose(x, y, message):
	'''This is a simple dialog that returns Yes or No.
	Pass it the location to place it (x,y) then the
	message to dislpay'''
		 
	# Create a yes/no dialog (overwrite?)
	dialog = gtk.Dialog ("!!!" , None, gtk.DIALOG_MODAL, ("yes", gtk.RESPONSE_ACCEPT, "No", gtk.RESPONSE_REJECT))
	label = gtk.Label(message)
	label.modify_font(pango.FontDescription("sans bold 12"))
	label.show()
	dialog.vbox.pack_start(label)
	dialog.move(x, y)   
	# Get the dialog information and return it
	response = dialog.run()
	if response == gtk.RESPONSE_ACCEPT:
		dialog.destroy()
		return "yes"
	else:
		dialog.destroy()
		return "no"


########################################################################
#						FIND DIALOG
########################################################################
def finder(x, y, selections):
	
	window = gtk.Window(gtk.WINDOW_TOPLEVEL)
	window.set_title("Claw3 File Finder")
	window.set_size_request(400, 400)
	
	box = gtk.VBox()
	window.add(box)
	
	entry = gtk.Entry()
	box.pack_start(entry, False, True, padding=0)
	
	scroll = gtk.ScrolledWindow()
	box.pack_start(scroll)
	
	list_store = gtk.ListStore(str)
	tree_view = gtk.TreeView(list_store)
	tree_view_column = gtk.TreeViewColumn("Results")
	tree_view.append_column(tree_view_column)
	cell_renderer = gtk.CellRendererText()
	tree_view_column.pack_start(cell_renderer, True)
	tree_view_column.set_attributes(cell_renderer, text=0)
	# tree_view.insert_column_with_attributes(0, "Results", gtk.CellRendererText())
	tree_view.set_search_column(0)
	scroll.add(tree_view)
	
	for selection in selections:
		list_store.append([selection])
		
	window.show_all()
