/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
 using Gee;
 using Sqlite;
 
 namespace Clinica {
 
    public enum Gender {
        MALE,
        FEMALE,
    }
    
    public class PatientIterator : SqlDataIterator {
		private ResourceManager resource_manager;
        public PatientIterator (ResourceManager resources) {
            base (resources.db, resources.patient_table);
            resource_manager = resources;
        }
        
        public new PatientIterator iterator () { return this; }
        
        public new Patient get () {
            return new Patient.with_id (resource_manager, base.get ());
        }
    }
 
    /**
     * @brief A Patient object represent a person
     * whose data are stored in the database (or
     * are going to be stored soon).
     */
    public class Patient : SqlDataType {
    
        /**
         * @brief Doctor associated to this patient. 
         */
        private Doctor? _doctor;
        public Doctor? doctor {
        	get {
        		/* Always reload the doctor to make edits available immediately */
        		if (get_integer ("doctor") == -1)
        		    return null;
          		this._doctor = new Doctor.with_id (resource_manager, get_integer ("doctor"));
        		return this._doctor;
        	}
        	set {
        		this._doctor = value;
        		if (_doctor == null)
        		    set_integer ("doctor", -1);
                else
            		set_integer("doctor", value.get_id ());
        	}
        }
        
        /* FIELDS */
        public string given_name {
        	get { return get_text ("given_name"); }
        	set { set_text ("given_name", value); }
        }
        
        public string surname {
        	get { return get_text ("surname"); }
			set { set_text ("surname", value); }
        }
        
        private DateTime _birth_date;
        public DateTime birth_date {
        	get {
        		_birth_date = get_date ("birth_date");
        		return _birth_date;
        	}
        	set { 
        		_birth_date = value;
        		set_date ("birth_date", _birth_date); 
        	}
        }
        
        public Gender gender {
        	get { 
        		if (get_text("gender") == "MALE")
        			return Gender.MALE;
        		else
        			return Gender.FEMALE;
        	}
        	set {
        		if (value == Gender.MALE)
        			set_text("gender", "MALE");
        		else
        			set_text("gender", "FEMALE");
        	}
        }
        
        public string phone {
        	get { return get_text ("phone"); }
        	set { set_text ("phone", value); }
        }
        
        public string residence_address {
        	get { return get_text ("residence_address"); }
        	set { set_text ("residence_address", value); }
        }
        
        public string identification_code {
        	get { return get_text ("identification_code"); }
        	set { set_text ("identification_code", value); }
        }
        
        
		private ResourceManager resource_manager;
    
        /**
         * @brief Create a new patient
         */
        public Patient (ResourceManager resources, Doctor? doctor) {
            base (resources.db);
            resource_manager = resources;
            table_name = resources.patient_table;
            error.connect ((t,l) => resources.error_callback(t,l));
            
            add_text_field ("given_name");
            add_text_field ("surname");
            add_date_field ("birth_date");
            add_text_field ("gender");
            add_text_field ("phone");
            add_text_field ("residence_address");
            add_text_field ("identification_code");
            add_integer_field ("doctor");

            this.doctor = doctor;
                
            init_resources ();
        }
        
        public Patient.with_id (ResourceManager resources, int ID) {
            this (resources, null);
            load (ID);
            if (doctor != null)
            {   
                doctor = new Doctor.with_id (resources, get_integer ("doctor"));
            }
        }
        
        public static new PatientIterator all (ResourceManager resources) {
            return new PatientIterator (resources);
        }
        
        public string get_complete_name () {
        	return string.join (" ", this.given_name, this.surname);
        }

		public int64 get_age () {
			DateTime today = new DateTime.now_utc ();
			var diff = today.difference (this.birth_date);

			/* TimeSpan seems not to work.. :( */
			return (diff / 31557600000000);
		}
		
		/**
		 * @brief Get the visit IDs for the selected patient.
		 * Actual visits can then be retrieved with 
		 * var visit = new Visit (ID);
		 */
		public GLib.List<int> get_visit_ids () {
			return associated_ids (resource_manager.visits_table, "patient");
		}
    }
 
 }
