# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import dbus

from cohoba.common.Utils import protocol_to_description
from telepathy.constants import (CONN_MGR_PARAM_FLAG_REQUIRED,
	CONN_MGR_PARAM_FLAG_REGISTER)

class Profile:
	def __init__(self, protocol, manager, bus_name, obj_path, reference_params):
		self.name = protocol
		self.protocol = protocol
		self.manager = manager
		
		self.bus_name = bus_name
		self.obj_path = obj_path
		self.reference_params = reference_params
	
	def get_description(self):
		return protocol_to_description(self.protocol)
		
	def has_required_parameters(self, params):
		req_flag = CONN_MGR_PARAM_FLAG_REQUIRED
		for name, param in self.reference_params.items():
			type, value, flags = param
			if (flags & req_flag) and name not in params:
				print 'Missing field to allow valid account:', name
				return False
		
		return True
	
	def prepare_params(self, params):
		return params
	
	def get_profile_keys(self):
		return []
		
	def get_id(self):
		return "%s_%s" % (self.protocol, self.manager)
		
	def __eq__(self, protocol):
		return (protocol != None and protocol.name == self.name and protocol.manager == self.manager)

class GmailProfile(Profile):
	def __init__(self, protocol, manager, bus_name, obj_path, params):
		Profile.__init__(self, protocol, manager, bus_name, obj_path, params)
		self.name = "gmail"

	def get_description(self):
		return "Google Talk"
	
	def get_profile_keys(self):
		return ("server", "old-ssl", "ignore-ssl-errors", "port")
		
	def prepare_params(self, params):
		params = params.copy()
		params["server"] = "talk.google.com"
		params["old-ssl"] = True
		params["ignore-ssl-errors"] = True
		params["port"] = dbus.UInt32(5223)
		return params

# Dictionary holding a mapping from protocol name to the different profiles for that protocol.
# If a protocol isn't in the dictionary, then the Profile base class is used.
# Each key is associated to a list of classes to be constructed with the manager parameters.

PROFILES_FOR_PROTOCOL = {
	"jabber": [GmailProfile, Profile],
}

def get_profiles_for_protocol(protocol):
	if protocol in PROFILES_FOR_PROTOCOL:
		return PROFILES_FOR_PROTOCOL[protocol]
	return [Profile]
