import gtk, gtk.glade, gobject
from os.path import join
import cohoba

from gettext import gettext as _
from cohoba.common.ui.ProfileComboBox import ProfileComboBox
from cohoba.common.ProfileList import ProfileList
from cohoba.common.Utils import singleton, dbus_type_to_python_type
from cohoba.houston.AccountList import AccountList
import cohoba.houston.AccountStore as AccountStore
from telepathy.constants import CONN_MGR_PARAM_FLAG_HAS_DEFAULT

class ProtocolParamsList(gtk.ListStore):
	def __init__(self, account_params, profile):
		gtk.ListStore.__init__(self, gobject.TYPE_PYOBJECT, gobject.TYPE_STRING, gobject.TYPE_STRING)
		self.account_params = account_params
		self.profile = profile
		self.update(profile)
		
	def update(self, profile):
		self.clear()
		if profile == None:
			return

		self.profile = profile
		for prop, val in profile.reference_params.items():
			# HACK: Filter out already presented fields
			if prop in ("account", "password"):
				continue
			
			# Filter out keys modified by the profile
			if prop in self.profile.get_profile_keys():
				continue
				
			dbus_type, default, flags = val

			python_type = dbus_type_to_python_type(dbus_type)

			value = ""
			# Insert default value
			if flags & CONN_MGR_PARAM_FLAG_HAS_DEFAULT:
				value = default
				
			# Insert stored value overriding default if necessary
			if prop in self.account_params:
				value = self.account_params[prop]
			
			self.append([python_type, prop, str(value)])
	
	def get_params(self):
		params = {}
		for t, key, val in self:
			if val == "":
				continue
				
			try:
				# First check if the value is default, then we ignore it.
				dbus_type, default, flags = self.profile.reference_params[key]
				if flags & CONN_MGR_PARAM_FLAG_HAS_DEFAULT:
					# This check is triky, since default can be anythng from str to dbus.Variant,
					# we convert to str first, then to requested type.
					if t(val) == t(str(default)):
						continue
				
				# Now check if the profile will override this key;
				if key in self.profile.get_profile_keys():
					continue
					
				params[key] = t(val)
			except Exception, e:
				raise e
		
		return params
		
class AccountEditor(gtk.Window):
	def __init__(self, account):
		gtk.Window.__init__(self)
		
		self.account = account
		
		self.set_property("border_width", 12)
		self.set_property("window_position", gtk.WIN_POS_CENTER)
		
		glade = gtk.glade.XML(join(cohoba.SHARED_DATA_DIR, "accounts.glade"), "account_modification_contents")
		self.add(glade.get_widget("account_modification_contents"))
		
		self.account_entry = glade.get_widget("account")
		self.password = glade.get_widget("password")
		self.save_password = glade.get_widget("save_password")
		self.close = glade.get_widget("close")
		
		self.advanced_settings = glade.get_widget("advanced_settings")
		self.advanced_settings.set_model(ProtocolParamsList(account.params, account.profile))
		cell = gtk.CellRendererText()
		column = gtk.TreeViewColumn("Key", cell)
		column.set_attributes(cell, text=1)
		self.advanced_settings.append_column(column)
	
		cell = gtk.CellRendererText()
		cell.set_property('editable', True)
		cell.connect('edited', self.advanced_cell_edited)
		column = gtk.TreeViewColumn("Value", cell)
		column.set_attributes(cell, text=2)
		self.advanced_settings.append_column(column)
		
		self.profile = ProfileComboBox()
		glade.get_widget("table").attach(self.profile, 1, 2, 0, 1, xoptions=gtk.FILL, yoptions=gtk.FILL)
		self.profile.connect('changed', self.on_profile_changed)
		self.on_profile_changed(self.profile)
		self.profile.show()
		
		self.connect('destroy', self.on_close)
		self.close.connect("clicked", lambda widget: self.destroy())
		AccountList().connect('account-deleted', self.on_account_deleted)
		self.sync_account()
	
	def advanced_cell_edited(self, cell, path, new_text):
		model = self.advanced_settings.get_model()
		iter = model.get_iter(path)
		try:
			if new_text != "":
				model[iter][0](new_text)
			model.set(iter, 2, new_text)
		except Exception, e:
			print e
		
	def sync_account(self):
		self.profile.set_active_profile(self.account.profile)
		if self.account.get_username() != None:
			self.account_entry.set_text(self.account.get_username())
		if self.account.get_password() != None:
			self.password.set_text(self.account.get_password())
		self.save_password.set_active(self.account.save_password)
		
	def on_account_deleted(self, account_list, deleted_account):
		if deleted_account == self.account:
			self.destroy()
			
	def on_close(self, *args):
		active_profile = self.profile.get_active_profile()
		if active_profile != None:
			self.account.set_profile(active_profile)
		
		# Warning in the meantime we use this precise order, but ideally set_username should disappear
		self.account.set_params(self.advanced_settings.get_model().get_params())
		self.account.set_username(self.account_entry.get_text())
		self.account.set_password(self.password.get_text())
		
		self.account.set_save_password(self.save_password.get_active())
		
		if self.account not in AccountList():
			self.account.generate_name()
			AccountList().add_account(self.account)
	
	def on_profile_changed(self, combo):
		profile = combo.get_active_profile()
		if profile != None:
			self.advanced_settings.get_model().update(profile)
			
class AccountEditorManager:
	def __init__(self):
		self.account_editors = {}

	def on_account_deleted(self, account):
		if account in self.account_editors:
			del self.account_editors[account]
							
	def get_editor_for_account(self, account=None):
		if account == None:
			account = AccountStore.create_new_account()
			
		if account not in self.account_editors:
			self.account_editors[account] = AccountEditor(account)
			self.account_editors[account].connect('destroy', lambda win: self.on_account_deleted(account))
		
		return self.account_editors[account]
	
	def show_account_editor(self, account=None):
		if self.is_connmgr_available():
			editor = self.get_editor_for_account(account)
			editor.present_with_time(gtk.get_current_event_time())
	
	def is_connmgr_available(self):
		if len(ProfileList()) == 0:
			d = gtk.MessageDialog(
				flags=gtk.DIALOG_MODAL,
				type=gtk.MESSAGE_WARNING,
				buttons=gtk.BUTTONS_CLOSE)
			d.set_markup("<b>%s</b>" % _("No connection manager found."))
			d.format_secondary_markup(_("Cohoba needs at least one connection manager.\nYou can try for example <i>telepathy-gabble</i>."))
			d.run()
			d.destroy()
			return False
			
		return True
AccountEditorManager = singleton(AccountEditorManager)
