/**************************************************************************\
 *
 *  This file is part of the Coin 3D visualization library.
 *  Copyright (C) 1998-2006 by Systems in Motion.  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *
 *  For using Coin with software that can not be combined with the GNU
 *  GPL, and for taking advantage of the additional benefits of our
 *  support services, please contact Systems in Motion about acquiring
 *  a Coin Professional Edition License.
 *
 *  See http://www.coin3d.org/ for more information.
 *
 *  Systems in Motion, Postboks 1283, Pirsenteret, 7462 Trondheim, NORWAY.
 *  http://www.sim.no/  sales@sim.no  coin-support@coin3d.org
 *
\**************************************************************************/

/*!
  \class SoTextureCoordinatePlane SoTextureCoordinatePlane.h Inventor/nodes/SoTextureCoordinatePlane.h
  \brief The SoTextureCoordinatePlane class generates texture coordinates by projecting onto a plane.
  \ingroup nodes

  SoTextureCoordinatePlane is used for generating texture coordinates
  by projecting the object onto a texture plane.  The s and t texture
  coordinates are computed as the distance from the origin to the
  projected point, in the respective directions. The texture plane is
  specified using two direction vectors, given as
  SoTextureCoordinatePlane::directionS and
  SoTextureCoordinatePlane::directionT in object space coordinates.
  The length of the vector determines the repeat interval of the
  texture per unit length. 

  A simple usage example:

  \code
  SoSeparator *root = new SoSeparator;
  root->ref();
  
  // the texture image
  SoTexture2 *tex = new SoTexture2;
  tex->filename.setValue("foo.png");
  root->addChild(tex);
  
  // the texture plane
  SoTextureCoordinatePlane *texPlane = new SoTextureCoordinatePlane;
  texPlane->directionS.setValue(SbVec3f(1,0,0));
  texPlane->directionT.setValue(SbVec3f(0,1,0));
  root->addChild(texPlane);
  
  // add a simple cube
  SoCube * c = new SoCube;
  c->width.setValue(1.0);
  c->height.setValue(1.0)
  c->depth.setValue(1.0);
  root->addChild(new SoCube);
  \endcode

  Here, we are projecting a texture onto a cube. The texture
  coordinate plane is specified by directionS = (1,0,0) and directionT
  = (0,1,0), meaning that it is parallel to the front face of the
  cube. Setting e.g. directionS = (0,1,0) and directionT = (-1,0,0)
  would rotate the texture counterclockwise by 90 degrees. Setting
  them to ((2,0,0), (0,2,0)) results to the texture being repeated twice 
  per unit, so the texture appears four times on the 1x1 face.

  Note that when you transform the cube, the transformation will also
  affect the texture - it will be transformed vs. the rest of the
  world, but appear "fixed" on the object. If you want to change the
  placement of the texture on the object, you have to insert a
  SoTexture2Transform node before the texture coordinate plane. For
  instance in the example above, since the cube is centered in its
  coordinate system, the lower left corner of the texture appears to
  be in the middle of the face. To move the texture's origin to
  coincide with the lower left corner of the face, insert

  \code
  SoTexture2Transform * tf = new SoTexture2Transform;
  tf->translation.setValue(-0.5,-0.5);
  root->addChild(tf);
  \endcode

  before adding the texture coordinate plane.

  <b>FILE FORMAT/DEFAULTS:</b>
  \code
    TextureCoordinatePlane {
        directionS 1 0 0
        directionT 0 1 0
    }
  \endcode
*/

// *************************************************************************

#include <Inventor/nodes/SoTextureCoordinatePlane.h>
#include <Inventor/nodes/SoSubNodeP.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/elements/SoGLTextureCoordinateElement.h>
#include <Inventor/elements/SoGLCacheContextElement.h>
#include <Inventor/elements/SoGLMultiTextureCoordinateElement.h>
#include <Inventor/elements/SoTextureUnitElement.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif // HAVE_CONFIG_H
#include <Inventor/system/gl.h>
#include <Inventor/C/glue/gl.h>

/*!
  \var SoSFVec3f SoTextureCoordinatePlane::directionS
  The S texture coordinate plane direction.
  The length of the vector determines the repeat interval of the
  texture per unit length. 

*/
/*!
  \var SoSFVec3f SoTextureCoordinatePlane::directionT
  The T texture coordinate plane direction.
  The length of the vector determines the repeat interval of the
  texture per unit length. 
*/

// *************************************************************************

SO_NODE_SOURCE(SoTextureCoordinatePlane);

/*!
  Constructor.
*/
SoTextureCoordinatePlane::SoTextureCoordinatePlane()
{
  SO_NODE_INTERNAL_CONSTRUCTOR(SoTextureCoordinatePlane);

  SO_NODE_ADD_FIELD(directionS, (1.0f, 0.0f, 0.0f));
  SO_NODE_ADD_FIELD(directionT, (0.0f, 1.0f, 0.0f));
}

/*!
  Destructor.
*/
SoTextureCoordinatePlane::~SoTextureCoordinatePlane()
{
}

// doc from parent
void
SoTextureCoordinatePlane::initClass(void)
{
  SO_NODE_INTERNAL_INIT_CLASS(SoTextureCoordinatePlane, SO_FROM_INVENTOR_2_0);
}

// generates texture coordinates for callback and raypick action
const SbVec4f &
SoTextureCoordinatePlane::generate(void * userdata,
                                   const SbVec3f &p,
                                   const SbVec3f & /* n */)
{
  SoTextureCoordinatePlane *thisp =
    (SoTextureCoordinatePlane*) userdata;

  thisp->gencache.ret.setValue(thisp->gencache.s.dot(p),
                               thisp->gencache.t.dot(p),
                               0.0f, 1.0f);
  return thisp->gencache.ret;
}

// doc from parent
void
SoTextureCoordinatePlane::doAction(SoAction * action)
{
  this->setupGencache();
  SoTextureCoordinateElement::setFunction(action->getState(), this,
                                          SoTextureCoordinatePlane::generate,
                                          this);
}

// doc from parent
void
SoTextureCoordinatePlane::GLRender(SoGLRenderAction * action)
{
  SoState * state = action->getState();
  int unit = SoTextureUnitElement::get(state);

  if (unit == 0) {
    SoTextureCoordinatePlane::doAction((SoAction *)action);
    SoGLTextureCoordinateElement::setTexGen(action->getState(),
                                            this,
                                            SoTextureCoordinatePlane::handleTexgen,
                                            this,
                                            SoTextureCoordinatePlane::generate,
                                            this);
  }
  else {
    const cc_glglue * glue = cc_glglue_instance(SoGLCacheContextElement::get(state));
    int maxunits = cc_glglue_max_texture_units(glue);
    if (unit < maxunits) {        
      this->setupGencache();
      SoGLMultiTextureCoordinateElement::setTexGen(action->getState(),
                                                   this, unit,
                                                   SoTextureCoordinatePlane::handleTexgen,
                                                   this,
                                                   SoTextureCoordinatePlane::generate,
                                                   this);
    }
  }
}

// doc from parent
void
SoTextureCoordinatePlane::callback(SoCallbackAction * action)
{
  SoTextureCoordinatePlane::doAction((SoAction *)action);
}

// doc from parent
void
SoTextureCoordinatePlane::pick(SoPickAction * action)
{
  SoTextureCoordinatePlane::doAction((SoAction *)action);
}

// texgen callback. Turns on plane texgen in OpenGL
void
SoTextureCoordinatePlane::handleTexgen(void *data)
{
  SoTextureCoordinatePlane *thisp = (SoTextureCoordinatePlane*)data;
  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni(GL_Q, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);

  float plane[4];
  plane[3] = 0.0f;
  const SbVec3f & s = thisp->directionS.getValue();
  plane[0] = s[0];
  plane[1] = s[1];
  plane[2] = s[2];
  glTexGenfv(GL_S, GL_OBJECT_PLANE, plane);
  const SbVec3f & t = thisp->directionT.getValue();
  plane[0] = t[0];
  plane[1] = t[1];
  plane[2] = t[2];
  glTexGenfv(GL_T, GL_OBJECT_PLANE, plane);

  // supply dummy plane for R and Q so that texture generation works
  // properly
  plane[0] = 0.0f;
  plane[1] = 0.0f;
  plane[2] = 0.0f;
  plane[3] = 1.0f;
  glTexGenfv(GL_R, GL_OBJECT_PLANE, plane);
  glTexGenfv(GL_Q, GL_OBJECT_PLANE, plane);
}

void 
SoTextureCoordinatePlane::setupGencache(void)
{
  this->gencache.s = this->directionS.getValue();
  this->gencache.t = this->directionT.getValue();
  // FIXME: the other variables in gencache are actually not needed.
  // Remove before Coin 3.0. pederb, 2004-04-14
}

