#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <X11/Xatom.h>
#include <X11/extensions/Xrender.h>

#include <compiz.h>


#define BS_SATURATION_INCREASE_BUTTON_DEFAULT       Button4
#define BS_SATURATION_INCREASE_MODIFIERS_DEFAULT    ControlMask
#define BS_SATURATION_DECREASE_BUTTON_DEFAULT       Button5
#define BS_SATURATION_DECREASE_MODIFIERS_DEFAULT    ControlMask

#define BS_BRIGHTNESS_INCREASE_BUTTON_DEFAULT       Button4
#define BS_BRIGHTNESS_INCREASE_MODIFIERS_DEFAULT    ShiftMask
#define BS_BRIGHTNESS_DECREASE_BUTTON_DEFAULT       Button5
#define BS_BRIGHTNESS_DECREASE_MODIFIERS_DEFAULT    ShiftMask

#define BS_SATURATION_STEP_DEFAULT                  5
#define BS_SATURATION_STEP_MAX                      10
#define BS_SATURATION_STEP_MIN                      1

#define BS_BRIGHTNESS_STEP_DEFAULT                  5
#define BS_BRIGHTNESS_STEP_MAX                      10
#define BS_BRIGHTNESS_STEP_MIN                      1

#define BS_DISPLAY_OPTION_SATURATION_INCREASE    0
#define BS_DISPLAY_OPTION_SATURATION_DECREASE    1
#define BS_DISPLAY_OPTION_BRIGHTNESS_INCREASE    2
#define BS_DISPLAY_OPTION_BRIGHTNESS_DECREASE    3
#define BS_DISPLAY_OPTION_NUM                    4

#define BS_SCREEN_OPTION_SATURATION_STEP        0
#define BS_SCREEN_OPTION_BRIGHTNESS_STEP        1
#define BS_SCREEN_OPTION_NUM                    2

static int displayPrivateIndex;

typedef struct _BSDisplay {
    int screenPrivateIndex;
    CompOption opt[BS_DISPLAY_OPTION_NUM];
} BSDisplay;


typedef struct _BSScreen {
    int brightnessStep;
    int saturationStep;
    CompOption opt[BS_SCREEN_OPTION_NUM];
} BSScreen;

#define GET_BS_DISPLAY(d) ((BSDisplay *) (d)->privates[displayPrivateIndex].ptr)
#define BS_DISPLAY(d) BSDisplay *bd = GET_BS_DISPLAY (d)
#define GET_BS_SCREEN(s, bd) ((BSScreen *) (s)->privates[(bd)->screenPrivateIndex].ptr)
#define BS_SCREEN(s) BSScreen *bs = GET_BS_SCREEN (s, GET_BS_DISPLAY (s->display))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static void
changeWindowSaturation (CompWindow *w, int direction)
{
    int step, saturation;

    BS_SCREEN (w->screen);

    step = (COLOR * bs->saturationStep / 100);         

    saturation = w->paint.saturation + step * direction;
    if (saturation > COLOR)
    {
	saturation = COLOR;
    }
    else if (saturation < 0)
    {
	saturation = 0;
    }

    if (w->paint.saturation != saturation)
    {
	w->paint.saturation = saturation;

	setWindowProp32 (w->screen->display, w->id,
			 w->screen->display->winSaturationAtom,
			 w->paint.saturation);
	addWindowDamage (w);
    }
}

static void
changeWindowBrightness (CompWindow *w, int direction)
{
    int step, brightness;

    BS_SCREEN (w->screen);

    step = (BRIGHT * bs->brightnessStep / 100);   

    brightness = w->paint.brightness + step * direction;
    if (brightness > BRIGHT)
    {
	brightness = BRIGHT;
    }
    else if (brightness < 0)
    {
	brightness = 0;
    }

    if (w->paint.brightness != brightness)
    {
	w->paint.brightness = brightness;

	setWindowProp32 (w->screen->display, w->id,
			 w->screen->display->winBrightnessAtom,
			 w->paint.brightness);
	addWindowDamage (w);
    }
}

static Bool 
increaseSaturation (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "window", 0);

    w = findTopLevelWindowAtDisplay (d, xid);
    if (w)
        changeWindowSaturation (w, 1);

    return TRUE;
}

static Bool 
decreaseSaturation (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "window", 0);

    w = findTopLevelWindowAtDisplay (d, xid);
    if (w)
        changeWindowSaturation (w, -1);

    return TRUE;
}

static Bool 
increaseBrightness (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "window", 0);

    w = findTopLevelWindowAtDisplay (d, xid);
    if (w)
        changeWindowBrightness (w, 1);

    return TRUE;
}

static Bool 
decreaseBrightness (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "window", 0);

    w = findTopLevelWindowAtDisplay (d, xid);
    if (w)
        changeWindowBrightness (w, -1);

    return TRUE;
}

static CompOption *
BSGetDisplayOptions (CompDisplay *display, int *count)
{
    BS_DISPLAY (display);

    *count = NUM_OPTIONS (bd);
    return bd->opt;
}

static Bool
BSSetDisplayOption (CompDisplay *display, char *name, CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    BS_DISPLAY (display);

    o = compFindOption (bd->opt, NUM_OPTIONS (bd), name, &index);
    if (!o)
	return FALSE;

    switch (index) {
        case BS_DISPLAY_OPTION_SATURATION_INCREASE:
        case BS_DISPLAY_OPTION_SATURATION_DECREASE:
        case BS_DISPLAY_OPTION_BRIGHTNESS_INCREASE:
        case BS_DISPLAY_OPTION_BRIGHTNESS_DECREASE:
            if (setDisplayAction (display, o, value))
                return TRUE;
	    break;

        default:
            break;
    }
    return FALSE;
}

static void
BSDisplayInitOptions (BSDisplay *bd, Display *display)
{
    CompOption *o;

    o = &bd->opt[BS_DISPLAY_OPTION_SATURATION_INCREASE];
    o->name                       = "saturation_increase";
    o->shortDesc                  = N_("Increase Saturation");
    o->longDesc                   = N_("Increase Saturation");
    o->type                       = CompOptionTypeAction;
    o->value.action.initiate      = increaseSaturation;
    o->value.action.terminate     = 0;
    o->value.action.bell          = FALSE;
    o->value.action.edgeMask      = 0;
    o->value.action.state         = CompActionStateInitKey;
    o->value.action.state        |= CompActionStateInitButton;
    o->value.action.type          = CompBindingTypeButton;
    o->value.action.button.modifiers = BS_SATURATION_INCREASE_MODIFIERS_DEFAULT;
    o->value.action.button.button    = BS_SATURATION_INCREASE_BUTTON_DEFAULT;

    o = &bd->opt[BS_DISPLAY_OPTION_SATURATION_DECREASE];
    o->name                       = "saturation_decrease";
    o->shortDesc                  = N_("Decrease Saturation");
    o->longDesc                   = N_("Decrease Saturation");
    o->type                       = CompOptionTypeAction;
    o->value.action.initiate      = decreaseSaturation;
    o->value.action.terminate     = 0;
    o->value.action.bell          = FALSE;
    o->value.action.edgeMask      = 0;
    o->value.action.state         = CompActionStateInitKey;
    o->value.action.state        |= CompActionStateInitButton;
    o->value.action.type          = CompBindingTypeButton;
    o->value.action.button.modifiers = BS_SATURATION_DECREASE_MODIFIERS_DEFAULT;
    o->value.action.button.button    = BS_SATURATION_DECREASE_BUTTON_DEFAULT;

    o = &bd->opt[BS_DISPLAY_OPTION_BRIGHTNESS_INCREASE];
    o->name                       = "brightness_increase";
    o->shortDesc                  = N_("Increase Brightness");
    o->longDesc                   = N_("Increase Brightness");
    o->type                       = CompOptionTypeAction;
    o->value.action.initiate      = increaseBrightness;
    o->value.action.terminate     = 0;
    o->value.action.bell          = FALSE;
    o->value.action.edgeMask      = 0;
    o->value.action.state         = CompActionStateInitKey;
    o->value.action.state        |= CompActionStateInitButton;
    o->value.action.type          = CompBindingTypeButton;
    o->value.action.button.modifiers = BS_BRIGHTNESS_INCREASE_MODIFIERS_DEFAULT;
    o->value.action.button.button    = BS_BRIGHTNESS_INCREASE_BUTTON_DEFAULT;

    o = &bd->opt[BS_DISPLAY_OPTION_BRIGHTNESS_DECREASE];
    o->name                       = "brightness_decrease";
    o->shortDesc                  = N_("Decrease Brightness");
    o->longDesc                   = N_("Decrease Brightness");
    o->type                       = CompOptionTypeAction;
    o->value.action.initiate      = decreaseBrightness;
    o->value.action.terminate     = 0;
    o->value.action.bell          = FALSE;
    o->value.action.edgeMask      = 0;
    o->value.action.state         = CompActionStateInitKey;
    o->value.action.state        |= CompActionStateInitButton;
    o->value.action.type          = CompBindingTypeButton;
    o->value.action.button.modifiers = BS_BRIGHTNESS_DECREASE_MODIFIERS_DEFAULT;
    o->value.action.button.button    = BS_BRIGHTNESS_DECREASE_BUTTON_DEFAULT;
}

static CompOption *
BSGetScreenOptions (CompScreen *screen, int *count)
{
    BS_SCREEN (screen);

    *count = NUM_OPTIONS (bs);
    return bs->opt;
}

static Bool
BSSetScreenOption (CompScreen *screen, char *name, CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    BS_SCREEN (screen);

    o = compFindOption (bs->opt, NUM_OPTIONS (bs), name, &index);
    if (!o)
	return FALSE;

    switch (index) {
        case BS_SCREEN_OPTION_BRIGHTNESS_STEP:
            if (compSetIntOption (o, value))
            {
                bs->brightnessStep = o->value.i;
                return TRUE;
            }
            break;

        case BS_SCREEN_OPTION_SATURATION_STEP:
            if (compSetIntOption (o, value))
            {
                bs->saturationStep = o->value.i;
                return TRUE;
            }
            break;

        default:
            break;
    }
    return FALSE;
}

static void
BSScreenInitOptions (BSScreen *bs, Display *display)
{
    CompOption *o;

    o = &bs->opt[BS_SCREEN_OPTION_BRIGHTNESS_STEP];
    o->name		= "brightness_step";
    o->shortDesc	= N_("Brightness Step");
    o->longDesc		= N_("Brightness change step");
    o->type		= CompOptionTypeInt;
    o->value.i		= BS_BRIGHTNESS_STEP_DEFAULT;
    o->rest.i.min	= BS_BRIGHTNESS_STEP_MIN;
    o->rest.i.max	= BS_BRIGHTNESS_STEP_MAX;

    o = &bs->opt[BS_SCREEN_OPTION_SATURATION_STEP];
    o->name		= "saturation_step";
    o->shortDesc	= N_("Saturation Step");
    o->longDesc		= N_("Saturation change step");
    o->type		= CompOptionTypeInt;
    o->value.i		= BS_SATURATION_STEP_DEFAULT;
    o->rest.i.min	= BS_SATURATION_STEP_MIN;
    o->rest.i.max	= BS_SATURATION_STEP_MAX;
}

static Bool
BSInitDisplay (CompPlugin  *p, CompDisplay *d)
{
    BSDisplay *bd;

    bd = malloc (sizeof (BSDisplay));
    if (!bd)
	return FALSE;

    
    bd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (bd->screenPrivateIndex < 0)
    {
	free (bd);
	return FALSE;
    }

    BSDisplayInitOptions (bd, d->display);

    d->privates[displayPrivateIndex].ptr = bd;

    return TRUE;
}

static void
BSFiniDisplay (CompPlugin  *p, CompDisplay *d)
{
    BS_DISPLAY (d);
    freeScreenPrivateIndex (d, bd->screenPrivateIndex);
    free (bd);
}

static Bool
BSInitScreen (CompPlugin *p, CompScreen *s)
{
    BSScreen *bs;
    BS_DISPLAY (s->display);

    bs = malloc (sizeof (BSScreen));
    if (!bs)
	return FALSE;

    bs->saturationStep = BS_SATURATION_STEP_DEFAULT;
    bs->brightnessStep = BS_BRIGHTNESS_STEP_DEFAULT;

    BSScreenInitOptions (bs, s->display->display);

    addScreenAction (s, &bd->opt[BS_DISPLAY_OPTION_BRIGHTNESS_INCREASE].value.action);
    addScreenAction (s, &bd->opt[BS_DISPLAY_OPTION_BRIGHTNESS_DECREASE].value.action);
    addScreenAction (s, &bd->opt[BS_DISPLAY_OPTION_SATURATION_INCREASE].value.action);
    addScreenAction (s, &bd->opt[BS_DISPLAY_OPTION_SATURATION_DECREASE].value.action);

    s->privates[bd->screenPrivateIndex].ptr = bs;
    return TRUE;
}

static void
BSFiniScreen (CompPlugin *p, CompScreen *s)
{
    BS_SCREEN (s);
    free (bs);
}

static Bool
BSInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
	return FALSE;

    return TRUE;
}

static void
BSFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
	freeDisplayPrivateIndex (displayPrivateIndex);
}

CompPluginVTable BSVTable = {
    "bs",
    N_("Brightness and Saturation"),
    N_("Brightness and Saturation adjustments"),
    BSInit,
    BSFini,
    BSInitDisplay,
    BSFiniDisplay,
    BSInitScreen,
    BSFiniScreen,
    0, /* InitWindow */
    0, /* FiniWindow */
    BSGetDisplayOptions,
    BSSetDisplayOption,
    BSGetScreenOptions,
    BSSetScreenOption,
    NULL,
    0
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &BSVTable;
}


