/*
 * Copyright © 2005 Novell, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation, and that the name of
 * Novell, Inc. not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.
 * Novell, Inc. makes no representations about the suitability of this
 * software for any purpose. It is provided "as is" without express or
 * implied warranty.
 *
 * NOVELL, INC. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL NOVELL, INC. BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: David Reveman <davidr@novell.com>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/cursorfont.h>

#include <compiz.h>

#define ResizeUpMask    (1L << 0)
#define ResizeDownMask  (1L << 1)
#define ResizeLeftMask  (1L << 2)
#define ResizeRightMask (1L << 3)

#define RESIZE_INITIATE_BUTTON_DEFAULT    Button2
#define RESIZE_INITIATE_BUTTON_MODIFIERS_DEFAULT CompAltMask

#define RESIZE_OPACITY_DEFAULT 100
#define RESIZE_OPACITY_MIN     1
#define RESIZE_OPACITY_MAX     100

#define RESIZE_OPACIFY_NON_OPAQUE_DEFAULT FALSE

#define RESIZE_OPACITY_DEFAULT 100
#define RESIZE_OPACITY_MIN     1
#define RESIZE_OPACITY_MAX     100

#define RESIZE_OPACIFY_MIN_OPACITY_DEFAULT 80
#define RESIZE_OPACIFY_MIN_OPACITY_MIN     1
#define RESIZE_OPACIFY_MIN_OPACITY_MAX     100

#define RESIZE_SYNC_WINDOW_DEFAULT FALSE

#define RESIZE_INITIATE_KEY_DEFAULT           "F8"
#define RESIZE_INITIATE_KEY_MODIFIERS_DEFAULT CompAltMask

struct _ResizeKeys {
    char	 *name;
    int		 dx;
    int		 dy;
    unsigned int warpMask;
    unsigned int resizeMask;
} rKeys[] = {
    { "Left",  -1,  0, ResizeLeftMask | ResizeRightMask, ResizeLeftMask },
    { "Right",  1,  0, ResizeLeftMask | ResizeRightMask, ResizeRightMask },
    { "Up",     0, -1, ResizeUpMask | ResizeDownMask,    ResizeUpMask },
    { "Down",   0,  1, ResizeUpMask | ResizeDownMask,    ResizeDownMask }
};

#define NUM_KEYS (sizeof (rKeys) / sizeof (rKeys[0]))

#define MIN_KEY_WIDTH_INC  24
#define MIN_KEY_HEIGHT_INC 24

#define RESIZE_DISPLAY_OPTION_INITIATE           0
#define RESIZE_DISPLAY_OPTION_OPACITY            1
#define RESIZE_DISPLAY_OPTION_OPACIFY_MIN_OPACITY 2
#define RESIZE_DISPLAY_OPTION_SYNC_WINDOW	3
#define RESIZE_DISPLAY_OPTION_USE_STRETCH_TEXTURE 4
#define RESIZE_DISPLAY_OPTION_NUM      5

static int displayPrivateIndex;

typedef struct _ResizeDisplay {
    CompOption opt[RESIZE_DISPLAY_OPTION_NUM];

    int		    screenPrivateIndex;
    HandleEventProc handleEvent;

    CompWindow	 *w;
    CompWindow	 *lastWindow;
    int		 lastWidth;
    int		 lastHeight;
    int		 currentWidth;
    int		 currentHeight;
    int		 currentX;
    int		 currentY;
    int		 releaseButton;
    unsigned int mask;
    int		 width;
    int		 height;
    KeyCode      key[NUM_KEYS];

    GLushort resizeOpacity;
    GLushort opacifyMinOpacity;

    Bool useStretchTexture;
    XWindowAttributes savedAttrib;
} ResizeDisplay;

#define RESIZE_USE_STRETCH_TEXTURE_DEFAULT FALSE

typedef struct _ResizeScreen {
    int grabIndex;

    PaintWindowProc paintWindow;
    PreparePaintScreenProc preparePaintScreen;
    PaintScreenProc paintScreen;
    DonePaintScreenProc donePaintScreen;

    Cursor leftCursor;
    Cursor rightCursor;
    Cursor upCursor;
    Cursor upLeftCursor;
    Cursor upRightCursor;
    Cursor downCursor;
    Cursor downLeftCursor;
    Cursor downRightCursor;
    Cursor middleCursor;
    Cursor cursor[NUM_KEYS];
} ResizeScreen;

#define GET_RESIZE_DISPLAY(d)				       \
    ((ResizeDisplay *) (d)->privates[displayPrivateIndex].ptr)

#define RESIZE_DISPLAY(d)		       \
    ResizeDisplay *rd = GET_RESIZE_DISPLAY (d)

#define GET_RESIZE_SCREEN(s, rd)				   \
    ((ResizeScreen *) (s)->privates[(rd)->screenPrivateIndex].ptr)

#define RESIZE_SCREEN(s)						      \
    ResizeScreen *rs = GET_RESIZE_SCREEN (s, GET_RESIZE_DISPLAY (s->display))

#define NUM_OPTIONS(d) (sizeof ((d)->opt) / sizeof (CompOption))

static Bool
resizeInitiate (CompDisplay     *d,
		CompAction      *action,
		CompActionState state,
		CompOption      *option,
		int	        nOption)
{
    CompWindow *w;
    Window     xid;

    RESIZE_DISPLAY (d);

    xid = getIntOptionNamed (option, nOption, "window", 0);

    w = findWindowAtDisplay (d, xid);
    if (w)
    {
	unsigned int mods;
	unsigned int mask;
	int          x, y;
	int	     button;
 
	RESIZE_SCREEN (w->screen);
 
	mods = getIntOptionNamed (option, nOption, "modifiers", 0);
  
	x = getIntOptionNamed (option, nOption, "x",
			       w->attrib.x + (w->width / 2));
	y = getIntOptionNamed (option, nOption, "y",
			       w->attrib.y + (w->height / 2));
 
	button = getIntOptionNamed (option, nOption, "button", -1);
 
	mask = getIntOptionNamed (option, nOption, "direction", 0);

	/* Initiate the resize in the direction suggested by the
	 * quarter of the window the mouse is in, eg drag in top left
	 * will resize up and to the left.  Keyboard resize starts out
	 * with the cursor in the middle of the window and then starts
	 * resizing the edge corresponding to the next key press. */
	if (state & CompActionStateInitKey)
	{
	    mask = 0;
	}
	else if (!mask)
	{
	    mask |= ((x - w->attrib.x) < (w->width / 2)) ?
		ResizeLeftMask : ResizeRightMask;
 
	    mask |= ((y - w->attrib.y) < (w->height / 2)) ?
		ResizeUpMask : ResizeDownMask;
	}
 
	if (otherScreenGrabExist (w->screen, "resize", 0))
	    return FALSE;
 
	if (rd->w)
	    return FALSE;
 
	if (w->type & (CompWindowTypeDesktopMask |
		       CompWindowTypeDockMask	 |
		       CompWindowTypeFullscreenMask))
	    return FALSE;
 
	if (w->attrib.override_redirect)
	    return FALSE;
 
	if (state & CompActionStateInitButton)
	    action->state |= CompActionStateTermButton;
  
	if (w->shaded)
	    mask &= ~(ResizeUpMask | ResizeDownMask);

	rd->w	   = w;
	rd->mask   = mask;
	rd->width  = w->attrib.width;
	rd->height = w->attrib.height;
    rd->savedAttrib = w->attrib;
    if (rd->useStretchTexture)
    {
        rd->currentX = w->attrib.x;
        rd->currentY = w->attrib.y;
        rd->currentWidth = rd->width;
        rd->currentHeight = rd->height;
        rd->lastWidth = rd->lastHeight = 0.0f;
    }
    else
        addWindowDamage(rd->w);

	lastPointerX = x;
	lastPointerY = y;

	if (!rs->grabIndex)
	{
	    Cursor cursor;

	    if (state & CompActionStateInitKey)
	    {
		cursor = rs->middleCursor;
	    }
	    else if (mask & ResizeLeftMask)
	    {
		if (mask & ResizeDownMask)
		    cursor = rs->downLeftCursor;
		else if (mask & ResizeUpMask)
		    cursor = rs->upLeftCursor;
		else
		    cursor = rs->leftCursor;
	    }
	    else if (mask & ResizeRightMask)
	    {
		if (mask & ResizeDownMask)
		    cursor = rs->downRightCursor;
		else if (mask & ResizeUpMask)
		    cursor = rs->upRightCursor;
		else
		    cursor = rs->rightCursor;
	    }
	    else if (mask & ResizeUpMask)
	    {
		cursor = rs->upCursor;
	    }
	    else
	    {
		cursor = rs->downCursor;
	    }

	    rs->grabIndex = pushScreenGrab (w->screen, cursor, "resize");
	}

  	if (rs->grabIndex)
  	{
	    rd->releaseButton = button;

	    (w->screen->windowGrabNotify) (w, x, y, state,
					   CompWindowGrabResizeMask |
					   CompWindowGrabButtonMask);
 
	    if (state & CompActionStateInitKey)
	    {
		int xRoot, yRoot;
 
		xRoot = w->attrib.x + (w->width  / 2);
		yRoot = w->attrib.y + (w->height / 2);
 
		warpPointer (d, xRoot - pointerX, yRoot - pointerY);
	    }
	}
    }
 
    return FALSE;
}

static void
resizeUpdateWindowRealSize (CompDisplay *d)
{
    int width, height;

    RESIZE_DISPLAY (d);

    if (!rd->w)
	return;

    if (rd->w->state & CompWindowStateMaximizedVertMask)
	rd->height = rd->w->attrib.height;

    if (rd->w->state & CompWindowStateMaximizedHorzMask)
	rd->width = rd->w->attrib.width;

    if (rd->width  == rd->w->attrib.width &&
	rd->height == rd->w->attrib.height)
	return;

    if (constrainNewWindowSize (rd->w,
				rd->width, rd->height,
				&width, &height))
    {
	XWindowChanges xwc;

	xwc.x = rd->w->attrib.x;
	if (rd->mask & ResizeLeftMask)
	    xwc.x -= width - rd->w->attrib.width;

	xwc.y = rd->w->attrib.y;
	if (rd->mask & ResizeUpMask)
	    xwc.y -= height - rd->w->attrib.height;

	xwc.width  = width;
	xwc.height = height;

	sendSyncRequest (rd->w);
	
	XConfigureWindow (d->display, rd->w->id,
			  CWX | CWY | CWWidth | CWHeight,
			  &xwc);
    }
}

static Bool
resizeTerminate (CompDisplay	 *d,
		 CompAction	 *action,
		 CompActionState state,
		 CompOption	 *option,
		 int		 nOption)
{
    RESIZE_DISPLAY (d);

    if (rd->w)
    {
	RESIZE_SCREEN (rd->w->screen);

	if (rd->useStretchTexture)
    {
        (*rd->w->screen->setWindowScale) (rd->w, 1.0f, 1.0f);
    	resizeUpdateWindowRealSize ( d );
    }

 	(rd->w->screen->windowUngrabNotify) (rd->w);
 
	if (state & CompActionStateCancel)
	{
	    XWindowChanges xwc;

	    sendSyncRequest (rd->w);

	    xwc.x      = rd->savedAttrib.x;
	    xwc.y      = rd->savedAttrib.y;
	    xwc.width  = rd->savedAttrib.width;
	    xwc.height = rd->savedAttrib.height;

	    XConfigureWindow (d->display, rd->w->id,
			      CWX | CWY | CWWidth | CWHeight,
			      &xwc);
	}
	else
	{
	    syncWindowPosition (rd->w);
	}

	if (rs->grabIndex)
	{
	    removeScreenGrab (rd->w->screen, rs->grabIndex, NULL);
	    rs->grabIndex = 0;
	}

	addWindowDamage(rd->w);

	if (rd->useStretchTexture)
        damageScreen ( rd->w->screen );

	rd->w		  = 0;
	rd->releaseButton = 0;
    }
    action->state &= ~(CompActionStateTermKey | CompActionStateTermButton);
  
    return FALSE;
}


static void
resizeUpdateWindowSize (CompDisplay *d)
{
    int width, height, dx, dy;

    RESIZE_DISPLAY (d);

    if (!rd->w)
        return;

    if (rd->w->state & CompWindowStateMaximizedVertMask)
        rd->height = rd->w->attrib.height;

    if (rd->w->state & CompWindowStateMaximizedHorzMask)
        rd->width = rd->w->attrib.width;

    if (rd->width  == rd->w->attrib.width &&
            rd->height == rd->w->attrib.height)
        return;

    if (rd->useStretchTexture)
    {
        dx = rd->width - rd->currentWidth;
        dy = rd->height - rd->currentHeight;
        if ( dx <= 2 && dx >= -2 &&
                dy <= 2 && dy >= -2 )
            return;
    }
    else
        if (rd->w->syncWait)
            return;

    if (constrainNewWindowSize (rd->w,
                rd->width, rd->height,
                &width, &height))
    {
        if (rd->useStretchTexture)
        {
            if ( rd->mask & ResizeLeftMask || rd->mask & ResizeUpMask )
            {
                dx = 0.0f;
                dy = 0.0f;

                if (rd->mask & ResizeLeftMask)
                    dx = rd->currentWidth - rd->width;

                if (rd->mask & ResizeUpMask)
                    dy = rd->currentHeight - rd->height;

                moveWindow ( rd->w, dx, dy, FALSE, FALSE );
            }
            else
            {
                moveWindow ( rd->w, -1, 0, FALSE, FALSE );
                moveWindow ( rd->w, 1, 0, FALSE, FALSE );
            }
            rd->currentWidth  = rd->width;
            rd->currentHeight = rd->height;

        }
        else
        {
            XWindowChanges xwc;

            xwc.x = rd->w->attrib.x;
            if (rd->mask & ResizeLeftMask)
                xwc.x -= width - rd->w->attrib.width;

            xwc.y = rd->w->attrib.y;
            if (rd->mask & ResizeUpMask)
                xwc.y -= height - rd->w->attrib.height;

            xwc.width  = width;
            xwc.height = height;

            if (rd->opt[RESIZE_DISPLAY_OPTION_SYNC_WINDOW].value.b) {
                sendSyncRequest (rd->w);
            }
            XConfigureWindow (d->display, rd->w->id,
                    CWX | CWY | CWWidth | CWHeight,
                    &xwc);

        }
    }
}

static void
resizeConstrainMinMax (CompWindow *w,
		       int        width,
		       int        height,
		       int        *newWidth,
		       int        *newHeight)
{
    const XSizeHints *hints = &w->sizeHints;
    int		     min_width = 0;
    int		     min_height = 0;
    int		     max_width = MAXSHORT;
    int		     max_height = MAXSHORT;

    if ((hints->flags & PBaseSize) && (hints->flags & PMinSize))
    {
	min_width = hints->min_width;
	min_height = hints->min_height;
    }
    else if (hints->flags & PBaseSize)
    {
	min_width = hints->base_width;
	min_height = hints->base_height;
    }
    else if (hints->flags & PMinSize)
    {
	min_width = hints->min_width;
	min_height = hints->min_height;
    }

    if (hints->flags & PMaxSize)
    {
	max_width = hints->max_width;
	max_height = hints->max_height;
    }

#define CLAMP(v, min, max) ((v) <= (min) ? (min) : (v) >= (max) ? (max) : (v))

    /* clamp width and height to min and max values */
    width  = CLAMP (width, min_width, max_width);
    height = CLAMP (height, min_height, max_height);

#undef CLAMP

    *newWidth  = width;
    *newHeight = height;
}


static void
resizeHandleKeyEvent (CompScreen *s,
		      KeyCode	 keycode)
{
    RESIZE_SCREEN (s);
    RESIZE_DISPLAY (s->display);

    if (rs->grabIndex && rd->w)
    {
	CompWindow *w;
	int i, widthInc, heightInc;

	w = rd->w;

	widthInc  = w->sizeHints.width_inc;
	heightInc = w->sizeHints.height_inc;

	if (widthInc < MIN_KEY_WIDTH_INC)
	    widthInc = MIN_KEY_WIDTH_INC;

	if (heightInc < MIN_KEY_HEIGHT_INC)
	    heightInc = MIN_KEY_HEIGHT_INC;

	for (i = 0; i < NUM_KEYS; i++)
	{
	    if (keycode != rd->key[i])
	      continue;

	    if (rd->mask & rKeys[i].warpMask)
	    {
		XWarpPointer (s->display->display, None, None, 0, 0, 0, 0,
			      rKeys[i].dx * widthInc,
			      rKeys[i].dy * heightInc);
	    }
	    else
	    {
		int x, y, left, top, width, height;

		left   = w->attrib.x - w->input.left;
		top    = w->attrib.y - w->input.top;
		width  = w->input.left + w->attrib.width  + w->input.right;
		height = w->input.top  + w->attrib.height + w->input.bottom;

		x = left + width  * (rKeys[i].dx + 1) / 2;
		y = top  + height * (rKeys[i].dy + 1) / 2;

		warpPointer (s->display, x - pointerX, y - pointerY);
		rd->mask = rKeys[i].resizeMask;
		updateScreenGrab (s, rs->grabIndex, rs->cursor[i]);
	    }
			    
	    break;
	}
    }
}

static void
resizeHandleMotionEvent (CompScreen *s,
			 int	    xRoot,
			 int	    yRoot)
{
    RESIZE_SCREEN (s);

    if (rs->grabIndex)
    {
	int pointerDx, pointerDy;

	RESIZE_DISPLAY (s->display);

	pointerDx = xRoot - lastPointerX;
	pointerDy = yRoot - lastPointerY;

	if (pointerDx || pointerDy)
	{
	    int w, h;

	    w = rd->width;
	    h = rd->height;

	    if (rd->mask & ResizeLeftMask)
		w -= pointerDx;
	    else if (rd->mask & ResizeRightMask)
		w += pointerDx;

	    if (rd->mask & ResizeUpMask)
		h -= pointerDy;
	    else if (rd->mask & ResizeDownMask)
		h += pointerDy;

	    resizeConstrainMinMax (rd->w, w, h, &w, &h);

	    rd->width  = w;
	    rd->height = h;

	    resizeUpdateWindowSize (s->display);
	}
    }
}

static void
resizeHandleEvent (CompDisplay *d,
		   XEvent      *event)
{
    CompScreen *s;

    RESIZE_DISPLAY (d);

    switch (event->type) {
    case KeyPress:
	s = findScreenAtDisplay (d, event->xkey.root);
	if (s)
	    resizeHandleKeyEvent (s, event->xkey.keycode);
	break;
    case KeyRelease:
	break;
    case ButtonRelease: {
	CompAction *action =
	    &rd->opt[RESIZE_DISPLAY_OPTION_INITIATE].value.action;

	if (action->state & CompActionStateTermButton)
	{
	    if (event->type == ButtonRelease &&
		(rd->releaseButton     == -1 ||
		 event->xbutton.button == rd->releaseButton))
	    {
		resizeTerminate (d,
				 action,
				 CompActionStateTermButton,
				 NULL,
				 0);
	    }
	}
    } break;
    case MotionNotify:
	s = findScreenAtDisplay (d, event->xmotion.root);
	if (s)
	    resizeHandleMotionEvent (s, pointerX, pointerY);
	break;
    case EnterNotify:
    case LeaveNotify:
	s = findScreenAtDisplay (d, event->xcrossing.root);
	if (s)
	    resizeHandleMotionEvent (s, pointerX, pointerY);
	break;
    case ClientMessage:
	if (event->xclient.message_type == d->wmMoveResizeAtom)
	{
	    CompWindow *w;

	    if (event->xclient.data.l[2] <= WmMoveResizeSizeLeft ||
		event->xclient.data.l[2] == WmMoveResizeSizeKeyboard)
	    {
		w = findWindowAtDisplay (d, event->xclient.window);
		if (w)
		{
		    CompOption o[6];
		    CompAction *action =
			&rd->opt[RESIZE_DISPLAY_OPTION_INITIATE].value.action;

		    o[0].type    = CompOptionTypeInt;
		    o[0].name    = "window";
		    o[0].value.i = event->xclient.window;

		    if (event->xclient.data.l[2] == WmMoveResizeSizeKeyboard)
		    {
			o[1].type    = CompOptionTypeInt;
			o[1].name    = "button";
			o[1].value.i = 0;

			resizeInitiate (d, action,
					CompActionStateInitKey,
					o, 2);
		    }
		    else
		    {
			static unsigned int mask[] = {
			    ResizeUpMask | ResizeLeftMask,
			    ResizeUpMask,
			    ResizeUpMask | ResizeRightMask,
			    ResizeRightMask,
			    ResizeDownMask | ResizeRightMask,
			    ResizeDownMask,
			    ResizeDownMask | ResizeLeftMask,
			    ResizeLeftMask,
			};
			unsigned int mods;
			Window	     root, child;
			int	     xRoot, yRoot, i;

			XQueryPointer (d->display, w->screen->root,
				       &root, &child, &xRoot, &yRoot,
				       &i, &i, &mods);

			/* TODO: not only button 1 */
			if (mods & Button1Mask)
			{
			    o[1].type	 = CompOptionTypeInt;
			    o[1].name	 = "modifiers";
			    o[1].value.i = mods;

			    o[2].type	 = CompOptionTypeInt;
			    o[2].name	 = "x";
			    o[2].value.i = event->xclient.data.l[0];

			    o[3].type	 = CompOptionTypeInt;
			    o[3].name	 = "y";
			    o[3].value.i = event->xclient.data.l[1];

			    o[4].type	 = CompOptionTypeInt;
			    o[4].name	 = "direction";
			    o[4].value.i = mask[event->xclient.data.l[2]];

			    o[5].type	 = CompOptionTypeInt;
			    o[5].name	 = "button";
			    o[5].value.i = event->xclient.data.l[3] ?
				event->xclient.data.l[3] : -1;

			    resizeInitiate (d,
					    action,
					    CompActionStateInitButton,
					    o, 6);

			    resizeHandleMotionEvent (w->screen, xRoot, yRoot);
			}
		    }
		}
	    }
	}
	break;
    case DestroyNotify:
	if (rd->w && rd->w->id == event->xdestroywindow.window)
	{
	    CompAction *action =
		&rd->opt[RESIZE_DISPLAY_OPTION_INITIATE].value.action;

	    resizeTerminate (d, action, 0, NULL, 0);
	}
	break;
    case UnmapNotify:
	if (rd->w && rd->w->id == event->xunmap.window)
	{
	    CompAction *action =
		&rd->opt[RESIZE_DISPLAY_OPTION_INITIATE].value.action;

	    resizeTerminate (d, action, 0, NULL, 0);
	}
	break;
    default:
	if (event->type == d->syncEvent + XSyncAlarmNotify)
	{
	    if (rd->w)
	    {
		XSyncAlarmNotifyEvent *sa;

		sa = (XSyncAlarmNotifyEvent *) event;

		if (rd->w->syncAlarm == sa->alarm)
		    resizeUpdateWindowSize (d);
	    }
	}
	break;
    }

    UNWRAP (rd, d, handleEvent);
    (*d->handleEvent) (d, event);
    WRAP (rd, d, handleEvent, resizeHandleEvent);
}

static Bool
resizePaintWindow (CompWindow              *w,
                 const WindowPaintAttrib *attrib,
                 Region                  region,
                 unsigned int            mask)
{
    WindowPaintAttrib sAttrib;
    CompScreen        *s = w->screen;
    Bool              status;

    RESIZE_SCREEN (s);

    if (rs->grabIndex)
    {
        RESIZE_DISPLAY (s->display);

        sAttrib = *attrib;

	if ( rd->w == w )
	{
	    if ( rd->useStretchTexture && (rd->lastWidth != rd->currentWidth || rd->lastHeight != rd->currentHeight ) )
	    	mask |= PAINT_WINDOW_TRANSFORMED_MASK;

	    if ( rd->resizeOpacity != OPAQUE && sAttrib.opacity >= rd->opacifyMinOpacity )
	    {
		attrib  = &sAttrib;
		sAttrib.opacity = (sAttrib.opacity * rd->resizeOpacity) >> 16;
	    }
	}
    }

    UNWRAP (rs, s, paintWindow);
    status = (*s->paintWindow) (w, attrib, region, mask);
    WRAP (rs, s, paintWindow, resizePaintWindow);

    return status;
}

static void
resizePreparePaintScreen (CompScreen *s,
			  int	     msSinceLastPaint)
{
    RESIZE_SCREEN (s);
    RESIZE_DISPLAY (s->display);

    if ( rd->useStretchTexture)
    {
    if ( rd->w != NULL && ( rd->lastWidth != rd->currentWidth || rd->lastHeight != rd->currentHeight ) )
    {
	float xScale, yScale;

	xScale = (float) rd->currentWidth / (float) rd->w->attrib.width;
	yScale = (float) rd->currentHeight / (float) rd->w->attrib.height;
	
	(*s->setWindowScale) ( rd->w, xScale, yScale );

	addWindowDamage ( rd->w );
	damageScreen ( s );
    }
    }
    
    UNWRAP (rs, s, preparePaintScreen);
    (*s->preparePaintScreen) (s, msSinceLastPaint);
    WRAP (rs, s, preparePaintScreen, resizePreparePaintScreen);
}

static Bool
resizePaintScreen (CompScreen		  *s,
		  const ScreenPaintAttrib *sAttrib,
		  Region		  region,
		  unsigned int		  mask)
{
    Bool status;

    RESIZE_SCREEN (s);
    RESIZE_DISPLAY (s->display);

    if (rd->useStretchTexture)
    {
    if ( rd->w != NULL && ( rd->lastWidth != rd->currentWidth || rd->lastHeight != rd->currentHeight ) )
    {
	mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;
    }
    }

    UNWRAP (rs, s, paintScreen);
    status = (*s->paintScreen) (s, sAttrib, region, mask);
    WRAP (rs, s, paintScreen, resizePaintScreen);

    return status;
}


static void
resizeDonePaintScreen (CompScreen *s)
{
    RESIZE_SCREEN (s);
    RESIZE_DISPLAY (s->display);

    if (rd->useStretchTexture)
    {
    	if ( rd->w != NULL && ( rd->lastWidth != rd->currentWidth || rd->lastHeight != rd->currentHeight ) )
    	{
		rd->lastWidth = rd->currentWidth;
		rd->lastHeight = rd->currentHeight;
    	}
    }

    UNWRAP (rs, s, donePaintScreen);
    (*s->donePaintScreen) (s);
    WRAP (rs, s, donePaintScreen, resizeDonePaintScreen);
}

static CompOption *
resizeGetDisplayOptions (CompDisplay *display,
			 int	     *count)
{
    RESIZE_DISPLAY (display);

    *count = NUM_OPTIONS (rd);
    return rd->opt;
}

static Bool
resizeSetDisplayOption (CompDisplay     *display,
			char	        *name,
			CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    RESIZE_DISPLAY (display);

    o = compFindOption (rd->opt, NUM_OPTIONS (rd), name, &index);
    if (!o)
	return FALSE;

    switch (index) {
    case RESIZE_DISPLAY_OPTION_INITIATE:
	if (setDisplayAction (display, o, value))
	    return TRUE;
    	break;
    case RESIZE_DISPLAY_OPTION_OPACITY:
        if (compSetIntOption (o, value))
        {
            rd->resizeOpacity = (o->value.i * OPAQUE) / 100;
            return TRUE;
        }
    	break;
     case RESIZE_DISPLAY_OPTION_OPACIFY_MIN_OPACITY:
         if (compSetIntOption (o, value))
         {
             rd->opacifyMinOpacity = (o->value.i * OPAQUE) / 100;
             return TRUE;
         }
         break;
     case RESIZE_DISPLAY_OPTION_USE_STRETCH_TEXTURE:
        if (compSetBoolOption (o, value))
        {
            rd->useStretchTexture = o->value.b;
    	    return TRUE;
        }
        break;
    case RESIZE_DISPLAY_OPTION_SYNC_WINDOW:
        if (compSetBoolOption (o, value))
	    return TRUE;
        break;
    default:
	break;
    }

    return FALSE;
}

static void
resizeDisplayInitOptions (ResizeDisplay *rd,
			  Display       *display)
{
    CompOption *o;

    o = &rd->opt[RESIZE_DISPLAY_OPTION_INITIATE];
    o->name			     = "initiate";
    o->shortDesc		     = N_("Initiate Window Resize");
    o->longDesc			     = N_("Start resizing window");
    o->type			     = CompOptionTypeAction;
    o->value.action.initiate	     = resizeInitiate;
    o->value.action.terminate	     = resizeTerminate;
    o->value.action.bell	     = FALSE;
    o->value.action.edgeMask	     = 0;
    o->value.action.type	     = CompBindingTypeButton;
    o->value.action.state	     = CompActionStateInitButton;
    o->value.action.button.modifiers = RESIZE_INITIATE_BUTTON_MODIFIERS_DEFAULT;
    o->value.action.button.button    = RESIZE_INITIATE_BUTTON_DEFAULT;
    o->value.action.type	    |= CompBindingTypeKey;
    o->value.action.state	    |= CompActionStateInitKey;
    o->value.action.key.modifiers    = RESIZE_INITIATE_KEY_MODIFIERS_DEFAULT;
    o->value.action.key.keycode      =
	XKeysymToKeycode (display,
			  XStringToKeysym (RESIZE_INITIATE_KEY_DEFAULT));

    o = &rd->opt[RESIZE_DISPLAY_OPTION_OPACITY];
    o->name       = "opacity";
    o->shortDesc  = "Opacity";
    o->longDesc   = "Opacity level of resizing windows";
    o->type       = CompOptionTypeInt;
    o->value.i    = RESIZE_OPACITY_DEFAULT;
    o->rest.i.min = RESIZE_OPACITY_MIN;
    o->rest.i.max = RESIZE_OPACITY_MAX;

    o = &rd->opt[RESIZE_DISPLAY_OPTION_OPACIFY_MIN_OPACITY];
    o->name	  = "opacify_min_opacity";
    o->shortDesc  = "Minimum opacity for opacify";
    o->longDesc	  = "Opacify only windows whose opacity is higher than this value";
    o->type	  = CompOptionTypeInt;
    o->value.i	  = RESIZE_OPACIFY_MIN_OPACITY_DEFAULT;
    o->rest.i.min = RESIZE_OPACIFY_MIN_OPACITY_MIN;
    o->rest.i.max = RESIZE_OPACIFY_MIN_OPACITY_MAX;

    o = &rd->opt[RESIZE_DISPLAY_OPTION_SYNC_WINDOW];
    o->name	  = "sync_window";
    o->shortDesc  = "Repaints the window on each resize step";
    o->longDesc	  = "If this is set to true the window will repaint itself during resize, which may cause some lag.";
    o->type	  = CompOptionTypeBool;
    o->value.b	  = RESIZE_SYNC_WINDOW_DEFAULT;

    o = &rd->opt[RESIZE_DISPLAY_OPTION_USE_STRETCH_TEXTURE];
    o->name	  = "stretch_texture";
    o->shortDesc  = "Uses a stretchy texture instead of actual window contents during resize.";
    o->longDesc	  = "This option makes window resizing a bit smoother and faster, though it doesn't look quite as cool (or maybe it does).";
    o->type	  = CompOptionTypeBool;
    o->value.b	  = RESIZE_USE_STRETCH_TEXTURE_DEFAULT;


}

static Bool
resizeInitDisplay (CompPlugin  *p,
		   CompDisplay *d)
{
    ResizeDisplay *rd;
    int	          i;

    rd = malloc (sizeof (ResizeDisplay));
    if (!rd)
	return FALSE;

    rd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (rd->screenPrivateIndex < 0)
    {
	free (rd);
	return FALSE;
    }

    rd->resizeOpacity = (RESIZE_OPACITY_DEFAULT * OPAQUE) / 100;

    rd->resizeOpacity = (RESIZE_OPACITY_DEFAULT * OPAQUE) / 100;

    rd->opacifyMinOpacity = (RESIZE_OPACIFY_MIN_OPACITY_DEFAULT * OPAQUE) / 100;

    rd->useStretchTexture = RESIZE_USE_STRETCH_TEXTURE_DEFAULT;

    resizeDisplayInitOptions (rd, d->display);

    rd->w = 0;

    rd->releaseButton = 0;

    for (i = 0; i < NUM_KEYS; i++)
	rd->key[i] = XKeysymToKeycode (d->display,
				       XStringToKeysym (rKeys[i].name));

    WRAP (rd, d, handleEvent, resizeHandleEvent);

    d->privates[displayPrivateIndex].ptr = rd;

    return TRUE;
}

static void
resizeFiniDisplay (CompPlugin  *p,
		   CompDisplay *d)
{
    RESIZE_DISPLAY (d);

    freeScreenPrivateIndex (d, rd->screenPrivateIndex);

    UNWRAP (rd, d, handleEvent);

    free (rd);
}

static Bool
resizeInitScreen (CompPlugin *p,
		  CompScreen *s)
{
    ResizeScreen *rs;

    RESIZE_DISPLAY (s->display);

    rs = malloc (sizeof (ResizeScreen));
    if (!rs)
	return FALSE;

    rs->grabIndex = 0;

    rs->leftCursor	= XCreateFontCursor (s->display->display, XC_left_side);
    rs->rightCursor	= XCreateFontCursor (s->display->display, XC_right_side);
    rs->upCursor	= XCreateFontCursor (s->display->display,
					     XC_top_side);
    rs->upLeftCursor	= XCreateFontCursor (s->display->display,
					     XC_top_left_corner);
    rs->upRightCursor	= XCreateFontCursor (s->display->display,
					     XC_top_right_corner);
    rs->downCursor	= XCreateFontCursor (s->display->display,
					     XC_bottom_side);
    rs->downLeftCursor	= XCreateFontCursor (s->display->display,
					     XC_bottom_left_corner);
    rs->downRightCursor = XCreateFontCursor (s->display->display,
					     XC_bottom_right_corner);
    rs->middleCursor    = XCreateFontCursor (s->display->display, XC_fleur);

    rs->cursor[0] = rs->leftCursor;
    rs->cursor[1] = rs->rightCursor;
    rs->cursor[2] = rs->upCursor;
    rs->cursor[3] = rs->downCursor;

    addScreenAction (s, &rd->opt[RESIZE_DISPLAY_OPTION_INITIATE].value.action);

    WRAP (rs, s, paintWindow, resizePaintWindow);
    WRAP (rs, s, preparePaintScreen, resizePreparePaintScreen);
    WRAP (rs, s, paintScreen, resizePaintScreen);
    WRAP (rs, s, donePaintScreen, resizeDonePaintScreen);

    s->privates[rd->screenPrivateIndex].ptr = rs;

    return TRUE;
}

static void
resizeFiniScreen (CompPlugin *p,
		  CompScreen *s)
{
    RESIZE_SCREEN (s);

    UNWRAP (rs, s, paintWindow);
    UNWRAP (rs, s, preparePaintScreen);
    UNWRAP (rs, s, paintScreen);
    UNWRAP (rs, s, donePaintScreen);

    free (rs);
}

static Bool
resizeInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
	return FALSE;

    return TRUE;
}

static void
resizeFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
	freeDisplayPrivateIndex (displayPrivateIndex);
}

CompPluginVTable resizeVTable = {
    "resize",
    N_("Resize Window"),
    N_("Resize window"),
    resizeInit,
    resizeFini,
    resizeInitDisplay,
    resizeFiniDisplay,
    resizeInitScreen,
    resizeFiniScreen,
    0, /* InitWindow */
    0, /* FiniWindow */
    resizeGetDisplayOptions,
    resizeSetDisplayOption,
    0, /* GetScreenOptions */
    0, /* SetScreenOption */
    NULL,
    0
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &resizeVTable;
}
