#include "c_defs.h"

/************************************************************************
 *
 * ibuf.c - input buffering to support Function key macros
 * Jon Trulson 3/30/96
 *
 * $Id: ibuf.c 762 2008-07-23 02:14:59Z jon13 $
 *
 * Copyright 1999-2004 Jon Trulson under the ARTISTIC LICENSE. (See LICENSE).
 **********************************************************************/

/**********************************************************************/
/* Unix/C specific porting and supporting code Copyright (C)1994-1996 */
/* by Jon Trulson <jon@radscan.com> under the same terms and          */
/* conditions of the original copyright by Jef Poskanzer and Craig    */
/* Leres.                                                             */
/*                                                                    */
/**********************************************************************/

#include "conf.h"
#include "global.h"
#include "defs.h"
#include "ibuf.h"
#include "rb.h"
#include "conqutil.h"

/* the input ringbuffer */
static ringBuffer_t *ibufRB = NULL;

/* we use the ringbuffer stuff in rb.c, however, the input
 *  buffer is really a buffer of integers, so we hide all that here, since
 *  the RB itself is byte oriented.
 */

/* convert the lengths into the 'real' lengths between the length we
 *  care about here (integers) vs the RB length (bytes).
 */
#define I2RB_LEN(x)       ((x) * sizeof(unsigned int))
#define RB2I_LEN(x)       ((x) / sizeof(unsigned int))

/* size of input ringbuffer (in integers) */
#define IBUF_RBMAX 1024


/* ibufInit - intialize the input buffer */

void ibufInit(void)
{
  ibufRB = rbCreate(I2RB_LEN(IBUF_RBMAX));

  return;
}

void ibufFlush(void)
{
  rbFlush(ibufRB);

  return;
}


int ibufCount(void)
{
  if (ibufRB)
    return RB2I_LEN(rbBytesUsed(ibufRB));
  else
    return 0;
}


/* ibufPut - put a string into the buffer */

void ibufPut(char *thestr)
{
  int i;
  int n = strlen(thestr);
  int c;

  /* cvt to int's and insert into rb */
  for (i=0; i<n; i++)
    {
      c = thestr[i] & 0xff;
      /* hopefully there is enough room, or... */
      rbPut(ibufRB, (Unsgn8 *)&c, I2RB_LEN(1));
    }

  return;
}

/* ibufPutc - put a char into the buffer */

void ibufPutc(unsigned int thechar)
{
  rbPut(ibufRB, (Unsgn8 *)&thechar, I2RB_LEN(1));

  return;
}


/* ibufGetc - return next char from the input buffer */

unsigned int ibufGetc(void)
{
  int c;

  if (!ibufCount())
    {
      return 0;
    }

  rbGet(ibufRB, (Unsgn8 *)&c, I2RB_LEN(1), TRUE);

  return c;
}
  
/* ibufExpandMacro - stuff the buffer if an fkey pressed */

int ibufExpandMacro(int fkey)
{
  if (fkey < 0 || fkey >= MAX_MACROS)
    return(FALSE);

  ibufPut(UserConf.MacrosF[fkey]);	

#ifdef DEBUG_MACROS
  utLog("ibufExpandMacro(): got an FKey: %d", ch);
#endif
  
  return(TRUE);

}
      
int ibufExpandMouseMacro(int but, Unsgn32 mods, real mangle)
{
  int myangle = ((mangle < 0.0) ? 0 : (int)utMod360(mangle));
  char *s;
  static char buf[MAX_MACRO_LEN];

  
  if (but < 0 || but >= CONF_MAXBUTTONS)
    return(FALSE);

  if (mods < 0 || mods >= CONF_MAXMODIFIERS)
    return(FALSE);

  /* we need to translate any occurances of \a into the mangle (angle) */

  s = UserConf.Mouse[but][mods];

#if defined(DEBUG_MACROS)
  utLog("ibufExpandMouseMacro(): got MOUSE Macro Key: %d, mod %d string = '%s'", but,
       mods, s);
#endif

  while (*s)
    {
      if (*s == '\\')
        {
          s++;

          if (*s == 'a')
            {
              s++;
              snprintf(buf, MAX_MACRO_LEN - 1, "%d",
                       myangle);
              ibufPut(buf); 
            }
          else
            ibufPutc('\\');
        }
      else
        {
          ibufPutc((int)*s);
          s++;
        }
    }

  return(TRUE);
}
      
