#!/usr/bin/env python

#****************************************************************************
# convertdlg.py, provides the main dialog and GUI interface
#
# ConvertAll, a units conversion program
# Copyright (C) 2004, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

try:
    from __main__ import __version__, __author__, helpFilePath
except ImportError:
    __version__ = __author__ = '??'
    helpFilePath = None
import sys, os.path
from unitdata import UnitData
from unitgroup import UnitGroup
from option import Option
from optiondlg import OptionDlg, OptionDlgInt, OptionDlgBool, OptionDlgPush
from unitedit import UnitEdit
from unitlistview import UnitListView
from numedit import NumEdit
from helpview import HelpView
from finddlg import FindDlg
from qt import *


class ModButton(QPushButton):
    """Button used to add operators, change exponent or clear unit"""
    def __init__(self, function, param, label, parent=None, \
                 name=None):
        QPushButton.__init__(self, label, parent, name)
        self.function = function
        self.param = param
        self.setFocusPolicy(QWidget.NoFocus)
        self.connect(self, SIGNAL('clicked()'), self.exe)

    def exe(self):
        """Execute function on button push"""
        if self.param != None:
            self.function(self.param)
        else:
            self.function()
        self.emit(PYSIGNAL('stateChg'), ())   # update listView

    def sizeHint(self):
        """Adjust width smaller"""
        oldSize = QPushButton.sizeHint(self)
        return QSize(oldSize.height(), oldSize.height())


class ConvertDlg(QWidget):
    """Main dialog for ConvertAll program"""
    defaultList = [\
        "# Options for the ConvertAll unit conversion program",
        "#",
        "# All options are set from within the program,",
        "# editing here is not recommended",
        "#",
        "BackgroundR         255",
        "BackgroundG         255",
        "BackgroundB         255",
        "ForegroundR         0",
        "ForegroundG         0",
        "ForegroundB         0",
        "#",
        "DecimalPlaces       8",
        "SciNotation         no",
        "FixedDecimals       no",
        "ShowOpButtons       yes",
        "#",
        "MainDlgXSize        0",
        "MainDlgYSize        0",
        "MainDlgXPos         0",
        "MainDlgYPos         0",
        "FinderXSize         0",
        "FinderYSize         0",
        "FinderXPos          0",
        "FinderYPos          0"]
    convert_xpm = [
        "16 16 5 1",
        " 	c None",
        ".	c #000000",
        "+	c #C9C12C",
        "@	c #C42727",
        "#	c #6F6FF2",
        "  @@@@@..#####  ",
        " @@@@@@..###### ",
        "@@...@@..#######",
        "@@.@@......#####",
        "@@..@@@..##...##",
        "@.@@.@@..####.##",
        "@.@@.@@..###..##",
        "@.@@.@@..##.##.#",
        ".@@@@.@..##.##.#",
        ".@@@@.@..##.##.#",
        ".@++@.@..#.####.",
        "......@..#.#++#.",
        "@....@@..#.#++#.",
        "@@..@@@..#......",
        " @@@@@@..##.... ",
        "  @@@@@..###..  "]
    option = Option('convertall', 20)
    unitData = UnitData()

    def __init__(self, parent=None, name=None):
        QWidget.__init__(self, parent, name, Qt.WType_TopLevel)
        if sys.platform == 'win32':
            self.setCaption('ConvertAll  (PyQt)')
        else:
            self.setCaption('ConvertAll')
        self.setIcon(QPixmap(ConvertDlg.convert_xpm))
        self.helpView = None
        self.findDlg = None
        ConvertDlg.option.loadAll(ConvertDlg.defaultList)
        try:
            ConvertDlg.unitData.readData()
        except UnitDataError, text:
            QMessageBox.warning(self, 'ConvertAll', \
                                'Error in unit data - %s' % text)
        self.fromGroup = UnitGroup(ConvertDlg.unitData, ConvertDlg.option)
        self.toGroup = UnitGroup(ConvertDlg.unitData, ConvertDlg.option)
        self.updateColors()
        self.textButtons = []

        topLayout = QHBoxLayout(self, 10)    # divide main, buttons
        mainLayout = QVBoxLayout(topLayout)
        unitLayout = QHBoxLayout(mainLayout) # unit selection

        fromBox = QVGroupBox('From Unit', self)
        unitLayout.addWidget(fromBox)
        self.fromUnitEdit = UnitEdit(self.fromGroup, fromBox)
        self.fromUnitEdit.setFocus()
        self.fromUnitListView = UnitListView(self.fromGroup, 0, fromBox)
        self.connect(self.fromUnitEdit, PYSIGNAL('currentChanged'), \
                     self.fromUnitListView.updateSelection)
        self.connect(self.fromUnitListView, PYSIGNAL('unitChanged'), \
                     self.fromUnitEdit.unitUpdate)
        self.fromUnitListView.setFocusProxy(self.fromUnitEdit)
        fromButtonBox = QHBox(fromBox)
        self.fromUnitListView.buttonBox = fromButtonBox
        self.addButtons(self.fromGroup, self.fromUnitListView, \
                        fromButtonBox, fromBox)

        toBox = QVGroupBox('To Unit', self)
        unitLayout.addWidget(toBox)
        self.toUnitEdit = UnitEdit(self.toGroup, toBox)
        self.toUnitListView = UnitListView(self.toGroup, 1, toBox)
        self.connect(self.toUnitEdit, PYSIGNAL('currentChanged'), \
                     self.toUnitListView.updateSelection)
        self.connect(self.toUnitListView, PYSIGNAL('unitChanged'), \
                     self.toUnitEdit.unitUpdate)
        self.toUnitListView.setFocusProxy(self.toUnitEdit)
        toButtonBox = QHBox(toBox)
        self.toUnitListView.buttonBox = toButtonBox
        self.addButtons(self.toGroup, self.toUnitListView, \
                        toButtonBox, toBox)
        self.showHideButtons()

        numberLayout = QHBoxLayout(mainLayout)
        statusLabel = QLabel('Set units', self)
        statusLabel.setFrameStyle(QFrame.Panel | QFrame.Sunken)
        mainLayout.addWidget(statusLabel)

        fromNumBox = QVGroupBox('No Unit Set', self)
        numberLayout.addWidget(fromNumBox)
        self.fromNumEdit = NumEdit(self.fromGroup, self.toGroup, fromNumBox, \
                                   statusLabel, 1, fromNumBox)
        self.connect(self.fromUnitEdit, PYSIGNAL('unitChanged'), \
                     self.fromNumEdit.unitUpdate)
        self.fromNumEdit.setEnabled(0)
        equalsLabel = QLabel(' = ', self)
        equalsLabel.setFont(QFont(self.font().family(), 30))
        numberLayout.addWidget(equalsLabel)

        toNumBox = QVGroupBox('No Unit Set', self)
        numberLayout.addWidget(toNumBox)
        self.toNumEdit = NumEdit(self.toGroup, self.fromGroup, toNumBox, \
                                 statusLabel, 0, toNumBox)
        self.connect(self.toUnitEdit, PYSIGNAL('unitChanged'), \
                     self.toNumEdit.unitUpdate)
        self.toNumEdit.setEnabled(0)
        self.connect(self.fromNumEdit, PYSIGNAL('convertNum'), \
                     self.toNumEdit.setNum)
        self.connect(self.toNumEdit, PYSIGNAL('convertNum'), \
                     self.fromNumEdit.setNum)
        self.connect(self.fromNumEdit, PYSIGNAL('convertRqd'), \
                     self.toNumEdit.convert)
        self.connect(self.toNumEdit, PYSIGNAL('convertRqd'), \
                     self.fromNumEdit.convert)

        buttonLayout = QVBoxLayout(topLayout)     # major buttons
        closeButton = QPushButton('&Close', self)
        buttonLayout.addWidget(closeButton)
        closeButton.setFocusPolicy(QWidget.NoFocus)
        self.connect(closeButton, SIGNAL('clicked()'), self.close)
        finderButton = QPushButton('&Unit Finder...', self)
        buttonLayout.addWidget(finderButton)
        finderButton.setFocusPolicy(QWidget.NoFocus)
        self.connect(finderButton, SIGNAL('clicked()'), self.showFinder)
        optionsButton = QPushButton('&Options...', self)
        buttonLayout.addWidget(optionsButton)
        optionsButton.setFocusPolicy(QWidget.NoFocus)
        self.connect(optionsButton, SIGNAL('clicked()'), self.changeOptions)
        helpButton = QPushButton('&Help...', self)
        buttonLayout.addWidget(helpButton)
        helpButton.setFocusPolicy(QWidget.NoFocus)
        self.connect(helpButton, SIGNAL('clicked()'), self.help)
        aboutButton = QPushButton('&About...', self)
        buttonLayout.addWidget(aboutButton)
        aboutButton.setFocusPolicy(QWidget.NoFocus)
        self.connect(aboutButton, SIGNAL('clicked()'), self.about)
        buttonLayout.addStretch(10)

        xSize = self.option.intData('MainDlgXSize', 0, 10000)
        ySize = self.option.intData('MainDlgYSize', 0, 10000)
        if xSize and ySize:
            self.resize(xSize, ySize)
        self.move(self.option.intData('MainDlgXPos', 0, 10000), \
                  self.option.intData('MainDlgYPos', 0, 10000))

    def addButtons(self, unitGroup, listView, buttonBox, groupBox):
        """Add buttons to unit selector"""
        buttons = []
        buttons.append(ModButton(unitGroup.addOper, 1, 'X', buttonBox))
        buttons.append(ModButton(unitGroup.addOper, 0, '/', buttonBox))
        buttons.append(ModButton(unitGroup.changeExp, 2, '^2', buttonBox))
        buttons.append(ModButton(unitGroup.changeExp, 3, '^3', buttonBox))
        buttons.append(ModButton(unitGroup.clearUnit, None, 'Clear Unit', \
                                 groupBox))
        buttonBox.setEnabled(0)
        for but in buttons:
            self.connect(but, PYSIGNAL('stateChg'), listView.relayChange)
            self.textButtons.append(but)

    def updateColors(self):
        """Adjust the colors to the current option settings"""
        pal = QApplication.palette()
        background = QColor(self.option.intData('BackgroundR', 0, 255), \
                            self.option.intData('BackgroundG', 0, 255), \
                            self.option.intData('BackgroundB', 0, 255))
        foreground = QColor(self.option.intData('ForegroundR', 0, 255), \
                            self.option.intData('ForegroundG', 0, 255), \
                            self.option.intData('ForegroundB', 0, 255))
        pal.setColor(QColorGroup.Base, background)
        pal.setColor(QPalette.Active, QColorGroup.Text, foreground)
        pal.setColor(QPalette.Inactive, QColorGroup.Text, foreground)
        QApplication.setPalette(pal, 1)

    def showFinder(self):
        """Show dialog for searhing and filtering units"""
        if not self.findDlg:
            self.findDlg = FindDlg(self)
        self.findDlg.show()

    def changeOptions(self):
        """Show dialog for option changes"""
        dlg = OptionDlg(ConvertDlg.option, self, None, 1)
        dlg.setIcon(QPixmap(ConvertDlg.convert_xpm))
        dlg.startGroupBox('Result Display')
        OptionDlgInt(dlg, 'DecimalPlaces', 'Decimal places', \
                     0, UnitGroup.maxDecPlcs)
        OptionDlgBool(dlg, 'SciNotation', 'Use scientific notation')
        OptionDlgBool(dlg, 'FixedDecimals', 'Use fixed decimal places')
        dlg.startGroupBox('Buttons')
        OptionDlgBool(dlg, 'ShowOpButtons', 'Show operator buttons')
        dlg.startGroupBox('Colors')
        OptionDlgPush(dlg, 'Set background color', self.backColor)
        OptionDlgPush(dlg, 'Set text color', self.textColor)
        if dlg.exec_loop() == QDialog.Accepted and ConvertDlg.option.chgList:
            self.option.writeChanges()
            self.showHideButtons()
            self.fromNumEdit.unitUpdate()
            self.toNumEdit.unitUpdate()

    def showHideButtons(self):
        """Show or hide text modify buttons"""
        visible = self.option.boolData('ShowOpButtons')
        for button in self.textButtons:
            if visible:
                button.show()
            else:
                button.hide()

    def backColor(self):
        """Allow user to set control background color"""
        background = QColor(self.option.intData('BackgroundR', 0, 255), \
                            self.option.intData('BackgroundG', 0, 255), \
                            self.option.intData('BackgroundB', 0, 255))
        newColor = QColorDialog.getColor(background, self)
        if newColor.isValid() and newColor != background:
            self.option.changeData('BackgroundR', `newColor.red()`, 1)
            self.option.changeData('BackgroundG', `newColor.green()`, 1)
            self.option.changeData('BackgroundB', `newColor.blue()`, 1)
            self.option.writeChanges()
            self.updateColors()

    def textColor(self):
        """Allow user to set control text color"""
        foreground = QColor(self.option.intData('ForegroundR', 0, 255), \
                            self.option.intData('ForegroundG', 0, 255), \
                            self.option.intData('ForegroundB', 0, 255))
        newColor = QColorDialog.getColor(foreground, self)
        if newColor.isValid() and newColor != foreground:
            self.option.changeData('ForegroundR', `newColor.red()`, 1)
            self.option.changeData('ForegroundG', `newColor.green()`, 1)
            self.option.changeData('ForegroundB', `newColor.blue()`, 1)
            self.option.writeChanges()
            self.updateColors()

    def findReadMeFile(self, pathList, fileName):
        """Search paths for file, return line list or None""" 
        for path in pathList:
            try:
                f = open(os.path.join(path, fileName), 'r')
                lineList = f.readlines()
                f.close()
                return lineList
            except IOError:
                pass
        return None

    def help(self):
        """View the ReadMe file"""
        if not self.helpView:
            modPath = os.path.abspath(sys.path[0])
            pathList = [helpFilePath, os.path.join(modPath, '../doc/'), \
                        modPath]
            lineList = self.findReadMeFile(filter(None, pathList), \
                                           'README.html')
            if not lineList:
                self.statusBar().message('Read Me file not found', 4000)
                return
            self.helpView = HelpView('\n'.join(lineList), \
                                     'ConvertAll README File')
            self.helpView.setIcon(QPixmap(ConvertDlg.convert_xpm))
        self.helpView.show()

    def about(self):
        """Show about info"""
        QMessageBox.about(self, 'ConvertAll', \
                          'ConvertAll Version %s\nby %s' % \
                          (__version__, __author__))
    
    def closeEvent(self, event):
        """Save window data on close"""
        self.option.changeData('MainDlgXSize', self.width(), 1)
        self.option.changeData('MainDlgYSize', self.height(), 1)
        self.option.changeData('MainDlgXPos', self.x(), 1)
        self.option.changeData('MainDlgYPos', self.y(), 1)
        if self.findDlg:
            self.option.changeData('FinderXSize', self.findDlg.width(), 1)
            self.option.changeData('FinderYSize', self.findDlg.height(), 1)
            self.option.changeData('FinderXPos', self.findDlg.x(), 1)
            self.option.changeData('FinderYPos', self.findDlg.y(), 1)
        self.option.writeChanges()
        event.accept()
