#!/usr/bin/env python

#****************************************************************************
# numedit.py, provides a number entry editor
#
# ConvertAll, a units conversion program
# Copyright (C) 2003, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

import re, sys
from qt import *

if sys.version_info[:2] >= (2, 2):
    from numeval2 import evalFloat
else:
    from numeval1 import evalFloat


class NumEdit(QLineEdit):
    "Number entry editor"
    def __init__(self, thisUnit, otherUnit, label, status, primary, \
                 parent=None, name=None):
        QLineEdit.__init__(self, parent, name)
        self.thisUnit = thisUnit
        self.otherUnit = otherUnit
        self.label = label
        self.status = status
        self.primary = primary
        self.onLeft = primary
        self.setValidator(FloatExprValidator(self))
        self.setText(self.thisUnit.formatNumStr(1.0))
        self.connect(self, SIGNAL('textChanged(const QString &)'), \
                     self.convert)

    def unitUpdate(self):
        "Update the editor and labels based on a unit change"
        if self.thisUnit.groupValid():
            self.label.setTitle(self.thisUnit.unitString())
            if self.otherUnit.groupValid():
                try:
                    self.thisUnit.reduceGroup()
                    self.otherUnit.reduceGroup()
                except UnitDataError, text:
                    QMessageBox.warning(self, 'ConvertAll', \
                                        'Error in unit data - %s' % text)
                    return
                if self.thisUnit.categoryMatch(self.otherUnit):
                    self.status.setText('Converting...')
                    if self.primary:
                        self.convert()
                    else:
                        self.emit(PYSIGNAL('convertRqd'), ())
                    return
                if self.onLeft:
                    self.status.setText('Units are not compatible '\
                               '(%s  vs.  %s)' % (self.thisUnit.compatStr(), \
                                                  self.otherUnit.compatStr()))
                else:
                    self.status.setText('Units are not compatible '\
                               '(%s  vs.  %s)' % (self.otherUnit.compatStr(), \
                                                  self.thisUnit.compatStr()))
            else:
                self.status.setText('Set units')
        else:
            self.status.setText('Set units')
            self.label.setTitle('No Unit Set')
        self.setEnabled(0)
        self.emit(PYSIGNAL('convertNum'), (None,))

    def convert(self):
        "Do conversion with self primary"
        self.primary = 1
        self.setEnabled(1)
        num = evalFloat(str(self.text()))
        if num == None:
            self.emit(PYSIGNAL('convertNum'), ('',))
            return
        try:
            numText = self.thisUnit.convertStr(num, self.otherUnit)
            self.emit(PYSIGNAL('convertNum'), (numText,))
        except UnitDataError, text:
            QMessageBox.warning(self, 'ConvertAll', \
                                'Error in unit data - %s' % text)

    def setNum(self, numText):
        "Set text based on conversion from other number editor"
        if numText != None:
            self.primary = 0
            self.setEnabled(1)
            self.blockSignals(1)
            self.setText(numText)
            self.blockSignals(0)
        else:
            self.setEnabled(0)


class FloatExprValidator(QValidator):
    "Validator for float python expressions typed into NumEdit"
    def __init__(self, parent=None, name=None):
        QValidator.__init__(self, parent, name)

    def validate(self, inputStr, pos):
        "Check for valid characters in entry"
        if re.search('[^\d\.eE\+\-\*/\(\)]', str(inputStr)):  # invalid char
            return (QValidator.Invalid, pos)
        return (QValidator.Acceptable, pos)
