#!/usr/bin/env python

#****************************************************************************
# unitlistview.py, provides a list view of available units
#
# ConvertAll, a units conversion program
# Copyright (C) 2003, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

import convertdlg
from qt import *


class UnitListView(QListView):
    "ListView of units available"
    def __init__(self, unitGroup, unitRefNum, parent=None, name=None):
        QListView.__init__(self, parent, name)
        self.unitRefNum = unitRefNum
        self.unitGroup = unitGroup
        self.buttonBox = None
        self.addColumn('Unit Name', 140)
        self.addColumn('Unit Type', 80)
        self.addColumn('Comments', 200)
        self.setSorting(-1)
        self.setAllColumnsShowFocus(1)
        self.showFocus = 0
        self.connect(self, SIGNAL('selectionChanged()'), self.replaceUnit)
        self.loadUnits()

    def loadUnits(self):
        "Load unit items"
        self.clear()
        tmpList = convertdlg.ConvertDlg.unitData.sortedKeys[:]
        tmpList.reverse()
        for name in tmpList:
            UnitListViewItem(convertdlg.ConvertDlg.unitData[name], \
                             self.unitRefNum, self)

    def relayChange(self):
        "Update list after buttons changed the unit group"
        self.updateSelection()
        self.setFocus()
        self.emit(PYSIGNAL('unitChanged'), ())   # update unitEdit

    def updateSelection(self):
        "Update list after change to line editor"
        self.blockSignals(1)
        self.showFocus = 0
        self.buttonBox.setEnabled(1)
        unit = self.unitGroup.currentUnit()
        if unit:
            self.setSelected(unit.viewLink[self.unitRefNum], 1)
        else:
            unit = self.unitGroup.currentPartialUnit()
            if unit:
                self.setCurrentItem(unit.viewLink[self.unitRefNum])
                self.showFocus = 1
            else:
                unit = self.unitGroup.currentSortPos()
                self.buttonBox.setEnabled(0)
            self.clearSelection()
            self.updateContents()
        self.center(0, unit.viewLink[self.unitRefNum].itemPos())
        self.blockSignals(0)

    def replaceUnit(self):
        "Replace current unit in response to a selection change"
        self.showFocus = 0
        item = self.selectedItem()
        if item:
            self.unitGroup.replaceCurrent(item.unit)
            self.emit(PYSIGNAL('unitChanged'), ())   # update unitEdit
            self.buttonBox.setEnabled(1)

    def sizeHint(self):
        "Adjust width smaller"
        return QSize(240, 170)


class UnitListViewItem(QListViewItem):
    "Item in list view, references unit"
    def __init__(self, unit, unitRefNum, parent=None):
        QListViewItem.__init__(self, parent)
        self.unit = unit
        unit.viewLink[unitRefNum] = self
        self.setText(0, unit.description())
        self.setText(1, unit.typeName)
        self.setText(2, unit.comments[1])

    def paintFocus(self, painter, color, rect):
        "Override to avoid painting of focus rectangle"
        if self.listView().showFocus:
            QListViewItem.paintFocus(self, painter, color, rect)
