/* $Id: cpl_fits-test.c,v 1.17 2010/11/11 09:23:18 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/11/11 09:23:18 $
 * $Revision: 1.17 $
 * $Name: cpl-5_3_0-BRANCH $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <assert.h>

#include "cpl_fits.h"
#include "cpl_memory.h"
#include "cpl_propertylist.h"
#include "cpl_test.h"
#include "cpl_propertylist.h"

#define CPL_FITS_NAME "cpl_fits-test.fits"

#ifndef CPL_FITS_NEXT
#define CPL_FITS_NEXT 12
#endif

/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{
    cpl_propertylist * pl;
    int                ivalue;
    cpl_error_code     error;
    int                iext;


    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */

    ivalue = cpl_fits_count_extensions(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(ivalue, -1);

    ivalue = cpl_fits_find_extension(CPL_FITS_NAME, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(ivalue, -1);

    ivalue = cpl_fits_find_extension(NULL, "EXT0");
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(ivalue, -1);

    ivalue = cpl_fits_count_extensions(".");
    cpl_test_error(CPL_ERROR_FILE_IO);
    cpl_test_eq(ivalue, -1);

    ivalue = cpl_fits_find_extension(".", "EXT0");
    cpl_test_error(CPL_ERROR_FILE_IO);
    cpl_test_eq(ivalue, -1);

    /* Create a test MEF */

    /* Empty primary unit */
    pl = cpl_propertylist_new();
    cpl_test_nonnull(pl);
    error = cpl_propertylist_save(pl, CPL_FITS_NAME, CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    ivalue = cpl_fits_count_extensions(CPL_FITS_NAME);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(ivalue);

    ivalue = cpl_fits_find_extension(CPL_FITS_NAME, "EXT0");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(ivalue, 0);

    for (iext = 1; iext <= CPL_FITS_NEXT; iext++) {
        const char * extname = cpl_sprintf("EXT%d", iext);
        int jext;

        error = cpl_propertylist_update_string(pl, "EXTNAME", extname);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        error = cpl_propertylist_save(pl, CPL_FITS_NAME, CPL_IO_EXTEND);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        ivalue = cpl_fits_count_extensions(CPL_FITS_NAME);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_eq(ivalue, iext);

        cpl_free((void*)extname);

        for (jext = 0; jext <= iext; jext++) {
            extname = cpl_sprintf("EXT%d", jext);

            ivalue = cpl_fits_find_extension(CPL_FITS_NAME, extname);
            cpl_test_error(CPL_ERROR_NONE);

            cpl_free((void*)extname);

            if (jext > 0) {
                cpl_test_eq(ivalue, jext);
            } else {
                /* EXT0 not found */
                cpl_test_eq(ivalue, 0);
            }
        }
    }

    cpl_propertylist_delete(pl);

    cpl_test_zero(remove(CPL_FITS_NAME));

    return cpl_test_end(0);
}
