#include "../config.h"

#include <string.h>
#include <gtk/gtklabel.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <glibtop.h>
#include <glibtop/cpu.h>
#include <libgnomeui/libgnomeui.h>
#include <math.h>
#include <glade/glade.h>

#include <gnome.h>
#include <libgnome/libgnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>

#define AVERAGE_NUM_TIMES ((int) 32)

typedef struct {
	PanelApplet base;
	GtkTooltips* tooltips;
	guchar* rgbbuf;
	guchar* firebuf;
	GtkWidget* drawing_area;
	GtkWidget* about_dialog;
	GtkWidget* prefs_dialog;

	guint timeout_handler_id;
	int ref_time_idx;
	int time_idx;
	int tooltip_update;
	glibtop_cpu* cpu;

	gboolean config_change;

	gint flame_sensitivity;
	gint update_interval;
	int decay;
	int border;
	int panel_border;
	gint new_image_width;
	gint new_image_height;
	gint image_width;
	gint image_height;
	// width or height depending on orientation
	gint image_size;
} CPUFireApplet;

static GladeXML *xml = NULL;
static gchar* glade_file=NULL;

static unsigned int cpufire_flame[256] = {
  0x000000, 0x000001, 0x000002, 0x000003, 0x000004, 0x000005, 0x000006, 0x000007,
  0x000009, 0x00000a, 0x00000b, 0x00000c, 0x00000d, 0x00000e, 0x00000f, 0x000010,
  0x000012, 0x000013, 0x000014, 0x000015, 0x000016, 0x000017, 0x000018, 0x000019,
  0x00001b, 0x00001c, 0x00001d, 0x00001e, 0x00001f, 0x000020, 0x000021, 0x000022,
  0x000024, 0x000025, 0x000026, 0x000027, 0x000028, 0x000029, 0x00002a, 0x00002b,
  0x00002d, 0x00002e, 0x00002f, 0x000030, 0x000031, 0x000032, 0x000033, 0x000034,
  0x000036, 0x000037, 0x000038, 0x000039, 0x00003a, 0x00003b, 0x00003c, 0x00003d,
  0x00003f, 0x000040, 0x000041, 0x000042, 0x000043, 0x000044, 0x000045, 0x000046,
  0x000048, 0x000049, 0x00004a, 0x00004b, 0x00004c, 0x00004d, 0x00004e, 0x00004f,
  0x000051, 0x000052, 0x000053, 0x000054, 0x010055, 0x010056, 0x010057, 0x010058,
  0x01005a, 0x01005b, 0x01005c, 0x01005d, 0x01005e, 0x01005f, 0x020060, 0x09005c,
  0x100058, 0x180054, 0x1f0050, 0x26004c, 0x2e0049, 0x350045, 0x3c0041, 0x44003d,
  0x4b0039, 0x520036, 0x590032, 0x61002e, 0x68002a, 0x6f0026, 0x770022, 0x7e001f,
  0x85001b, 0x8d0017, 0x940013, 0x9b000f, 0xa3000b, 0xaa0008, 0xb10004, 0xb80000,
  0xbb0301, 0xbe0802, 0xc00c04, 0xc21106, 0xc41507, 0xc61a09, 0xc91e0a, 0xcb230c,
  0xcd270e, 0xcf2b0f, 0xd23011, 0xd43412, 0xd63914, 0xd83d16, 0xda4217, 0xdc4619,
  0xde491a, 0xdf4b1b, 0xe14e1c, 0xe2511d, 0xe3541e, 0xe5571f, 0xe65920, 0xe85c21,
  0xe95f22, 0xea6223, 0xec6524, 0xed6825, 0xef6a26, 0xf06d27, 0xf17028, 0xf37329,
  0xf4762a, 0xf6782b, 0xf77b2c, 0xf87e2d, 0xfa812e, 0xfb842f, 0xfd8730, 0xfe8931,
  0xfe8e30, 0xfe942e, 0xfd9a2d, 0xfda12b, 0xfda729, 0xfcad27, 0xfcb325, 0xfbb923,
  0xfbbf21, 0xfac51f, 0xfacb1e, 0xfad01c, 0xf9d61a, 0xf9dc18, 0xf8e217, 0xf8e715,
  0xf8ed13, 0xf8ef1a, 0xf8ef25, 0xf8f030, 0xf8f03a, 0xf8f145, 0xf8f150, 0xf8f15a,
  0xf8f265, 0xf8f270, 0xf8f37a, 0xf8f384, 0xf8f48e, 0xf8f498, 0xf8f4a2, 0xf8f5ac,
  0xf8f5b6, 0xf8f6c0, 0xf8f6ca, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4,
  0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4, 0xf9f6d4
};

static void start_procman_cb (BonoboUIComponent* uic, CPUFireApplet* applet);
static void cpufire_applet_properties_dialog (BonoboUIComponent* uic, CPUFireApplet* applet);
static void cpufire_applet_help_cb (BonoboUIComponent* uic, CPUFireApplet* applet);
static void cpufire_applet_about_cb (BonoboUIComponent* uic, CPUFireApplet* applet);
static void cpufire_applet_prefs_response_cb (GtkDialog* dialog, gint response, CPUFireApplet* applet);
static void cpufire_applet_set_size(CPUFireApplet* applet);

static const BonoboUIVerb cpufire_menu_verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("MultiLoadRunProcman", start_procman_cb),
	BONOBO_UI_UNSAFE_VERB ("CPUFirePreferences", cpufire_applet_properties_dialog),
	BONOBO_UI_UNSAFE_VERB ("CPUFireHelp", cpufire_applet_help_cb),
	BONOBO_UI_UNSAFE_VERB ("CPUFireAbout", cpufire_applet_about_cb),
	BONOBO_UI_VERB_END
};


static GType
cpufire_applet_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (PanelAppletClass),
			NULL, NULL, NULL, NULL, NULL,
			sizeof (CPUFireApplet),
			0, NULL, NULL
		};

		type = g_type_register_static (
				PANEL_TYPE_APPLET, "CPUFireApplet", &info, 0);
	}

	return type;
}

static void
cpufire_applet_load_properties (CPUFireApplet *applet)
{
	applet->update_interval = panel_applet_gconf_get_int (PANEL_APPLET (applet), "update_interval", NULL);
	applet->flame_sensitivity = panel_applet_gconf_get_int (PANEL_APPLET (applet), "flame_sensitivity", NULL);
	applet->image_size = panel_applet_gconf_get_int (PANEL_APPLET (applet), "image_size", NULL);


	/* when gconf_get_int failes */
	if (!applet->update_interval)
		applet->update_interval = 50;
	if (!applet->flame_sensitivity)
		applet->flame_sensitivity = 1000;
	if (!applet->image_size)
		applet->image_size = 64;

	/* check range */
	if (applet->update_interval < 10)
		applet->update_interval = 10;
	if (applet->flame_sensitivity < 10)
		applet->flame_sensitivity = 10;
	if (applet->image_size < 16)
		applet->image_size = 16;
}

static void
cpufire_applet_save_properties (CPUFireApplet *applet)
{
	panel_applet_gconf_set_int (PANEL_APPLET (applet), "update_interval",   (gint) (applet->update_interval), NULL);
	panel_applet_gconf_set_int (PANEL_APPLET (applet), "flame_sensitivity", (gint) (applet->flame_sensitivity), NULL);
	panel_applet_gconf_set_int (PANEL_APPLET (applet), "image_size",       (gint) (applet->image_size), NULL);
}

static void
cpufire_applet_option_change (GtkWidget *widget, gpointer user_data)
{
	GtkWidget* w;
	CPUFireApplet* applet = (CPUFireApplet*) user_data;

	w= g_object_get_data (G_OBJECT (applet->prefs_dialog), "size-spinbutton");
	applet->image_size = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));
	w= g_object_get_data (G_OBJECT (applet->prefs_dialog), "update-interval-spinbutton");
	applet->update_interval = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));
	w= g_object_get_data (G_OBJECT (applet->prefs_dialog), "sensitivity-spinbutton");
	applet->flame_sensitivity = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

	/* Save the properties */
	cpufire_applet_save_properties (applet);
	cpufire_applet_set_size(applet);
}


static void
cpufire_applet_properties_dialog (BonoboUIComponent *uic, CPUFireApplet *applet)
{
	static GtkWidget* global_property_box = NULL;
	static GtkWidget* glade_property_box = NULL;
	GtkWidget* spin_button;

	g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));

	if (applet->prefs_dialog != NULL)
	{
		gtk_widget_show (applet->prefs_dialog);
		gtk_window_present (GTK_WINDOW (applet->prefs_dialog));
		return;
	}

	if (global_property_box == NULL) {
		xml = glade_xml_new (glade_file, NULL, NULL);
		glade_property_box = glade_xml_get_widget (xml,"dialog1");
	}

	applet->prefs_dialog = glade_property_box;
	gtk_window_set_resizable (GTK_WINDOW (applet->prefs_dialog), FALSE);
	
	spin_button = glade_xml_get_widget (xml, "size-spinbutton");
	gtk_object_set_data (GTK_OBJECT (applet->prefs_dialog),
			"size-spinbutton", spin_button);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button), applet->image_size);
	g_signal_connect(GTK_OBJECT(spin_button), "value_changed",
			GTK_SIGNAL_FUNC(cpufire_applet_option_change), applet);

	spin_button = glade_xml_get_widget (xml, "update-interval-spinbutton");
	gtk_object_set_data (GTK_OBJECT (applet->prefs_dialog),
			"update-interval-spinbutton", spin_button);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button), applet->update_interval);
	g_signal_connect(GTK_OBJECT(spin_button), "value_changed",
			GTK_SIGNAL_FUNC(cpufire_applet_option_change), applet);

	spin_button = glade_xml_get_widget (xml, "sensitivity-spinbutton");
	gtk_object_set_data (GTK_OBJECT (applet->prefs_dialog),
			"sensitivity-spinbutton", spin_button);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button), applet->flame_sensitivity);
	g_signal_connect(GTK_OBJECT(spin_button), "value_changed",
			GTK_SIGNAL_FUNC(cpufire_applet_option_change), applet);

	g_signal_connect (GTK_OBJECT (applet->prefs_dialog),
			"response", 
			G_CALLBACK (cpufire_applet_prefs_response_cb),
			applet);
	g_signal_connect (GTK_OBJECT (applet->prefs_dialog),
			"destroy",
			GTK_SIGNAL_FUNC (gtk_widget_destroy),
			NULL);
	g_object_add_weak_pointer (G_OBJECT (applet->prefs_dialog),
			(void**)&(applet->prefs_dialog));
	gtk_window_set_screen (GTK_WINDOW (applet->prefs_dialog),
			       gtk_widget_get_screen (GTK_WIDGET (applet)));
	gtk_widget_show_all (applet->prefs_dialog);
	return;
}

static void
cpufire_applet_destroy (CPUFireApplet *applet, gpointer horse)
{
	if (applet->timeout_handler_id > 0) {
		gtk_timeout_remove (applet->timeout_handler_id);
		applet->timeout_handler_id = 0;
	}

	if (applet->about_dialog) {
		gtk_widget_destroy (applet->about_dialog);
		applet->about_dialog = NULL;
	}

	if (applet->prefs_dialog) {
		gtk_widget_destroy (applet->prefs_dialog);
		applet->prefs_dialog = NULL;
	}

	if (applet->drawing_area) {
		gtk_widget_destroy (applet->drawing_area);
		applet->drawing_area = NULL;
	}

	if (applet->tooltips)
		g_object_unref (applet->tooltips);

	g_free(applet->rgbbuf);
	applet->rgbbuf=NULL;

	g_free(applet->firebuf);
	applet->firebuf=NULL;

	g_free(applet->cpu);
	applet->cpu=NULL;
}

static void
cpufire_applet_set_size(CPUFireApplet* applet)
{
	gint panel_size = panel_applet_get_size (PANEL_APPLET (applet)) - 2*applet->panel_border;
	switch (panel_applet_get_orient(PANEL_APPLET (applet))) {
	case PANEL_APPLET_ORIENT_LEFT:
	case PANEL_APPLET_ORIENT_RIGHT:
		applet->new_image_width = panel_size;
		applet->new_image_height = applet->image_size;
		break;
	case PANEL_APPLET_ORIENT_UP:
	case PANEL_APPLET_ORIENT_DOWN:
		applet->new_image_width = applet->image_size;
		applet->new_image_height = panel_size;
		break;
	}

	applet->config_change = TRUE;
}

static void
cpufire_applet_change_size_cb(PanelApplet *pa, gint s, CPUFireApplet *applet)
{
	cpufire_applet_set_size(applet);
}

static void
cpufire_applet_change_orient_cb(PanelApplet *pa, gint s, CPUFireApplet *applet)
{
	cpufire_applet_set_size(applet);
}

static void change_background_cb (
                PanelApplet *a,
                PanelAppletBackgroundType type,
                GdkColor *color, GdkPixmap *pixmap,
                CPUFireApplet *applet)
{
    GtkRcStyle *rc_style;
    GtkStyle *style;

    /* reset style */
    gtk_widget_set_style (GTK_WIDGET (&(applet->base)), NULL);
    rc_style = gtk_rc_style_new ();
    gtk_widget_modify_style (GTK_WIDGET (&(applet->base)), rc_style);
	gtk_rc_style_unref (rc_style);

    switch (type) {
        case PANEL_COLOR_BACKGROUND:
            gtk_widget_modify_bg (GTK_WIDGET (&(applet->base)),
                    GTK_STATE_NORMAL, color);
            break;

        case PANEL_PIXMAP_BACKGROUND:
            style = gtk_style_copy (GTK_WIDGET (&(applet->base))->style);
            if (style->bg_pixmap[GTK_STATE_NORMAL])
                g_object_unref(style->bg_pixmap[GTK_STATE_NORMAL]);

            style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref(pixmap);
            gtk_widget_set_style (GTK_WIDGET (&(applet->base)), style);
			g_object_unref(style);
            break;

        case PANEL_NO_BACKGROUND:
        default:
            break;
    }
}

static void
start_procman_cb (BonoboUIComponent* uic, CPUFireApplet* applet)
{
	GError *error = NULL;

	gdk_spawn_command_line_on_screen (
			gtk_widget_get_screen (GTK_WIDGET (applet)),
			"gnome-system-monitor", &error);

	if (error) {
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new (NULL,
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("There was an error executing '%s': %s"),
				"gnome-system-monitor",
				error->message);

		g_signal_connect (dialog, "response",
				G_CALLBACK (gtk_widget_destroy),
				NULL);

		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
		gtk_window_set_screen (GTK_WINDOW (dialog),
				gtk_widget_get_screen (GTK_WIDGET (applet)));

		gtk_widget_show (dialog);

		g_error_free (error);
	}
}

static void
cpufire_applet_help_cb (BonoboUIComponent* uic, CPUFireApplet* applet)
{
	GError *error = NULL;

    static GnomeProgram *applet_program = NULL;
        
    if (!applet_program) {
        int argc = 1; 
        char *argv[2] = { "cpufire" };
        applet_program = gnome_program_init ("cpufire", VERSION,
                              LIBGNOME_MODULE, argc, argv,
                              GNOME_PROGRAM_STANDARD_PROPERTIES, NULL);
    }           

    gnome_help_display_desktop_on_screen (
            applet_program, "cpufire", "cpufire",NULL,
            gtk_widget_get_screen (GTK_WIDGET(applet)),
            &error);         

	if (error)
	{
		GtkWidget *dialog =
		gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
					_("There was an error displaying help: %s"),
					error->message);
		g_signal_connect (G_OBJECT (dialog), "response",
				  G_CALLBACK (gtk_widget_destroy), NULL);
		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
		gtk_window_set_screen (GTK_WINDOW (dialog),
				       gtk_widget_get_screen (GTK_WIDGET (
							      applet)));
		gtk_widget_show (dialog);
		g_error_free (error);
	}
}

static void
cpufire_applet_about_cb (BonoboUIComponent *uic, CPUFireApplet *applet)
{
	const gchar* authors[] = { "Soeren Sonnenburg <debian@nn7.de>", NULL };
	const gchar* documenters[] = { NULL }; 
	const gchar* translator_credits = NULL;
	GError* error = NULL;
	GdkPixbuf* logo = NULL;

	if (applet->about_dialog != NULL) {
		gtk_window_set_screen (GTK_WINDOW (applet->about_dialog),
				       gtk_widget_get_screen (GTK_WIDGET (&applet->base)));
		
		gtk_window_present (GTK_WINDOW (applet->about_dialog));
		return;
	}

	logo=gdk_pixbuf_new_from_file(DATADIR "/cpufire.png", &error);

	applet->about_dialog = gnome_about_new (
			_("CPU Fire Applet"),
			VERSION,
			_("(C) 2004-2007 Soeren Sonnenburg"),
			_("This utility shows the current CPU usage as a fire."),
			authors,
			documenters,
			translator_credits,
			logo);

	gtk_window_set_screen (GTK_WINDOW (applet->about_dialog),
			       gtk_widget_get_screen (GTK_WIDGET (applet)));

	g_signal_connect (applet->about_dialog, "destroy",
			  G_CALLBACK (gtk_widget_destroyed),
			  &applet->about_dialog);
 
	gtk_widget_show (applet->about_dialog);

	return;
}

static void
cpufire_applet_prefs_response_cb (GtkDialog *dialog, gint response, CPUFireApplet* applet)
{
	GError*  error = NULL;

	if (response == GTK_RESPONSE_HELP) {

		gnome_help_display_desktop_on_screen (
				NULL, "cpufire", "cpufire",NULL,
				gtk_widget_get_screen (GTK_WIDGET(applet)),
				&error);         

		if (error)
		{
			GtkWidget *dlg = gtk_message_dialog_new (GTK_WINDOW (dialog),
					GTK_DIALOG_DESTROY_WITH_PARENT
					|| GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE,
					_("There was an error displaying help: %s"),
					error->message);
			g_signal_connect (G_OBJECT (dlg), "response",
					G_CALLBACK (gtk_widget_destroy),
					NULL);
			gtk_window_set_resizable (GTK_WINDOW (dlg), FALSE);
			gtk_window_set_screen (GTK_WINDOW (dlg),
					gtk_widget_get_screen (
						GTK_WIDGET (dialog)));
			gtk_widget_show (dlg);
			g_error_free (error);
		}
	}
	else
		gtk_widget_hide (GTK_WIDGET (dialog));
}

gboolean
on_darea_expose (GtkWidget *widget,
		GdkEventExpose *event,
		CPUFireApplet* applet)
{
	if (widget && widget->window && applet && applet->rgbbuf)
	{
		gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
				0, applet->panel_border, applet->image_width, applet->image_height,
				GDK_RGB_DITHER_MAX, applet->rgbbuf, applet->image_width * 3);
		return TRUE;
	}
	else
		return FALSE;
}

gboolean
burn_flame (CPUFireApplet* applet)
{
	gint x, y;
	gint image_width=applet->image_width;
	gint image_height=applet->image_height;
	int rnd_color=0;
	int max_color=0;
	double total_avg=0;

	guchar* rgbbuf = applet->rgbbuf;
	guchar* firebuf = applet->firebuf;
	glibtop_cpu* cpu = &(applet->cpu[applet->time_idx]);

	if (applet->config_change)
	{
		gtk_timeout_remove (applet->timeout_handler_id);
		applet->timeout_handler_id = 0;
		applet->config_change = FALSE;
		g_free(applet->rgbbuf);
		g_free(applet->firebuf);
		applet->image_width=applet->new_image_width;
		applet->image_height=applet->new_image_height;
		applet->firebuf= g_malloc0((applet->image_width + 2*applet->border) * (applet->image_height + 2*applet->border));
		applet->rgbbuf= g_malloc0(applet->image_width * applet->image_height * 3);
		gtk_widget_set_size_request (applet->drawing_area, applet->image_width, applet->image_height);
		applet->timeout_handler_id = g_timeout_add (applet->update_interval, (GSourceFunc) burn_flame, applet);
		return FALSE;
	}

	glibtop_get_cpu (cpu);

	glibtop_cpu* c = &(applet->cpu[applet->time_idx]);
	glibtop_cpu* c_ref = &(applet->cpu[applet->ref_time_idx]);

	if (c->total - c_ref->total)
	{
		total_avg =  (c->user - c_ref->user +
			c->sys  - c_ref->sys +
			c->nice - c_ref->nice) / ((double) (c->total - c_ref->total));
	}

	max_color= (int) 255.0/log(applet->flame_sensitivity)*log(1+applet->flame_sensitivity*total_avg);

	if (max_color > 255)
		max_color = 255;

	applet->time_idx++;
	if (applet->time_idx >= AVERAGE_NUM_TIMES)
		applet->time_idx=0;

	if (applet->time_idx == applet->ref_time_idx)
		applet->ref_time_idx = (applet->ref_time_idx-1);

	if (applet->ref_time_idx < 0)
		applet->ref_time_idx= AVERAGE_NUM_TIMES-1;

	for (x = 0; x < image_width+2*applet->border; x++)
	{
		for (y=0; y<applet->border; y++)
		{
			if (random()>random())
				rnd_color=max_color;
			else
				rnd_color=0;

			firebuf[(image_height+applet->border+y)*(image_width+2*applet->border)+x]=rnd_color;
		}
	}

	for (y = image_height-2 + 2*applet->border ; y >= 1; y--)
	{
		for (x = 1 ; x < image_width + 2*applet->border-1; x++)
		{
				
				int val= (firebuf[y  *(image_width + 2 * applet->border)+(x)]+
						firebuf[(y+1)*(image_width + 2 * applet->border)+(x-1)]+
						firebuf[(y+1)*(image_width + 2 * applet->border)+(x)]+
						firebuf[(y+1)*(image_width + 2 * applet->border)+(x+1)])/4 - applet->decay;
			if (val < 0)
				val=0;
			applet->firebuf[y * (image_width + 2 * applet->border) + x]=val;
		}
	}

	for (y = 0; y < image_height; y++)
	{
		for (x = 0; x < image_width; x++)
		{
			unsigned int col=cpufire_flame[firebuf[y * (image_width + 2 * applet->border) + (x + applet->border)]];
			rgbbuf[3*(y * image_width + x)+2]=   col         & 0xff;
			rgbbuf[3*(y * image_width + x)+1]= (col >> 8)  & 0xff;
			rgbbuf[3*(y * image_width + x)+0]= (col >> 16) & 0xff;
		}
	}

	if (applet->drawing_area && applet->drawing_area->window)
	{
		if (!applet->tooltip_update--)
		{
			gchar *tooltip_text;
			tooltip_text = g_strdup_printf(_("Processor:\n%4.1f%% in use"), 100.0 * total_avg);
			if (applet->tooltips)
				gtk_tooltips_set_tip(applet->tooltips, GTK_WIDGET (applet), tooltip_text, tooltip_text);
			g_free(tooltip_text);
			applet->tooltip_update=20;
		}

		gdk_draw_rgb_image (applet->drawing_area->window, applet->drawing_area->style->fg_gc[GTK_STATE_NORMAL],
				0, 0, image_width, image_height,
				GDK_RGB_DITHER_MAX, applet->rgbbuf, image_width * 3);
	}

	return TRUE;
}

static gboolean
cpufire_applet_fill (CPUFireApplet* applet)
{

	applet->tooltips = gtk_tooltips_new();
	g_object_ref (applet->tooltips);
	gtk_object_sink (GTK_OBJECT (applet->tooltips));
	gtk_tooltips_set_tip(applet->tooltips, GTK_WIDGET (applet), "Starting...", "Starting...");

	applet->cpu= g_malloc0(sizeof(glibtop_cpu)*AVERAGE_NUM_TIMES);

	applet->rgbbuf = NULL;
	applet->firebuf = NULL;
	applet->drawing_area = NULL;

	applet->ref_time_idx = 0;
	applet->time_idx = 0;
	applet->tooltip_update = 0;
	applet->config_change = FALSE;

	applet->flame_sensitivity = 1000;
	applet->update_interval = 50;
	applet->decay = 0;
	applet->border = 1;
	applet->panel_border = 2;
	applet->image_size = 40;

	cpufire_applet_load_properties(applet);
	cpufire_applet_set_size(applet);

	applet->drawing_area = gtk_drawing_area_new ();

	gtk_container_add (GTK_CONTAINER (applet), applet->drawing_area);
	g_signal_connect (GTK_OBJECT (applet),"destroy",
			   GTK_SIGNAL_FUNC (cpufire_applet_destroy),NULL);
	g_signal_connect (G_OBJECT (applet), "change_size",
			  G_CALLBACK (cpufire_applet_change_size_cb), applet);
	g_signal_connect (G_OBJECT (applet), "change_orient",
			  G_CALLBACK (cpufire_applet_change_orient_cb), applet);
 	g_signal_connect (G_OBJECT (applet), "change_background",
			  G_CALLBACK (change_background_cb), applet);
	gtk_signal_connect (GTK_OBJECT (applet->drawing_area), "expose-event", GTK_SIGNAL_FUNC (on_darea_expose), applet);
	applet->timeout_handler_id = g_timeout_add (applet->update_interval, (GSourceFunc) burn_flame, applet);
	gtk_tooltips_enable(applet->tooltips);

	glade_gnome_init ();
	glade_file = gnome_program_locate_file
		(NULL, GNOME_FILE_DOMAIN_DATADIR,
		 "cpufire-applet/cpufire-applet.glade", FALSE, NULL);

	panel_applet_setup_menu_from_file (PANEL_APPLET (applet),
			NULL,
			"GNOME_CPUFireApplet.xml",
			NULL,
			cpufire_menu_verbs,
			applet);

	gtk_widget_show_all (GTK_WIDGET (applet));

	return TRUE;
}

static gboolean
cpufire_applet_factory (CPUFireApplet *applet,
		const gchar          *iid,
		gpointer              data)
{
	gboolean retval = FALSE;

	if (iid && !strncmp (iid, "OAFIID:CPUFire_Applet", strlen("OAFIID:CPUFire_Applet")))
		retval = cpufire_applet_fill (applet);

	return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:CPUFire_Applet_Factory",
		cpufire_applet_get_type (),
		"cpufire",
		"0",
		(PanelAppletFactoryCallback) cpufire_applet_factory,
		NULL);
