/*
 *  methods for encryption/security mechanisms for cryptmount
 *  $Revision: 158 $, $Date: 2007-04-10 07:27:01 +0100 (Tue, 10 Apr 2007) $
 *  Copyright 2005-2007 RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exemption, permission is granted to link cryptmount
    with the OpenSSL project's "OpenSSL" library and distribute
    the linked code without invoking clause 2(b) of the GNU GPL version 2.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with cryptmount; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>

#include <ctype.h>
#if HAVE_DLFCN && USE_MODULES
#  include <dlfcn.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "armour.h"
#include "cryptmount.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif



/*
 *  ==== libgcrypt key-management routines ====
 */

#if HAVE_LIBGCRYPT
#  if !USE_MODULES || defined(AS_MODULE)
#    include <gcrypt.h>

#if defined(TESTING) && defined(AS_MODULE)
cm_testinfo_t *test_ctxtptr;
#endif


/*
 *  keyfile format is:
 *      char magic[7]="cm-gcry";
 *      char version;
 *      uint16{LSB-first} cipher_blocklength, keylength;
 *      char salt[kmgcry_saltlen];
 *      [block][block][block];
 *      (last block ends with uint32 xor-checksum of key
 *      (post-padded with zeros to next 4-byte boundary),
 *      post-padded with zeros to next cipher_blocklength boundary);
 */

const char kmgcry_magstr[]="cm-gcry";
const char kmgcry_version = (char)0;
static const size_t kmgcry_maglen = 7;  /* = strlen(kmgcry_magstr) */
enum {
    kmgcry_saltlen = 12
};


static struct kmgcry_mode {
    char *name;
    unsigned mode; } kmgcry_modes[] = {
    { "ecb",    GCRY_CIPHER_MODE_ECB },
    { "cfb",    GCRY_CIPHER_MODE_CFB },
    { "cbc",    GCRY_CIPHER_MODE_CBC },
    { "ofb",    GCRY_CIPHER_MODE_OFB },
    { "cfb",    GCRY_CIPHER_MODE_CFB },
    { NULL, GCRY_CIPHER_MODE_NONE }
};


static int kmgcry_get_algos(const keyinfo_t *keyinfo,
                    int *cipher, int *ciphermode, int *digest)
    /* get libgcrypt algorithms for encoding key */
{   char *buff=NULL,*pos;
    struct kmgcry_mode *cmd;
    int eflag=ERR_NOERROR;
    const char *algstr="blowfish", *mdstr="cbc", *dgststr="md5";

    if (keyinfo->cipheralg != NULL && keyinfo->cipheralg[0] != '\0') {
        buff = (char*)realloc(buff, strlen(keyinfo->cipheralg) + 1);
        strcpy(buff, keyinfo->cipheralg);
        algstr = buff;
        if ((pos = strchr(buff,'-')) != NULL) {
            *pos = '\0';
            mdstr = pos + 1;
            for (++pos; *pos!='\0'; ++pos) {
                *pos = tolower((int)*pos);
            }
        }
    }
    *cipher = gcry_cipher_map_name(algstr);
    if (*cipher == 0) {
        fprintf(stderr, _("couldn't find libgcrypt cipher \"%s\"\n"), algstr);
        eflag = ERR_BADALGORITHM;
        goto bail_out;
    }

    for (cmd=kmgcry_modes; cmd->name!=NULL; ++cmd) {
        if (strcmp(cmd->name,mdstr) == 0) break;
    }
    *ciphermode = cmd->mode;

    if (keyinfo->digestalg != NULL && keyinfo->digestalg[0] != '\0') {
        buff = (char*)realloc(buff, strlen(keyinfo->digestalg) + 1);
        strcpy(buff, keyinfo->digestalg);
        dgststr = buff;
    }
    *digest = gcry_md_map_name(dgststr);
    if (*digest == 0) {
        fprintf(stderr, _("couldn't find libgcrypt digest \"%s\"\n"), dgststr);
        eflag = ERR_BADALGORITHM;
        goto bail_out;
    }

  bail_out:

    if (buff != NULL) free((void*)buff);

    return eflag;
}


#  ifdef TESTING

static int kmgcry_test_getalgos()
{   keyinfo_t keyinfo;
    int cipher=0, mode=0, digest=0;

    CM_TEST_START("libgcrypt algorithm-identification");

    keyinfo.cipheralg = NULL;
    keyinfo.digestalg = NULL;
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));
    CM_ASSERT_DIFFERENT(0, cipher);
    CM_ASSERT_DIFFERENT(0, mode);
    CM_ASSERT_DIFFERENT(0, digest);

    keyinfo.cipheralg = "";
    keyinfo.digestalg = "";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));

    keyinfo.cipheralg = "twofish";
    keyinfo.digestalg = "sha1";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));
    CM_ASSERT_EQUAL(GCRY_CIPHER_TWOFISH, cipher);
    CM_ASSERT_EQUAL(GCRY_CIPHER_MODE_CBC, mode);
    CM_ASSERT_EQUAL(GCRY_MD_SHA1, digest);

    keyinfo.cipheralg = "CAST5-CFB";
    keyinfo.digestalg = "ripemd160";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));
    CM_ASSERT_EQUAL(GCRY_CIPHER_CAST5, cipher);
    CM_ASSERT_EQUAL(GCRY_CIPHER_MODE_CFB, mode);
    CM_ASSERT_EQUAL(GCRY_MD_RMD160, digest);

    CM_TEST_OK();
}

#  endif    /* TESTING */


static int kmgcry_initcipher(int cipher, int ciphermode, int digest,
            const unsigned char *salt, const char *pass, size_t passlen,
            gcry_cipher_hd_t *hd)
{   gcry_md_hd_t md_hand;
    size_t ckeysz,cblksz,kpos,ivpos,pos,mdlen;
    unsigned char *ckey=NULL,*civ=NULL,*buff;
    int eflag=ERR_BADALGORITHM;

    if (gcry_cipher_open(hd, cipher, ciphermode, 0) != 0) {
        fprintf(stderr, "cannot open libgcrypt cipher[%d,%d]\n",
                cipher, ciphermode);
        goto bail_out;
    }

    (void)gcry_cipher_algo_info(cipher, GCRYCTL_GET_KEYLEN, NULL, &ckeysz);
    ckey = (unsigned char*)sec_realloc(ckey, ckeysz);
    (void)gcry_cipher_algo_info(cipher, GCRYCTL_GET_BLKLEN, NULL, &cblksz);
    civ = (unsigned char*)sec_realloc(civ, cblksz);

    /* generate cipher key & iv by hashing password: */
    kpos = ivpos = 0;
    if (gcry_md_open(&md_hand, digest, 0) != 0) {
        fprintf(stderr, "cannot open libgcrypt digest[%d]\n", digest);
        goto bail_out;
    }
    mdlen = gcry_md_get_algo_dlen(digest);
    do {
        gcry_md_reset(md_hand);

        gcry_md_write(md_hand, (const void*)salt, (size_t)kmgcry_saltlen);
        gcry_md_write(md_hand, (const void*)pass, passlen);
        if (kpos > 0) {
            gcry_md_write(md_hand, (const void*)ckey, kpos); }
        if (ivpos > 0) {
            gcry_md_write(md_hand, (const void*)civ, ivpos); }

        buff = gcry_md_read(md_hand, digest);
        pos = 0;
        while (kpos < ckeysz && pos < mdlen) {
            ckey[kpos++] = buff[pos++]; }
        while (ivpos < cblksz && pos < mdlen) {
            civ[ivpos++] = buff[pos++]; }
    } while (kpos < ckeysz || ivpos < cblksz);
    gcry_md_close(md_hand);

    /* setup cipher initial state: */
    if (gcry_cipher_setkey(*hd, (void*)ckey, ckeysz) != 0
      || gcry_cipher_setiv(*hd, (void*)civ, cblksz) != 0) {
        fprintf(stderr, "failed to setup libgcrypt cipher iv[%d,%d]\n",
                (int)ckeysz, (int)cblksz);
        goto bail_out;
    }
    sec_free(ckey);
    sec_free(civ);

    eflag = ERR_NOERROR;

  bail_out:

    return eflag;
}


static int kmgcry_init_algs()
{
    /* nothing needed */
    return 0;
}


static int kmgcry_free_algs()
{
    /* nothing needed */
    return 0;
}


void kmgcry_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("md5");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("blowfish");
    }
}


static int kmgcry_is_compat(const keyinfo_t *keyinfo, FILE *fp_key)
{   char buff[32];

    if (keyinfo->format != NULL) {
        return (strcmp(keyinfo->format, "libgcrypt") == 0);
    } else {
        if (fp_key != NULL) {
            /* check header of existing key-file: */
            fread((void*)buff, kmgcry_maglen, (size_t)1, fp_key);
            return (strncmp(buff, kmgcry_magstr, kmgcry_maglen) == 0);
        }
    }

    return 0;
}


static int kmgcry_needs_pw(const keyinfo_t *keyinf)
{
    return 1;
}


static int kmgcry_get_key(const char *ident, const keyinfo_t *keyinfo,
            unsigned char **key, int *keylen, FILE *fp_key)
    /* extract key from libgcrypt-encrypted file */
{   gcry_cipher_hd_t chd;
    char *passwd=NULL;
    unsigned char hbuff[kmgcry_maglen+4], salt[kmgcry_saltlen],
            *buff=NULL, *bptr;
    size_t cblksz;
    uint32_t chksum, chksum0;
    int cnt, cipher, ciphermode, digest, eflag=ERR_NOERROR;

    *key = NULL; *keylen = 0;

    eflag = kmgcry_get_algos(keyinfo, &cipher, &ciphermode, &digest);
    if (eflag != ERR_NOERROR) goto bail_out;
    gcry_cipher_algo_info(cipher, GCRYCTL_GET_BLKLEN, NULL, &cblksz);

#ifdef TESTING
    passwd = (char*)malloc((size_t)1024);
    strncpy(passwd, (test_ctxtptr->argpassword[0] != NULL ? test_ctxtptr->argpassword[0] : ""),
            1024);
#else
    if (km_get_passwd(ident, &passwd, 0, 0) != ERR_NOERROR) goto bail_out;
#endif

    /* read key header: */
    fread((void*)hbuff, (size_t)1, kmgcry_maglen, fp_key);
    if (strncmp((const char*)hbuff, kmgcry_magstr, kmgcry_maglen) != 0) {
        fprintf(stderr, _("bad keyfile format (libgcrypt)\n"));
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    fread((void*)hbuff, (size_t)1, (size_t)1, fp_key);
    if (hbuff[0] != '\0') {
        fprintf(stderr, "bad keyfile version [%d]\n", (int)buff[0]);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    fread((void*)hbuff, (size_t)4, (size_t)1, fp_key);
    if (hbuff[0] != (cblksz & 0xff) || hbuff[1] != ((cblksz & 0xff00) >> 8)) {
        fprintf(stderr, "mismatched cipher block size\n");
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    *keylen = ((unsigned)hbuff[2]) | (((unsigned)hbuff[3]) << 8);

    /* read salt from keyfile: */
    fread((void*)salt, (size_t)1, sizeof(salt), fp_key);

    /* read encrypted key from keyfile: */
    eflag = kmgcry_initcipher(cipher, ciphermode, digest,
                    salt, passwd, strlen(passwd), &chd);
    if (eflag != ERR_NOERROR) goto bail_out;
    cnt = km_aug_keysz((unsigned)*keylen, (unsigned)cblksz) / cblksz;
    buff = sec_realloc(buff, cnt * cblksz);
    bptr = buff;
    while (cnt--) {
        fread((void*)bptr, cblksz, (size_t)1, fp_key);
        gcry_cipher_decrypt(chd, (void*)bptr, cblksz, NULL, 0);
        bptr += cblksz;
    }
    gcry_cipher_close(chd);

    /* verify checksum: */
    if (!km_aug_verify(buff, (unsigned)*keylen, &chksum0, &chksum)) {
        fprintf(stderr, "checksum mismatch in keyfile (gcry, %x != %x)\n",
            (unsigned)chksum, (unsigned)chksum0);
        eflag = ERR_BADFILE;
    }

    if (keyinfo->maxlen > 0 && *keylen > keyinfo->maxlen) {
        *keylen = keyinfo->maxlen;
    }
    *key = (unsigned char*)sec_realloc((void*)*key, (size_t)*keylen);
    memcpy(*key, buff, (size_t)*keylen);

    if (ferror(fp_key) != 0) {
        fprintf(stderr, _("key-extraction failed for \"%s\"\n"),
                keyinfo->filename);
        eflag = ERR_BADDECRYPT;
    }

  bail_out:

    if (buff != NULL) sec_free(buff);

    return eflag;
}


static int kmgcry_put_key(const char *ident, const keyinfo_t *keyinfo,
            const unsigned char *key, const int keylen, FILE *fp_key)
    /* store key in libgcrypt-encrypted file */
{   gcry_cipher_hd_t chd;
    char *passwd=NULL;
    unsigned char hbuff[4], salt[kmgcry_saltlen], *buff=NULL, *bptr;
    size_t buffsz, cblksz, wrcnt;
    int cnt, cipher, ciphermode, digest, eflag=ERR_NOERROR;

    eflag = kmgcry_get_algos(keyinfo, &cipher, &ciphermode, &digest);
    if (eflag != ERR_NOERROR) goto bail_out;
    gcry_cipher_algo_info(cipher, GCRYCTL_GET_BLKLEN, NULL, &cblksz);

#ifdef TESTING
    passwd = sec_realloc((void*)passwd, (size_t)1024);
    strncpy(passwd, (test_ctxtptr->argpassword[1] != NULL ? test_ctxtptr->argpassword[1] : ""),
            1024);
#else
    if (km_get_passwd(ident, &passwd, 1, 1) != ERR_NOERROR) goto bail_out;
#endif

    /* write key header: */
    wrcnt = 0;
    wrcnt += fwrite((const void*)kmgcry_magstr, kmgcry_maglen, (size_t)1, fp_key);
    wrcnt += fwrite((const void*)&kmgcry_version, (size_t)1, (size_t)1, fp_key);
    hbuff[0] = (cblksz & 0xff); hbuff[1] = (cblksz & 0xff00) >> 8;
    hbuff[2] = (keylen & 0xff); hbuff[3] = (keylen & 0xff00) >> 8;
    wrcnt += fwrite((const void*)hbuff, (size_t)4, (size_t)1, fp_key);

    /* generate salt & record in keyfile: */
    get_randkey(salt, sizeof(salt));
    wrcnt += fwrite((const void*)salt, sizeof(salt), (size_t)1, fp_key);

    /* augment key with simple checksum: */
    buff = km_aug_key(key, (unsigned)keylen, (unsigned)cblksz, &buffsz);

    /* write encrypted key into keyfile: */
    eflag = kmgcry_initcipher(cipher, ciphermode, digest,
                    salt, passwd, strlen(passwd), &chd);
    if (eflag != ERR_NOERROR) goto bail_out;
    cnt = buffsz / cblksz;
    bptr = buff;
    while (cnt--) {
        gcry_cipher_encrypt(chd, (void*)bptr, cblksz, NULL, 0);
        wrcnt += fwrite((const void*)bptr, cblksz, (size_t)1, fp_key);
        bptr += cblksz;
    }
    gcry_cipher_close(chd);

    if (wrcnt != (4 + buffsz / cblksz)) {
        fprintf(stderr, _("failed to create new key file\n"));
	fprintf(stderr, "wrcnt=%d\n", (int)wrcnt);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

  bail_out:
    if (buff != NULL) sec_free(buff);
    if (passwd != NULL) sec_free(passwd);

    return eflag;
}


static void *kmgcry_md_prepare(void)
{   gcry_md_hd_t *handle;

    handle = (gcry_md_hd_t*)malloc(sizeof(gcry_md_hd_t));
    (void)gcry_md_open(handle, GCRY_MD_SHA1, 0);

    return (void*)handle;
}


static void kmgcry_md_block(void *state, unsigned char *buff, size_t len)
{   gcry_md_hd_t *handle=(gcry_md_hd_t*)state;

    gcry_md_write(*handle, (const void*)buff, len);
}


static void kmgcry_md_final(void *state, unsigned char **mdval, size_t *mdlen)
{   gcry_md_hd_t *handle=(gcry_md_hd_t*)state;
    unsigned char *buff;
    int algo;

    gcry_md_final(*handle);
    algo = gcry_md_get_algo(*handle);
    *mdlen = gcry_md_get_algo_dlen(algo);

    buff = gcry_md_read(*handle, algo);
    *mdval = (unsigned char*)malloc(*mdlen);
    memcpy((void*)*mdval, (const void*)buff, *mdlen);
}


static void kmgcry_md_release(void *state)
{   gcry_md_hd_t *handle=(gcry_md_hd_t*)state;

    gcry_md_close(*handle);
    free((void*)handle);
}

#  ifdef TESTING

static int kmgcry_test_hash()
{   void *mdcontext;
    unsigned char *mdval=NULL;
    size_t mdlen, i;
    unsigned q;
    const char *str="noisy\n";
    const char *hash="7c1c9261fa774475ec1c0d887eaf00c19b0eb218";

    CM_TEST_START("libgcrypt hashing");

    mdcontext = kmgcry_md_prepare();
    CM_ASSERT_DIFFERENT(NULL, mdcontext);
    kmgcry_md_block(mdcontext, (unsigned char*)str, strlen(str));
    kmgcry_md_final(mdcontext, &mdval, &mdlen);
    CM_ASSERT_DIFFERENT(NULL, mdval);
    CM_ASSERT_EQUAL(strlen(hash)/2, mdlen);
    for (i=0; i<mdlen; ++i) {
        sscanf(hash+2*i, "%2x", &q);
        CM_ASSERT_EQUAL(q, (unsigned)mdval[i]);
    }

    kmgcry_md_release(mdcontext);

    CM_TEST_OK();
}

static void kmgcry_testctxt(cm_testinfo_t *context)
{
    test_ctxtptr = context;
}

static int kmgcry_runtests()
{
    kmgcry_test_hash();
    kmgcry_test_getalgos();

    return 0;
}

#  endif    /* TESTING */


keymanager_t keymgr_gcry = {
    "libgcrypt", 0,   kmgcry_init_algs, kmgcry_free_algs,
                      kmgcry_mk_default, kmgcry_is_compat, kmgcry_needs_pw,
                      kmgcry_get_key, kmgcry_put_key,
                      kmgcry_md_prepare, kmgcry_md_block,
                      kmgcry_md_final, kmgcry_md_release,
    NULL
#ifdef TESTING
    , kmgcry_testctxt, kmgcry_runtests
#endif
};

#  endif    /* !USE_MODULES || defined(AS_MODULE) */
#endif  /* HAVE_LIBGCRYPT */


#ifndef AS_MODULE

#if defined(TESTING)
#  define MOD_PATH "./cm-gcry.so"
#else
#  define MOD_PATH CM_MODULE_DIR "/cm-gcry.so"
#endif

keymanager_t *kmgcry_gethandle()
{
#if HAVE_LIBGCRYPT
#  if USE_MODULES
    KM_GETHANDLE(MOD_PATH, "keymgr_gcry");
#  else
    return &keymgr_gcry;
#  endif
#else
    return NULL;
#endif
}

#endif  /* !AS_MODULE */

/*
 *  (C)Copyright 2005-2007, RW Penney
 */
