/*
 *  miscellaneous utilitie functions for cryptmount
 *  $Revision: 134 $, $Date: 2007-02-19 17:18:07 +0000 (Mon, 19 Feb 2007) $
 *  Copyright 2005-2007, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exemption, permission is granted to link cryptmount
    with the OpenSSL project's "OpenSSL" library and distribute
    the linked code without invoking clause 2(b) of the GNU GPL version 2.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with cryptmount; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>

#include <ctype.h>
#include <linux/major.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/times.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "cryptmount.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif



int cm_path(char **buff, const char *file)
    /* create full pathname of config file */
{   size_t pfxlen, sfxlen;
    const char *pfx=NULL;

    if (buff == NULL || file == NULL) return 0;

#ifdef TESTING
    pfx = (test_ctxtptr->argconfigdir != NULL ? test_ctxtptr->argconfigdir : "/nowhere");
#else
    pfx = CM_CONFIG_DIR;
#endif

    pfxlen = strlen(pfx);
    sfxlen = strlen(file);
    *buff = (char*)realloc((void*)(*buff), (pfxlen + sfxlen + 2));

    snprintf(*buff, (pfxlen + sfxlen + 2), "%s/%s", pfx, file);

    return (int)(pfxlen + sfxlen + 1);
}


char *cm_strdup(const char *orig)
    /* make duplicate of existing string, reserving memory */
{   char *cpy;

    if (orig == NULL) return NULL;
    cpy = (char*)malloc((size_t)(strlen(orig) + 1));
    return strcpy(cpy, orig);
}


void *sec_realloc(void *ptr, size_t size)
    /* slightly more secure version of realloc() */
{   size_t cnt, *memarr;

    cnt = (size + 2 * sizeof(size_t) - 1) / sizeof(size_t);
    memarr = (size_t*)calloc(cnt, sizeof(size_t));

    if (memarr == NULL) {
        fprintf(stderr, _("unable to allocated memory\n"));
        abort();
        return NULL;
    }

    /* prepend usable memory chunk with record of size of chunk: */
    memarr[0] = (cnt - 1) * sizeof(size_t);

    if (ptr != NULL) {
        size_t oldsz;

        /* copy (usable) part of old memory block into new: */
        oldsz = *(((size_t*)ptr) - 1);
        if (oldsz > size) oldsz = size;
        memcpy((void*)(memarr + 1), (const void*)ptr, size);

        /* dispose of old memory block: */
        sec_free(ptr);
    }

    return (void*)(memarr + 1);
}


void mem_cleanse(unsigned char *addr, size_t sz)
    /* overwrite memory with (weak) pseudo-random numbers */
{   size_t i;
    static unsigned long salt=0x917c;

    salt ^= (unsigned long)addr;

    for (i=0; i<sz; ++i) {
        addr[i] = (i % 21) ^ (salt % 221);
        salt += 4;
    }
}


void sec_free(void *ptr)
    /* slightly more secure version of free() */
{   size_t *memarr, sz;

    if (ptr == NULL) return;

    memarr = ((size_t*)ptr) - 1;
    sz = memarr[0];

    mem_cleanse((unsigned char*)(memarr + 1), sz);

    free((void*)memarr);
}


int get_randkey(unsigned char *buff, size_t len)
    /* generate random string of bytes */
{   struct rnddev {
        const char *name;
        unsigned devmaj, devmin;
        FILE *fp; };
    struct rnddev devs[] = {
#if 0 || !defined(TESTING)
        { "/dev/hwrng",     MISC_MAJOR, 183,    NULL },
        { "/dev/random",    MEM_MAJOR, 8,       NULL },
#endif
        { "/dev/urandom",   MEM_MAJOR, 9,       NULL },
        { NULL, 0, 0, NULL } };
    cm_sha1_ctxt_t *mdcontext=NULL;
    unsigned char *mdval,*devbuff=NULL;
    size_t pos, sz, blklen, mdlen;
    int i,cnt,eflag=ERR_NOERROR;
    struct stat sbuff;
    pid_t pid;
    struct tms tbuff;
    clock_t clk;
    static unsigned seed=17;


    /* try to find good-quality random-number devices: */
    for (i=0,cnt=0; devs[i].name!=NULL; ++i) {
        if (stat(devs[i].name, &sbuff) != 0) continue;
        if ((unsigned)major(sbuff.st_rdev) != devs[i].devmaj
          || (unsigned)minor(sbuff.st_rdev) != devs[i].devmin) continue;
        if ((devs[i].fp = fopen(devs[i].name,"rb")) != NULL) ++cnt;
    }
    if (cnt == 0) {
        fprintf(stderr, _("no random-number devices found\n"));
        eflag = WRN_LOWENTROPY;
    }
    blklen = (len > CM_SHA1_SIZE ? (size_t)CM_SHA1_SIZE : len);
    devbuff = (unsigned char*)sec_realloc(NULL, (size_t)blklen);


    pid = getpid();

    /*  combine multiple sources of entropy
        (should still be effective if subset aren't viable): */
    pos = 0;
    while (pos < len) {
        mdcontext = cm_sha1_init();

        /* fold-in entropy from random-number devices: */
        for (i=0; devs[i].name!=NULL; ++i) {
            if (devs[i].fp == NULL) continue;
            (void)fread((void*)devbuff, (size_t)1, (size_t)blklen, devs[i].fp);
            cm_sha1_block(mdcontext, devbuff, blklen);
        }

        /* fold-in existing bytes from key: */
        if (pos > 0) {
            cm_sha1_block(mdcontext, buff, pos);
        }

        /* fold-in some (weak) sources of entropy: */
        cm_sha1_block(mdcontext, (unsigned char*)&pid, sizeof(pid));
        clk = times(&tbuff);
        cm_sha1_block(mdcontext, (unsigned char*)&clk, sizeof(clk));
        cm_sha1_block(mdcontext, (unsigned char*)&seed, sizeof(seed));
        cm_sha1_block(mdcontext, (unsigned char*)&tbuff, sizeof(tbuff));

        cm_sha1_final(mdcontext, &mdval, &mdlen);

        sz = ((pos + mdlen) > len ? (len - pos) : mdlen);
        memcpy((void*)(buff + pos),(const void*)mdval, (size_t)sz);

        pos += sz;
        seed = seed * 20 + 1;

        cm_sha1_free(mdcontext);
        sec_free(mdval);
    }

    sec_free((void*)devbuff);
    for (i=0; devs[i].name!=NULL; ++i) {
        if (devs[i].fp != NULL) fclose(devs[i].fp);
    }

    return eflag;
}


int km_get_passwd(const char *ident, char **passwd, int isnew, int verify)
    /* read password from terminal, possibly asking for confirmation */
{   enum { BUFFSZ=2048 };
    char *tmppass,buff[BUFFSZ];
    size_t plen;
    int eflag=ERR_NOERROR;

    snprintf(buff, sizeof(buff),
            (isnew ? _("enter new password for target \"%s\": ")
                : _("enter password for target \"%s\": ")),
            ident);

    tmppass = getpass(buff);
    plen = strlen(tmppass);
    *passwd = (char*)sec_realloc((void*)*passwd, (plen + 1));
    strcpy(*passwd, tmppass);
    mem_cleanse((unsigned char*)tmppass, plen + 1);

    if (verify) {
        snprintf(buff, sizeof(buff), _("confirm password: "));
        tmppass = getpass(buff);
        plen = strlen(tmppass);
        if (strcmp(*passwd, tmppass) != 0) {
            fprintf(stderr, _("password mismatch\n"));
            sec_free(*passwd);
            *passwd = NULL;
            eflag = ERR_BADPASSWD;
        }
        mem_cleanse((unsigned char*)tmppass, plen);
    }

    return eflag;
}


unsigned km_aug_keysz(unsigned keylen, unsigned blksz)
    /* calculate size of augmented cipher-key after appending checksum etc */
{
    return blksz * ((keylen + 2 * sizeof(uint32_t) + blksz - 1) / blksz);
}


unsigned char *km_aug_key(const unsigned char *key, unsigned keylen,
                    unsigned blocksz, size_t *buffsz)
    /* augment cipher key with checksum prior to encryption & storage */
{   unsigned char *buff=NULL;
    uint32_t chksum, *kptr=NULL;
    size_t idx, cnt;

    *buffsz = km_aug_keysz(keylen, blocksz);
    buff = (unsigned char*)sec_realloc(buff, *buffsz);

    /* copy key into zero-padded buffer: */
    memset(buff, 0, (size_t)*buffsz);
    memcpy(buff, key, (size_t)keylen);

    /* compute crude EOR checksum (invariant to byte-ordering): */
    cnt = (keylen + sizeof(chksum) - 1) / sizeof(chksum);
    chksum = 0;
    kptr = (uint32_t*)buff;
    for (idx=0; idx<cnt; ++idx) {
        chksum ^= *kptr;
        ++kptr;
    }

    /* install checksum at next 4-byte boundary & pad with noise: */
    *kptr = chksum;
    idx = (idx + 1) * sizeof(chksum);
    if (idx < *buffsz) {
        get_randkey((buff + idx), (*buffsz - idx));
    }

    return buff;
}


int km_aug_verify(const unsigned char *buff, unsigned keylen,
                uint32_t *expected, uint32_t *actual)
    /* check augmented cipher key against simple checksum */
{   unsigned cnt;
    uint32_t *kptr;

    cnt = (keylen + sizeof(*expected) - 1) / sizeof(*expected);
    *actual = 0;
    kptr = (uint32_t*)buff;
    while (cnt--) {
        *actual ^= *kptr;
        ++kptr;
    }

    *expected = *kptr;
    return (*expected == *actual);
}


/*
 *  sha1 message-digest algorithm
 *  - based on "Cryptography - theory & practice" (2nd Ed.), DR Stinson, 2002
 *  (rather inefficient, but good enough for short messages)
 */

static const uint32_t
    SHA1_H0 = 0x67452301U,
    SHA1_H1 = 0xEFCDAB89U,
    SHA1_H2 = 0x98BADCFEU,
    SHA1_H3 = 0x10325476U,
    SHA1_H4 = 0xC3D2E1F0U,
    SHA1_K0 = 0x5A827999U,
    SHA1_K1 = 0x6ED9EBA1U,
    SHA1_K2 = 0x8F1BBCDCU,
    SHA1_K3 = 0xCA62C1D6U;


cm_sha1_ctxt_t *cm_sha1_init(void)
{   cm_sha1_ctxt_t *ctxt;
    unsigned idx;

    ctxt = (cm_sha1_ctxt_t*)sec_realloc(NULL, sizeof(cm_sha1_ctxt_t));

    ctxt->msglen = 0;
    ctxt->buffpos = 0;
    ctxt->H[0] = SHA1_H0;
    ctxt->H[1] = SHA1_H1;
    ctxt->H[2] = SHA1_H2;
    ctxt->H[3] = SHA1_H3;
    ctxt->H[4] = SHA1_H4;
    for (idx=0; idx<16; ++idx) ctxt->buff[idx] = 0;

    return ctxt;
}


void cm_sha1_block(cm_sha1_ctxt_t *ctxt, const unsigned char *buff, size_t len)
{   uint32_t W[80], A, B, C, D, E, q;
    unsigned idx, round;

    while (len > 0) {
        /* accumulate bytes into buffer (respecting endianess): */
        idx = ctxt->buffpos >> 2;
        round = 3 - (ctxt->buffpos & 0x03);
        ctxt->buff[idx] |= ((uint32_t)*buff) << (round * 8);
        ctxt->msglen += 8;
        ++ctxt->buffpos;
        ++buff;
        --len;

        if (ctxt->buffpos >= 64) {
            /* whole 512-bit string is ready - apply SHA1 update to block: */
            for (idx=0; idx<16; ++idx) W[idx] = ctxt->buff[idx];
            for (idx=16; idx<80; ++idx) {
                q = W[idx-3] ^ W[idx-8] ^ W[idx-14] ^ W[idx-16];
                W[idx] = ((q & 0x7fffffff)) << 1 | ((q & 0x80000000) >> 31);
            }

            A = ctxt->H[0];
            B = ctxt->H[1];
            C = ctxt->H[2];
            D = ctxt->H[3];
            E = ctxt->H[4];

            for (round=0; round<80; ++round) {
                q = (((A & 0x07ffffff) << 5) | ((A & 0xf8000000) >> 27))
                            + E + W[round];
                switch (round / 20) {
                case 0:
                    q += ((B & C) | ((~B) & D)) + SHA1_K0;
                    break;
                case 1:
                    q += (B ^ C ^ D) + SHA1_K1;
                    break;
                case 2:
                    q += ((B & C) | (B & D) | (C & D)) + SHA1_K2;
                    break;
                case 3:
                    q += (B ^ C ^ D) + SHA1_K3;
                    break;
                }
                E = D;
                D = C;
                C = ((B & 0xfffffffc) >> 2) | ((B & 0x03) << 30);
                B = A;
                A = q;
            }

            ctxt->H[0] += A;
            ctxt->H[1] += B;
            ctxt->H[2] += C;
            ctxt->H[3] += D;
            ctxt->H[4] += E;
            ctxt->buffpos = 0;
            for (idx=0; idx<16; ++idx) ctxt->buff[idx] = 0;
        }
    }
}


void cm_sha1_final(cm_sha1_ctxt_t *ctxt, unsigned char **mdval, size_t *mdlen)
{   unsigned char *cptr, buff[64], mrk=0x80;
    unsigned idx, padlen;
    uint32_t msglen;

    /* add closing sequence onto message string: */
    msglen = ctxt->msglen;
    for (idx=0; idx<64; ++idx) buff[idx] = 0;
    padlen = (ctxt->buffpos < 56 ? 55 - ctxt->buffpos : 119 - ctxt->buffpos);
    cm_sha1_block(ctxt, &mrk, (size_t)1);
    if (padlen > 0) cm_sha1_block(ctxt, buff, (size_t)padlen);
    buff[4] = (msglen & 0xff000000) >> 24;
    buff[5] = (msglen & 0xff0000) >> 16;
    buff[6] = (msglen & 0xff00) >> 8;
    buff[7] = msglen & 0xff;
    cm_sha1_block(ctxt, buff, (size_t)8);

    /* transcribe internal state into array of bytes: */
    *mdval = (unsigned char*)sec_realloc(NULL, (size_t)CM_SHA1_SIZE);
    *mdlen = CM_SHA1_SIZE;
    cptr = *mdval;
    for (idx=0; idx<5; ++idx) {
        cptr[0] = (ctxt->H[idx] & 0xff000000) >> 24;
        cptr[1] = (ctxt->H[idx] & 0xff0000) >> 16;
        cptr[2] = (ctxt->H[idx] & 0xff00) >> 8;
        cptr[3] = (ctxt->H[idx] & 0xff);
        cptr += 4;
    }
}


void cm_sha1_free(cm_sha1_ctxt_t *ctxt)
{
    sec_free((void*)ctxt);
}


#ifdef TESTING

int ut_test_sha1()
    /* check internal SHA1 hashing algorithm against known test-vectors */
{   cm_sha1_ctxt_t *ctxt;
    unsigned char *mdval;
    unsigned b, q, idx;
    size_t mdlen;
    struct {
        char *input, *hash; } cases[] = {
    { "",                       "da39a3ee5e6b4b0d3255bfef95601890afd80709" },
    { "a",                      "86f7e437faa5a7fce15d1ddcb9eaeaea377667b8" },
    { "alpha",                  "be76331b95dfc399cd776d2fc68021e0db03cc4f" },
    { "alphabetti spaghetti",   "c5fe3361dfdf6f17706cbe3ab0cc6419d057c329" },
    { "Cryptography - theory and practice, by Douglas R Stinson, Chapman & Hall/CRC",  "2e0e07901c8460bc57b0097a66c7086ed5e97808" },
    { NULL, NULL } };

    CM_TEST_START("internal SHA1");
    CM_ASSERT_EQUAL(CM_SHA1_SIZE, 5 * sizeof(uint32_t));

    idx = 0;
    while (cases[idx].input != NULL) {
        ctxt = cm_sha1_init();
        cm_sha1_block(ctxt, (const unsigned char*)cases[idx].input,
                            strlen(cases[idx].input));
        cm_sha1_final(ctxt, &mdval, &mdlen);
        cm_sha1_free(ctxt);

        for (b=0; b<CM_SHA1_SIZE; ++b) {
            sscanf((cases[idx].hash + 2*b), "%02x", &q);
            CM_ASSERT_EQUAL(q, (unsigned)mdval[b]);
        }

        sec_free((void*)mdval);
        ++idx;
    }

    CM_TEST_OK();
}

#endif  /* TESTING */

/*
 *  utils.c
 *  (C)Copyright 2005-2007, RW Penney
 */
