/*
 *  Filesystem-related utilities for cryptmount
 *  $Revision: 258 $, $Date: 2009-05-04 07:07:04 +0100 (Mon, 04 May 2009) $
 *  (C)Copyright 2005-2009, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <ctype.h>
#include <fcntl.h>
#include <mntent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "cryptmount.h"
#include "delegates.h"
#include "dmutils.h"
#include "fsutils.h"
#if WITH_CSWAP
#  include <sys/swap.h>
#endif


#define ETCMTAB     "/etc/mtab"
#define ETCMTABTMP  "/etc/mtab.cm"


enum {
    F_MATCHUID = 0x01, F_CLOSE1 = 0x02 };
static int run_sucommand(const char *path, const char *argv[],
                        unsigned switches);
int do_addmntent(const tgtdefn_t *tgt);
int do_rmvmntent(const tgtdefn_t *tgt);


#if ERSATZ_MOUNT

int fs_mount(const char *mntdev, const tgtdefn_t *tgt)
    /* Fallback version of mount(8) using mount(2) */
{   unsigned long mflags;
    int eflag=ERR_NOERROR;
    char *errstr=NULL;

    if (parse_fsoptions(tgt->fsoptions, &mflags) != 0) {
        return ERR_BADMOUNT;
    }

    errstr = (char*)malloc((size_t)(strlen(mntdev) + strlen(tgt->dir) + 64));
    sprintf(errstr, "mounting \"%s\" on \"%s\" failed", mntdev, tgt->dir);

    if (mount(mntdev, tgt->dir, tgt->fstype, mflags, NULL) == 0) {
        (void)do_addmntent(tgt);
    } else {
        perror(errstr);
        eflag = ERR_BADMOUNT;
    }

    free((void*)errstr);

    return eflag;
}

#else   /* !ERSATZ_MOUNT */

int fs_mount(const char *dev, const tgtdefn_t *tgt)
    /* Delegate filesystem mounting to mount(8) */
{   int idx, stat, eflag=ERR_NOERROR;
    const char *argv[16];

    /* Construct argument list for mount -t ... -o ... <dev> <dir> */
    idx = 0;
    argv[idx++] = "[cryptmount-mount]";
    if (tgt->fstype != NULL) {
        argv[idx++] = "-t"; argv[idx++] = tgt->fstype;
    }
    if (tgt->fsoptions != NULL) {
        argv[idx++] = "-o"; argv[idx++] = tgt->fsoptions;
    }
    argv[idx++] = dev; argv[idx++] = tgt->dir;
    argv[idx++] = NULL;

    stat = run_sucommand(DLGT_MOUNT, argv, F_MATCHUID);
    eflag = (stat != 0 ? ERR_BADMOUNT: ERR_NOERROR);

    return eflag;
}

#endif  /* ERSATZ_MOUNT */


#if ERSATZ_UMOUNT

int fs_unmount(const tgtdefn_t *tgt)
    /* Fallback version of umount(8) using umount(2) */
{   int eflag=ERR_NOERROR;
    char *errstr=NULL;

    errstr = (char*)malloc((size_t)(strlen(tgt->dir) + 64));
    sprintf(errstr, "unmounting \"%s\" failed", tgt->dir);

    if (umount(tgt->dir) == 0) {
        (void)do_rmvmntent(tgt);
    } else {
        perror(errstr);
        eflag = ERR_BADMOUNT;
    }

    free((void*)errstr);

    return eflag;
}

#else   /* !ERSATZ_UMOUNT */

int fs_unmount(const tgtdefn_t *tgt)
    /* Delegate filesystem mounting to umount(8) */
{   int idx, stat, eflag=ERR_NOERROR;
    const char *argv[16];

    /* Construct argument list for umount -t ... <dir> */
    idx = 0;
    argv[idx++] = "[cryptmount-umount]";
#ifdef PARANOID
    if (tgt->fstype != NULL) {
        argv[idx++] = "-t"; argv[idx++] = tgt->fstype;
    }
    if (tgt->fsoptions != NULL) {
        argv[idx++] = "-O"; argv[idx++] = tgt->fsoptions;
    }
#endif
    argv[idx++] = tgt->dir;
    argv[idx++] = NULL;

    stat = run_sucommand(DLGT_UMOUNT, argv, F_MATCHUID);
    eflag = (stat != 0 ? ERR_BADMOUNT: ERR_NOERROR);

    return eflag;
}

#endif  /* ERSATZ_UMOUNT */


#if WITH_CSWAP

int fs_swapon(const char *mntdev, const tgtdefn_t *tgt)
{   int idx, stat, eflag=ERR_NOERROR;
    const char *argv[8];
    int prio, rawprio=0x100;

    if (strcmp(tgt->fstype,"swap") != 0) {
        fprintf(stderr, _("Unsuitable filesystem type \"%s\" for swapping\n"),
                tgt->fstype);
        eflag = ERR_BADSWAP;
        goto bail_out;
    }

    if ((tgt->flags & FLG_MKSWAP) != 0) {
        /* construct argument list for mkswap <dev> */
        idx = 0;
        argv[idx++] = "[cryptmount-mkswap]";
        argv[idx++] = mntdev;
        argv[idx++] = NULL;

        stat = run_sucommand(DLGT_MKSWAP, argv, F_CLOSE1);
        eflag = (stat != 0 ? ERR_BADSWAP : ERR_NOERROR);
        if (eflag != ERR_NOERROR) goto bail_out;
    }

    if (tgt->fsoptions != NULL
      && sscanf(tgt->fsoptions, "pri=%i", &prio) == 1) {
        rawprio = prio;
    }

    prio = ( SWAP_FLAG_PREFER |
            ((rawprio << SWAP_FLAG_PRIO_SHIFT) & SWAP_FLAG_PRIO_MASK) );
    if (swapon(mntdev, prio) != 0) {
        eflag = ERR_BADSWAP;
        goto bail_out;
    }

  bail_out:

    return eflag;
}


int fs_swapoff(const tgtdefn_t *tgt)
{   char *mntdev=NULL;
    int eflag=ERR_NOERROR;

    if (strcmp(tgt->fstype,"swap") != 0) {
        eflag = ERR_BADSWAP;
        goto bail_out;
    }

    devmap_path(&mntdev, tgt->ident);

    if (swapoff(mntdev) != 0) {
        eflag = ERR_BADSWAP;
        goto bail_out;
    }

  bail_out:

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}

#endif  /* WITH_CSWAP */


int parse_fsoptions(const char *buff, unsigned long *mflags)
    /* Convert string of mount-options into binary flags */
{   struct fsopt_t {
        const char *str;
        unsigned long mask; };
    struct fsopt_t fsopts[] = {
        { "defaults",   0 },
        { "noatime",    MS_NOATIME },
        { "nodev",      MS_NODEV },
        { "noexec",     MS_NOEXEC },
        { "nosuid",     MS_NOSUID },
        { "ro",         MS_RDONLY },
        { "sync",       MS_SYNCHRONOUS },
        { NULL, 0 } };
    unsigned idx,len;

    *mflags = 0;
    if (buff == NULL) return 0;

    for (;;) {
        for (len=0; buff[len]!='\0' && buff[len]!=','; ++len);
        for (idx=0; fsopts[idx].str!=NULL; ++idx) {
            if (strncmp(buff, fsopts[idx].str, (size_t)len) == 0) {
                *mflags |= fsopts[idx].mask;
                break;
            }
        }
        if (fsopts[idx].str == NULL) {
            fprintf(stderr, "bad option \"%s\"\n", buff);
            return 1;
        }

        if (buff[len] == '\0') break;
        buff += len + 1;
    }

    return 0;
}


int do_addmntent(const tgtdefn_t *tgt)
    /* Add entry into /etc/mtab for newly-mounted filing system */
{   char *mntdev=NULL;
    struct mntent mntinfo;
    FILE *fp;
    int eflag=ERR_NOERROR;

    devmap_path(&mntdev, tgt->ident);

    mntinfo.mnt_fsname = mntdev;
    mntinfo.mnt_dir = tgt->dir;
    mntinfo.mnt_type = tgt->fstype;
    mntinfo.mnt_opts = "none";
    mntinfo.mnt_freq = 0;
    mntinfo.mnt_passno = 0;

    fp = setmntent(ETCMTAB, "a");
    if (fp == NULL) {
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    (void)addmntent(fp, &mntinfo);
    endmntent(fp);

  bail_out:

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


int do_rmvmntent(const tgtdefn_t *tgt)
    /* Remove entry from /etc/mtab after unmounting filing system */
{   char *mntdev=NULL;
    struct mntent *mntinfo;
    FILE *fp_in=NULL,*fp_out=NULL;
    struct stat sbuff;
    int i,eflag=ERR_NOERROR,found=0;

    /* FIXME - add lots more checks on integrity: */

    devmap_path(&mntdev, tgt->ident);

    /* Open old /etc/mtab & create temporary replacement: */
    fp_in = setmntent(ETCMTAB, "r");
    fp_out = setmntent(ETCMTABTMP, "w");
    if (fp_in == NULL || fp_out == NULL) {
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    i = stat(ETCMTAB, &sbuff);
    if (i != 0 || !S_ISREG(sbuff.st_mode)) {
        fprintf(stderr, "%s is not a valid file\n", ETCMTAB);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    /* Transfer entries from old /etc/mtab to prototype replacement file: */
    while ((mntinfo = getmntent(fp_in)) != NULL) {
        if (strcmp(mntinfo->mnt_fsname, mntdev) == 0
          && strcmp(mntinfo->mnt_dir, tgt->dir) == 0) {
            ++found;
        } else {
            addmntent(fp_out, mntinfo);
        }
    }

    /* Transfer ownership & permissions from old /etc/mtab to new: */
    if (chown(ETCMTABTMP, sbuff.st_uid, sbuff.st_gid) != 0
      || chmod(ETCMTABTMP, sbuff.st_mode) != 0) {
        fprintf(stderr, "cannot transfer ownership/modes to \"%s\"\n",
                ETCMTABTMP);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    endmntent(fp_in); fp_in = NULL;

    if (rename(ETCMTABTMP, ETCMTAB)) {
        fprintf(stderr, "Failed to recreate %s\n", ETCMTAB);
    }

  bail_out:

    if (fp_in != NULL) endmntent(fp_in);
    if (fp_out != NULL) endmntent(fp_out);
    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


int is_mounted(const tgtdefn_t *tgt)
    /* Check if target is mounted */
{   int mounted=0;
    char *mntdev=NULL;
    struct mntent *mntinfo;
    struct stat st_mtb, st_tgt;
    FILE *fp;

    /* check if underlying device has been configured at all: */
    if (!is_configured(tgt->ident, NULL)) return 0;

    /* find path to device that would have been mounted & device info: */
    devmap_path(&mntdev, tgt->ident);
    if (stat(mntdev, &st_tgt) != 0) return 0;

    /* check entries in /etc/mtab: */
    fp = setmntent(ETCMTAB, "r");
    if (fp == NULL) {
        return 0;       /* indeterminate case - assume not mounted */
    }
    while ((mntinfo = getmntent(fp)) != NULL && !mounted) {
        if (stat(mntinfo->mnt_fsname, &st_mtb) != 0) continue;

        /* compare to mounted device on basis of kernel device maj/min: */
        if (major(st_mtb.st_rdev) == major(st_tgt.st_rdev)
          && minor(st_mtb.st_rdev) == minor(st_tgt.st_rdev)) {
            mounted = 1;
        }
    }
    endmntent(fp);

    return mounted;
}


int is_readonlyfs(const char *path)
    /* check if filesystem containing *path is read-only */
{   struct statvfs sbuff;

    return (path == NULL
            || (statvfs(path, &sbuff) != 0
            || (sbuff.f_flag & ST_RDONLY) != 0));
}


#if WITH_FSCK

int fs_check(const char *dev, const tgtdefn_t *tgt)
{   int idx, stat, eflag=ERR_NOERROR;
    const char *argv[16];

    /* construct argument list for fsck -T -t ... <dev> */
    idx = 0;
    argv[idx++] = "[cryptmount-fsck]";
    argv[idx++] = "-T";
    if (tgt->fstype != NULL) {
        argv[idx++] = "-t"; argv[idx++] = tgt->fstype;
    }
    argv[idx++] = dev;
    argv[idx++] = NULL;

    stat = run_sucommand(DLGT_FSCK, argv, F_MATCHUID);
    eflag = ((stat == 0 || stat == 1) ? ERR_NOERROR : ERR_BADFSCK);

    return eflag;
}

#endif  /* WITH_FSCK */


#if !ERSATZ_MOUNT || !ERSATZ_UMOUNT || WITH_FSCK || WITH_CSWAP

static int run_sucommand(const char *path, const char **argv,
                        unsigned switches)
    /* fork (& wait for) system-command as root */
{   pid_t child;
    int stat=-1, fd;

    switch ((child = fork())) {
        case -1:        /* fork failed */
            fprintf(stderr, "failed to fork (%s)\n", path);
            break;
        case 0:         /* child fork */
            if ((switches & F_MATCHUID) != 0) {
                /* change real UID to match effective UID
                   (probably only useful if euid==root): */
                (void)setuid(geteuid());
            }
            if ((switches & F_CLOSE1) != 0) {
                /* redirect standard output to /dev/null */
                fd = open("/dev/null", O_WRONLY);
                if (fd >= 0) (void)dup2(fd, STDOUT_FILENO);
                else (void)close(STDOUT_FILENO);
            }

            execv(path, (char *const *)argv);
            fprintf(stderr, "failed to invoke \"%s\"\n", path);
            exit(EXIT_BADEXEC);
            break;
        default:        /* parent fork */
            if (waitpid(child, &fd, 0) == child) {
                stat = fd;
            }
            break;
    }

    return stat;
}

#endif  /* !ERSATZ_MOUNT ... */

/* vim: set ts=4 sw=4 et: */

/*
 *  (C)Copyright 2005-2009, RW Penney
 */
