/*
  Copyright (C) 2003 by Marten Svanfeldt

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef __CS_RENDERMESHLIST_H__
#define __CS_RENDERMESHLIST_H__

#include "csextern.h"

#include "csutil/array.h"
#include "csutil/parray.h"
#include "csutil/ref.h"
#include "csgeom/box.h"
#include "csgeom/sphere.h"
#include "ivideo/graph3d.h"

struct iEngine;
struct iObjectRegistry;
struct csRenderMesh;
struct iMeshWrapper;

/**
 * This class is used when we need to store, sort and then render a list of
 * rendermeshes.
 */
class CS_CSTOOL_EXPORT csRenderMeshList
{
public:
  /**
   * Constructor. 
   * It needs objectregistry becouse the meshsorter needs to get the 
   * renderpriorities sortingoptions from the engine.
   */
  csRenderMeshList (iEngine* engine);
  
  /**
   * Destructor. Clean up the list
   */
  ~csRenderMeshList ();

  /**
   * Add a new set of rendermeshes to the lists
   */
  void AddRenderMeshes (csRenderMesh** meshes, int num, long renderPriority,
	csZBufMode z_buf_mode, const csBox3& bbox);

  /**
   * Remove all rendermeshes that are outside the specified sphere.
   * Note: the sphere's radius has to be the square of it's actual value.
   * (For optimization purposes.)
   */
  void CullToSphere (const csSphere& sphere);

  /**
   * Sort the list of meshes by render priority and within every render
   * priority. Return number of total meshes.
   */
  int SortMeshLists ();

  /**
   * After sorting the meshes fetch them with this function.
   */
  void GetSortedMeshes (csRenderMesh** meshes);

  /**
   * Empty the meshlist. It will still hold the list of renderpriorities.
   */
  void Empty ();

private:
  struct meshListEntry
  {
    csRenderMesh* rm;
    csBox3 bbox;

    meshListEntry (csRenderMesh* mesh, const csBox3& bb) : 
      rm(mesh), bbox(bb) {}
  };

  /// This struct contains one entry in the RP infoqueue
  struct renderMeshListInfo 
  {
    /// RP number
    long renderPriority;

    /// Sorting options
    int sortingOption;

    /// list of rendermeshes
    csArray<meshListEntry> meshList;

  };

  csPDelArray < renderMeshListInfo > renderList;
  iEngine* engine;

  static int SortMeshMaterial (meshListEntry const& me1, meshListEntry const& me2);
  static int SortMeshBack2Front (meshListEntry const& me1, meshListEntry const& me2);
  static int SortMeshFront2Back (meshListEntry const& me1, meshListEntry const& me2);
};

#endif //__CS_RENDERMESHLIST_H__
