/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/** @file cssedwindow.h
 *  @brief Header file for the cssed main window object. 
 *  Those functions are the interface for cssed's ::_CssedWindow object.
 */
 
/** a macro to cast a pointer to ::CssedWindow */
#define CSSED_WINDOW(n) (CssedWindow*) n

/** @enum CssedEolMode cssed's end of line modes */
typedef enum {
	/** Windows end of line Carriage return / Line feed */
	EOL_MODE_CRLF, 	
	/** Mac end of line Carriage return */
	EOL_MODE_CR,	
	/** Unix end of line Line feed */
	EOL_MODE_LF			
} CssedEolMode;

/** @enum CssedCodePage cssed's code pages */
typedef enum {
	/** Default code page */
	CSSED_CODEPAGE_DEFAULT,
	/** UTF8 code page */
	CSSED_CODEPAGE_UTF8,
	/** DBCS code page (Thai, Hangul) */
	CSSED_CODEPAGE_DBCS
} CssedCodePage;

/** @enum CssedFileType cssed's built in file types */
typedef enum {
	/** CSS style sheet (core language) */
	CSSED_FILETYPE_CSS,	
	/** hypertext,  this is not just HTML but ASP and PHP */
	CSSED_FILETYPE_HTML,	
	/** XML, cssed also recognizes *.svg file as XML */
	CSSED_FILETYPE_XML,		
	/** plain text, so no style */
	CSSED_FILETYPE_TEXT,	
	/** shell and bash scripts */
	CSSED_FILETYPE_SH,		
	/** not just C but also CPP */
	CSSED_FILETYPE_C,		
	/** Apache configuration and .htaccess files */
	CSSED_FILETYPE_APACHE_CONF,	
	/** Lisp files, mostly for emacs.el files */
	CSSED_FILETYPE_LISP,	
	/** Perl scripts */
	CSSED_FILETYPE_PERL,	
	/** Python scripts */
	CSSED_FILETYPE_PYTHON,	
	/** SQL files */
	CSSED_FILETYPE_SQL,	
	/** unused right now, don't use it */
	CSSED_FILETYPE_BYNARY,		
	/** internal use, don't use it ever */
	CSSED_FILETYPE_UNKNOW	
} CssedFileType;

// those are opaque use only provided functions
/** @struct _CssedDoc.  
 *  @brief This object implements a cssed document. This is opaque and should be accessed with the functions provided in document.h
 */
/** @typedef CssedDoc.  
 *  @brief A utility data type for ::_CssedDoc structures.
 */
typedef struct _CssedDoc CssedDoc;
	
/** @struct _CssedWindow.  
 *  @brief This object represents the cssed main window object. This is opaque and should be accessed with the functions provided in cssedwindow.h
 */	
/** @typedef CssedWindow.  
 *  @brief A utility data type for ::_CssedWindow structures.
 */
typedef struct _CssedWindow CssedWindow;
	

/** Gets the width of the window.
 *	This function returns the main window's widget pixel width. 
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@return An integer containing the width of the window's widget in pixels.
 */
gint cssed_window_get_width( CssedWindow* window );
/** Gets the height of the window.
 *	This function returns the main window's widget pixel height. 
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@return An integer containing the height of the window's widget in pixels.
 */
gint cssed_window_get_height( CssedWindow* window );
/** Returns the main ::CssedWindow widget.
 *	This function returns one GtkWidget's pointer, containing the main window widget ( a GtkWindow ).
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@return A GtkWidget pointer.
 */
GtkWidget* cssed_window_get_window_widget ( CssedWindow* window );
/** Returns the editor control contextual menu widget.
 *	This function returns one GtkWidget's pointer, containing the editor control contextual menu  ( a GtkMenu ).
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@return A GtkWidget pointer.
 */
GtkWidget* cssed_window_get_pop_menu ( CssedWindow* window );
/** Returns the last directory where a file has been opened.
 *	This function returns a buffer containing the full path of the last directory where a file has been opened. You should free it with g_free when no longer needed.
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@return A newly allocated buffer with the full path to the directory or NULL. 
 */
gchar* cssed_window_get_last_open_dir( CssedWindow* window );
/** Returns the last directory where a file has been saved.
 *	This function returns a buffer containing the full path of the last directory where a file has been saved. You should free it with g_free when no longer needed.
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@return A newly allocated buffer with the full path to the directory or NULL. 
 */
gchar* cssed_window_get_last_save_dir( CssedWindow* window );
/** Writes a message to the program output.
 *	This function writes a message to the program output. You can use Pango markup if you wish here.
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@param message - the text to be written.
 */
void cssed_window_output_write( CssedWindow* window, gchar* message );
/** Deletes all text at the program output.
 *	This function deletes all text in the program output window.
 *	@param window - one pointer to the main ::CssedWindow object.
 */
void cssed_window_output_clear( CssedWindow* window );
/** Gets the length of the document's list.
 *	This function retrieves the number of opened documents in cssed.
 *	@param window - one pointer to the main ::CssedWindow object.
 */
gint cssed_window_get_document_list_length( CssedWindow* window );
/** Gets the first element on the document's list
 *	This function retrieves the first element in a doubly linked list - implemented with GList.
 *	@param window - one pointer to the main ::CssedWindow object.
 *	@returns A pointer to the first element in the list
 *	@code
	#include <gtk/gtk.h>
	#include <cssedwindow.h>
	#include <document.h>

	CssedPlugin* plugin;
	GList* doc_list;
	CssedWindow* win;
	CssedDoc* doc;
	gchar* doc_name;

	// build and load the plugin
	
	// get and iterate through the document's list
	win = (CssedWindow*) cssed_plugin_get_window (plugin);
	doc_list = cssed_window_get_document_list_first( win );
	while( doc_list ){
		doc = (CssedDoc*) doc_list->data;
		doc_name = document_get_filename( doc );
		if( doc_name ){
			g_print( "Found document: %s\n", doc_name );
			g_free( doc_name );
		}else{
			g_print( "Found unnamed document\n");
		}
		doc_list = g_list_next( doc_list );	
	}
	@endcode
 */
GList* cssed_window_get_document_list_first( CssedWindow* window );
/** Inserts a css dialog to be used by autocompletion.
 *	This function inserts a css dialog as explained in @ref dialog_class.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@param key - the autocompletion trigger key word.
 *	@param class - a ::CssDialogClass implementing the dialog.
 */
void cssed_window_css_dialog_insert( CssedWindow* window, gchar* key, gpointer class );
/** Removes a css dialog used by auto completion.
 *	This function removes a css dialog as explained in @ref dialog_class.<br>
 *	If the keyword is in use, the new dialog will replace the old one.<br>
 *	If you don't like this behaviour, use ::cssed_window_css_dialog_keyword_in_use to check if the keyword is being used and act at your convenience.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@param key - the autocompletion trigger keyword.
 */
void cssed_window_css_dialog_remove_by_keyword( CssedWindow* window, gchar* key );
/** Searches a css dialog by the auto completion trigger keyword.
 *	This returns a pointer to a ::CssDialogClass implementing the dialog identified by the keyword, as explained in @ref dialog_class.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@param key - the autocompletion trigger keyword.
 *	@return A void pointer to a ::CssDialogClass - cast it to use it - or NULL if no dialog is found.
 */
gpointer cssed_window_css_dialog_lookup_by_keyword( CssedWindow* window, gchar* key );
/** Searches an auto completion trigger keyword being used.
 *	This returns a boolean indicating if a keyword is being used, as explained in @ref dialog_class.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@param key - the autocompletion trigger keyword.
 *	@return A boolean indicating if the keyword is in use.
 */
gboolean cssed_window_css_dialog_keyword_in_use( CssedWindow* window, gchar* key );
/** Searches if a file name is opened in cssed.
 *	It searches an exact match between the file name passed and the document's file name, usually the full path.<br>
 *	If the file is found, it becomes the current document. This function is mainly to avoid to reopen a file and used
 *	from some file selector callbacks.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@param file - The file name ( full path ).
 *	@return A boolean indicating if the file is opened.
 */
gboolean cssed_window_is_file_opened( CssedWindow* window, gchar* file );
/** Gets the number of documents opened in cssed.
 *	It returns the same value as ::cssed_window_get_document_list_length.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@return An integer with the number of open documents.
 */
gint cssed_window_get_num_docs( CssedWindow* window );
/** Makes active a document by number.
 *	It selects and makes the current active document, the one in the same position in the document's list, as the number passed.
 *	@param window - A pointer to the main ::CssedWindow object.
 *	@param doc_number - A number with the position of the document in the document's list ( starting at 0 ).
 */
void cssed_window_set_ndoc_active( CssedWindow* window, gint doc_number );



