/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifndef PLAT_GTK
#   define PLAT_GTK 1
#endif

// this is the line numbers width template
#define MARGIN_LINES_WIDTH_STRING "_9999"
// markers margin width in pixels
#define MARGIN_MARKERS_WIDTH_PIXELS 14
// folding margin width in pixels
#define MARGIN_FOLDING_WIDTH_PIXELS 12

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>

#ifndef WIN32
# include <sys/mman.h>
# include <libgen.h>
#endif

#ifdef WITH_TERMINAL
 #include <vte/vte.h>
#endif

#include "cssedwindow.h"
#include "callbacks.h"
#include "cssdialogs-interface.h"
#include "interface.h"
#include "support.h"
#include "document.h"
#include "debug.h"
#include "cssdialogs-class.h"
#include "utils.h"

#include <Scintilla.h>
#include <SciLexer.h>
#include <ScintillaWidget.h>

struct _CssedDoc
{
	CssedWindow* window;	// main window object
	GtkWidget* editor;		// scintilla editor
	gint scid;				// scintilla id
	GtkWidget* notebook; 	// main notebook where this doc is attached
	gchar* filename;		// full path to file, it is dynamically allocated so dispose mem when not needed
	CssedFileType filetype;	// one of the supported filetypes
	gboolean modified; 		// modified flag, will be updated when the document changes
	gboolean autocompletion;// set autocompletion mode for the document
};

/* some prototypes will be moved to sciwrappers.h and functions to sciwrappers.c */
void 				on_editor_notification		(GtkWidget* editor,gint scn, gpointer lscn, gpointer user_data);
void 				sci_set_css_style 			(ScintillaObject * sci);
ScintillaObject*	sci_get_from_document		(CssedDoc* doc);
ScintillaObject* 	sci_get_from_notebook_page	(GtkNotebook* notebook, gint 		page_number);
void 				sci_set_line_numbers		(ScintillaObject* sci,  gboolean 	set );
void 				sci_set_keyworkds 			(ScintillaObject* sci,  gchar* 		keywords);
void 				sci_add_text 				(ScintillaObject* sci,  gchar* 		text);
gboolean			sci_can_redo				(ScintillaObject* sci);
gboolean			sci_can_undo				(ScintillaObject* sci);
void 				sci_undo					(ScintillaObject* sci);
void 				sci_redo					(ScintillaObject* sci);
void 				sci_empty_undo_buffer		(ScintillaObject* sci);
void 				sci_end_undo_action			(ScintillaObject* sci);
void 				sci_start_undo_action		(ScintillaObject* sci);
void				sci_set_visible_eols		(ScintillaObject* sci, gboolean set );
gboolean			sci_get_visible_eols		(ScintillaObject* sci);
void				sci_set_lines_wrapped		(ScintillaObject* sci, gboolean set );
gboolean 			sci_get_lines_wrapped		(ScintillaObject* sci);
void				sci_set_visible_white_spaces(ScintillaObject* sci, gboolean set );
gboolean 			sci_get_visible_white_spaces(ScintillaObject* sci);
void 				sci_convert_eols			(ScintillaObject* sci, gint eolmode);
gint				sci_get_eol_mode			(ScintillaObject* sci);
void 				sci_set_eol_mode			(ScintillaObject* sci, gint eolmode);
void 				sci_zoom_in					(ScintillaObject* sci);
void 				sci_zoom_out				(ScintillaObject* sci);
void 				sci_zoom_off				(ScintillaObject* sci);
void				sci_set_marker_at_line		(ScintillaObject* sci, gint line_number, gboolean set, gint marker );
gboolean 			sci_is_marker_set_at_line	(ScintillaObject* sci, gint line, gint marker);
gboolean 			sci_marker_next				(ScintillaObject* sci, gint line, gint marker_mask);
gboolean 			sci_marker_prev				(ScintillaObject* sci, gint line, gint marker_mask);

gint 				sci_get_line_from_position	(ScintillaObject* sci, gint position);
gint 				sci_get_position_from_line	(ScintillaObject* sci, gint line );
gint 				sci_get_current_position	(ScintillaObject* sci);
void 				sci_set_current_position	(ScintillaObject* sci, gint position);
void 				sci_set_current_line		(ScintillaObject* sci, gint line);

void 				sci_cut						(ScintillaObject* sci);
void 				sci_copy					(ScintillaObject* sci);
void 				sci_paste					(ScintillaObject* sci);

gint				sci_get_selection_start		(ScintillaObject* sci);
gint				sci_get_selection_end		(ScintillaObject* sci);
void 				sci_replace_sel				(ScintillaObject* sci, gchar* text);

gint				sci_get_length				(ScintillaObject* sci);
void				sci_get_text				(ScintillaObject* sci,gint len,gchar* text);
void				sci_get_selected_text		(ScintillaObject* sci, gchar* text);
void				sci_get_line				(ScintillaObject* sci, gint line, gchar* text);
gint 				sci_line_length				(ScintillaObject* sci, gint line);
gint				sci_get_line_count			( ScintillaObject* sci );
void 				sci_get_xy_from_position	(ScintillaObject* sci,gint pos, gint* x, gint* y);

void 				sci_set_undo_collection		(ScintillaObject* sci, gboolean set);
gboolean			sci_get_undo_collection		(ScintillaObject* sci);

void 				sci_set_selection_start		(ScintillaObject* sci, gint position);
void				sci_set_selection_end		(ScintillaObject* sci, gint position);
gint				sci_get_line_end_from_position	(ScintillaObject* sci, gint position);

void 				sci_toggle_fold				(ScintillaObject* sci, gint line);
gboolean			sci_get_line_is_visible		(ScintillaObject* sci, gint line);
void				sci_ensure_line_is_visible	(ScintillaObject* sci, gint line);
gint				sci_get_fold_level			(ScintillaObject* sci, gint line);

void 				sci_set_folding_margin_visible (ScintillaObject * sci, gboolean set );
gboolean			sci_get_folding_margin_visible(ScintillaObject * sci);
gboolean			sci_get_fold_expanded(ScintillaObject* sci, gint line);
void				sci_colourise(ScintillaObject* sci, gint start, gint end);

gboolean			sci_get_line_numbers(ScintillaObject * sci);
int					get_new_sci_number ( void );


/*
	create an empty doc with a scintilla editor
*/
CssedDoc *
cssed_create_new_doc ( CssedWindow* window )
{
	GtkWidget *editor;
	ScintillaObject *sci;
	CssedDoc *newdoc;
	gint sciid;
	CssedConfig* cfg;

	newdoc = g_malloc (sizeof (CssedDoc));
	cfg = cssed_window_get_config( window );

	editor = scintilla_new ();
	sci = SCINTILLA (editor);
	sciid = get_new_sci_number ();
	scintilla_set_id (sci, sciid);

	DBGMSG ("document.c:cssed_create_new_doc() - created doc, editor %x, siid %d", editor, sciid);
	DBGMSG ("document.c:cssed_create_new_doc() - in list %d", cssed_window_get_document_list_length ( window ));

	g_signal_connect( editor, "sci-notify", G_CALLBACK(on_editor_notification), (gpointer) window );
	/* This is unused now but will be used in the future to track function key press events and others
	 * The callback func is in callbacks.c
	g_signal_connect( editor, "key-press-event", G_CALLBACK(on_editor_key_press_event), (gpointer) window );
	*/
	g_signal_connect( editor, "button-press-event", G_CALLBACK(on_editor_button_press_event), (gpointer) window );
	g_signal_connect( editor, "grab-focus" , G_CALLBACK(on_editor_grab_focus), (gpointer) window );

	gtk_widget_set_usize (editor, 300, 300);
	newdoc->editor = editor;					  		// scintilla editor
	newdoc->scid = sciid;						  	// scintilla id
	newdoc->notebook = 							  	// main notebook where this doc is attached
				cssed_window_get_document_notebook ( window );
	newdoc->filename = NULL;    				  	// full path to file if exists
	newdoc->modified = FALSE;					  	// fresh document, so unmodified
	newdoc->autocompletion = cfg->autocompletion; // autocompletion from config
	newdoc->window = window;
	newdoc->filetype = CSSED_FILETYPE_UNKNOW;

	document_set_default_configuration( newdoc );
	document_set_undo_collection( newdoc, TRUE );
	document_start_undo_action(newdoc);

#ifdef WIN32
	document_set_eol_mode(newdoc, EOL_MODE_CRLF);
#endif

	return newdoc;
}
//
void
document_save(CssedDoc* doc)
{
	CssedWindow* window;
	GtkWidget *file_selector;
	gchar *doc_buffer;
	gint doclen;
	size_t nwriten;
	FILE *f;
	gchar *msg;
#ifdef GTK_ATLEAST_2_4
	gchar *filename;
	gchar *last_save_dir;
#endif

	window = CSSED_WINDOW(doc->window);

	if (doc->filename == NULL)
	{
#ifndef GTK_ATLEAST_2_4
		file_selector = create_save_as_fileselection (window);
		gtk_widget_show (file_selector);
		return;
#else
		file_selector = gtk_file_chooser_dialog_new(_("Select file"), NULL,
																GTK_FILE_CHOOSER_ACTION_SAVE,
																GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
																GTK_STOCK_SAVE, GTK_RESPONSE_OK,
																NULL );
		last_save_dir = cssed_window_get_last_save_dir(window);
		if( last_save_dir != NULL ){
			gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (file_selector),	last_save_dir );
			g_free(last_save_dir);
		}
		if( gtk_dialog_run(GTK_DIALOG(file_selector)) ==  GTK_RESPONSE_OK ){
			filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_selector));
			gtk_widget_destroy(file_selector);
			document_save_as(doc, filename);		
			g_free(filename);	
			return;
		}else{
			gtk_widget_destroy(file_selector);
			return;			
		}	
#endif	
	}

	DBGMSG ("document.c: document_save() - Trying to save a document called %s", doc->filename);
	f = fopen (doc->filename, "w");
	if (f == NULL)
	{
		msg = g_strdup_printf
			(_("Unable to open the file\n%s\nfor writing."),doc->filename);
		cssed_error_message ( _("Cannot open file"),msg );
		g_free (msg);
	}
	else
	{
#ifdef WIN32
		gint eols;
		eols = document_get_eol_mode(doc);
		if( eols == EOL_MODE_CRLF ){
			document_convert_eols ( doc, EOL_MODE_LF);
		}
#endif
		doclen = document_get_length (doc);
		doc_buffer = g_malloc0 (++doclen);
		document_get_text (doc, doclen, doc_buffer);

		nwriten =
			fwrite (doc_buffer, sizeof (gchar), strlen(doc_buffer), f);

#ifdef WIN32
		if( eols == EOL_MODE_CRLF ){
			document_convert_eols ( doc, EOL_MODE_CRLF);
		}
#endif

		if ( nwriten != strlen(doc_buffer) )
		{
			cssed_error_message
				(_("Error while writing to file"),
				 _("There is a mismatch between the file size\nand the bytes written.\nThe file may be damaged.") );
		}
		document_set_save_point ( doc );
		g_free (doc_buffer);
		fclose (f);
	}
}


void
document_save_as(CssedDoc* doc, gchar* filename)
{
	CssedWindow *window;	// main window object
	GtkWidget *window_widget;
	GtkWidget *notebook;
	GtkWidget *editor;
	GtkWidget* new_label;
	GtkWidget* dialog;
	gint response;
	gchar *doc_buffer;
	gint doclen;
	gint current_index;
	FILE *f;
	gchar *msg;
	gchar* dir;
	CssedFileType filetype;
	gchar* basename;
	GList* list;
	GtkWidget* hbox;
#ifdef WIN32
	int nwriten;
#else
	size_t nwriten;
#endif

	window = document_get_window(doc) ;
	window_widget = cssed_window_get_window_widget(window);
	notebook = cssed_window_get_document_notebook(window);

	if( g_file_test( filename , G_FILE_TEST_EXISTS) ){
		dialog = create_yes_no_cancel_dialog (_("File exists and will be overwritten"),
										  _("Do you want to overwrite the existing file with the current one?") );
		response = gtk_dialog_run( GTK_DIALOG(dialog) );
		gtk_widget_destroy( dialog );
		if( response != GTK_RESPONSE_YES ){
			return;
		}
	}
	
	f = fopen (filename, "w");
	if (f == NULL)
	{
		msg = g_strdup_printf (_("The file %s\ncannot be opened for writing."),
					   filename);
		cssed_error_message (msg,_("Cannot open file"));
		g_free (msg);
	}
	else
	{
		doclen = document_get_length (doc);
		doc_buffer = g_malloc (++doclen);
		document_get_text (doc, doclen, doc_buffer);
	
		nwriten = fwrite (doc_buffer, sizeof (gchar), doclen, f);
		if (nwriten != doclen)
		{
			cssed_error_message
			(
			_("There is a mismatch between the file size\nand the bytes written.\nThe file may be damaged."),
			_("Error while writing to file")
			);
		}
		document_empty_undo_buffer(doc);
		document_set_save_point (doc);
		current_index =	gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook));
		editor = gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), current_index);
		
		hbox = gtk_notebook_get_tab_label(GTK_NOTEBOOK (notebook), editor);
		list = gtk_container_get_children( GTK_CONTAINER(hbox) );
		list = g_list_first( list );
		new_label = GTK_WIDGET( list->data );
		g_list_free( list );
		
		if( new_label != NULL ){
			basename = g_path_get_basename  (filename);
			gtk_label_set_text(GTK_LABEL(new_label),basename);
			g_free(basename);
		}
	
		gtk_notebook_set_menu_label_text(GTK_NOTEBOOK (notebook), editor, filename );		
		document_set_filename( doc, filename );		
		filetype = document_get_filetype_from_filename( filename );
		
		if( filetype != CSSED_FILETYPE_UNKNOW ){
			document_set_filetype( 	doc, filetype );
		}
	
		gtk_window_set_title(GTK_WINDOW(window_widget),filename);
	
#ifdef WIN32
		dir = g_strconcat( g_dirname( filename ), "\\", NULL );
#else
		dir = g_strconcat( g_dirname( filename ), "/", NULL );
#endif
	
		cssed_window_set_last_save_dir( window, dir);
		
		g_free ( dir );
		g_free ( doc_buffer );
		fclose ( f );
	}
}

/*  create an empty doc with an scintilla editor and
	attach to the notebook passed as argument, puts the
	string title on notebook label.
*/
void
create_and_attach_new_doc (CssedWindow* window , const gchar * title)
{
	CssedDoc *newdoc;
	GtkWidget *label;
	GtkWidget *notebook;
	// testing here
	GtkWidget *hbox;
	GtkWidget *close_button;
	GtkWidget *icon;
	GtkWidget *label_menu;

	newdoc = cssed_create_new_doc (window);
	if (newdoc == NULL)
	{ // FIXME say something to user
		DBGMSG ("document.c:create_and_attach_new_doc() - passed null doc !!!");
	}
	else
	{
		DBGMSG ("document.c:create_and_attach_new_doc() - doc passed %x", newdoc);
		DBGMSG ("document.c:create_and_attach_new_doc() - in list %d", cssed_window_get_document_list_length ( window ));
		document_set_filetype( newdoc, CSSED_FILETYPE_TEXT );
#ifdef WIN32
		document_set_codepage( newdoc, CSSED_CODEPAGE_DEFAULT );
#else
		document_set_codepage( newdoc, CSSED_CODEPAGE_UTF8 );// empty docs are UTF8 by default
#endif
		cssed_window_add_document_to_list( window, newdoc );
		gtk_widget_show ( newdoc->editor );

		hbox = gtk_hbox_new( FALSE, 0 );
		gtk_widget_show( hbox );
		close_button = gtk_button_new();
		gtk_widget_show( close_button );
		icon = create_pixmap (NULL,"doc_close.png");
		gtk_widget_show( icon );
		gtk_container_add( GTK_CONTAINER(close_button), icon );

		label = gtk_label_new ( title );
		gtk_widget_show( label );
		label_menu = gtk_label_new ( title );
		gtk_widget_show( label_menu );

		gtk_box_pack_start  (GTK_BOX(hbox), label,FALSE,FALSE,0);
		gtk_box_pack_end    (GTK_BOX(hbox), close_button,FALSE,FALSE,0);

		g_signal_connect_swapped ((gpointer) close_button, "clicked",
			  G_CALLBACK (document_close), newdoc);

		notebook = cssed_window_get_document_notebook( window );
		gtk_notebook_append_page_menu ( GTK_NOTEBOOK( notebook ),  newdoc->editor, hbox, label_menu);

		gtk_widget_queue_draw ( notebook );
	}
}

void
create_and_attach_new_named_doc_from_fileselector_cb (GtkButton * button, gpointer user_data)
{
	GtkWidget* file_selector;
	CssedFileSelectorData* data;
	CssedWindow* window;
	const gchar *selected_filename;
	gchar* filename;

	data = (CssedFileSelectorData*) user_data;
	window = (CssedWindow *) data->main_window;
	file_selector = GTK_WIDGET (data->fileselector);

	selected_filename =	gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_selector));
	filename = g_strdup( selected_filename ); // avoid compiler warnings
	gtk_widget_destroy (gtk_widget_get_toplevel (file_selector));
	create_and_attach_new_named_doc( window, filename );
	g_free( filename );
	g_free (data);
}

void
create_and_attach_new_named_doc(CssedWindow* window, gchar* filename)
{
	CssedDoc* newdoc;
	CssedDoc* curdoc;
	GtkWidget* label;
	GtkWidget* notebook;
	gint pnum;
	struct stat ststat;
	gint _retval;
#ifdef WIN32
	GError* error = NULL;
	gchar* map;
#else
	gint filedes;
	void* map;
#endif
	gchar* ptr;
	gchar* block_end;
	gchar* line_start;
	gchar* line_buffer;
	gint line_size;
	gchar* file_name;
	gchar* filenamecp;
	gchar* errormsg;
	gboolean needs_new;
	GtkWidget* window_widget;
	gint current_index;
	GtkWidget *editor;
	gchar* dir;
	CssedEolMode mode;
	/*
	G_CONST_RETURN gchar *charset;

	if( g_get_charset( &charset) ){
		g_print("%s\n", charset );
	}*/
	if( cssed_window_is_file_opened( window, filename ) ){
		return;
	}

	notebook = cssed_window_get_document_notebook( window );

	DBGMSG ("document.c - create_and_attach_new_named_doc - Filename: %s",
							filename);
	if (!g_file_test (filename, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR))
	{
		errormsg = g_strdup_printf(_("File %s\ndoes not exist or is not a regular file."),filename);
		cssed_error_message (errormsg,"Error opening file");
		g_free(errormsg);
		return;
	}

	_retval = stat (filename, &ststat);
	if (_retval != 0)
	{
		errormsg = g_strdup_printf(_("Cannot get information about file\n%s."),filename);
		cssed_error_message (errormsg,_("Error opening file"));
		g_free(errormsg);
		return;
	}
#ifdef WIN32
	if( !g_file_get_contents (filename, &map, NULL,  &error) ){
		errormsg = g_strdup_printf(_("Cannot open the file\n%s."),filename);
		cssed_error_message (errormsg,_("Error opening file") );
		g_free(errormsg);
		return;
	}
#else
	if ((filedes = open (filename, O_RDONLY)) < 0)
	{
		errormsg = g_strdup_printf(_("Cannot open the file\n%s."),filename);
		cssed_error_message (errormsg,_("Error opening file") );
		g_free(errormsg);
		return;
	}
	if ((map =
	     mmap (0, ststat.st_size, PROT_READ, MAP_SHARED, filedes, 0)) == MAP_FAILED)
	{
		errormsg = g_strdup_printf(_("Cannot map to memory the file\n%s."),filename);
		cssed_error_message (errormsg,_("Error opening file") );
		close (filedes);
		g_free(errormsg);
		return;
	}
#endif
/**/
	ptr = map;		// alocate the pointer at start of memory block
	block_end = map + ststat.st_size;	     // end of memory block
	line_start = map;	                     // start of first line

	curdoc = document_get_current(window);

	if ( curdoc->filename == NULL && document_get_length(curdoc) == 0){
		needs_new = FALSE;
	}else{
		needs_new = TRUE;
	}

	if( needs_new ){
		newdoc = cssed_create_new_doc (window);
		cssed_window_add_document_to_list( window, newdoc );
	}else{
		newdoc = curdoc;
	}


	if (newdoc == NULL) // sanity check
	{
	/* FIXME; say something to user */
		DBGMSG ("document.c create_and_attach_new_named_doc(), ERR _ Passed null doc !!!");
#ifndef WIN32
		close (filedes);
		munmap(map, ststat.st_size);
#endif
		return;
	}
	else
	{
		DBGMSG ("document.c create_and_attach_new_named_doc(), Doc passed %p", newdoc);
		DBGMSG ("document.c create_and_attach_new_named_doc(), in list %d", cssed_window_get_document_list_length( window ));
		// add the file to the editor
		line_size = 0;
		// if empty just return
		if( ststat.st_size == 0 ){
			errormsg = g_strdup_printf("open file: the file '%s' is empty",filename);
			document_write_error_to_program_output(newdoc, errormsg,ERROR_COLOR_STRING);
			g_free(errormsg);
		}else{
			while( ptr <= block_end ){
				if( ( (char) *ptr == '\n' ) || (ptr == block_end) ){
					// add line
					line_buffer = (char*) calloc( line_size + 1, sizeof(char));
					if( line_buffer == NULL ){ /* FIXME; say something to user */
						DBGMSG ("document.c create_and_attach_new_named_doc() - Cannot reserve mem, calloc returned %p", line_buffer);
#ifndef WIN32
						close (filedes);
						munmap(map, ststat.st_size);
#endif
						return;
					}else{
						line_buffer = (gchar*) strncpy(line_buffer,line_start , line_size * sizeof(char));
						document_add_text_line( newdoc,(char*)  line_buffer);
						free(line_buffer);
					}
					line_size = 0;
					line_start = ptr + 1;
				}else{
					line_size++;
				}
				ptr++;
			}
		}
		// get and set eol mode
		mode = buffer_get_first_eol( map, block_end );
		document_set_eol_mode( newdoc, mode );

		filenamecp = (char*) g_strdup( filename );
		file_name = g_path_get_basename( filenamecp );

		document_set_filetype(
								newdoc,
								document_get_filetype_from_filename( filename )
							);

		if( needs_new ){
			newdoc->notebook = notebook;
			newdoc->filename = filenamecp;
			newdoc->modified = FALSE;
		}else{
			newdoc->filename = filenamecp;
			newdoc->modified = FALSE;
		}

		if( needs_new ){
			GtkWidget *hbox;
			GtkWidget *close_button;
			GtkWidget* icon;
			GtkWidget* label_menu;

			gtk_widget_show (newdoc->editor);

			hbox = gtk_hbox_new( FALSE, 0 );
			close_button = gtk_button_new();
			icon = create_pixmap (NULL,"doc_close.png");
			gtk_widget_show( hbox );
			gtk_container_add( GTK_CONTAINER(close_button), icon );

			gtk_widget_show( icon );
			gtk_widget_show( close_button );

			label = gtk_label_new (file_name);
			gtk_widget_show( label );

			label_menu= gtk_label_new (filename);
			gtk_widget_show( label_menu );

			gtk_box_pack_start  (GTK_BOX(hbox), label,FALSE,FALSE,0);
			gtk_box_pack_end    (GTK_BOX(hbox), close_button,FALSE,FALSE,0);

			g_signal_connect_swapped ((gpointer) close_button, "clicked",
				  G_CALLBACK (document_close), newdoc);

			gtk_notebook_append_page_menu ((GtkNotebook *) notebook, newdoc->editor, hbox, label_menu);

#ifdef GTK_IS_2_2
			pnum = gtk_notebook_get_n_pages (GTK_NOTEBOOK(notebook));
#else
			pnum = g_list_length(GTK_NOTEBOOK(notebook)->children);
#endif
			gtk_notebook_set_current_page   (GTK_NOTEBOOK(notebook), --pnum);
			gtk_widget_queue_draw (GTK_WIDGET (notebook));
			document_colourise_all(newdoc);
			document_empty_undo_buffer(newdoc);
			document_set_save_point(newdoc);
		}else{
			GList* list;
			GtkWidget* hbox;

			current_index =
				gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook));
			editor = gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
						    current_index);

			hbox = gtk_notebook_get_tab_label(GTK_NOTEBOOK (notebook), editor);
			list = gtk_container_get_children( GTK_CONTAINER(hbox) );
			list = g_list_first( list );
			label = GTK_WIDGET( list->data );

			gtk_label_set_text(GTK_LABEL(label), file_name);
			gtk_notebook_set_menu_label_text(GTK_NOTEBOOK (notebook), editor, filename );

			window_widget = cssed_window_get_window_widget( window );
			gtk_window_set_title(GTK_WINDOW(window_widget), filename);
			document_colourise_all(newdoc);
			document_empty_undo_buffer(newdoc);
		}
		

#ifdef WIN32
		dir = g_strconcat( g_dirname( filename ), "\\", NULL );
		document_set_codepage( newdoc, CSSED_CODEPAGE_DEFAULT );
#else
		dir = g_strconcat( g_dirname( filename ), "/", NULL );
		// testing here, does it slow things too much ? FIXME && test please
		if( g_utf8_validate(map, ststat.st_size-1, NULL) ){
				document_set_codepage( newdoc, CSSED_CODEPAGE_UTF8 );
		}
#endif
		cssed_window_set_last_open_dir(window, dir);
		cssed_window_disable_save(window);
		cssed_window_disable_undo(window);
		cssed_window_disable_redo(window);
		document_set_current_position(newdoc, 0);
		g_free(dir);
		g_free(file_name);
	}
#ifndef WIN32
	close (filedes);
	munmap(map, ststat.st_size);
#endif
}

/*  unused	- flaged to be skipped

	create one alone editor used as a callback from create_window
	NEDDS FIX don't use in other created documents.
	use create_and_attach_new_doc() instead

GtkWidget *
create_initial_blank_editor (CssedWindow* window)
{
	CssedDoc *newdoc;

	newdoc = cssed_create_new_doc (window);
	cssed_window_add_document_to_list( window, newdoc );
	DBGMSG ("document.c:create_initial_blank_editor() - Initial editor created");
	DBGMSG ("document.c:create_initial_blank_editor() - in list %d", cssed_window_get_document_list_length ( window ));
	cssed_window_disable_save( window );
	cssed_window_disable_undo( window );
	cssed_window_disable_redo( window );
	return newdoc->editor;
}*/

/*  Get diferent - incrementally - integers as sci ids
	so, right now ,we are limited in docs opened in a
	single editing session. Even if you closed them.
	FIXME
*/
int
get_new_sci_number ()
{
	static int initialized = 0;
	static unsigned int count;

	if (initialized != 1)
	{
		initialized = 1;
		count = 0;
		return ++count;	// id 0 created on startup
	}
	else
	{
		return ++count;
	}
}

/* close doc - check if it's saved and ask to user if not */
void
document_close( CssedDoc* doc )
{
	GtkWidget* okcancel;
	gint response;

	if ( document_get_modified( doc ) )
	{
		okcancel = create_ok_cancel_dialog(
				_("File is not saved, do you want to close it?"),
				_("The file has changed and the changes will be lost if you close it. Click the  <b>\"Cancel\"</b> button if you want to skip to close the file, click the <b>\"Ok\"</b> button if you want to close the file unsaved.")
				);
		 response = gtk_dialog_run( GTK_DIALOG(okcancel) );
		if( response == GTK_RESPONSE_OK ){
			document_dispose (doc);
		}
		gtk_widget_destroy( GTK_WIDGET(okcancel) );
	}else{
		document_dispose (doc);
	}
}

/* validator arrow */
void
document_set_validator_arrow_at_line( CssedDoc* doc, gint line )
{
	ScintillaObject * sci;
	gint pos;
	sci =  sci_get_from_document( doc );

	SSM(sci, SCI_MARKERDELETEALL,1,0);
	SSM(sci, SCI_MARKERADD, line-1, 1);
	pos = document_get_position_from_line( doc, --line);
	document_ensure_line_is_visible( doc, line );
	document_set_current_position(doc, pos);
}
void
document_clear_validator_arrow( CssedDoc* doc )
{
	ScintillaObject * sci;
	sci =  sci_get_from_document( doc );

	SSM(sci, SCI_MARKERDELETEALL,1,0);
}
/* scroll to the caret
	when you adds text, to the scintilla control, if the
	edge bound of text is down the editor window, no automatic
	scrolling is performed so you must call this func
	to scroll to the caret before text insertion.
*/
void
document_scroll_to_caret( CssedDoc* doc )
{
	ScintillaObject * sci;
	sci = SCINTILLA( doc->editor );
	SSM (sci, SCI_MOVECARETINSIDEVIEW, 0, 0);
}
/* recolourise the document from start to end */
void
document_colourise(CssedDoc* doc, gint start, gint end)
{
	ScintillaObject * sci;
	sci = (ScintillaObject *) sci_get_from_document(  doc );
	sci_colourise( sci, start, end );
}
void
document_colourise_all(CssedDoc* doc)
{
	gint end;
	end = document_get_length( doc );
	document_colourise(doc , 0, end);
}
void
document_style_clear_all(CssedDoc* doc) // no wrapper for this yet
{
	ScintillaObject * sci;
	sci = (ScintillaObject *) sci_get_from_document(  doc );
	SSM(sci, SCI_STYLECLEARALL, 0 , 0);
}

void
document_set_css_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );
	//sci_set_css_style (sci);

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_CSS, 0);

	SSM (sci, SCI_STYLESETFORE, SCE_CSS_DEFAULT, cfg->DEFAULT_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_DEFAULT, cfg->DEFAULT_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_DEFAULT, cfg->DEFAULT_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_TAG, cfg->TAG_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_TAG, cfg->TAG_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_TAG, cfg->TAG_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_PSEUDOCLASS, cfg->PSEUDOCLASS_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_PSEUDOCLASS, cfg->PSEUDOCLASS_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_PSEUDOCLASS, cfg->PSEUDOCLASS_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_UNKNOWN_PSEUDOCLASS, cfg->UNKNOWN_PSEUDOCLASS_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_UNKNOWN_PSEUDOCLASS, cfg->UNKNOWN_PSEUDOCLASS_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_UNKNOWN_PSEUDOCLASS, cfg->UNKNOWN_PSEUDOCLASS_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_OPERATOR, cfg->OPERATOR_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_OPERATOR, cfg->OPERATOR_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_OPERATOR, cfg->OPERATOR_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_IDENTIFIER, cfg->IDENTIFIER_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_IDENTIFIER, cfg->IDENTIFIER_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_IDENTIFIER, cfg->IDENTIFIER_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_UNKNOWN_IDENTIFIER, cfg->UNKNOWN_IDENTIFIER_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_UNKNOWN_IDENTIFIER, cfg->UNKNOWN_IDENTIFIER_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_UNKNOWN_IDENTIFIER, cfg->UNKNOWN_IDENTIFIER_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_VALUE, cfg->VALUE_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_VALUE, cfg->VALUE_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_VALUE, cfg->VALUE_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_COMMENT, cfg->COMMENT_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_COMMENT, cfg->COMMENT_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_COMMENT, cfg->COMMENT_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_IMPORTANT, cfg->IMPORTANT_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_IMPORTANT, cfg->IMPORTANT_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_IMPORTANT, cfg->IMPORTANT_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_DIRECTIVE, cfg->DIRECTIVE_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_DIRECTIVE, cfg->DIRECTIVE_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_DIRECTIVE, cfg->DIRECTIVE_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_DOUBLESTRING, cfg->DOUBLESTRING_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_DOUBLESTRING, cfg->DOUBLESTRING_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_DOUBLESTRING, cfg->DOUBLESTRING_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_SINGLESTRING, cfg->SINGLESTRING_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_SINGLESTRING, cfg->SINGLESTRING_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_SINGLESTRING, cfg->SINGLESTRING_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_CLASS, cfg->CLASS_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_CLASS, cfg->CLASS_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_CLASS, cfg->CLASS_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_ID, cfg->ID_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_ID, cfg->ID_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_ID, cfg->ID_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_VALID_VALUE, cfg->VALID_VALUE_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_VALID_VALUE, cfg->VALID_VALUE_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_VALID_VALUE, cfg->VALID_VALUE_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_FUNCTION, cfg->FUNCTION_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_FUNCTION, cfg->FUNCTION_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_FUNCTION, cfg->FUNCTION_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_NUMBER, cfg->NUMBER_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_NUMBER, cfg->NUMBER_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_NUMBER, cfg->NUMBER_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_UNIT, cfg->UNIT_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_UNIT, cfg->UNIT_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_UNIT, cfg->UNIT_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_COLOR, cfg->COLOR_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_COLOR, cfg->COLOR_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_COLOR, cfg->COLOR_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_HEXACOLOR, cfg->HEXACOLOR_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_HEXACOLOR, cfg->HEXACOLOR_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_HEXACOLOR, cfg->HEXACOLOR_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_ATTR_MATCH, cfg->ATTR_MATCH_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_ATTR_MATCH, cfg->ATTR_MATCH_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_ATTR_MATCH, cfg->ATTR_MATCH_bold);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_LANGUAGE, cfg->LANGUAGE_fore_color);
	SSM (sci, SCI_STYLESETBACK, SCE_CSS_LANGUAGE, cfg->LANGUAGE_back_color);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_LANGUAGE, cfg->LANGUAGE_bold);

	sci_set_keyworkds ( sci, (gchar*) cssed_window_get_keyword_list( window ) );
}
void
document_set_font_by_name(CssedDoc* doc, gchar* font_name)
{
	ScintillaObject * sci;
	CssedWindow* window;
	gint style;
	window = CSSED_WINDOW(doc->window);
	sci = SCINTILLA( doc->editor );

	switch( document_get_filetype( doc ) ){
		case CSSED_FILETYPE_CSS:
			for( style = SCE_CSS_DEFAULT; style <= SCE_CSS_LANGUAGE; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_HTML:
			for( style = SCE_H_DEFAULT; style <= SCE_HPHP_OPERATOR; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_SH:
			for( style = SCE_SH_DEFAULT; style <= SCE_SH_HERE_Q; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_C:
			for( style = SCE_C_DEFAULT; style <= SCE_C_GLOBALCLASS; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
		case CSSED_FILETYPE_APACHE_CONF:
			for( style = SCE_CONF_DEFAULT; style <= SCE_CONF_DIRECTIVE; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_PERL:
			for( style = SCE_PL_DEFAULT; style <= SCE_PL_STRING_QW; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_PYTHON:
			for( style = SCE_P_DEFAULT; style <= SCE_P_STRINGEOL; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_XML:
			for( style = SCE_H_DEFAULT; style <= SCE_H_SGML_BLOCK_DEFAULT; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_DIFF:
			for( style = SCE_DIFF_DEFAULT; style <= SCE_DIFF_ADDED; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		case CSSED_FILETYPE_MAKE:
			for( style = SCE_MAKE_DEFAULT; style <= SCE_MAKE_DEFAULT; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
		default:
			for( style = 0; style <= STYLE_DEFAULT; style++){
				SSM (sci, SCI_STYLESETFONT, style,(sptr_t) font_name );
			}
			break;
	}
}

void
document_set_codepage( CssedDoc* doc, CssedCodePage codepage )
{
	ScintillaObject * sci;

	sci = SCINTILLA( doc->editor );
	switch( codepage ){
		case CSSED_CODEPAGE_DEFAULT:
			SSM( sci, SCI_SETCODEPAGE, 0, 0);
			break;
		case CSSED_CODEPAGE_UTF8:
			SSM( sci, SCI_SETCODEPAGE, SC_CP_UTF8, 0);
			break;
		case CSSED_CODEPAGE_DBCS:
			SSM( sci, SCI_SETCODEPAGE, SC_CP_DBCS, 0);
			break;
	}
}

CssedCodePage
document_get_codepage( CssedDoc* doc )
{
	ScintillaObject * sci;
	gint scintilla_codepage;

	sci = SCINTILLA( doc->editor );

	scintilla_codepage = SSM( sci,  SCI_GETCODEPAGE, 0,0);

	switch( scintilla_codepage ){
		case 0:
			return CSSED_CODEPAGE_DEFAULT;
		case SC_CP_UTF8:
			return CSSED_CODEPAGE_UTF8;
		case SC_CP_DBCS:
			return CSSED_CODEPAGE_DBCS;
	}
	return CSSED_CODEPAGE_DEFAULT;
}

void
document_set_character_set( CssedDoc* doc, gint charset )
{
	ScintillaObject * sci;
	CssedWindow* window;
	gint style;
	window = CSSED_WINDOW(doc->window);
	sci = SCINTILLA( doc->editor );

	switch( document_get_filetype( doc ) ){
		case CSSED_FILETYPE_CSS:
			for( style = SCE_CSS_DEFAULT; style <= SCE_CSS_LANGUAGE; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_HTML:
			for( style = SCE_H_DEFAULT; style <= SCE_HPHP_OPERATOR; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_SH:
			for( style = SCE_SH_DEFAULT; style <= SCE_SH_HERE_Q; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_C:
			for( style = SCE_C_DEFAULT; style <= SCE_C_GLOBALCLASS; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
		case CSSED_FILETYPE_APACHE_CONF:
			for( style = SCE_CONF_DEFAULT; style <= SCE_CONF_DIRECTIVE; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_PERL:
			for( style = SCE_PL_DEFAULT; style <= SCE_PL_STRING_QW; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_PYTHON:
			for( style = SCE_P_DEFAULT; style <= SCE_P_STRINGEOL; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_XML:
			for( style = SCE_H_DEFAULT; style <= SCE_H_SGML_BLOCK_DEFAULT; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_DIFF:
			for( style = SCE_DIFF_DEFAULT; style <= SCE_DIFF_ADDED; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		case CSSED_FILETYPE_MAKE:
			for( style = SCE_MAKE_DEFAULT; style <= SCE_MAKE_DEFAULT; style++){
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
		default:
			for( style = 0; style <= STYLE_DEFAULT; style++){ // default to ANSI
				SSM (sci, SCI_STYLESETCHARACTERSET, style, charset );
			}
			break;
	}
}

void
document_set_font(CssedDoc* doc)
{
	CssedWindow* window;
	CssedConfig* cfg;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	document_set_font_by_name(doc, cfg->font);
}

void
document_set_current_font_size(CssedDoc* doc)
{
	CssedWindow* window;
	CssedConfig* cfg;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	document_set_font_size(doc, cfg->font_size);
}

void
document_set_font_size(CssedDoc* doc, gint size)
{
	ScintillaObject * sci;
	CssedWindow* window;
	gint style;
	window = CSSED_WINDOW(doc->window);
	sci = SCINTILLA( doc->editor );

	switch( document_get_filetype( doc ) ){
		case CSSED_FILETYPE_TEXT:
			for( style = 0; style <= STYLE_DEFAULT; style++){
				SSM (sci, SCI_STYLESETSIZE, style, size);
			}
			break;
		case CSSED_FILETYPE_CSS:
			for( style = SCE_CSS_DEFAULT; style <= SCE_CSS_LANGUAGE; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_HTML:
			for( style = SCE_H_DEFAULT; style <= SCE_HPHP_OPERATOR; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
		case CSSED_FILETYPE_SH:
			for( style = SCE_SH_DEFAULT; style <= SCE_SH_HERE_Q; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_C:
			for( style = SCE_C_DEFAULT; style <= SCE_C_GLOBALCLASS; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_APACHE_CONF:
			for( style = SCE_CONF_DEFAULT; style <= SCE_CONF_DIRECTIVE; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_PERL:
			for( style = SCE_PL_DEFAULT; style <= SCE_PL_STRING_QW; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_XML:
			for( style = SCE_H_DEFAULT; style <= SCE_H_SGML_BLOCK_DEFAULT; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_PYTHON:
			for( style = SCE_P_DEFAULT; style <= SCE_P_STRINGEOL; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_DIFF:
			for( style = SCE_DIFF_DEFAULT; style <= SCE_DIFF_ADDED; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		case CSSED_FILETYPE_MAKE:
			for( style = SCE_MAKE_DEFAULT; style <= SCE_MAKE_IDEOL; style++){
				SSM ( sci, SCI_STYLESETSIZE, style, size );
			}
			break;
		default:
			for( style = 0; style <= STYLE_DEFAULT; style++){
				SSM (sci, SCI_STYLESETSIZE, style, size);
			}
			break;
	}
}


void
document_set_default_configuration( CssedDoc* doc )
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );

	sci = SCINTILLA( doc->editor );
	SSM (sci, SCI_SETTABWIDTH, 4, 0);
	SSM (sci, SCI_SETREADONLY, 0, 0);
	SSM (sci, SCI_USEPOPUP, FALSE, 0);
	SSM (sci, SCI_SETMARGINTYPEN, 0, SC_MARGIN_NUMBER);
	SSM (sci, SCI_SETMARGINTYPEN, 1, SC_MARGIN_SYMBOL);
	SSM (sci, SCI_SETMARGINWIDTHN, 1, MARGIN_MARKERS_WIDTH_PIXELS); // always opened
	SSM (sci, SCI_SETMARGINSENSITIVEN,1, TRUE);
	SSM (sci, SCI_MARKERDEFINE,0, SC_MARK_ROUNDRECT); // marker 0 for bookmarks
	SSM (sci, SCI_MARKERDEFINE,1, SC_MARK_ARROWS); // marker 1 for validator
	SSM (sci, SCI_MARKERSETFORE,1, 0x0000ff);
	SSM (sci,SCI_SETUNDOCOLLECTION,1,0);
	SSM (sci,SCI_SETMODEVENTMASK, 	SC_MOD_INSERTTEXT | SC_MOD_DELETETEXT |
									SC_PERFORMED_UNDO |  SC_PERFORMED_REDO |
									SC_LASTSTEPINUNDOREDO ,
									0);
									/* SC_MOD_CHANGESTYLE | SC_MOD_CHANGEFOLD | */
									/* SC_MOD_CHANGEMARKER | SC_MOD_BEFOREINSERT  */
									/* SC_MOD_BEFOREDELETE */
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.comment", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.compact", (sptr_t) "1");
 	SSM (sci, SCI_SETFOLDFLAGS, 16,0);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDEROPEN, SC_MARK_BOXMINUS);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDEROPEN, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDEROPEN, 0x000000);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDER, SC_MARK_BOXPLUS);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDER, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDER, 0x000000);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDERSUB, SC_MARK_VLINE);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDERSUB, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDERSUB, 0x000000);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDERTAIL, SC_MARK_LCORNER);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDERTAIL, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDERTAIL, 0x000000);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDEREND, SC_MARK_BOXPLUSCONNECTED);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDEREND, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDEREND, 0x000000);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDEROPENMID, SC_MARK_BOXMINUSCONNECTED);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDEROPENMID, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDEROPENMID, 0x000000);
    SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDERMIDTAIL, SC_MARK_TCORNER);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDERMIDTAIL, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDERMIDTAIL, 0x000000);
	SSM (sci, SCI_SETMARGINSENSITIVEN,2, TRUE);
	SSM (sci, SCI_SETMARGINMASKN,2, SC_MASK_FOLDERS);

	if( cfg->folding ){
		SSM (sci, SCI_SETMARGINWIDTHN, 2, MARGIN_FOLDING_WIDTH_PIXELS );
	}else{
		SSM (sci, SCI_SETMARGINWIDTHN, 2, 0 );
	}

	if( cfg->linenumbers ){
		SSM (sci, SCI_SETMARGINWIDTHN, 0,
			SSM(sci, SCI_TEXTWIDTH, STYLE_LINENUMBER,(sptr_t) MARGIN_LINES_WIDTH_STRING)
			);
	}else{
		SSM (sci, SCI_SETMARGINWIDTHN, 0, 0 );
	}

	if( cfg->lineendings ){
		sci_set_visible_eols( sci, TRUE );
	}

	if( cfg->whitespaces ){
		sci_set_visible_white_spaces(sci, TRUE);
	}

	if( cfg->lineswraped ){
		sci_set_lines_wrapped( sci, TRUE );
	}

	if( cfg->autocompletion ){
		document_set_autocompletion_enabled( doc, TRUE );
	}
}

/* ****************** TESTING NEW FILE TYPES HERE ************************** */
void
document_set_html_style( CssedDoc* doc )
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;
	gint bstyle;
	gint bastyle;
	gint jstyle;
	gint jastyle;
	gint hpstyle;

	const gchar htmlKeyWords[] =
		"a abbr acronym address applet area b base basefont bdo big blockquote body br button caption center cite code col colgroup dd del dfn dir div dl dt em embed fieldset font form frame frameset h1 h2 h3 h4 h5 h6 head hr html i iframe img input ins isindex kbd label legend li link map menu meta noframes noscript object ol optgroup option p param pre q quality s samp script select small span strike strong style sub sup table tbody td textarea tfoot th thead title tr tt u ul var xmlns leftmargin topmargin abbr accept-charset accept accesskey action align alink alt archive axis background bgcolor border cellpadding cellspacing char charoff charset checked cite class classid clear codebase codetype color cols colspan compact content coords data datafld dataformatas datapagesize datasrc datetime declare defer dir disabled enctype face for frame frameborder selected headers height href hreflang hspace http-equiv id ismap label lang language link longdesc marginwidth marginheight maxlength media framespacing method multiple name nohref noresize noshade nowrap object onblur onchange onclick ondblclick onfocus onkeydown onkeypress onkeyup onload onmousedown onmousemove onmouseover onmouseout onmouseup onreset onselect onsubmit onunload profile prompt pluginspage readonly rel rev rows rowspan rules scheme scope scrolling shape size span src standby start style summary tabindex target text title type usemap valign value valuetype version vlink vspace width text password checkbox radio submit reset file hidden image public !doctype xml";

	const gchar jsKeyWords[] =
		"break this for while null else var false void new delete typeof if in continue true function with return case super extends do const try debugger catch switch finally enum export default class throw import length concat join pop push reverse shift slice splice sort unshift Date Infinity NaN undefined escape eval isFinite isNaN Number parseFloat parseInt string unescape Math abs acos asin atan atan2 ceil cos exp floor log max min pow random round sin sqrt tan MAX_VALUE MIN_VALUE NEGATIVE_INFINITY POSITIVE_INFINITY toString valueOf String length anchor big bold charAt charCodeAt concat fixed fontcolor fontsize fromCharCode indexOf italics lastIndexOf link slice small split strike sub substr substring sup toLowerCase toUpperCase";

	const gchar vbsKeyWords[] =
		"and as byref byval case call const continue dim do each else elseif end error exit false for function global goto if in loop me new next not nothing on optional or private public redim rem resume select set sub then to true type while with boolean byte currency date double integer long object single string type variant";

	const gchar pyKeyWords[] = 	"and assert break class continue complex def del elif else except exec finally for from global if import in is int lambda not or pass print raise return tuple try unicode while yield long float str list";


	const gchar phpKeyWords[] = "and or xor FILE exception php_user_filter LINE array as break case cfunction class const continue declare default die do echo else elseif empty enddeclare endfor endforeach endif endswitch endwhile eval exit extends for foreach function global if include include_once isset list new old_function print require require_once return static switch unset use var while FUNCTION CLASS	METHOD";

	const gchar sgmlKeyWords[] = "DOCTYPE";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.html", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.html.preprocessor", (sptr_t) "1");

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_HTML, 0);
	SSM (sci,SCI_SETSTYLEBITS, 7, 0);

	SSM(sci, SCI_SETKEYWORDS, 0, (sptr_t) htmlKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 1, (sptr_t) jsKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 2, (sptr_t) vbsKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 3, (sptr_t) pyKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 4, (sptr_t) phpKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 5, (sptr_t) sgmlKeyWords);

	// Unknown tags and attributes are highlighed in red.
	// If a tag is actually OK, it should be added in lower case to the htmlKeyWords string.

	SSM(sci, SCI_STYLESETFORE, SCE_H_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_TAG, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_TAG, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_TAGUNKNOWN, 0x0000ff);
	SSM(sci, SCI_STYLESETBACK, SCE_H_TAGUNKNOWN, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ATTRIBUTE, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_ATTRIBUTE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ATTRIBUTEUNKNOWN, 0x00ff);
	SSM(sci, SCI_STYLESETBACK, SCE_H_ATTRIBUTEUNKNOWN, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_NUMBER, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_DOUBLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SINGLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SINGLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_OTHER, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_OTHER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ENTITY, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_ENTITY, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_TAGEND, 0x800000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_TAGEND, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_XMLSTART, 0x990000);// <?
	SSM(sci, SCI_STYLESETBACK, SCE_H_XMLSTART, 0xf0f0f0);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_H_XMLSTART, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_H_XMLEND, 0x990000);// ?>
	SSM(sci, SCI_STYLESETBACK, SCE_H_XMLEND, 0xf0f0f0);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SCRIPT, 0x800000);// <script
	SSM(sci, SCI_STYLESETBACK, SCE_H_SCRIPT, 0xf0f0f0);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ASP, 0x4f4f00);	// <% ... %>
	SSM(sci, SCI_STYLESETBACK, SCE_H_ASP, 0xf0f0f0);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_H_ASP, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ASPAT, 0x4f4f00);	// <%@ ... %>
	SSM(sci, SCI_STYLESETBACK, SCE_H_ASPAT, 0xf0f0f0);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_H_ASPAT, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_H_CDATA, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_CDATA, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_QUESTION, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_QUESTION, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_VALUE, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_VALUE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_XCCOMMENT, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_XCCOMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_DEFAULT, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_COMMAND, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_COMMAND, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_1ST_PARAM, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_1ST_PARAM, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_DOUBLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_SIMPLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_SIMPLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_ERROR, 0x000080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_ERROR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_SPECIAL, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_SPECIAL, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_ENTITY, 0x808060);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_ENTITY, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_COMMENT, 0x202020);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_1ST_PARAM_COMMENT, 0x202020);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_1ST_PARAM_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_BLOCK_DEFAULT, 0x808020);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_BLOCK_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HB_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HB_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HB_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HB_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HB_NUMBER, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_HB_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HB_WORD, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_HB_WORD, 0xffffff);
	SSM(sci, SCI_STYLESETBOLD, SCE_HB_WORD, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_HB_STRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_HB_STRING, 0x008000);

	SSM(sci, SCI_STYLESETFORE, SCE_HB_IDENTIFIER, 0x103000);
	SSM(sci, SCI_STYLESETBACK, SCE_HB_IDENTIFIER, 0xffffff);
//~ #define SCE_HB_START 70

	// Show the whole section of VBScript
	for (bstyle=SCE_HB_DEFAULT; bstyle<=SCE_HB_STRINGEOL; bstyle++) {
		SSM(sci, SCI_STYLESETBACK, bstyle, 0xf5f5f5);
		// This call extends the backround colour of the last style on the line to the edge of the window
		SSM(sci, SCI_STYLESETEOLFILLED, bstyle, 1);
	}
	SSM(sci,SCI_STYLESETBACK, SCE_HB_STRINGEOL, 0x7F7FFF);

	SSM(sci, SCI_STYLESETFORE, SCE_HBA_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HBA_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HBA_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HBA_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HBA_NUMBER, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_HBA_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HBA_WORD, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_HBA_WORD, 0xffffff);
	SSM(sci, SCI_STYLESETBOLD, SCE_HBA_WORD, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_HBA_STRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_HBA_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HBA_IDENTIFIER, 0x103000);
	SSM(sci, SCI_STYLESETBACK, SCE_HBA_IDENTIFIER, 0xffffff);

	// Show the whole section of ASP VBScript
	for (bastyle=SCE_HBA_DEFAULT; bastyle<=SCE_HBA_STRINGEOL; bastyle++) {
		SSM(sci, SCI_STYLESETBACK, bastyle, 0xf5f5f5);
		// This call extends the backround colour of the last style on the line to the edge of the window
		SSM(sci, SCI_STYLESETEOLFILLED, bastyle, 1);
	}
	//SSM(sci, SCI_STYLESETBACK, SCE_HBA_STRINGEOL, 0xCFCF7F);

//~ #define SCE_HBA_START 80

	// If there is no need to support embedded Javascript, the following code can be dropped.
	// Javascript will still be correctly processed but will be displayed in the default style.
	SSM(sci, SCI_STYLESETFORE, SCE_HJ_START, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_START, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_COMMENT, 0x222222);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_COMMENTDOC, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_COMMENTDOC, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_NUMBER, 0x606000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_WORD, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_KEYWORD, 0x001050);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_KEYWORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_DOUBLESTRING, 0x080080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_DOUBLESTRING, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJ_SYMBOLS, 0x501000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_SYMBOLS, 0xffffff);

//~ #define SCE_HJ_REGEX 52
	SSM(sci, SCI_STYLESETFORE, SCE_HJA_START, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_START, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_COMMENT, 0x606060);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_COMMENTDOC, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_COMMENTDOC, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_NUMBER, 0x408000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_WORD, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_DOUBLESTRING, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_SINGLESTRING, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_SINGLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_SYMBOLS, 0x005000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_SYMBOLS, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HJA_KEYWORD, 0x005000);
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_KEYWORD, 0xffffff);

	// Show the whole section of Javascript with silver background
	for (jstyle=SCE_HJ_DEFAULT; jstyle<=SCE_HJ_SYMBOLS; jstyle++) {
		SSM(sci, SCI_STYLESETBACK, jstyle, 0xf5f5f5);
		SSM(sci, SCI_STYLESETEOLFILLED, jstyle, 1);
	}
	SSM(sci, SCI_STYLESETBACK, SCE_HJ_STRINGEOL, 0xDFDF7F);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_HJ_STRINGEOL, 1);

	// Show the whole section of Javascript with silver background
	for (jastyle=SCE_HJA_DEFAULT; jastyle<=SCE_HJA_SYMBOLS; jastyle++) {
		SSM(sci, SCI_STYLESETBACK, jastyle, 0xf5f5f5);
		SSM(sci, SCI_STYLESETEOLFILLED, jastyle, 1);
	}
	SSM(sci, SCI_STYLESETBACK, SCE_HJA_STRINGEOL, 0x00AF5F);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_HJA_STRINGEOL, 1);
//~ #define SCE_HJA_REGEX 67
	SSM(sci, SCI_STYLESETFORE, SCE_HP_START, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_START, 0xf0f0f0);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_HP_START, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_NUMBER, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_WORD, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_STRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_CHARACTER, 0x006060);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_CHARACTER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_TRIPLE, 0x002060);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_TRIPLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_TRIPLEDOUBLE, 0x002060);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_TRIPLEDOUBLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_CLASSNAME, 0x202010);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_CLASSNAME, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_CLASSNAME, 0x102020);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_CLASSNAME, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_OPERATOR, 0x602020);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_IDENTIFIER, 0x001060);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_START, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_START, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_NUMBER, 0x408000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_STRING, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_CHARACTER, 0x505080);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_CHARACTER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_WORD, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_TRIPLE, 0x002060);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_TRIPLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_TRIPLEDOUBLE, 0x002060);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_TRIPLEDOUBLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_CLASSNAME, 0x202010);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_CLASSNAME, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_DEFNAME, 0x102020);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_DEFNAME, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_OPERATOR, 0x601010);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPA_IDENTIFIER, 0x105010);
	SSM(sci, SCI_STYLESETBACK, SCE_HPA_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HP_NUMBER, 0x408000);
	SSM(sci, SCI_STYLESETBACK, SCE_HP_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_SIMPLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_SIMPLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_HSTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_HSTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_NUMBER, 0x006060);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_WORD, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_VARIABLE, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_VARIABLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_OPERATOR, 0x602010);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_HSTRING_VARIABLE, 0x601010);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_HSTRING_VARIABLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_HPHP_COMPLEX_VARIABLE, 0x105010);
	SSM(sci, SCI_STYLESETBACK, SCE_HPHP_COMPLEX_VARIABLE, 0xffffff);

	// Show the whole section of php with silver bg
	for (hpstyle=SCE_HP_START; hpstyle<=SCE_HPHP_OPERATOR; hpstyle++) {
		SSM(sci, SCI_STYLESETBACK, hpstyle, 0xf5f5f5);
		SSM(sci, SCI_STYLESETEOLFILLED, hpstyle, 1);
	}
}

void
document_set_sh_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	const gchar shKeyWords[] = "[ ] test function bash alias bg bind break builtin cd command compgen \
complete continue declare dirs disown echo  enable  eval  exec \
exit  export  for fc  fg getopts hash help history in jobs kill let \
local logout popd printf pushd pwd read readonly  return  set \
shift  shopt  source  suspend  test  times  trap  type typeset \
ulimit umask unalias unset wait if then else while do case done in esac fi";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_BASH, 0);

	SSM(sci, SCI_SETKEYWORDS, 0, (sptr_t) shKeyWords);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_ERROR, 0x0000ff);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_ERROR, 0x0000ff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_NUMBER, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_WORD, 0x800000);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_STRING, 0x404000);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_CHARACTER, 0x004040);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_CHARACTER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_OPERATOR, 0x004000);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_IDENTIFIER, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_SCALAR, 0x204070);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_SCALAR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_PARAM, 0x704020);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_PARAM, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_BACKTICKS, 0x205020);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_BACKTICKS, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_HERE_DELIM, 0x601060);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_HERE_DELIM, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_SH_HERE_Q, 0x303090);
	SSM(sci, SCI_STYLESETBACK, SCE_SH_HERE_Q, 0xffffff);
}

void
document_set_c_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	const gchar primaryKeyWords[] = "typedef struct char int float double void unsigned signed long short extern static register auto const if else switch for while do break continue return";

	const gchar secondaryKeyWords[] = "gconstpointer gchar guchar gint guint gint8 gint16 \
gint32 gint64 gshort gushort glong gulong gfloat gdouble gboolean gpointer NULL FALSE TRUE \
GObject GtkAccelGroup GtkObject GtkWidget GtkMisc GtkLabel GtkAccelLabel GtkTipsQuery GtkArrow \
GtkImage GtkPixmap GtkContainer GtkBin GtkAlignment GtkFrame GtkAspectFrame GtkButton \
GtkToggleButton GtkCheckButton GtkRadioButton GtkOptionMenu GtkItem GtkMenuItem GtkCheckMenuItem \
GtkRadioMenuItem GtkImageMenuItem GtkSeparatorMenuItem GtkTearoffMenuItem GtkListItem GtkTreeItem \
GtkWindow GtkDialog GtkColorSelectionDialog GtkFileSelection GtkFontSelectionDialog \
GtkInputDialog GtkMessageDialog GtkPlug GtkEventBox GtkHandleBox GtkScrolledWindow GtkViewport \
GtkBox GtkButtonBox GtkHButtonBox GtkVButtonBox GtkVBox GtkColorSelection GtkFontSelection \
GtkGammaCurve GtkHBox GtkCombo GtkStatusbar GtkCList GtkCTree GtkFixed GtkPaned GtkHPaned \
GtkVPaned GtkLayout GtkList GtkMenuShell GtkMenuBar GtkMenu GtkNotebook GtkSocket GtkTable \
GtkTextView GtkToolbar GtkTree GtkTreeView GtkCalendar GtkDrawingArea GtkCurve GtkEntry \
GtkSpinButton GtkRuler GtkHRuler GtkVRuler GtkRange GtkScale GtkHScale GtkVScale GtkScrollbar \
GtkHScrollbar GtkVScrollbar GtkSeparator GtkHSeparator GtkVSeparator GtkInvisibleGtkOldEditable \
GtkText GtkPreview GtkProgress GtkProgressBar" "GtkAdjustment GtkCellRenderer \
GtkCellRendererPixbuf GtkCellRendererText GtkCellRendererToggle GtkItemFactory GtkTooltips \
GtkTreeViewColumn AtkObject GtkAccessible GtkIconFactory GtkIMContext GtkIMContextSimple \
GtkIMMulticontext GtkListStore GtkRcStyle GtkSettings GtkSizeGroup GtkStyle GtkTextBuffer \
GtkTextChildAnchor GtkTextMark GtkTextTag GtkTextTagTable GtkTreeModelSort GtkTreeSelection \
GtkTreeStore GtkWindowGroup GdkDragContext GdkScreen GdkPixbuf GdkDrawable GdkPixmap GdkImage \
GdkPixbufAnimation GdkDevice GInterface AtkImplementorIface GtkEditable GtkCellEditable \
GtkTreeModel GtkTreeDragSource GtkTreeDragDest GtkTreeSortable GdkPixbuf GdkPixbufAnimation \
GtkTreeModel GtkTreeSelection GtkTreeViewColumn \
GtkTreeView GtkTreeView GtkTreeSortable GtkTreeModelSort GtkCellRenderer GtkCellEditable \
GtkCellRendererPixbuf GtkCellRendererText GtkCellRendererToggle GtkListStore GtkTreeStore \
GtkTreeViewColumnSizingGtkTreeCellDataFunc GtkTreePath" ;

	const gchar docCommentKeyWords[] = "cssed";

	const gchar typedefsKeyWords[] = "GtkAccelGroup GtkObject GTK_OBJECT_TYPE GTK_OBJECT_TYPE_NAME \
GtkObjectFlags GTK_OBJECT_FLAGS GTK_OBJECT_FLOATING GtkArgFlags GtkWidgetClass GtkWidgetFlags \
GTK_WIDGET_TYPE GTK_WIDGET_STATE GTK_WIDGET_SAVED_STATE GTK_WIDGET_FLAGS GTK_WIDGET_TOPLEVEL \
GTK_WIDGET_NO_WINDOW GTK_WIDGET_REALIZED GTK_WIDGET_MAPPED GTK_WIDGET_VISIBLE GTK_WIDGET_DRAWABLE \
GTK_WIDGET_SENSITIVE GTK_WIDGET_PARENT_SENSITIVE GTK_WIDGET_IS_SENSITIVE GTK_WIDGET_CAN_FOCUS \
GTK_WIDGET_HAS_FOCUS GTK_WIDGET_CAN_DEFAULT GTK_WIDGET_RECEIVES_DEFAULT GTK_WIDGET_HAS_DEFAULT \
GTK_WIDGET_HAS_GRAB GTK_WIDGET_RC_STYLE GTK_WIDGET_COMPOSITE_CHILD GTK_WIDGET_APP_PAINTABLE \
GTK_WIDGET_DOUBLE_BUFFERED GTK_WIDGET_SET_FLAGS GTK_WIDGET_UNSET_FLAGS GtkCallback \
GtkRequisition GtkAllocation GtkSelectionData GtkWidgetAuxInfo GtkWidgetShapeInfo \
GtkWidgetHelpType G_CONST_RETURN GtkImageType GTK_IS_RESIZE_CONTAINER \
GTK_CONTAINER_WARN_INVALID_CHILD_PROPERTY_ID GtkDialogFlags GtkResponseType GtkBoxChild \
GTK_BUTTONBOX_DEFAULT GTK_TREE_VIEW_COLUMN_GROW_ONLY GTK_TREE_VIEW_COLUMN_AUTOSIZE \
GTK_TREE_VIEW_COLUMN_FIXED GtkTreeViewDropPosition GtkTreeViewPrivate GtkTreeViewColumnDropFunc \
GtkTreeViewMappingFunc GtkTreeViewSearchEqualFunc GTK_STOCK_ADD GTK_STOCK_APPLY GTK_STOCK_BOLD \
GTK_STOCK_CANCEL GTK_STOCK_CDROM GTK_STOCK_CLEAR GTK_STOCK_CLOSE GTK_STOCK_COLOR_PICKER \
GTK_STOCK_CONVERT GTK_STOCK_COPY GTK_STOCK_CUT GTK_STOCK_DELETE GTK_STOCK_DIALOG_AUTHENTICATION \
GTK_STOCK_DIALOG_ERROR GTK_STOCK_DIALOG_INFO GTK_STOCK_DIALOG_QUESTION GTK_STOCK_DIALOG_WARNING \
GTK_STOCK_DND GTK_STOCK_DND_MULTIPLE GTK_STOCK_EXECUTE GTK_STOCK_FIND GTK_STOCK_FIND_AND_REPLACE \
GTK_STOCK_FLOPPY GTK_STOCK_GOTO_BOTTOM GTK_STOCK_GOTO_FIRST GTK_STOCK_GOTO_LAST \
GTK_STOCK_GOTO_TOP GTK_STOCK_GO_BACK GTK_STOCK_GO_DOWN GTK_STOCK_GO_FORWARD GTK_STOCK_GO_UP \
GTK_STOCK_HARDDISK GTK_STOCK_HELP GTK_STOCK_HOME GTK_STOCK_INDENT GTK_STOCK_INDEX \
GTK_STOCK_ITALIC GTK_STOCK_JUMP_TO GTK_STOCK_JUSTIFY_CENTER GTK_STOCK_JUSTIFY_FILL \
GTK_STOCK_JUSTIFY_LEFT GTK_STOCK_JUSTIFY_RIGHT GTK_STOCK_MISSING_IMAGE GTK_STOCK_NETWORK " "\
GTK_STOCK_NEW GTK_STOCK_NO GTK_STOCK_OK GTK_STOCK_OPEN GTK_STOCK_PASTE GTK_STOCK_PREFERENCES \
GTK_STOCK_PRINT GTK_STOCK_PRINT_PREVIEW GTK_STOCK_PROPERTIES GTK_STOCK_QUIT GTK_STOCK_REDO \
GTK_STOCK_REFRESH GTK_STOCK_REMOVE GTK_STOCK_REVERT_TO_SAVED GTK_STOCK_SAVE GTK_STOCK_SAVE_AS \
GTK_STOCK_SELECT_COLOR GTK_STOCK_SELECT_FONT GTK_STOCK_SORT_ASCENDING GTK_STOCK_SORT_DESCENDING \
GTK_STOCK_SPELL_CHECK GTK_STOCK_STOP GTK_STOCK_STRIKETHROUGH GTK_STOCK_UNDELETE \
GTK_STOCK_UNDERLINE GTK_STOCK_UNDO GTK_STOCK_UNINDENT GTK_STOCK_YES GTK_STOCK_ZOOM_100 \
GTK_STOCK_ZOOM_FIT GTK_STOCK_ZOOM_IN GTK_STOCK_ZOOM_OUT";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.preprocessor", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.at.else", (sptr_t) "1");

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_CPP, 0);

	SSM(sci, SCI_SETKEYWORDS, 0, (sptr_t) primaryKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 1, (sptr_t) secondaryKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 2, (sptr_t) docCommentKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 4, (sptr_t) typedefsKeyWords);

	SSM(sci, SCI_STYLESETFORE, SCE_C_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_C_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_C_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_C_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_COMMENTDOC, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_C_COMMENTDOC, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_NUMBER, 0x408000);
	SSM(sci, SCI_STYLESETBACK, SCE_C_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_WORD, 0x802080);
	SSM(sci, SCI_STYLESETBACK, SCE_C_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_WORD2, 0x800020);
	SSM(sci, SCI_STYLESETBACK, SCE_C_WORD2, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_STRING, 0x404000);
	SSM(sci, SCI_STYLESETBACK, SCE_C_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_CHARACTER, 0x004040);
	SSM(sci, SCI_STYLESETBACK, SCE_C_CHARACTER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_UUID, 0x804040);
	SSM(sci, SCI_STYLESETBACK, SCE_C_UUID, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_PREPROCESSOR, 0x804060);
	SSM(sci, SCI_STYLESETBACK, SCE_C_PREPROCESSOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_OPERATOR, 0x101030);
	SSM(sci, SCI_STYLESETBACK, SCE_C_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_IDENTIFIER, 0x100000);
	SSM(sci, SCI_STYLESETBACK, SCE_C_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_STRINGEOL, 0x104010);
	SSM(sci, SCI_STYLESETBACK, SCE_C_STRINGEOL, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_VERBATIM, 0x906040);
	SSM(sci, SCI_STYLESETBACK, SCE_C_VERBATIM, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_REGEX, 0x905010);
	SSM(sci, SCI_STYLESETBACK, SCE_C_REGEX, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_COMMENTLINEDOC, 0x005020);
	SSM(sci, SCI_STYLESETBACK, SCE_C_COMMENTLINEDOC, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_COMMENTDOCKEYWORD, 0x400010);
	SSM(sci, SCI_STYLESETBACK, SCE_C_COMMENTDOCKEYWORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_COMMENTDOCKEYWORDERROR, 0x001020);
	SSM(sci, SCI_STYLESETBACK, SCE_C_COMMENTDOCKEYWORDERROR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_C_GLOBALCLASS, 0x109040);
	SSM(sci, SCI_STYLESETBACK, SCE_C_GLOBALCLASS, 0xffffff);
}
//Apache configuration files
void
document_set_conf_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	const gchar directiveKeyWords[] = "acceptfilter acceptmutex accessconfig accessfilename \
action addalt addaltbyencoding addaltbytype addcharset adddefaultcharset adddescription \
addencoding addhandler addicon addiconbyencoding addiconbytype addlanguage addmodule \
addmoduleinfo addtype agentlog alias aliasmatch allow allowconnect allowoverride \
anonymous anonymous_authoritative anonymous_logemail anonymous_mustgiveemail \
anonymous_nouserid anonymous_verifyemail authauthoritative authdbauthoritative \
authdbgroupfile authdbmauthoritative authdbmgroupfile authdbmgroupfile authdbuserfile \
authdbmuserfile authdigestfile authgroupfile authname authtype authuserfile bindaddress \
browsermatch browsermatchnocase bs2000account cachedefaultexpire cachedirlength cachedirlevels \
cacheforcecompletion cachegcinterval cachelastmodifiedfactor cachemaxexpire cachenegotiateddocs \
cacheroot cachesize cgicommandargs checkspelling clearmodulelist contentdigest cookiedomain \
cookieexpires cookieformat cookielog cookielog cookieprefix cookiestyle cookietracking \
coredumpdirectory customlog defaulticon defaultlanguage defaulttype deny directory \
directorymatch directoryindex documentroot ebcdicconvert ebcdicconvertbytype ebcdickludge \
enableexceptionhook errordocument errorheader errorlog example expiresactive expiresbytype \
expiresdefault extendedstatus fancyindexing fileetag files filesmatch forcetype forensiclog \
group header headername hostnamelookups identitycheck ifdefine ifmodule imapbase imapdefault \
imapmenu include indexignore indexoptions indexorderdefault isapireadaheadbuffer \
isapilognotsupported isapiappendlogtoerrors isapiappendlogtoquery keepalive keepalivetimeout \
languagepriority limit limitexcept limitinternalrecursion limitrequestbody limitrequestfields \
limitrequestfieldsize limitrequestline listen listenbacklog loadfile loadmodule location \
locationmatch lockfile logformat loglevel " "maxclients maxkeepaliverequests maxrequestsperchild \
maxspareservers metadir metafiles metasuffix mimemagicfile minspareservers mmapfile \
namevirtualhost nocache options order passenv pidfile port protocolreqcheck \
proxyblock proxydomain proxypass proxypassreverse proxyreceivebuffersize proxyremote \
proxyrequests proxyvia readmename redirect redirectmatch redirectpermanent redirecttemp \
refererignore refererlog removeencoding removehandler removetype require resourceconfig \
rewritebase rewritecond rewriteengine rewritelock rewritelog rewriteloglevel rewritemap \
rewriteoptions rewriterule rlimitcpu rlimitmem rlimitnproc satisfy scoreboardfile script \
scriptalias scriptaliasmatch scriptinterpretersource scriptlog scriptlogbuffer \
scriptloglength sendbuffersize serveradmin serveralias servername serverpath serverroot \
serversignature servertokens servertype setenv setenvif setenvifnocase sethandler \
shmemuidisuser startservers threadsperchild timeout transferlog typesconfig unsetenv \
usecanonicalname user userdir virtualhost virtualdocumentroot virtualdocumentrootip \
virtualscriptalias virtualscriptaliasip xbithack numservers startthreads minsparethreads \
maxsparethreads maxthreadsperchild addoutputfilter from";

	const gchar paramaterKeyWords[] = "on off uslock pthread sysvsem fcntl flock os2sem tpfcore none default all none basic digest double registry script on off email minimal productonly os full standalone dns indexes multiviews includes followsymlinks execcgi";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_CONF, 0);

	SSM(sci, SCI_SETKEYWORDS, 0, (sptr_t) directiveKeyWords);
	SSM(sci, SCI_SETKEYWORDS, 1, (sptr_t) paramaterKeyWords);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_NUMBER, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_IDENTIFIER, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_EXTENSION, 0x801060);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_EXTENSION, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_PARAMETER, 0x606000);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_PARAMETER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_STRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_OPERATOR, 0x801000);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_IP, 0x008040);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_IP, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_CONF_DIRECTIVE, 0x800000);
	SSM(sci, SCI_STYLESETBACK, SCE_CONF_DIRECTIVE, 0xffffff);
}
void
document_set_perl_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	const gchar plKeyWords[] = "if else elsif for while until loop foreach do label last use my sub package defined require bless shift chomp chop chr crypt hex index lc lcfirst length oct ord pack reverse rindex sprintf substr uc ucfirst pos quotemeta split study abs atan2 cos exp hex int log oct rand sin sqrt srandpop push shift splice unshift grep join map reverse sort unpack delete each exists keys values binmode close closedir dbmclose dbmopen die eof fileno flock format getc print printf read readdir rewinddir seek seekdir select syscall sysread sysseek syswrite tell telldir truncate warn write pack read syscall sysread syswrite unpack vec chdir chmod chown chroot fcntl glob ioctl link lstat mkdir open opendir readlink rename rmdir stat symlink sysopen umask unlink utime caller continue die do dump eval exit goto last next redo return sub wantarray caller import local my our package use defined dump eval formline local my our reset scalar undef wantarray alarm exec fork getpgrp getppid getpriority kill pipe setpgrp setpriority sleep system times wait waitpid do import no package require use bless dbmclose dbmopen package ref tie tied untie use accept bind connect getpeername getsockname getsockopt listen recv send setsockopt shutdown socket socketpair msgctl msgget msgrcv msgsnd semctl semget semop shmctl shmget shmread shmwrite endgrent endhostent endnetent endpwent getgrent getgrgid getgrnam getlogin getpwent getpwnam getpwuid setgrent setpwent endprotoent endservent gethostbyaddr gethostbyname gethostent getnetbyaddr getnetbyname getnetent getprotobyname getprotobynumber getprotoent getservbyname getservbyport getservent sethostent setnetent setprotoent setservent gmtime localtime time times abs bless chomp chr exists formline glob import lc lcfirst map my no our prototype qx qw readline readpipe ref sub sysopen tie tied uc ucfirst untie us";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_PERL, 0);

	SSM(sci, SCI_SETKEYWORDS, 0, (sptr_t) plKeyWords);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_ERROR, 0x0000ff);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_ERROR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_POD, 0x004080);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_POD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_NUMBER, 0x800040);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_WORD, 0x800000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_STRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_CHARACTER, 0x006000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_CHARACTER, 0xffffff);
	SSM(sci, SCI_STYLESETFORE, SCE_PL_PUNCTUATION, 0x804000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_PUNCTUATION, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_PREPROCESSOR, 0x606000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_PREPROCESSOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_OPERATOR, 0x101010);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_IDENTIFIER, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_SCALAR, 0x800050);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_SCALAR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_ARRAY, 0x001050);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_ARRAY, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_HASH, 0x000080);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_HASH, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_SYMBOLTABLE, 0x106020);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_SYMBOLTABLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_REGEX, 0x404000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_REGEX, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_REGSUBST, 0x404000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_REGSUBST, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_LONGQUOTE, 0x601030);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_LONGQUOTE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_BACKTICKS, 0x309090);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_BACKTICKS, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_DATASECTION, 0x104000);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_DATASECTION, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_HERE_DELIM, 0x007020);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_HERE_DELIM, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_HERE_Q, 0x004040);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_HERE_Q, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_HERE_QQ, 0x104040);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_HERE_QQ, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_HERE_QX, 0x204040);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_HERE_QX, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_STRING_Q, 0x008010);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_STRING_Q, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_STRING_QQ, 0x108010);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_STRING_QQ, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_STRING_QX, 0x208010);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_STRING_QX, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_STRING_QR, 0x308010);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_STRING_QR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_PL_STRING_QW, 0x408010);
	SSM(sci, SCI_STYLESETBACK, SCE_PL_STRING_QW, 0xffffff);
}

void
document_set_python_style( CssedDoc* doc )
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	const gchar pyKeyWords[] = "and assert break class continue def del elif else except exec finally for from global if import in is lambda not or pass print raise return try while yield";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.commnent.python", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.quotes.python", (sptr_t) "1");

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_PYTHON, 0);

	SSM(sci, SCI_SETKEYWORDS, 0, (sptr_t) pyKeyWords);

	SSM(sci, SCI_STYLESETFORE, SCE_P_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_COMMENTLINE, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_P_COMMENTLINE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_NUMBER, 0x800040);
	SSM(sci, SCI_STYLESETBACK, SCE_P_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_STRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_STRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_CHARACTER, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_CHARACTER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_WORD, 0x800060);
	SSM(sci, SCI_STYLESETBACK, SCE_P_WORD, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_TRIPLE, 0x208000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_TRIPLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_TRIPLEDOUBLE, 0x004040);
	SSM(sci, SCI_STYLESETBACK, SCE_P_TRIPLEDOUBLE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_CLASSNAME, 0x303000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_CLASSNAME, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_DEFNAME, 0x800000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_DEFNAME, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_OPERATOR, 0x800030);
	SSM(sci, SCI_STYLESETBACK, SCE_P_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_IDENTIFIER, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_COMMENTBLOCK, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_P_COMMENTBLOCK, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_P_STRINGEOL, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_P_STRINGEOL, 0xffffff);
}

void
document_set_xml_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	const gchar sgmlKeyWords[] = "ELEMENT DOCTYPE ATTLIST ENTITY NOTATION";

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.html", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.html.preprocessor", (sptr_t) "1");

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_XML, 0);

	SSM(sci, SCI_SETKEYWORDS, 5, (sptr_t) sgmlKeyWords);

	SSM(sci, SCI_STYLESETFORE, SCE_H_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_TAG, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_TAG, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_TAGUNKNOWN, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_TAGUNKNOWN, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ATTRIBUTE, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_ATTRIBUTE, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ATTRIBUTEUNKNOWN, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_ATTRIBUTEUNKNOWN, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_NUMBER, 0x008040);
	SSM(sci, SCI_STYLESETBACK, SCE_H_NUMBER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_DOUBLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SINGLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SINGLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_OTHER, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_OTHER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_ENTITY, 0x800080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_ENTITY, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_TAGEND, 0x800000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_TAGEND, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_XMLSTART, 0x990000);// <?
	SSM(sci, SCI_STYLESETBACK, SCE_H_XMLSTART, 0xf0f0f0);
	SSM(sci, SCI_STYLESETEOLFILLED, SCE_H_XMLSTART, 1);

	SSM(sci, SCI_STYLESETFORE, SCE_H_XMLEND, 0x990000);// ?>
	SSM(sci, SCI_STYLESETBACK, SCE_H_XMLEND, 0xf0f0f0);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_DEFAULT, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_COMMAND, 0x990000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_COMMAND, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_1ST_PARAM, 0x808000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_1ST_PARAM, 0xf0f0f0);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_DOUBLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_DOUBLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_SIMPLESTRING, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_SIMPLESTRING, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_ERROR, 0x000080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_ERROR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_SPECIAL, 0x008080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_SPECIAL, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_ENTITY, 0x808060);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_ENTITY, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_COMMENT, 0xf0f0f0);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_1ST_PARAM_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_1ST_PARAM_COMMENT, 0xf0f0f0);

	SSM(sci, SCI_STYLESETFORE, SCE_H_SGML_BLOCK_DEFAULT, 0x808040);
	SSM(sci, SCI_STYLESETBACK, SCE_H_SGML_BLOCK_DEFAULT, 0xffffff);
}

void
document_set_diff_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_DIFF, 0);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_COMMENT, 0x400000);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_COMMAND, 0x606060);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_COMMAND, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_HEADER, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_HEADER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_POSITION, 0x008040);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_POSITION, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_DELETED, 0x0000ff);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_DELETED, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_DIFF_ADDED, 0x900000);
	SSM(sci, SCI_STYLESETBACK, SCE_DIFF_ADDED, 0xffffff);
}

void
document_set_make_style(CssedDoc* doc)
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_MAKEFILE, 0);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_DEFAULT, 0x000000);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_DEFAULT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_COMMENT, 0x808080);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_COMMENT, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_PREPROCESSOR, 0x606060);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_PREPROCESSOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_IDENTIFIER, 0x990066);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_IDENTIFIER, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_OPERATOR, 0x008040);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_OPERATOR, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_TARGET, 0x600000);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_TARGET, 0xffffff);

	SSM(sci, SCI_STYLESETFORE, SCE_MAKE_IDEOL, 0x008000);
	SSM(sci, SCI_STYLESETBACK, SCE_MAKE_IDEOL, 0xffffff);
}

/* null style means no style so it is applied to text filetype */
void
document_set_null_style( CssedDoc* doc )
{
	ScintillaObject * sci;
	CssedWindow* window;
	CssedConfig* cfg;
	gint style;

	window = CSSED_WINDOW(doc->window);
	cfg = cssed_window_get_config( window );
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_NULL, 0);

	for( style = 0; style <= STYLE_DEFAULT; style++){
		SSM(sci, SCI_STYLESETFORE, style, 0x000000);
		SSM(sci, SCI_STYLESETBACK, style, 0xffffff);
	}
}
/* ************************** TEST END ************************************* */

CssedDoc*
document_get_from_editor( CssedWindow* window, GtkWidget* editor )
{
	GList* tmplist;
	CssedDoc* tmpdoc;

	tmplist = cssed_window_get_document_list_first( window );
	while(tmplist){
		tmpdoc = (CssedDoc*) tmplist->data;
		if( editor == (GtkWidget*) tmpdoc->editor ){
			return tmpdoc;
		}else{
			tmplist = g_list_next(tmplist);
		}
	}

	return NULL;
}
CssedDoc*
document_get_from_notebook_page_num(CssedWindow* window, gint page_number)
{
	GtkWidget* editor;
	//CssedDoc* tmpdoc;
	GtkWidget* notebook;

	notebook = cssed_window_get_document_notebook( window );

	editor = gtk_notebook_get_nth_page (GTK_NOTEBOOK(notebook), page_number);
	return document_get_from_editor( window, editor );
}

CssedDoc*
document_get_current(CssedWindow* window)
{
	GtkWidget* notebook;
	CssedDoc* tmpdoc;
	gint current_index;

	notebook = cssed_window_get_document_notebook( window );

	current_index = gtk_notebook_get_current_page   ( GTK_NOTEBOOK(notebook) );
	tmpdoc = document_get_from_notebook_page_num(window, current_index);

	return tmpdoc;
}

void
document_set_current(CssedDoc* doc)
{
	CssedWindow* win;
	GtkWidget* notebook;
	GtkWidget* editor;
	gint doc_index;

	win = document_get_window( doc );
	notebook = cssed_window_get_document_notebook( win );
	editor = doc->editor;
	doc_index = gtk_notebook_page_num  ( GTK_NOTEBOOK(notebook), editor );

	if( doc_index > -1 ){
		gtk_notebook_set_current_page   (GTK_NOTEBOOK(notebook), doc_index);
	}
}

gboolean
document_get_modified( CssedDoc* doc )
{
	ScintillaObject * sci;
	gint modified;

	sci = (ScintillaObject *) sci_get_from_document(  doc );
	modified = SSM (sci, SCI_GETMODIFY, 0, 0);
	if( modified == 0) return FALSE;
	else return TRUE;
}

void// FIXME: it's not being used (?)
document_set_modified( CssedDoc* doc , gboolean modified)
{
	ScintillaObject * sci;

	if( modified ){
		doc->modified = TRUE; // FIXME: it's not being used (?)
	}else{
		sci = (ScintillaObject *) sci_get_from_document(  doc );
		SSM (sci, SCI_SETSAVEPOINT, 0, 0);
		doc->modified = FALSE; // FIXME: it's not being used (?)
	}
}

void
document_set_save_point( CssedDoc* doc )
{
	ScintillaObject * sci;
	CssedWindow* window;

	sci = (ScintillaObject *) sci_get_from_document(  doc );
	window = document_get_window( doc );

	SSM (sci, SCI_SETSAVEPOINT, 0, 0);
	cssed_window_disable_save( window );
}

/*  Detach the doc from the notebook , pops it out the list
	and clear all memory allocated. If last doc in queue it
	must open a clean one.
	Save before it or changes will be lost.
*/
gboolean
document_dispose( CssedDoc* doc )
{
	GtkWidget* notebook;
	GtkWidget* editor;
	CssedWindow* window;
	gint document_page_number;
	gint notebook_total_pages;
	gboolean needs_new = FALSE;

	notebook = doc->notebook;
	editor = doc->editor;

	document_page_number = gtk_notebook_page_num(GTK_NOTEBOOK(notebook),editor);

	if( document_page_number == -1 ){ // editor does not exist
		return FALSE;
	}else{
		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), document_page_number);
#ifdef GTK_IS_2_2
		notebook_total_pages = gtk_notebook_get_n_pages (GTK_NOTEBOOK(notebook));
#else
		notebook_total_pages = g_list_length(GTK_NOTEBOOK(notebook)->children);
#endif
		if( notebook_total_pages == 0 ){
			needs_new = TRUE;
			DBGMSG("document.c:document_dispose() - Last notebook page. Will create new one ...");
		}
		// clear doc allocated mem and pop out the doc list
		if( doc->filename != NULL ){
			g_free(doc->filename);
		}
		window = doc->window;
		cssed_window_delete_document_from_list( window ,doc );
		g_free(doc);
	}

	if( needs_new )	create_and_attach_new_doc (window,_("Untitled"));
	return TRUE;
}

/* line numbers visibility */
gboolean
document_get_line_numbers_visible( CssedDoc* doc )
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	return sci_get_line_numbers(sci);
}

void
document_set_line_numbers_visible( CssedDoc* doc, gboolean visible )
{
	ScintillaObject* sci;

	DBGMSG("Going to set line numbers visibility to %s in doc %p",visible?"true":"false",doc);
	sci = sci_get_from_document(doc);
	sci_set_line_numbers(sci,visible);
}
/* folding marging visibility */
gboolean
document_get_folding_margin_visible( CssedDoc* doc )
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	return sci_get_folding_margin_visible(sci);
}

void
document_set_folding_margin_visible( CssedDoc* doc, gboolean visible )
{
	ScintillaObject* sci;

	DBGMSG("Going to set folding margin visibility to %s in doc %p",visible?"true":"false",doc);
	sci = sci_get_from_document(doc);
	sci_set_folding_margin_visible(sci,visible);
}
/* line endings visibility */
gboolean
document_get_line_endings_visible( CssedDoc* doc)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	return sci_get_visible_eols(sci);
}
void
document_set_line_endings_visible( CssedDoc* doc, gboolean visible)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	sci_set_visible_eols(sci,visible);
}
/* line wrapping */
gboolean
document_get_lines_wrapped( CssedDoc* doc)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	return sci_get_lines_wrapped(sci);
}
void
document_set_lines_wrapped( CssedDoc* doc, gboolean wrapped)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	sci_set_lines_wrapped(sci,wrapped);
}
/* white spaces */
gboolean
document_get_white_spaces_visible( CssedDoc* doc)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	return sci_get_visible_white_spaces(sci);
}
void
document_set_white_spaces_visible( CssedDoc* doc, gboolean visible)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	sci_set_visible_white_spaces(sci,visible);
}
/* EOL mode */
CssedEolMode
document_get_eol_mode( CssedDoc* doc )
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	return (CssedEolMode) sci_get_eol_mode	(sci);
}

void
document_set_eol_mode(CssedDoc* doc, CssedEolMode eolmode)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	sci_set_eol_mode(sci, (gint) eolmode);
}

void
document_convert_eols(CssedDoc* doc, CssedEolMode eolmode)
{
	ScintillaObject* sci;
	sci = sci_get_from_document(doc);
	sci_convert_eols(sci, (gint) eolmode);
}
/* text */
void
document_add_text( CssedDoc* doc, gchar* text )
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	sci_add_text(sci,text);
}

void/* adds text to the document editor ending in line feed */
document_add_text_line( CssedDoc * doc, char* text )
{
	ScintillaObject * sci;
	sci = SCINTILLA( doc->editor );

	SSM (sci, SCI_ADDTEXT, strlen(text) , (sptr_t) text );
	SSM (sci, SCI_ADDTEXT, strlen("\n") , (sptr_t) "\n" );
}
void // no wrapper function for this
document_search_next( CssedDoc* doc, gchar* text, gint flags )
{
	ScintillaObject* sci;
	gint selection_start;
	gint selection_end;
	gint search_pos;
	gint line;
#ifdef DEBUG
	gint current_pos;
	gint anchor_pos;
#endif

	sci = sci_get_from_document( doc );
	// look if any selection, if so move the caret out it can
	// it can be selected by a prevous search.
	// right now, cssed cannot search into a block of text selected by
	// the user. The user must know that.

	selection_start = SSM( sci, SCI_GETSELECTIONSTART, 0, 0);
	selection_end   = SSM( sci, SCI_GETSELECTIONEND, 0, 0);

#ifdef DEBUG
	current_pos		= SSM( sci, SCI_GETCURRENTPOS, 0, 0);
	anchor_pos		= SSM( sci, SCI_GETANCHOR, 0, 0);
	DBGMSG( "document_search_next(): Selection start %d end %d pos %d anchor %d",selection_start,selection_end,current_pos,anchor_pos);
#endif

	if( selection_end > selection_start ){ // there's selection
		// FIXME: build a wrapper function
		SSM( sci, SCI_GOTOPOS, selection_end + 1, 0);
	}
	// set the anchor
	// FIXME: build a wrapper function
	SSM( sci, SCI_SEARCHANCHOR,0,0);
	// FIXME: build a wrapper function
	search_pos = SSM( sci, SCI_SEARCHNEXT,flags,(sptr_t) text );
	if( search_pos != -1 ){
		line = document_get_line_from_position( doc, search_pos );
		if( !document_get_line_is_visible( doc, line ) ){
			document_ensure_line_is_visible( doc, line );
		}
		// FIXME: build a wrapper function
		SSM( sci, SCI_SCROLLCARET,0,0);
	}else{
		cssed_error_message
		(
		_("The text cannot be found or end of document reached.\n\nIf you are \
sure the text exists, then the document is inactive.\nClick on a document to make it active"),
		_("End of document reached")
		);
	}
}

void // no wrapper function for this
document_search_prev( CssedDoc* doc, gchar* text, gint flags )
{
	ScintillaObject* sci;
	gint selection_start;
	gint selection_end;
	gint search_pos;
	gint line;
#ifdef DEBUG
	gint current_pos;
	gint anchor_pos;
#endif

	sci = sci_get_from_document( doc );
	// look if any selection, if so move the caret out it can
	// it can be selected by a prevous search.
	// right now, cssed cannot search into a block of text selected by
	// the user. The user must know that.

	selection_start = SSM( sci, SCI_GETSELECTIONSTART, 0, 0);
	selection_end   = SSM( sci, SCI_GETSELECTIONEND, 0, 0);

#ifdef DEBUG
	current_pos		= SSM( sci, SCI_GETCURRENTPOS, 0, 0);
	anchor_pos		= SSM( sci, SCI_GETANCHOR, 0, 0);
	DBGMSG( "document_search_prev(): Selection start %d end %d pos %d anchor %d",selection_start,selection_end,current_pos,anchor_pos);
#endif

	if( selection_end > selection_start ){ // there's selection
		// FIXME: build a wrapper function
		SSM( sci, SCI_GOTOPOS, selection_start - 1, 0);
	}
	// set the anchor
	// FIXME: build a wrapper function
	SSM( sci, SCI_SEARCHANCHOR,0,0);
	// FIXME: build a wrapper function
	search_pos = SSM( sci, SCI_SEARCHPREV, flags,(sptr_t) text );
	if( search_pos != -1 ){
		line = document_get_line_from_position( doc, search_pos );
		if( !document_get_line_is_visible( doc, line ) ){
			document_ensure_line_is_visible( doc, line );
		}
		// FIXME: build a wrapper function
		SSM( sci, SCI_SCROLLCARET,0,0);
	}else{
		cssed_error_message
		(
		_("The text cannot be found or start of document reached.\n\nIf you are \
sure the text exists, then the document is inactive.\nClick on a document to make it active"),
		_("Start of document reached")
		);
	}
}

gboolean
document_can_redo(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_can_redo( sci );
}
gboolean
document_can_undo(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_can_undo( sci );
}

void
document_undo(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_undo(sci);
}
void
document_redo(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_redo(sci);
}

void
document_end_undo_action(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_end_undo_action(sci);
}

void
document_start_undo_action (CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_start_undo_action(sci);
}

void
document_new_undo_action(CssedDoc* doc)
{
	document_end_undo_action(doc);
	document_start_undo_action (doc);
}

void
document_empty_undo_buffer(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_empty_undo_buffer(sci);
}

void
document_set_undo_collection(CssedDoc* doc, gboolean set)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_set_undo_collection(sci,set);
}

gboolean
document_get_undo_collection(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_undo_collection(sci);
}

void
document_zoom_in(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_zoom_in(sci);
}

void
document_zoom_out(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_zoom_out(sci);
}

void
document_zoom_off(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_zoom_off(sci);
}

void
document_cut (CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_cut (sci);
}
void
document_copy (CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_copy (sci);
}
void
document_paste (CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_paste (sci);
}

void
document_set_marker_at_line( CssedDoc* doc, gint line, gboolean set, gint marker)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_set_marker_at_line( sci, line, set, marker);
}
gboolean
document_is_marker_set_at_line( CssedDoc* doc, gint line, gint marker)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_is_marker_set_at_line( sci, line, marker);
}
gboolean
document_marker_next(CssedDoc* doc, gint line, gint marker_mask)
{
	ScintillaObject* sci;
	//gint position;

	sci = sci_get_from_document( doc );
	return sci_marker_next( sci, line, marker_mask );
}


gboolean
document_marker_prev(CssedDoc* doc, gint line, gint marker_mask)
{
	ScintillaObject* sci;
	//gint position;

	sci = sci_get_from_document( doc );
	return sci_marker_prev( sci, line, marker_mask );
}

gint
document_get_line_from_position( CssedDoc* doc, gint position)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_line_from_position(sci, position );
}
gint
document_get_position_from_line( CssedDoc* doc, gint line)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_position_from_line(sci, line );
}
gint
document_get_current_position	(CssedDoc* doc )
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_current_position	(sci);
}

void
document_set_current_position	(CssedDoc* doc, gint position)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_set_current_position(sci, position);
	document_scroll_to_caret( doc );
}

void
document_set_current_line(CssedDoc* doc, gint line)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_set_current_line(sci, line);
}
gint
document_get_selection_start(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_selection_start	(sci);
}

gint
document_get_selection_end(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_selection_end(sci);
}

void
document_set_selection_start(CssedDoc* doc, gint position)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_set_selection_start	(sci, position);
}

void
document_set_selection_end(CssedDoc* doc, gint position)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_set_selection_end(sci, position);
}

gint
document_get_line_end_from_position(CssedDoc* doc, gint position)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_line_end_from_position(sci, position);
}

void  // make a sci wrapper for this
document_set_selection_range(CssedDoc* doc,gint start, gint end)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	SSM(sci,SCI_SETSEL, end, start);
}

void
document_replace_sel(CssedDoc* doc, gchar* text)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_replace_sel(sci, text);
}
void
document_clear_sel(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_replace_sel(sci, "");
}


gint
document_get_length	(CssedDoc* doc)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	return sci_get_length(sci);
}
/* you should allocate a buffer enought to contain the text plus a null char */
void
document_get_text(CssedDoc* doc, gint len, gchar* text)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_get_text(sci, len, text);
}
/* you should allocate a buffer enought to contain the text plus a null char */
void
document_get_selected_text(CssedDoc* doc, gchar* text)
{
	ScintillaObject* sci;

	sci = sci_get_from_document( doc );
	sci_get_selected_text(sci, text);
}

void
document_write_error_to_program_output(CssedDoc* doc, gchar* error_message, gchar* color_desc)
{
	CssedWindow *window;
	gchar* markup;

	window = CSSED_WINDOW(doc->window);

	markup = g_strdup_printf("<span color='%s'>%s</span>",color_desc,error_message);
	cssed_window_output_write( window, markup );
	g_free( markup );

	gtk_notebook_set_current_page ( GTK_NOTEBOOK(cssed_window_get_footer_notebook( window )),0);
}
// FIXME fails if a string with a brace is present in the declaration
gboolean
document_current_pos_is_into_selector(CssedDoc* doc)
{
	ScintillaObject *sci;
	gint position;
	gint open_brace;
	gint close_brace;
	//gchar *buffer;

	sci = (ScintillaObject *) sci_get_from_document (doc);
	position = document_get_current_position (doc);

	SSM (sci, SCI_SEARCHANCHOR, 0, 0);
	open_brace = SSM (sci, SCI_SEARCHPREV, 0, (sptr_t) "{");

#ifdef DEBUG
	if (open_brace == -1)
		DBGMSG ("document.c: document_current_pos_is_into_selector() - No opening brace before\n");
	else
		DBGMSG ("document.c: document_current_pos_is_into_selector() -Opening brace at %d\n", open_brace);
#endif

	SSM (sci, SCI_SEARCHANCHOR, 0, 0);
	close_brace = SSM (sci, SCI_SEARCHNEXT, 0, (sptr_t) "}");

#ifdef DEBUG
	if (close_brace == -1)
		DBGMSG ("document.c: document_current_pos_is_into_selector() -No closing brace after\n");
	else
		DBGMSG ("document.c: document_current_pos_is_into_selector() -Closing brace at %d\n", close_brace);
#endif

	if (close_brace < position)
	{
		document_set_current_position(doc,position);
		return FALSE;
	}
	else
	{
		if ((open_brace != -1) && (close_brace != -1))
		{
			document_set_current_position(doc,position);
			return TRUE;
		}else{
			document_set_current_position(doc,position);
			return FALSE;
		}
	}
}
gint
document_get_line_length(CssedDoc* doc, gint line)
{
	ScintillaObject *sci;
	sci = sci_get_from_document (doc);
	return sci_line_length(sci,line);
}

gint
document_get_line_count( CssedDoc* doc )
{
	ScintillaObject *sci;
	sci = sci_get_from_document (doc);
	return sci_get_line_count( sci );
}

void
document_get_text_line(CssedDoc* doc, gint line, gchar* buffer)
{
	ScintillaObject *sci;

	sci = (ScintillaObject *) sci_get_from_document (doc);
	sci_get_line(sci, line, buffer);
}
void  // not in document.h
document_get_xy_from_position(CssedDoc* doc,gint pos, gint* x, gint* y)
{
	ScintillaObject *sci;

	sci = (ScintillaObject *) sci_get_from_document (doc);
	sci_get_xy_from_position(sci, pos, x, y);
}
void   // not in document.h
document_gtk_popmenu_position_func(GtkMenu *menu, gint *x, gint *y,
				gboolean *push_in,  gpointer user_data)
{
	CssedPoint* point;
	CssedWindow* window;
	CssedDoc* doc;
	GtkWidget* notebook;
	GtkWidget* editor;
	GtkWidget* window_widget;
	GdkWindow* gdkwindow;
	gint new_x;
	gint new_y;
	gint root_x;
	gint root_y;

	point = (CssedPoint*) user_data;
	window = CSSED_WINDOW(point->main_window);
	doc = document_get_current(window);
	notebook = cssed_window_get_document_notebook( window );
	editor = GTK_WIDGET(doc->editor);
	//window_widget = gtk_widget_get_toplevel(notebook);
	window_widget = cssed_window_get_window_widget( window );
	gdkwindow = gtk_widget_get_parent_window (editor);
	DBGMSG("document.c: document_gtk_popmenu_position_func - is notebook %d",GTK_IS_NOTEBOOK(notebook));
	gtk_widget_translate_coordinates        (editor,
                                             window_widget,
                                             point->x,
                                             point->y,
                                             &new_x,
                                             &new_y);
	DBGMSG("document.c: document_gtk_popmenu_position_func -Translated coordinates from editor to window x = %d , y = %d",
			new_x, new_y);
	gdk_window_get_root_origin(gdkwindow, &root_x, &root_y);
	DBGMSG("document.c: document_gtk_popmenu_position_func -Root window coordinates x = %d , y = %d",
			root_x, root_y);
	*x = new_x + root_x + 5;  // +5  add a little padding to right
	*y = new_y + root_y + 30; // +30 add a little padding down
	DBGMSG("document.c: document_gtk_popmenu_position_func -Menu coordinates x = %d , y = %d",
			*x, *y);

	g_free(user_data);
}
/* It will scan the second treeview colum ( in the css definitions treeview )
	trying to find the text passed. If it's found, it builds a popmenu with the
	third column contents and connect the appropiated callbacks to get
	autocompletion to work */
void // not in document.h
document_pop_submenu_from_keyword_in_treeview(CssedDoc* doc,gchar* text)
{
	CssedWindow* window;
	CssedPoint* point;
	CssedPopmenuData* popmenu_data;
	GtkWidget* treeview;
	GtkTreeModel* model;
	GtkWidget* menu;
	GtkWidget* menu_items;
	GtkTreeIter iter;
	gboolean valid;
	gchar* keyword;
	gchar* value;
	gchar* path_str;
	GList* menu_item_list = NULL;
	gint nrow;
	gint x;
	gint y;

	window = CSSED_WINDOW(doc->window);
	treeview = cssed_window_get_css_definition_treeview( window );

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));

	if ( model == NULL ){
		return;
	}
	valid = gtk_tree_model_get_iter_from_string(model,&iter,"0:0");
	if(!valid){
		return;
	}
	nrow = 0;
	while(valid){
		gtk_tree_model_get (model, &iter, 1, &keyword, -1);
        DBGMSG ("scanning %s", keyword);
		if( strcmp(keyword,text)==0 ){
			DBGMSG("found !!\n");
			path_str = g_strdup_printf("0:%d:0",nrow);
			valid =  gtk_tree_model_get_iter_from_string(model,&iter,path_str);
			DBGMSG("trying to get child on %s",path_str);
			g_free(path_str);
			if(!valid ){
				DBGMSG("Cannot get child.");
				g_free(keyword);
				return;
			}
			DBGMSG("got 3rd child!");
			menu = gtk_menu_new ();

			while( valid ){
				popmenu_data = g_malloc(sizeof(	CssedPopmenuData ));
				gtk_tree_model_get (model, &iter, 2, &value, -1);
				menu_items = gtk_menu_item_new_with_label (value);
				popmenu_data->main_window = window;
				popmenu_data->document = document_get_current(window);
				popmenu_data->attribute = g_strdup(value);
				popmenu_data->property = g_strdup(keyword);
				popmenu_data->add_property_string = FALSE;

				menu_item_list = g_list_append(menu_item_list, popmenu_data);
				gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_items);

				// callback will call fire_dialog_from_strings_or_write_output()
				g_signal_connect (G_OBJECT (menu_items), "activate",
                                      G_CALLBACK (popmenu_autocomp_activate_cb),
                                      popmenu_data );

				gtk_widget_show (menu_items);
				g_free(value);
				valid = gtk_tree_model_iter_next (model,&iter);
			}

			// this will dispose all memory allocated parsing the list
			// passed on unmap event (event->type ==  GDK_UNMAP).
			g_signal_connect (G_OBJECT (menu), "event-after",
                               G_CALLBACK (popmenu_autocomp_event_after_cb),
                               menu_item_list );
			DBGMSG("Sending list %p as user_data for callback",menu_item_list);
			document_get_xy_from_position(doc,document_get_current_position(doc),&x,&y);
			DBGMSG("Current point into editor x=%d,y=%d",x,y);
			point = g_malloc(sizeof(CssedPoint));
			point->main_window = window;
			point->x = x;
			point->y = y;
			gtk_menu_popup (GTK_MENU(menu),NULL, NULL,
										 document_gtk_popmenu_position_func  ,
                                         (gpointer) point,// freed by callback
                                         0,
                                         gtk_get_current_event_time());
			g_free(keyword);
			return;
		}
		nrow++;
        g_free (keyword);
		valid = gtk_tree_model_iter_next (model,&iter);
     }
}

void
document_show_autocompletion_list(CssedDoc* doc, gint identifier, gchar* list)
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	SSM(sci,SCI_USERLISTSHOW, identifier, (sptr_t) list);
	//SSM(sci,SCI_AUTOCSHOW,0, (sptr_t) list);
}
/* folding */

void
document_toggle_fold_at_line(CssedDoc* doc, gint line)
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	sci_toggle_fold(sci,line);
}

gboolean
document_get_line_is_visible(CssedDoc* doc, gint line)
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	return sci_get_line_is_visible(sci,line);
}

void
document_ensure_line_is_visible(CssedDoc* doc, gint line)
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	sci_ensure_line_is_visible( sci, line);
}

gint
document_get_fold_level(CssedDoc* doc, gint line)
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	return sci_get_fold_level(sci,line);
}

void
document_unfold_all( CssedDoc* doc )
{
	gint lines;
	gint pos;
	int i;
	lines = document_get_line_count( doc );
	pos = document_get_current_position( doc );
	for( i=0; i<lines;i++){
		document_ensure_line_is_visible( doc, i );
	}
	//document_set_current_position( doc,  0 );
}

void
document_fold_all( CssedDoc* doc )
{
	gint lines;
	gint pos;
	int i;

	if(!document_get_folding_enabled( doc ) ){
		document_set_folding_margin_visible( doc ,TRUE );
	}

	lines = document_get_line_count( doc );
	pos = document_get_current_position( doc );
	for( i=0; i<lines;i++){
		int level =document_get_fold_level(doc,i);
		if (level & SC_FOLDLEVELHEADERFLAG) {
			if( document_get_fold_expanded_at_line(doc, i) ){
				document_toggle_fold_at_line(doc, i);
			}
		}
	}
	//document_set_current_position( doc,  0 );
}
gboolean
document_get_fold_expanded_at_line( CssedDoc* doc, gint line)
{
	ScintillaObject* sci;
	sci = sci_get_from_document( doc );
	return sci_get_fold_expanded(sci, line);
}

gboolean
document_get_folding_enabled( CssedDoc* doc )
{
	return document_get_folding_margin_visible( doc );
}

CssedFileType
document_get_filetype( CssedDoc* doc )
{
	return doc->filetype;
}

CssedFileType
document_get_filetype_from_filename( gchar* filename )
{
	GPatternSpec *css_pattern;
	GPatternSpec *htm_pattern;
	GPatternSpec *html_pattern;
	GPatternSpec *php_pattern;
	GPatternSpec *php2_pattern;
	GPatternSpec *asp_pattern;
	GPatternSpec *c_pattern;
	GPatternSpec *cpp_pattern;
	GPatternSpec *cpp2_pattern;
	GPatternSpec *h_pattern;
	GPatternSpec *perl_pattern;
	GPatternSpec *cgi_pattern;
	GPatternSpec *py_pattern;
	GPatternSpec *xml_pattern;
	GPatternSpec *svg_pattern;
	GPatternSpec *diff_pattern;
	GPatternSpec *patch_pattern;
	GPatternSpec *make_pattern;

	// check file type
	css_pattern = g_pattern_spec_new ( "*.css" );
	htm_pattern = g_pattern_spec_new ( "*.htm" );
	html_pattern = g_pattern_spec_new ( "*.html" );
	php_pattern = g_pattern_spec_new ( "*.php" );
	php2_pattern = g_pattern_spec_new ( "*.php?" );
	asp_pattern = g_pattern_spec_new ( "*.asp?" );
	c_pattern = g_pattern_spec_new ( "*.c" );
	cpp_pattern = g_pattern_spec_new ( "*.cpp" );
	cpp2_pattern = g_pattern_spec_new ( "*.cxx" );
	h_pattern = g_pattern_spec_new ( "*.h" );
	perl_pattern = g_pattern_spec_new ( "*.pl" );
	cgi_pattern = g_pattern_spec_new ( "*.cgi" );
	py_pattern = g_pattern_spec_new ( "*.py" );
	xml_pattern = g_pattern_spec_new ( "*.xml" );
	svg_pattern = g_pattern_spec_new ( "*.svg" );
	diff_pattern = g_pattern_spec_new ( "*.diff" );
	patch_pattern = g_pattern_spec_new ( "*.patch" );
	make_pattern = g_pattern_spec_new ( "*akefile*" );

  	if (g_pattern_match (css_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_CSS;
	}else if ( g_pattern_match (htm_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (html_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (php_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (php2_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (asp_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_HTML;
	}else if ( g_pattern_match (c_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (cpp_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (cpp2_pattern, strlen (filename), filename, NULL) ||
			  g_pattern_match (h_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_C;
	}else if ( g_pattern_match (perl_pattern, strlen (filename), filename, NULL) ||
			    g_pattern_match (cgi_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_PERL;
	}else if ( g_pattern_match (xml_pattern, strlen (filename), filename, NULL) ||
			    g_pattern_match (svg_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_XML;
	}else if ( g_pattern_match (py_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_PYTHON;
	}else if ( g_pattern_match (diff_pattern, strlen (filename), filename, NULL) ||
			    g_pattern_match (patch_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_DIFF;
	}else if ( g_pattern_match (make_pattern, strlen (filename), filename, NULL)){
		return CSSED_FILETYPE_MAKE;
	}else{
		return CSSED_FILETYPE_TEXT; // default to plain text
	}
}
/* this sets the style and the filetype filed */
void
document_set_filetype( CssedDoc* doc, CssedFileType filetype )
{
	CssedWindow* window;
	CssedConfig* cfg;

	if( document_get_filetype( doc ) == filetype ) {
		document_set_font( doc );
		document_set_current_font_size( doc );
		return;
	}

	window = document_get_window( doc );
	cfg = cssed_window_get_config( window );
	document_style_clear_all( doc );

	switch( filetype ){
		case CSSED_FILETYPE_CSS:
			document_set_css_style( doc );
			doc->filetype = CSSED_FILETYPE_CSS;
			document_set_folding_margin_visible( doc, cfg->folding );// Can fold
			break;
		case CSSED_FILETYPE_HTML:
			document_set_html_style( doc );
			doc->filetype = CSSED_FILETYPE_HTML;
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, cfg->folding );// Can fold
			break;
		case CSSED_FILETYPE_C:
			document_set_c_style( doc );
			doc->filetype = CSSED_FILETYPE_C;
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, cfg->folding );// Can fold
			break;
		case CSSED_FILETYPE_SH:
			document_set_sh_style( doc );
			doc->filetype = CSSED_FILETYPE_SH;
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, FALSE );// No folding
			break;
		case CSSED_FILETYPE_PERL:
			document_set_perl_style( doc );
			doc->filetype = CSSED_FILETYPE_PERL;
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, cfg->folding );// Can fold
			break;
		case CSSED_FILETYPE_APACHE_CONF:
			document_set_conf_style( doc );
			doc->filetype = CSSED_FILETYPE_APACHE_CONF;
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, FALSE );// No folding
			break;
		case CSSED_FILETYPE_PYTHON:
			document_set_python_style( doc );
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			doc->filetype = CSSED_FILETYPE_PYTHON;
			document_set_folding_margin_visible( doc, cfg->folding );// Can fold
			break;
		case CSSED_FILETYPE_XML:
			document_set_xml_style( doc );
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			doc->filetype = CSSED_FILETYPE_XML;
			document_set_folding_margin_visible( doc, cfg->folding );// Can fold
			break;
		case CSSED_FILETYPE_UNKNOW:
			document_set_null_style( doc ); // no style
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, FALSE );// No folding
			doc->filetype = CSSED_FILETYPE_UNKNOW;
			document_set_folding_margin_visible( doc, FALSE );// No folding
			break;
		case CSSED_FILETYPE_TEXT:
			document_set_null_style( doc ); // no style
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, FALSE );// No folding
			doc->filetype = CSSED_FILETYPE_TEXT;
			break;
		case CSSED_FILETYPE_DIFF:
			document_set_diff_style( doc ); // diff / patch files
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, FALSE );// No folding
			doc->filetype = CSSED_FILETYPE_DIFF;
			break;
		case CSSED_FILETYPE_MAKE:
			document_set_make_style( doc );
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			document_set_folding_margin_visible( doc, FALSE );// No folding
			doc->filetype = CSSED_FILETYPE_DIFF;
			break;
		default:
			document_set_null_style( doc ); // no style
			//doc->autocompletion = FALSE;	// no autocompletion for this file type yet
			doc->filetype = CSSED_FILETYPE_TEXT;
			document_set_folding_margin_visible( doc, FALSE );// No folding
			break;
	}
	document_set_font( doc );
	document_set_current_font_size( doc );
	document_colourise_all(  doc );
}
// ACCESS TO DOCUMENT'S FIELDS
CssedWindow*
document_get_window( CssedDoc* doc )
{
	return doc->window;
}


gchar*
document_get_filename( CssedDoc* doc )
{
	if( doc == NULL ) return NULL;

	if( doc->filename != NULL ){
		return g_strdup( doc->filename );
	}else{
		return NULL;
	}
}

void
document_set_filename( CssedDoc* doc , gchar* filename )
{
	g_return_if_fail( filename != NULL );
	if( doc->filename != NULL ) g_free( doc->filename );
	doc->filename = g_strdup( filename );
}

gboolean
document_get_autocompletion_enabled( CssedDoc* doc )
{
	return doc->autocompletion;
}

void
document_set_autocompletion_enabled ( CssedDoc* doc, gboolean enabled )
{
	doc->autocompletion = enabled;
}

// FIXME: this must be moved to a window function
void
fire_dialog_from_strings_or_write_output(CssedWindow* window,
										CssedDoc* tmpdoc,
										gchar* attribute,
										gchar* property ,
										gboolean add_previous)
{
	gchar *line_to_insert;
	gchar* strdialog;
	GtkWidget *dialog;
	CssDialogClass* class;

	if( attribute[0] == '(' ){ // if it does not start with '(' is not a dialog
		if( (class = (CssDialogClass*) cssed_window_css_dialog_lookup_by_keyword (window, attribute)) != NULL ){
			dialog = class->create_function(window, g_strdup (property) ,
												add_previous,
												TRUE);
			gtk_widget_show (dialog);
		}
		else
		{ // dialog does not exist yet
			strdialog = g_strdup_printf(_("The dialog %s does not exist."),attribute);
			document_write_error_to_program_output(tmpdoc,strdialog,"black");

			if( add_previous ){
				line_to_insert =
				g_strconcat (property, ": ", attribute, ";",
					     	NULL);
				document_add_text_line (tmpdoc, line_to_insert);
				document_add_text (tmpdoc, "\t");
				g_free (line_to_insert);
			}else{
				document_add_text (tmpdoc, attribute);
			}

			g_free(strdialog);
		}
	}
	else
	{// no dialog to show, write the selected strings
		if (tmpdoc != NULL)
		{
			if( add_previous ){
				line_to_insert = g_strconcat (property, ":", attribute, ";", NULL);
				document_add_text_line (tmpdoc, line_to_insert);
				document_add_text (tmpdoc, "\t");
				g_free (line_to_insert);
			}else{
				document_add_text (tmpdoc, attribute);
			}
		}
	}
}

// scintilla control access
GtkWidget* // you must use the SCINTILLA() macro to get the ScintillaObject and include scintilla headers
document_get_scintilla_widget ( CssedDoc* doc )
{
	return doc->editor;
}

// callback func called by all editors when a signals arises
void
on_editor_notification( GtkWidget* editor,gint scn, gpointer lscn, gpointer user_data)
{
	struct SCNotification *nt;
	CssedDoc* tmpdoc;
	CssedWindow* window;
	// used if lastchar == ':' in Autocompletion mode
	gint linestart;
	gchar* buffer;
	gint line;
	gint linelength;
	gchar lastchar;
	gint position;
	gint margin;
	gchar** uris;
	gint uris_index;
	gchar* uri;

	window = CSSED_WINDOW( user_data );
	tmpdoc = document_get_from_editor( window, editor );

	nt = lscn;
	switch (nt->nmhdr.code){
		// open "file:///" uris dropped from file browser
		case SCN_URIDROPPED:
			uris_index = 0;
			uris = g_strsplit(nt->text,"\n",-1);
			while(uris[uris_index] != NULL){
#ifdef GTK_IS_2_2
				if( g_str_has_prefix( uris[uris_index],"file://") ){
#else
				if( strncmp( uris[uris_index], "file://",7 * sizeof(gchar)) == 0 ){
#endif
#ifdef WIN32 // windows passes file:/// as unix but the root dir must be a drive
				  uri = g_strdup(uris[uris_index] + 8);
#else
				  uri = g_strdup(uris[uris_index] + 7);
#endif
				  g_strstrip( uri );
					if( g_file_test(uri, G_FILE_TEST_IS_REGULAR ) ){
						create_and_attach_new_named_doc(window, uri);
					}else{
						if( g_file_test(uri, G_FILE_TEST_IS_DIR ) ){
							buffer = g_strdup_printf(_("Cannot open %s, is a directory"),uri);
							document_write_error_to_program_output(tmpdoc,buffer,"brown");
							g_free(buffer);
						}else{
							buffer = g_strdup_printf(_("Cannot open %s, is not a regular file"),uri);
							document_write_error_to_program_output(tmpdoc,buffer,"brown");
							g_free(buffer);
						}
					}
					g_free(uri);
				}else{
					if( strlen(uris[uris_index]) > 0 ){
						buffer = g_strdup_printf(_("Cannot open file %s, protocol not supported"),uris[uris_index]);
						document_write_error_to_program_output(tmpdoc,buffer,"brown");
						g_free(buffer);
					}
				}
				uris_index++;
			}
			g_strfreev(uris);
			break;
		case SCN_CHARADDED:
			 lastchar = nt->ch;
			if( tmpdoc->filetype==CSSED_FILETYPE_CSS && tmpdoc->autocompletion ){
				// add close brace if open brace is added
				if( lastchar == '{' ){
					document_add_text(tmpdoc,"\n\t\n}\n");
					document_set_current_position(tmpdoc,
						(document_get_current_position(tmpdoc)-3) );
					document_show_autocompletion_list(tmpdoc,0, cssed_window_get_keyword_list( window ) );
				}
				if( lastchar == ';'){
					document_add_text(tmpdoc,"\n\t");
					document_show_autocompletion_list(tmpdoc,0,cssed_window_get_keyword_list( window ));
					}

				if( lastchar == ':'){
					// get if in selector
					if(document_current_pos_is_into_selector(tmpdoc)){
						DBGMSG("into a selector");
						position = document_get_current_position(tmpdoc);
						line = document_get_line_from_position(tmpdoc,position);
						linestart  = document_get_position_from_line(tmpdoc,line);

						// we can not assume the user is at the end of a line
						// so check if we are. reserve memory for the whole line
						linelength = document_get_line_length(tmpdoc,line);
						DBGMSG("Line %d start %d pos %d length %d",line,linestart,position,linelength);

						if( position > linestart){
							buffer = g_malloc0(sizeof(gchar)*( linelength )+1);
							document_get_text_line(tmpdoc,line,buffer);
							DBGMSG("Line contains: \"%s\"", buffer);
							buffer[position-linestart-1] = '\0';
							g_strstrip( buffer );
							DBGMSG("Cleaned: \"%s\"", buffer);
							document_pop_submenu_from_keyword_in_treeview(tmpdoc,buffer);
							g_free(buffer);
						}
					}
				}
			}
			break;
		case SCN_UPDATEUI:
			document_new_undo_action(tmpdoc);
			break;
		case SCN_KEY:
			if( (nt->ch=='a') && (nt->modifiers==4) ){ // Alt + A fire autocompletion
				document_show_autocompletion_list(tmpdoc,0, cssed_window_get_keyword_list( window ));
			}else if( (nt->ch=='l') && (nt->modifiers==4) ){ // Alt + l selects current line
				gint kill_line;
				gint start_pos;
				gint line_len;
				gint end_pos;
				gint curpos;

				curpos = document_get_current_position(tmpdoc);
				kill_line = document_get_line_from_position(tmpdoc, curpos);
				start_pos = document_get_position_from_line(tmpdoc, kill_line);
				line_len = document_get_line_length(tmpdoc, kill_line);
				end_pos = start_pos + line_len;
				document_set_selection_range(tmpdoc, start_pos, end_pos);
			}else if((nt->ch==';') && ((nt->modifiers==5)||(nt->modifiers==4))){ // Alt + ";" adds ; (no auto complete)
				document_add_text(tmpdoc,";");
			}else if( (nt->ch=='s') && (nt->modifiers==4) ){ // Alt + s scan selector
				on_menu_scan_selector_activate (NULL, window);
			}
			DBGMSG("Pressed char %d modifier %d", nt->ch, nt->modifiers );
			break;
		case SCN_MODIFIED:
			if (nt->modificationType & SC_LASTSTEPINUNDOREDO) {
				if(! document_can_undo(tmpdoc) ){
					cssed_window_disable_save ( window );
					cssed_window_disable_undo( window );
				}else{
					cssed_window_enable_undo( window );
				}
				if( ! document_can_redo(tmpdoc) ){
					cssed_window_disable_redo( window );
				}else{
					cssed_window_enable_redo( window );
				}
			} else if (nt->modificationType & (SC_MOD_INSERTTEXT | SC_MOD_DELETETEXT)) {
				cssed_window_enable_save ( window );
				cssed_window_enable_undo ( window );
				if( document_can_redo( tmpdoc ) ){
					cssed_window_enable_redo ( window );
				}else{
					cssed_window_disable_redo( window );
				}
			} else if ( nt->modificationType & SC_PERFORMED_UNDO ){
				if( document_get_modified( tmpdoc ) ){
					cssed_window_enable_save( window );
				}
				if( document_can_redo( tmpdoc ) ){
					cssed_window_enable_redo ( window );
				}else{
					cssed_window_disable_redo( window );
				}
			} else if ( nt->modificationType & SC_PERFORMED_REDO ){
				if( document_get_modified( tmpdoc ) ){
					cssed_window_enable_save( window );
				}
				if( document_can_redo( tmpdoc ) ){
					cssed_window_enable_redo ( window );
				}else{
					cssed_window_disable_redo( window );
				}
			}
			break;
		case SCN_MARGINCLICK:
			position = nt->position;
			margin = nt->margin;
			DBGMSG("Clicked margin %d",margin);
			line = document_get_line_from_position( tmpdoc, position );
			if( margin == 1){
				if( document_is_marker_set_at_line( tmpdoc, line, 0 ) ){
					document_set_marker_at_line(tmpdoc,line,TRUE, 0 );
				}else{
					document_set_marker_at_line(tmpdoc,line,FALSE, 0 );
				}
			}else if( margin == 2 ){
				int level =document_get_fold_level(tmpdoc,line);
				if (level & SC_FOLDLEVELHEADERFLAG) {
					DBGMSG("Fold toggled at line %d",line);
					document_toggle_fold_at_line(tmpdoc, line);
                }
 			}
			break;
		case SCN_USERLISTSELECTION:
			 /* no user list it's just a test */
			switch( nt->wParam ){
				case 1:{
					g_print("Selected from list 1: %s\n",(gchar*) nt->text);
					}
				break;
				default:
					g_print("Unknown list selected\n");
				break;
			}
			break;
	}
}
/*






*/

/* scintilla object related functions */
ScintillaObject*
sci_get_from_notebook_page(GtkNotebook* notebook, gint page_number)
{
	ScintillaObject* sci;
	GtkWidget* editor;

	editor = gtk_notebook_get_nth_page (notebook, page_number);
	sci = SCINTILLA(editor);
	return sci;
}

ScintillaObject*
sci_get_from_document( CssedDoc* doc )
{
	ScintillaObject * sci;
	sci = SCINTILLA( doc->editor );
	return sci;
}

/* it's unused right now see document_set_css_style*/
void
sci_set_css_style (ScintillaObject * sci)
{
	//gint cred = 0x00ff00;
	gint colive = 0x004400;
	gint cnavy = 0x990000;
	//gint cblack = 0x000000;
	//gint cyellow = 0x009999;
	//gint cblue = 0x993300;
	//gint cred = 0x00ff00;

	SSM (sci, SCI_STYLECLEARALL, 0, 0);
	SSM (sci, SCI_SETLEXER, SCLEX_CSS, 0);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_DEFAULT, 0x000077);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_TAG, 0xA46621 );
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_TAG, 1);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_PSEUDOCLASS, 0x990066);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_UNKNOWN_PSEUDOCLASS, 0x0000ff);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_OPERATOR, 0x000099);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_IDENTIFIER, 0x990000);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_IDENTIFIER, 1);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_UNKNOWN_IDENTIFIER , 0x0000ff);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_VALUE, 0x222222);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_COMMENT, 0x666666);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_IMPORTANT, 0x000099);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_IMPORTANT, 1);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_DIRECTIVE, 0x660066);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_DOUBLESTRING, 0x660033);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_SINGLESTRING, 0x660033);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_CLASS, colive);
	SSM (sci, SCI_STYLESETBOLD,	SCE_CSS_CLASS, 1);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_ID, colive); // 0x009999);
	SSM (sci, SCI_STYLESETBOLD, SCE_CSS_ID, 1);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_VALID_VALUE, 0x003399);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_FUNCTION,  0x993300);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_NUMBER, 0x330099);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_UNIT, 0x990000);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_COLOR, 0x006600);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_HEXACOLOR, 0x006600);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_ATTR_MATCH, 0x004444);
	SSM (sci, SCI_STYLESETFORE, SCE_CSS_LANGUAGE, cnavy);
	SSM (sci, SCI_SETTABWIDTH, 4, 0);
	SSM (sci, SCI_SETREADONLY, 0, 0);
	SSM (sci, SCI_SETMARGINTYPEN, 0, SC_MARGIN_NUMBER);
	SSM (sci, SCI_SETMARGINWIDTHN, 0,
				SSM(sci, SCI_TEXTWIDTH, STYLE_LINENUMBER,
				(sptr_t) MARGIN_LINES_WIDTH_STRING));
	SSM (sci, SCI_SETMARGINTYPEN, 1, SC_MARGIN_SYMBOL);
	SSM (sci, SCI_SETMARGINWIDTHN, 1, MARGIN_MARKERS_WIDTH_PIXELS );
	SSM (sci, SCI_SETMARGINSENSITIVEN,1, TRUE);
	SSM (sci, SCI_MARKERDEFINE,0, SC_MARK_ROUNDRECT); // marker 0 for bookmarks
	SSM (sci,SCI_SETUNDOCOLLECTION,1,0);
	SSM (sci,SCI_SETMODEVENTMASK, SC_MODEVENTMASKALL ,0);
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold", (sptr_t) "1");
	SSM (sci, SCI_SETPROPERTY, (sptr_t) "fold.compact", (sptr_t) "1");
 	SSM (sci, SCI_SETFOLDFLAGS, 16,0);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDEROPEN, SC_MARK_BOXMINUS);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDEROPEN, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDEROPEN, 0x000000);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDER, SC_MARK_BOXPLUS);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDER, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDER, 0x000000);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDERSUB, SC_MARK_VLINE);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDERSUB, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDERSUB, 0x000000);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDERTAIL, SC_MARK_LCORNER);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDERTAIL, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDERTAIL, 0x000000);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDEREND, SC_MARK_BOXPLUSCONNECTED);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDEREND, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDEREND, 0x000000);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDEROPENMID, SC_MARK_BOXMINUSCONNECTED);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDEROPENMID, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDEROPENMID, 0x000000);
 	SSM (sci,SCI_MARKERDEFINE,  SC_MARKNUM_FOLDERMIDTAIL, SC_MARK_TCORNER);
	SSM (sci,SCI_MARKERSETFORE, SC_MARKNUM_FOLDERMIDTAIL, 0xffffff);
	SSM (sci,SCI_MARKERSETBACK, SC_MARKNUM_FOLDERMIDTAIL, 0x000000);
	SSM (sci, SCI_SETMARGINSENSITIVEN,2, TRUE);
	SSM (sci, SCI_SETMARGINMASKN,2, SC_MASK_FOLDERS);
	SSM (sci, SCI_SETMARGINWIDTHN, 2, 0 );
}


void
sci_set_keyworkds (ScintillaObject * sci, gchar* keywords)
{
	enum {
		SC_KEYS_PROPERTIES,
		SC_KEYS_PSEUDO,
		SC_KEYS_VALUES,
		SC_KEYS_FUNCS,
		SC_KEYS_UNITS,
		SC_KEYS_COLORS
	};

	const gchar* values = "above absolute always armenian auto avoid baseline behind below bidi-override \
blink block bold bolder both bottom capitalize caption center center-left center-right child circle cjk-ideographic close-quote \
code collapse collapse compact condensed continuous crop cross crosshair dashed decimal decimal-leading-zero default digits \
disc dotted double embed e-resize expanded extra-condensed extra-expanded far-left far-right fast female fixed \
georgian groove hebrew help hide high hiragana hidden higher hiragana-iroha icon inherit inline inset inside invert inline-table faster \
invert italic justify katakana katakana-iroha large larger landscape left left-side leftwards level lighter line-through list-item loud \
low lower lowercase lower-greek lower-alpha lower-latin lower-roman ltr male marker medium menu message-box middle mix move ne-resize \
no-close-quote none no-open-quote no-repeat normal nowrap n-resize nw-resize oblique once open-quote outset outside overline \
pointer portrait pre relative repeat repeat-x repeat-y ridge right rightwards right-side rtl run-in scroll semi-condensed semi-expanded \
separate se-resize show silent slow slower small smaller small-caps small-caption side soft solid spell-out square s-resize static status-bar sub super sw-resize \
table table-caption table-cell table-column table-column-group table-footer-group table-header-group table-row table-row-group text thick thin \
top text-bottom text-top transparent ultra-condensed ultra-expanded underline upper-alpha uppercase upper-latin upper-roman visible wait wider w-resize x-fast x-high x-loud x-low \
x-slow x-soft  xx-small x-large xx-small x-small xx-large serif sans-serif cursive fantasy monospace";

	const gchar* namedColors = "aqua black blue fuchsia gray green lime maroon  navy olive purple red silver teal white yellow activeborder \
activecaption appworkspace background buttonface buttonhighlight buttonshadow buttontext captiontext graytext highlight highlighttext \
inactiveborder inactivecaption inactivecaptiontext infobackground infotext menu menutext scrollbar threeddarkshadow threedface threedhighlight \
threedlightshadow threedshadow window windowframe windowtext";

	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_PROPERTIES, (sptr_t) keywords);
	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_PSEUDO, (sptr_t) "link hover active visited first-letter first-line after before print visual aural first left first-child");
	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_VALUES, (sptr_t) values);
	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_FUNCS, (sptr_t) "counter attr url rgb rect");
#ifdef WIN32
	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_UNITS, (sptr_t) "em ex px in cm mm pt pc \\% mhz hz deg rad gra s ms");
#else
	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_UNITS, (sptr_t) "em ex px in cm mm pt pc \% mhz hz deg rad gra s ms");
#endif
	SSM (sci, SCI_SETKEYWORDS, SC_KEYS_COLORS, (sptr_t) namedColors);
}


/* line numbers visibility */
void
sci_set_line_numbers(ScintillaObject * sci, gboolean set )
{
	if( set ){
			SSM (sci, SCI_SETMARGINWIDTHN, 0,
				SSM(sci, SCI_TEXTWIDTH, STYLE_LINENUMBER,
				(sptr_t) MARGIN_LINES_WIDTH_STRING));
	}else{
		SSM (sci, SCI_SETMARGINWIDTHN, 0, 0 );
	}
}
gboolean
sci_get_line_numbers(ScintillaObject * sci)
{
	gint margin_width;

	margin_width = SSM(sci, SCI_GETMARGINWIDTHN, 0, 0);
	if( margin_width > 0 ) return TRUE;
	else return FALSE;
}
/* folding margin visibility */
void
sci_set_folding_margin_visible (ScintillaObject * sci, gboolean set )
{
	if( set ){
		SSM (sci, SCI_SETMARGINWIDTHN, 2, MARGIN_FOLDING_WIDTH_PIXELS );
	}else{
		SSM (sci, SCI_SETMARGINWIDTHN, 2, 0 );
	}
}
gboolean
sci_get_folding_margin_visible(ScintillaObject * sci)
{
	gint margin_width;

	margin_width = SSM(sci, SCI_GETMARGINWIDTHN, 2, 0);
	if( margin_width > 0 ) return TRUE;
	else return FALSE;
}
/* end of lines */
void
sci_set_visible_eols(ScintillaObject* sci, gboolean set )
{
	SSM(sci,SCI_SETVIEWEOL,set,0);
}

gboolean
sci_get_visible_eols(ScintillaObject* sci)
{
	return SSM( sci, SCI_GETVIEWEOL,0,0);
}
void
sci_set_visible_white_spaces(ScintillaObject* sci, gboolean set )
{
	if(set){
		SSM(sci,SCI_SETVIEWWS,SCWS_VISIBLEALWAYS,0);
	}else{
		SSM(sci,SCI_SETVIEWWS,SCWS_INVISIBLE,0);
	}
}
gboolean
sci_get_visible_white_spaces(ScintillaObject* sci)
{
	return SSM( sci, SCI_GETVIEWWS,0,0);
}
void
sci_set_lines_wrapped(ScintillaObject* sci, gboolean set )
{
	if( set ){
		SSM(sci,SCI_SETWRAPMODE,SC_WRAP_WORD,0);
	}else{
		SSM(sci,SCI_SETWRAPMODE,SC_WRAP_NONE,0);
	}
}
gboolean
sci_get_lines_wrapped(ScintillaObject* sci)
{
	return SSM( sci, SCI_GETWRAPMODE,0,0);
}

gint
sci_get_eol_mode( ScintillaObject* sci)
{
	return SSM( sci, SCI_GETEOLMODE, 0, 0);
}

void
sci_set_eol_mode( ScintillaObject* sci, gint eolmode)
{
	SSM( sci, SCI_SETEOLMODE, eolmode, 0);
}

void
sci_convert_eols( ScintillaObject* sci, gint eolmode)
{
	SSM( sci, SCI_CONVERTEOLS, eolmode,0);
}


void
sci_add_text(ScintillaObject* sci, gchar* text)
{
	if( text != NULL ){// if null text is passed to scintilla will segfault
		SSM( sci, SCI_ADDTEXT, strlen(text) , (sptr_t) text);
	}
}

gboolean
sci_can_undo( ScintillaObject* sci )
{
	return SSM( sci, SCI_CANUNDO, 0, 0);
}

gboolean
sci_can_redo( ScintillaObject* sci )
{
	return SSM( sci, SCI_CANREDO, 0, 0);
}

void
sci_undo( ScintillaObject* sci )
{
	if( sci_can_undo(sci) ){
		SSM( sci, SCI_UNDO, 0, 0);
	}else{ // change it to a document function
		cssed_error_message(_("There is no action in the undo list."),
							_("Undo limit reached"));
	}
}

void
sci_redo( ScintillaObject* sci )
{
	if( sci_can_redo( sci ) ){
		SSM( sci, SCI_REDO,0,0);
	}else{// change it to a document function
		cssed_error_message(_("There is no action in the redo list."),
							_("Redo limit reached"));
	}
}

void
sci_start_undo_action( ScintillaObject* sci )
{
	SSM( sci,SCI_BEGINUNDOACTION,0,0 );
}

void
sci_end_undo_action( ScintillaObject* sci )
{
	SSM( sci, SCI_ENDUNDOACTION,0,0);
}
void
sci_set_undo_collection( ScintillaObject* sci, gboolean set )
{
	SSM( sci, SCI_SETUNDOCOLLECTION,set,0);
}

gboolean
sci_get_undo_collection( ScintillaObject* sci )
{
	return SSM( sci, SCI_GETUNDOCOLLECTION,0,0);
}

void
sci_empty_undo_buffer( ScintillaObject* sci )
{
	SSM( sci, SCI_EMPTYUNDOBUFFER,0,0);
}

void
sci_zoom_in( ScintillaObject* sci )
{
	SSM( sci, SCI_ZOOMIN,0,0);
}

void
sci_zoom_out( ScintillaObject* sci )
{
	SSM( sci, SCI_ZOOMOUT,0,0);
}

void
sci_zoom_off( ScintillaObject* sci )
{
	SSM( sci, SCI_SETZOOM,0,0);
}
void
sci_set_marker_at_line( ScintillaObject* sci, gint line_number, gboolean set, gint marker )
{
	DBGMSG("Set marker to %d line %d \n",set,line_number);

	if( set ){
		SSM( sci, SCI_MARKERADD,line_number, marker);
	}else{
		SSM( sci, SCI_MARKERDELETE,line_number, marker);
	}
}
gboolean
sci_is_marker_set_at_line(ScintillaObject* sci, gint line, gint marker)
{
	gint state;

	state = SSM( sci, SCI_MARKERGET, line, marker );
	return(!(state & (1 << marker)));
}

gboolean
sci_marker_next(ScintillaObject* sci, gint line, gint marker_mask)
{
	gint marker_line;

	marker_line = SSM(sci, SCI_MARKERNEXT, line, marker_mask);
	DBGMSG( "Marker next: line %d next marker %d",line,marker_line);

	if( marker_line != -1 ){
		SSM(sci,SCI_GOTOLINE,marker_line,0);
		return TRUE;
	}else{
		return FALSE;
	}
}

gboolean
sci_marker_prev(ScintillaObject* sci, gint line, gint marker_mask)
{
	gint marker_line;

	marker_line = SSM(sci, SCI_MARKERPREVIOUS, line, marker_mask);
	DBGMSG( "Marker previous: line %d marker %d",line,marker_line);

	if( marker_line != -1 ){
		SSM(sci,SCI_GOTOLINE,marker_line,0);
		return TRUE;
	}else{
		return FALSE;
	}
}

gint
sci_get_line_from_position(ScintillaObject* sci, gint position )
{
	return SSM(sci, SCI_LINEFROMPOSITION, position, 0);
}

gint
sci_get_position_from_line(ScintillaObject* sci, gint line )
{
	return SSM(sci, SCI_POSITIONFROMLINE, line, 0);
}

gint
sci_get_current_position(ScintillaObject* sci )
{
	return SSM(sci, SCI_GETCURRENTPOS, 0, 0);
}

void
sci_set_current_position(ScintillaObject* sci, gint position )
{
	SSM(sci, SCI_GOTOPOS, position, 0);
}

void
sci_set_current_line(ScintillaObject* sci, gint line )
{
	SSM(sci, SCI_GOTOLINE, line, 0);
}

gint
sci_get_line_count( ScintillaObject* sci )
{
	return SSM(sci, SCI_GETLINECOUNT, 0, 0);
}

void
sci_set_selection_start(ScintillaObject* sci, gint position)
{
	SSM(sci, SCI_SETSELECTIONSTART, position, 0);
}

void
sci_set_selection_end(ScintillaObject* sci, gint position)
{
	SSM(sci, SCI_SETSELECTIONEND, position, 0);
}

gint
sci_get_line_end_from_position(ScintillaObject* sci, gint position)
{
	return SSM(sci, SCI_GETLINEENDPOSITION, position, 0);
}

void
sci_cut(ScintillaObject* sci)
{
	SSM(sci, SCI_CUT, 0, 0);
}

void
sci_copy(ScintillaObject* sci)
{
	SSM(sci, SCI_COPY, 0, 0);
}
void
sci_paste(ScintillaObject* sci)
{
	SSM(sci, SCI_PASTE, 0, 0);
}
gint
sci_get_selection_start(ScintillaObject* sci)
{
	return SSM(sci, SCI_GETSELECTIONSTART,0,0);
}

gint
sci_get_selection_end(ScintillaObject* sci)
{
	return SSM(sci, SCI_GETSELECTIONEND,0,0);
}

void
sci_replace_sel(ScintillaObject* sci, gchar* text)
{
	SSM(sci, SCI_REPLACESEL,0, (sptr_t) text);
}

gint
sci_get_length(ScintillaObject* sci)
{
	return SSM(sci,SCI_GETLENGTH,0,0);
}
gint
sci_line_length(ScintillaObject* sci,gint line)
{
	return SSM(sci,SCI_LINELENGTH, line,0);
}
void // will not be null terminated
sci_get_line(ScintillaObject* sci, gint line, gchar* text)
{
	SSM(sci,SCI_GETLINE,line, (sptr_t) text);
}
void // the last char will be null terminated
sci_get_text(ScintillaObject* sci, gint len, gchar* text)
{
	SSM( sci, SCI_GETTEXT, len,(sptr_t) text );
}

void
sci_get_selected_text(ScintillaObject* sci, gchar* text)
{
	SSM( sci, SCI_GETSELTEXT, 0, (sptr_t) text);
}

void
sci_get_xy_from_position(ScintillaObject* sci,gint pos, gint* x, gint* y)
{
	*x = SSM(sci, SCI_POINTXFROMPOSITION,0, (int) pos);
 	*y = SSM(sci, SCI_POINTYFROMPOSITION,0, (int) pos);
}
/* folding */
gboolean
sci_get_line_is_visible(ScintillaObject* sci, gint line)
{
	return SSM(sci,SCI_GETLINEVISIBLE, line,0);
}
void
sci_ensure_line_is_visible( ScintillaObject* sci, gint line)
{
	 SSM(sci,SCI_ENSUREVISIBLE,line,0);
}

gint
sci_get_fold_level(ScintillaObject* sci, gint line)
{
	return SSM(sci,SCI_GETFOLDLEVEL, line,0);
}

void
sci_toggle_fold(ScintillaObject* sci, gint line)
{
	SSM( sci, SCI_TOGGLEFOLD, line, 1);
}

gboolean
sci_get_fold_expanded(ScintillaObject* sci, gint line)
{
	return SSM( sci, SCI_GETFOLDEXPANDED, line, 0);
}

void
sci_colourise( ScintillaObject* sci, gint start, gint end)
{
	SSM( sci, SCI_COLOURISE, start, end);
}








