# containers.rb: plottable elements that hold other plottable elements
# copyright (c) 2006,2007,2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Dobjects/Dvector'
require 'CTioga/debug'
require 'CTioga/log'
require 'CTioga/plot_style'
require 'CTioga/layout'
require 'CTioga/legends'

module CTioga

  Version::register_svn_info('$Revision: 839 $', '$Date: 2008-10-11 14:47:08 +0200 (Sat, 11 Oct 2008) $')

  # A class that can hold several 'child' elements, such as
  # the SubPlot class and the Region class. A missed funcall will
  # result in a try to call the parent, if it exists.
  class Container < TiogaElement
    # The various properties defining the class
    attr_reader :elements, :funcalls

    # A rescale attribute
    attr_accessor :rescale

    # Whether or not to show legend for the current plot
    attr_accessor :show_legend

    # If this attribute is set, all legends here will be completely
    # ignored.
    attr_accessor :disable_legend

    # The frame of the element, if that element is the root. Expressed
    # in big points. Supposedly, nothing should stick out of this frame.
    # If the element is not root, this should be nil.
    #
    # As usual, [left, right, top, bottom]
    attr_accessor :root_frame

    # The layout preferences for the object, that is, useful
    # information that will be used by the object to make decisions
    # about layouts.
    attr_accessor :layout_preferences

    # The layout !!! This is the one handling the object, but
    # that does not necessarily mean that self == layout.root_object.
    # This might even seldom be the case.
    attr_accessor :layout

    # Override the layouts opinion about where the
    # object should be placed. Please note that there is no
    # need for a layout in the case when this attribute is not nil
    # and not empty. Used with care ;-) !
    attr_accessor :force_position

    # If true, the plot will keep its legend element, and
    # the ones communicated by the children for itself.
    # If false, legend informations are sent to the parent and simply
    # ignored otherwise.
    attr_accessor :accept_legend

    def initialize(p = nil)
      @parent = p

      # elements to be given to tioga
      @elements = []

      # Used to add initial function calls.
      @funcalls = []

      # Used to store legends internally. It is an array of CurveStyle
      @legend_info = []
      # Whether to accept legends or forward them to the parent
      @accept_legend = false
      # Whether or not to show them
      @show_legend = false

      # A rescale factor:
      @rescale = false

      # The root frame:
      @root_frame = nil

      @layout_preferences = LayoutPreferences.new
    end

    # Converts the current layout using Layout#convert_layout
    def convert_layout(cls)
      if @layout
        @layout = @layout.convert_layout(cls)
      else
        @layout = cls.new(self)
      end
    end

    def add_elem(elem)
      elem.parent = self
      @elements << elem

      # If the element has a style, then we assume it is a curve,
      # and add legend accordingly
      if elem.respond_to?(:style) and elem.style.has_legend?
        add_legend_info(CurveLegend.new(elem.style))
      end
    end

    def add_funcall(funcall)
      funcall.parent = self
      @funcalls << funcall
    end

    # If legends are disable, we drop the legend altogether.
    # If we accept legends, add the legend, if it is not false.
    # Else, forward it to the parents.
    #
    # _info_ may be either a CurveStyle or a LegendLine object.
    def add_legend_info(info)
      return unless info.is_a?(LegendItem)
      return if @disable_legend
      if @accept_legend
        debug "Adding legend information #{info.inspect}"
        @legend_info << info if info
      elsif parent
        debug "Forwarding legend to the parent"
        parent.add_legend_info(info)
      else
        info "Legend information dropped - this might not be expected"
      end
    end

    # Do the funcalls and some other initialization as well
    def make_funcalls(t)
      t.rescale(@rescale) if @rescale
      for f in @funcalls
        f.do(t)
      end
    end

    def has_plots?
      @elements.each {|e| 
        return true if e.is_a?(Curve2D)
        if e.respond_to? :has_plots? and e.has_plots?
          return true
        end
      }
      return false
    end

    # The internal_get_boundaries is the function responsible for
    # doing the real work behind getting the boundaries for objects.
    # Some objects might want to use some boundaries but report other
    # to their parents; they can rely on internal_get_boundaries to
    # give the right result for them and modify get_boundaries to suit
    # their needs.
    def internal_get_boundaries
      bounds = []
      @elements.each do |e|
        bounds.push(e.get_boundaries) if e.respond_to? :get_boundaries
      end
      b = Curve2D.compute_boundaries(bounds)
      return b
    end

    # This function can be used by external objects to query
    # the boundaries this object would like to report. The
    # default behaviour is to not report anything to the parent.
    def get_boundaries
      return [0.0/0.0, 0.0/0.0, 0.0/0.0, 0.0/0.0]
    end
    

    # We redirect missed funcall to the @parent
    def method_missing(meth, *args)
      if @parent
        debug "Redirecting call #{meth} to parent"
        @parent.send(meth,*args)
      else
        raise NoMethodError.new("No such method: #{meth}",meth,args)
      end
    end

    # Whether or not we should display a legend
    def display_legend?
      @show_legend && has_plots?  && has_legends? && ( not @disable_legend)
    end

  end

  # This class represent a subplot, a subfigure or the main plot.

  class SubPlot < Container
    
#     # Various textual objects laying around:
#     attr_accessor :title, :xlabel, :ylabel

#     # Ticks
#     attr_accessor :xticks, :yticks

    # various important things:
    attr_accessor :legend_specs
    
    # The actual boundaries used for making the plot. This element is only
    # filled in at the exact moment of the plotting. It is provided so that
    # subplots or curves may use it.
    attr_reader :effective_bounds
    
    # The amount of space that should be left between the given curve and
    # the box. It is an array [left,right,top,bottom].
    attr_accessor :plot_margins

    # A PlotSyle object carrying information about the style of the plot,
    # such as titles/legends/background color...
    attr_accessor :plot_style


#     # The appearance of the edgesax
#     attr_accessor :edges

    def number
      return @elements.length
    end

    # Returns a purified version of #legend_specs with only plot margins
    def plot_margins(specs)
      ret = {}
      for k in specs.keys
        if k =~ /plot_(left|right|top|bottom)_margin/
          ret[$1] = specs[k]
        end
      end
      return ret
    end
    
    def initialize(type = :subplot, parent = nil)
      super(parent)
      # @type is the type of the element, :subfigure, :subplot or nil
      # to indicate that it is the main plot ;-) !

      # the direct parent of that element. If nil, it means that it's
      # directly a subchild of the main plot.
      @parent = parent

      # Defaults to no margins (looks somehow less nice in many cases)
      @plot_margins = [0.0,0.0,0.0,0.0]
      
      # Some legend tweaking:
      if type == :subplot || type == nil
        @show_legend = true
        @accept_legend = true
      else
        @show_legend = false
        @accept_legend = false
      end

#       @title = Label.new(:title)
#       @title.label = "A nice plot" unless parent

#       @xlabel = Label.new(:xlabel)
#       @xlabel.label = "$x$" unless parent

#       @ylabel = Label.new(:ylabel)
#       @ylabel.label = "$y$" unless parent

#       @xticks = TickLabels.new(:xaxis_numeric_label)
#       @yticks = TickLabels.new(:yaxis_numeric_label)


      # the hash used by show_plot_with_legend: we inherit it from the parent
      # if it exists
      @legend_specs = if parent 
                        parent.legend_specs.dup
                      else
                        {}
                      end
     
      # User-specified boundaries (left, right, top, bottom)
      @bounds = [nil,nil,nil,nil]
      @effective_bounds = nil

#      @edges = EdgesAndAxes.new(@xticks, @yticks)

      @plot_style = PlotStyle.new(self, parent ? true : false)
    end

    # Returns the inner frames of the object in terms of absolute
    # big points.
    def inner_frame(t)
      frame = outer_frame(t)

      h = t.legend_defaults.dup
      h.update @legend_specs
      frame = Utils::apply_margin_to_frame(frame, h) if display_legend?
      return frame
    end

    # The outer frame is the reference frame for the object. It is
    # either the root frame (the whole plot) or the inner frame of the
    # parent object. Objects should refrain from writing anything
    # outside this outer frame.
    def outer_frame(t)
      # The outer frame is either  the root frame
      # or the inner_frame of the parent.
      return @root_frame || parent.inner_frame(t)
    end

    def bound_left=(a)
      @bounds[0] = a
    end

    def bound_right=(a)
      @bounds[1] = a
    end

    def bound_top=(a)
      @bounds[2] = a
    end

    def bound_bottom=(a)
      @bounds[3] = a
    end

    # Used by children to notify that an axis shoudn't be used
    # anymore.
    #
    # TODO: move this to plot_style...
    def disable_axis(axis)
      case axis
      when :x
        add_funcall(TiogaFuncall.new(:top_edge_type=, AXIS_LINE_ONLY))
      when :y
        add_funcall(TiogaFuncall.new(:right_edge_type=, AXIS_LINE_ONLY))
      end
    end


    def set_margin(which, what)
      @margins[which.to_s] = what.to_f
    end

    # this function computes the boundaries for the given plot...
    def compute_boundaries
      # We want to use the internal version of get_boundaries
      bounds = internal_get_boundaries

      # Increase the boundary with the margins
      width = bounds[1] - bounds[0]
      bounds[0] -= width * @plot_margins[0]
      bounds[1] += width * @plot_margins[1]
      height = bounds[2] - bounds[3]
      bounds[2] += height * @plot_margins[2]
      bounds[3] -= height * @plot_margins[3]

      # overriding with user-specified values if applicable
      nb_nans = 0
      4.times do |i|
        if @bounds[i] 
          bounds[i] = @bounds[i] 
        end
        # Replace NaNs with 0
        if bounds[i].nan?
          bounds[i] = 0 
          nb_nans += 1
        end
      end
      # In the case all boundaries are NaN, we provide a 0->1 mapping
      if nb_nans == 4
        return [0,1,1,0]
      end
      return bounds
    end

    # Does the plot have legends ?
    def has_legends?
      return (not @legend_info.empty?)
    end


    # this function takes a FigureMaker object and returns the block of
    # instructions corresponding to the current contents of the SubPlot.
    # To actually plot it correctly, it just needs a further wrapping
    # in subplot or subfigure.

    def make_main_block(t)
      if has_plots?
        block = proc {
          t.context {
            # First compute the boundaries:
            # We first check if the object was required to be somewhere
            if @force_position and !@force_position.empty?
              legend_specs = @force_position.to_frame("plot_%s_margin")
            else
              # Ask the layout.
              legend_specs = @layout.legend_specs(t)
            end
            # We always wrap the plot in a show_plot_with_legend,
            # even if there is no legends. That, anyway, is the
            # job of the layout to see this.
            debug "Plot specs: #{legend_specs.inspect}"
            debug "Plot legends #{display_legend?.inspect}: "
            debug " - @show_legend = #{@show_legend.inspect}"
            debug " - @disable_legend = #{@disable_legend.inspect}"
            debug " - has_plots? = #{has_plots?.inspect}"
            debug " - has_legends? = #{has_legends?.inspect}"
            debug "Reference frame (#{self.class}): #{Utils::frames_str(t)}"

            margins = plot_margins(legend_specs)
            
            t.context do
              # The call has to be wrapped within a
              # context call, as it turns the frame reference into
              # the legend subframe specification afterwards...
              t.set_subframe(margins)
              @effective_bounds = compute_boundaries
              debug "Plot boundaries: #{@effective_bounds.inspect}"
              debug "Box frame (#{self.class}): #{Utils::frames_str(t)}"
              
              @plot_style.show_edges(t, self)
              
              t.show_plot(@effective_bounds) {
                # output frame debug information./
                debug "Plot frame (#{self.class}): #{Utils::frames_str(t)}"
                if root_frame
                  debug "Root frame: #{Utils::framespec_str(root_frame)}"
                end
                
                debug "Inner frame: #{Utils::framespec_str(inner_frame(t))}"

                @plot_style.show_background(t, self)
                @plot_style.show_labels(t, self)

                @elements.each do |e|
                  e.do(t)
                end
                # We remove all the legends here, as they should all be
                # handled by the children, or forwarded here.
                t.reset_legend_info
                debug "Current figure has #{@legend_info.size} legends"
                debug "Legends #{@legend_info.inspect}"
                #                 for s in @legend_info
                #                   t.save_legend_info(s.legend_info)
                #                 end
              } 
                
              # Now, additional axes, if applicable
              t.context do 
                t.set_bounds(@effective_bounds)
                # Call the stuff for additional axes.
                @plot_style.edges.show_additional_axes(t, self)
              end

            end
            @plot_style.legend_style.show_legends(t, legend_specs, 
                                                  margins, @legend_info)
          }
        }
      else
        block = proc do
          @elements.each do |e|
            e.do(t)
          end
        end
      end

      return proc do
        make_funcalls(t)
        block.call
      end

    end

    # this function sets up the appropriate environment and calls the block...
    def do(t)
      debug "Making main block for object: #{identify(self)}"
      make_main_block(t).call
    end

  end

  # The Region class handles the cases where the user wants some parts
  # between curves to be coloured. In this case, all the curves which
  # are included between the --region and --end options will
  class Region < Container

    # Style attributes
    attr_accessor :region_color, :region_transparency
    

    # Some general options
    attr_accessor :dont_display, :region_debug

    # Filling rules:
    attr_accessor :invert_rule, :fill_twice 
    
    # We want to call the parent's disable legend.
    undef :disable_legend, :disable_legend=
    # And as well for rescale.
    undef :rescale=, :rescale
    
    def initialize(parent = nil)
      super
      @region_color = [0.9,0.9,0.9] # Very light gray.
      @region_transparency = 0
      @dont_display = false
    end

    # Fills the regions using the index to choose the closing path
    # for the first curve and opposed ones for next curves.
    def fill_region(t,i)
      # We need to wrap in a context, else the clipping path screws up
      # everyting
      t.context do 
        debug "Filling with parameter #{i}"
        bound = internal_get_boundaries
        for el in @elements
          if el.respond_to?(:make_path)
            el.make_path(t)
            # We close the path with a line coming back alternatively to
            # the top and the bottom
            close_to = bound[i % 2 + 2]
            el.close_path(t, close_to)
            i += 1
            
            # If the debugging flag is set, we heavily tamper with the
            # style of the elements to force the fill where
            # we would clip.
            if @region_debug
              el.style.fill_type = close_to
              el.style.fill_transparency = 0.8
              el.style.fill_color = el.style.color || el.style.marker_color
            end
          end
          t.clip
        end
        
        t.fill_color = @region_color
        t.fill_transparency = @region_transparency
        t.fill_rect(bound[0], bound[3], bound[1] - bound[0], 
                    bound[2] - bound[3])
      end
    end

    # Region must return its true boundaries.
    def get_boundaries
      internal_get_boundaries
    end
    
    def do(t)
      debug "Region: plotting elements"
      make_funcalls(t)
      # We fill first, as it looks way better
      if @fill_twice or (! @invert_rule)
        fill_region(t,0)
      end
      if @fill_twice or @invert_rule
        fill_region(t,1)
      end
      # Then, we plot the elements, unless we were told not to
      # by --region-dont-display
      unless @dont_display
        @elements.each do |e|
          e.do(t) 
        end
      end

    end

  end

end
