# legends.rb: legend implementation of CTioga
# copyright (c) 2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/log'
require 'CTioga/debug'
require 'CTioga/curve_style'
require 'CTioga/dimension'

module CTioga

  Version::register_svn_info('$Revision: 839 $', '$Date: 2008-10-11 14:47:08 +0200 (Sat, 11 Oct 2008) $')

  # The base class for all legends items (real legends, spacing, lines).
  class LegendItem

    include Debug
    include Log

    # This class-wide variable is used to number text
    # in a unique fashion
    @@legend_item_numbering = 0

    def initialize
      @legend_number = @@legend_item_numbering
      @@legend_item_numbering += 1
    end

    # Returns the (width, height) in figure coordinates
    # of the legend element with the given LegendStyle
    # and FigureMaker reference objects.
    #
    # The returned values can be inaccurate to some extent.
    def size(t, legend_style)
      return [0, 0]
    end

    # Draws the legend at the given top left position (x,y)
    # in figure coordinates.
    def draw(t, legend_style, x, y)
    end

    # For internal use
    def legend_name
      return "legend-#{@legend_number}"
    end

    # We put the baseline so that the space above and below in the
    # box of height legend_style.dy is even.
    def get_baseline_y(t, legend_style, y)
      return y - 0.5 * ( TextDimension.new(1.0).to_figure(t, :y)  +
                         legend_style.dy.to_figure(t, :y))
    end


  end

  # A class representing the style of a single legend line (unrelated
  # to a curve)
  class LegendLine < LegendItem

    # The text of the line
    attr_accessor :text
    
    def initialize(text = "")
      super()
      @text = text
    end

    # Draw one single text line
    def draw(t, legend_style, x, y)
      y = get_baseline_y(t, legend_style, y) 
      t.show_text('x' => x, 'y' => y, 
                  'text' => @text,
                  'justification' => LEFT_JUSTIFIED,
                  'measure' => legend_name
                  )
    end

    # Computes the size of the line. Height should always
    # be accurate, but width can be 0 sometimes...
    def size(t, legend_style)
      height = legend_style.dy.to_figure(t, :y)
      info = t.get_text_size(legend_name)
      if info.key? 'width'
        width = t.convert_output_to_figure_dx(10*info['width'])
      else
        width = 0
      end

      return [ width, height ]
    end
      
  end

  # The class handling the drawing of one Curve
  class CurveLegend < LegendItem
    
    attr_accessor :curve_style

    def initialize(style)
      super()
      @curve_style = style
    end
    
    # Draw one single text line
    def draw(t, legend_style, x, y)
      y = get_baseline_y(t, legend_style, y) 
      t.context do 
        # Position specification for the legend pictogram
        margin_specs = { 'left' => x,
          'right' => 1 - x - legend_style.picto_width.to_figure(t, :x),
          'bottom' => y,
          'top' => 1 - y - legend_style.picto_height.to_figure(t, :y)
        }
        debug "Legend margins for '#{@curve_style.legend}' : #{margin_specs.inspect}"
        t.subfigure(margin_specs) do
          # We scale the text back to normal so the markers have the right
          # size
          # t.line_width = 0.1
          # t.stroke_rect(0,0,1,1)
          t.rescale_text(1/legend_style.text_scale)
          @curve_style.output_legend_pictogram(t)
        end
      end
      t.show_text('x' => x + 
                  legend_style.picto_width.to_figure(t, :x) + 
                  legend_style.picto_to_text.to_figure(t, :x), 
                  'y' => y, 'text' => @curve_style.legend,
                  'measure' => legend_name,
                  'justification' => LEFT_JUSTIFIED)
    end

    # Computes the size of the line. Height should always
    # be accurate, but width can be 0 sometimes...
    def size(t, legend_style)
      height = legend_style.dy.to_figure(t, :y)

      width = legend_style.picto_width.to_figure(t, :x) + 
        legend_style.picto_to_text.to_figure(t, :x) 

      info = t.get_text_size(legend_name)
      
      if info.key? 'width'
        width += t.convert_output_to_figure_dx(10*info['width'])
      end

      return [ width, height ]
    end
    
    
  end


end
