# legends.rb: legend implementation of CTioga
# copyright (c) 2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/log'
require 'CTioga/debug'
require 'CTioga/curve_style'
require 'CTioga/dimension'

module CTioga

  Version::register_svn_info('$Revision: 858 $', '$Date: 2009-01-07 13:39:07 +0100 (Wed, 07 Jan 2009) $')

  # A class that provides style information and display facilities
  # for legends.
  #
  # It does not provide contents for the legends themselves
  class LegendStyle
    include Debug
    include Log

    # The distance between two legend lines
    attr_accessor :dy

    # The legend scale
    attr_accessor :scale

    # The padding between the legend and anything else around
    attr_accessor :padding

    # The width of the pictogram
    attr_accessor :picto_width

    # The height of the pictogram
    attr_accessor :picto_height

    # The distance between the pictogram and the text
    attr_accessor :picto_to_text

    # Scales all the text by that factor:
    attr_accessor :text_scale

    # Style of the box around the legend : (none/round/box)
    attr_accessor :box_style

    # The amount of space that should be left around the legend
    # when drawing the box.
    attr_accessor :margin

    # The color of the background. false means no background
    # Only meaningful when the #box_style is set to something
    # different of none.
    attr_accessor :background_color

    # The color for the lines
    attr_accessor :line_color

    # The color for the lines
    attr_accessor :line_width

    # The color for the lines
    attr_accessor :line_style
    
    # The transparency of the background
    attr_accessor :background_transparency

    def initialize
      # All in relative units ;-)...
      @dy = TextDimension.new(1.9)
      @padding = TextDimension.new(0.5)
      @picto_width = TextDimension.new(1.5)
      @picto_height = TextDimension.new(0.75)
      @picto_to_text = TextDimension.new(0.5)
      @scale = 1.0

      @box_style = false        # For no box.

      # Frame styles:
      @background_color = false
      @background_transparency = 0.0
      @line_color = Tioga::FigureConstants::Black
      @line_width = 1.0
      @line_style = false


      @margin = TextDimension.new(0.4)

    end



    # Prepares the context in which the legend is made, and run
    # the corresponding code
    def legend_context(t, legend_specs,plot_margins)
      subfigure_margins = {}

      if legend_specs.key?('legend_left_margin')
        # Now, we enter a subfigure:
        for pos in %w{left right top bottom}
          subfigure_margins[pos] = legend_specs["legend_#{pos}_margin"]
        end
        debug "Legend margins #{subfigure_margins.inspect}"
      else                      # We make a subfigure relative to the plot
        if plot_margins
          subfigure_margins = plot_margins.dup
        else
          plot_margins = false
        end
      end

      t.context do
        t.set_subframe(subfigure_margins) if subfigure_margins
        t.rescale(@scale)
        t.rescale_text(@text_scale)
        yield 
      end

    end

    # Prepare the frame path
    def prepare_frame_path(t,xl,yt, total_width, total_height)
      case @box_style
      when :square
        t.append_rect_to_path(xl - @margin.to_figure(t, :x), 
                              yt + @margin.to_figure(t, :y),
                              total_width + 2 * @margin.to_figure(t, :x), 
                              - total_height - 2 * @margin.to_figure(t, :y))   
      when :round
        # For some reason, the append_rounded_rect_to_path does not
        # work so well...
        dx = @margin.to_figure(t, :x)
        dy = @margin.to_figure(t, :y)
        xr = xl + total_width
        yb = yt - total_height
        t.move_to_point(xl - dx, yt)
        t.append_curve_to_path(xl - dx, yt + 0.5 * dy, # First control point
                               xl - 0.5 * dx, yt + dy, 
                               xl, yt + dy)
        t.append_point_to_path(xr, yt + dy)
        t.append_curve_to_path(xr + 0.5 * dx, yt + dy, # First control point
                               xr + dx, yt + 0.5 * dy, 
                               xr + dx, yt)
        t.append_point_to_path(xr + dx, yb)
        t.append_curve_to_path(xr + dx, yb - 0.5 * dy, # First control point
                               xr + 0.5 * dx, yb - dy, 
                               xr, yb - dy)
        t.append_point_to_path(xl, yb - dy)
        t.append_curve_to_path(xl - 0.5 * dx, yb - dy, # First control point
                               xl - dx, yb - 0.5 * dy, 
                               xl - dx, yb)
        t.close_path
      end
    end

    # Displays legends _legend_info_ to the FigureMaker object _t_ using
    # boundaries given by _legend_specs_ for the plot with the
    # given _plot_margins_
    def show_legends(t, legend_specs, plot_margins, legend_info)
      debug "Plotting legends: #{legend_specs.inspect}"
      @text_scale ||= t.legend_scale

      # First, we collect data on the legend_info
      widths = []
      heights = []
      total_height = 0
      total_width = 0
      legend_context(t, legend_specs, plot_margins) do 
        for item in legend_info
          w, h = item.size(t, self)
          widths << w
          heights << h
          total_height += h
          total_width = w if total_width < w
        end
      end

      debug "Total height: #{total_height}\tTotal width: #{total_width}"

      # Need to scope them now, so the context down actually modifies them !
      xl = 0
      yl = 0
      if legend_specs.key?('legend_left_margin')
        legend_context(t, legend_specs, plot_margins) do 
          yl = 1
          xl = @padding.to_figure(t, :x)
        end
      elsif legend_specs.key?('legend_left') or 
          legend_specs.key?('legend_right') or 
          legend_specs.key?('legend_hcenter')

        legend_context(t, legend_specs, plot_margins) do 
          if legend_specs.key?('legend_left')
            xl = t.convert_frame_to_figure_x(legend_specs['legend_left'])
          elsif legend_specs.key?('legend_right')
            xl = t.convert_frame_to_figure_x(legend_specs['legend_right']) - 
              total_width
          else
            xl = t.convert_frame_to_figure_x(legend_specs['legend_hcenter']) - 
              total_width * 0.5
          end
          
          if legend_specs.key?('legend_top')
            yl = t.convert_frame_to_figure_y(legend_specs['legend_top'])
          elsif legend_specs.key?('legend_bottom')
            yl = t.convert_frame_to_figure_y(legend_specs['legend_bottom']) + 
              total_height
          else
            yl = t.convert_frame_to_figure_y(legend_specs['legend_vcenter']) + 
              total_height * 0.5
          end
        end
      else
        # Silently do nothing:
        return
      end

      # Always wrapping within a context call ;-)...
      legend_context(t, legend_specs, plot_margins) do 
        # First, we draw the box if necessary 
        if @box_style
          t.context do 
            if @background_color
              t.fill_color = @background_color
              t.fill_transparency = @background_transparency
              prepare_frame_path(t, xl, yl, total_width, total_height)
              t.fill
            end
            if @line_color
              t.stroke_color = @line_color
              t.line_width = @line_width
              t.line_type = @line_style if @line_style
              prepare_frame_path(t, xl, yl, total_width, total_height)
              t.stroke
            end
          end
        end
        
        for item in legend_info
          w, h = item.size(t, self)
          item.draw(t, self, xl, yl)
          yl -= h
        end
      end
    end

  end

end
