/* file.cxx
     $Id: file.cxx,v 1.10 2001/11/27 23:57:36 elf Exp $

   written by Marc Singer
   25 October 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   Code to do file handling.  This, presently, includes editing
   filesystem files.

*/

#include "std.h"
#include "preferences.h"	// Everyone MUST get into the pref scene

#include <sys/stat.h>
#include "file.h"
#include "mman.h"
#include "path.h"

#define CB_COPY_BUFFER		(65536)
#define CB_GUESS_BUFFER		(4096) 	// 4K, one page?  8K, two pages?

/* file_copy

   copies the contents of the source file to the destination file.
   The return value is the number of bytes copies, or -1 if there is
   an error.

*/

long file_copy (const char* szFileDst, const char* szFileSrc)
{
  LPath pathSrc (szFileSrc);
  LPath pathDst (szFileDst);

  int fhSrc = -1;
  int fhDst = -1;
  void* pv = NULL;
  struct stat stat;
  long result = -1;

  if (!::stat (pathSrc, &stat)
      && stat.st_size
      && ((fhSrc = open (pathSrc, O_RDONLY)) != -1)
      && ((fhDst = open (pathDst, O_WRONLY | O_TRUNC | O_CREAT, 0644)) != -1)
      && (pv = malloc (CB_COPY_BUFFER))) {
    result = 0;
    while (stat.st_size) {
      long cb = stat.st_size;
      if (cb > CB_COPY_BUFFER)
	cb = CB_COPY_BUFFER;
      read (fhSrc, pv, cb);
      result += write (fhDst, pv, cb);
      stat.st_size -= cb;
    }
  }
  if (fhSrc != -1)
    close (fhSrc);
  if (fhDst != -1)
    close (fhDst);
  if (pv)
    free (pv);
  return result;
}

/* file_edit

   runs the user's preferred editor on the named file.  We copy the
   file to a temporary before running the editor, run it, and return
   the temporary filename if the user edited the file.  DO NOT delete
   the memory for this temporary filename.  Copy it if you need to
   hang onto it.

*/

const char* file_edit (const char* szFile)
{
  LPath path (szFile);

  const char* szEditor = g_preferences.fetch ("Editor");
  if (!szEditor)
    szEditor = getenv ("EDITOR");
  if (!szEditor)
    szEditor = getenv ("VISUAL");
  if (!szEditor)
    return NULL;

     // *** Replaced tmpnam with mkstemp, but did not smooth over the
     // code that uses it.  This isn't really cool since mkstemp is
     // designed to open the file along with making a name for it.
     // I've done this to eliminate a compiler warning.
     // Unfortunately, I'm doing an end-run around the real problem.
  static char szFileTemp[sizeof (P_tmpdir) + 20];
  strcpy (szFileTemp, P_tmpdir);
  strcat (szFileTemp, "/curvesXXXXXX");
  int fhTemp = mkstemp (szFileTemp);
  if (fhTemp != -1)
    close (fhTemp);
//  char* szFileTemp = tmpnam (NULL); 	// FIXME: this returns static buffer

  file_copy (szFileTemp, path);
  
  struct stat stat;
  ::stat (szFileTemp, &stat);
  long time = stat.st_mtime;

  char sz[256];
  sprintf (sz, "%s %s", szEditor, szFileTemp);
  system (sz);

  ::stat (szFileTemp, &stat);
  if (stat.st_mtime == time) {
    unlink (szFileTemp);
    return NULL;
    //    szFileTemp = NULL;
  }
  return szFileTemp;
}

bool file_guess_binary (const char* szFile)
{
  bool fResult = false;

  int fh = -1;
  char* pbMap = NULL;
  struct stat stat;
  long cb = CB_GUESS_BUFFER;
  if (!::stat (szFile, &stat) && stat.st_size
      && ((fh = open (szFile, O_RDONLY)) != -1)
      && (cb = ((stat.st_size < cb) ? stat.st_size : cb))
      && (pbMap = (char*) mmap (NULL, CB_GUESS_BUFFER, PROT_READ, 
				MAP_FILE | MAP_PRIVATE, fh, 0))) {
    int cNul       = 0;
    int cPrintable = 0;
    int cBinary    = 0;
    for (char* pb = pbMap; cb-- > 0; ++pb) {
      if (!*pb)
	++cNul;
      else if (isprint (*pb) || isspace (*pb))
	++cPrintable;
      else
	++cBinary;
    }    
    if (cNul > 1 || cBinary > cPrintable/3)
      fResult = true;
    {
      LPath path (szFile);
      TRACE((T_FILE_TRACE, "'%-15s' cPrintable %4d  cBinary %4d  %s", 
	 path.tail (), cPrintable, cBinary, fResult ? "Binary" : "Text"));
    }
  }
  if (pbMap)
    munmap (pbMap, CB_GUESS_BUFFER);
  if (fh != -1)
    close (fh);
  return fResult;
}
