/* Copyright 2008 Simon Richter <Simon.Richter@hogyros.de>
 *
 * Released under the GNU General Public Licence version 3.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "damnfile.hpp"

#include "integration_project.hpp"
#include "build_project.hpp"

#include "unknown_project_type.hpp"

#include <util/deb822.h>

#include <fstream>
#include <sstream>

namespace dammit {

namespace {

std::string trim_whitespace(std::string const &s)
{
	return s.substr(s.find_first_not_of(" \r\n"), s.find_last_not_of(" \r\n") + 1);
}

}

std::auto_ptr<project> load_projects_recursive(path const &root)
{
	using boost::filesystem::directory_iterator;

	std::auto_ptr<project> thisdir;

	if(exists(root / "Damnfile"))
		thisdir = load_project_file(root);

	std::auto_ptr<integration_project> everything(new integration_project("", root));

	for(directory_iterator i(root); i != directory_iterator(); ++i)
	{
		if(is_symlink(*i))
			continue;
		if(is_directory(*i))
		{
			std::auto_ptr<project> subdir(load_projects_recursive(*i));
			if(!subdir.get())
				continue;
			everything->subprojects.push_back(subdir);
		}
	}

	if(everything->subprojects.empty())
		return thisdir;

	if(thisdir.get())
		everything->subprojects.push_back(thisdir);
	
	return std::auto_ptr<project>(everything);
}

std::auto_ptr<project> load_project_file(path const &project_path)
try
{
	path const project_file = project_path / "Damnfile";

	std::ifstream s;

	s.exceptions(std::ios::badbit);

	s.open(project_file.string().c_str());

	if(s)
	{
		util::deb822_parser parser(s);

		if(parser->key != "Project")
			// TODO: error handling
			return std::auto_ptr<project>();

		std::auto_ptr<build_project> project(new build_project(parser->value, project_path));

		project->project_type = build_project::program;

		for(boost::filesystem::directory_iterator i(project_path); i != boost::filesystem::directory_iterator(); ++i)
		{
			if(!is_directory(*i))
				project->input_files.push_back(i->leaf());
		}

		++parser;

		while(parser)
		{
			if(parser->key == "Type")
			{
				if(parser->value == "program")
					project->project_type = build_project::program;
				else if(parser->value == "library")
					project->project_type = build_project::library;
				else
					throw unknown_project_type(parser->value);
			}
			else if(parser->key == "Libraries")
			{
				std::istringstream lib_stream(parser->value);
				for(std::string lib; std::getline(lib_stream, lib, ',');)
					project->libraries.push_back(trim_whitespace(lib));
			}
			else if(parser->key == "Platform:")
			{
				std::istringstream pf_stream(parser->value);
				for(std::string pf; std::getline(pf_stream, pf, ',');)
					project->platforms.push_back(trim_whitespace(pf));
			}
			else
				// TODO: error handling
				break;
			++parser;
		}
		return std::auto_ptr<dammit::project>(project);
	}
	return std::auto_ptr<project>();
}
catch(std::ios::failure &)
{
	return std::auto_ptr<project>();
}

}
