/*
 * Copyright (C) 2003 David Roundy
 * Most of the UTF code is Copyright (C) 1999-2001 Free Software Foundation, Inc.
 * This file is part of darcs.
 *
 * Darcs is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Library General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the GNU LIBICONV Library; see the file COPYING.LIB.
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place -
 * Suite 330, Boston, MA 02111-1307, USA.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>

#ifdef _WIN32
#include <windows.h>
#else
#include <sys/mman.h>
#endif

#include "fpstring.h"

int wfindps_helper(char c, const char *s, int len) {
  int i;
  for (i=0;i<len;i++)
    if (s[i] == c) return i;
  return -1;
}

static int my_is_space(char c) {
  /* I don't use the C library isspace function because I don't want to
     have the answer depend on the locale, or a patch file could mean
     different things on different computers.

     Also, as long as I'm writing my own isspace, I've decided to leave out
     '\f' and '\v' for just a tad more speed, and since I don't anticipate
     using these characters in patches.

     And yes, I'm assuming the compiler will inline this function.
  */
  return c == ' ' || c == '\n' || c == '\t' || c == '\r';
}

int first_white(const char *s, int len) {
  int i;
  for (i=0;i<len;i++)
    if (my_is_space(s[i])) return i;
  return len;
}

int first_nonwhite(const char *s, int len) {
  int i;
  for (i=0;i<len;i++)
    if (!my_is_space(s[i])) return i;
  return len;
}

static int is_funky(char c) {
  /* these are chars we don't want in text files. */
  return c == 0 || c == 26;
}

int has_funky_char(const char *s, int len) {
  int i;
  for (i=0;i<len;i++)
    if (is_funky(s[i])) return 1;
  return 0;
}

// mmapping...

#ifdef _WIN32

/* I have no idea if this works or not, and it is very tied to the usage
 * of mmap in FastPackedString. Most arguments are ignored...
 */

char *my_mmap(int length, int fd)
{
    HANDLE file = (HANDLE)_get_osfhandle(fd);
    HANDLE hnd = CreateFileMapping(file, 0, PAGE_READONLY, 0, 0, 0);
    DWORD offhi = 0, offlo = 0;
    SIZE_T size = 0;
    char *p = MapViewOfFile(hnd, FILE_MAP_READ, offhi, offlo, size);
    return p;
}

int munmap(void *start, size_t length)
{
    UnmapViewOfFile(start);
}

#else

char *my_mmap(int len, int fd) {
  void *maybeok = mmap(NULL, len, PROT_READ, MAP_SHARED, fd, 0);
  if (maybeok == MAP_FAILED) return NULL;
  else return (char *)maybeok;
}

#endif

#define GAP (500*1024*1024)
#define MAX(a,b) ((a>b)?a:b)
static char *border = NULL;

char *dumb_malloc(int len) {
  char *filler, *out;
  if (!border) {
    filler = (char *) malloc(GAP);
    border = filler + GAP;
    free(filler);
  }
  filler = (char *) malloc(MAX(GAP,len));
  if (filler >= border) {
    filler = (char *) realloc(filler,len);
    if (filler < border) printf("Argh!!!\n");
    return filler;
  }
  out = (char *) malloc(MAX(GAP,len));
  out = (char *) realloc(out,len);
  free(filler);
  return out;
}

// ForeignPtr debugging stuff...

static int num_alloced = 0;

void debug_free(void *p) {
  num_alloced--;
  fprintf(stderr, "Freeing %p (%d left)\n", p, num_alloced);
}

void debug_alloc(void *p, const char *name) {
  num_alloced++;
  fprintf(stderr, "Allocating %p named %s (%d left)\n",
          p, name, num_alloced);
}

/* Specification: RFC 2279 */

int utf8_to_ints(HsInt *pwc, const unsigned char *s, int n) {
  /* returns number of unicode chars in the output.  The output array is
     assumed to have the same number of elements as the input array, which
     is n. */

  HsInt *pwc_original = pwc;
  while (n > 0) {
    unsigned char c = s[0];

    if (c < 0x80) {
      *pwc++ = c;
      n--;
      s++;
    } else if (c < 0xc2) {
      return -1;
    } else if (c < 0xe0) {
      if (n < 2) return -1;
      if (!((s[1] ^ 0x80) < 0x40)) return -1;
      *pwc++ = ((unsigned) (c & 0x1f) << 6)
        | (unsigned) (s[1] ^ 0x80);
      n -= 2;
      s += 2;
    } else if (c < 0xf0) {
      if (n < 3) return -1;
      if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40
            && (c >= 0xe1 || s[1] >= 0xa0)))
        return -1;
      *pwc++ = ((unsigned) (c & 0x0f) << 12)
        | ((unsigned) (s[1] ^ 0x80) << 6)
        | (unsigned) (s[2] ^ 0x80);
      n -= 3;
      s += 3;
    } else if (c < 0xf8 && sizeof(unsigned)*8 >= 32) {
      if (n < 4) return -1;
      if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40
            && (s[3] ^ 0x80) < 0x40
            && (c >= 0xf1 || s[1] >= 0x90)))
        return -1;
      *pwc++ = ((unsigned) (c & 0x07) << 18)
        | ((unsigned) (s[1] ^ 0x80) << 12)
        | ((unsigned) (s[2] ^ 0x80) << 6)
        | (unsigned) (s[3] ^ 0x80);
      n -= 4;
      s += 4;
    } else if (c < 0xfc && sizeof(unsigned)*8 >= 32) {
      if (n < 5) return -1;
      if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40
            && (s[3] ^ 0x80) < 0x40 && (s[4] ^ 0x80) < 0x40
            && (c >= 0xf9 || s[1] >= 0x88)))
        return -1;
      *pwc++ = ((unsigned) (c & 0x03) << 24)
        | ((unsigned) (s[1] ^ 0x80) << 18)
        | ((unsigned) (s[2] ^ 0x80) << 12)
        | ((unsigned) (s[3] ^ 0x80) << 6)
        | (unsigned) (s[4] ^ 0x80);
      n -= 5;
      s += 5;
    } else if (c < 0xfe && sizeof(unsigned)*8 >= 32) {
      if (n < 6) return -1;
      if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40
            && (s[3] ^ 0x80) < 0x40 && (s[4] ^ 0x80) < 0x40
            && (s[5] ^ 0x80) < 0x40
            && (c >= 0xfd || s[1] >= 0x84)))
        return -1;
      *pwc++ = ((unsigned) (c & 0x01) << 30)
        | ((unsigned) (s[1] ^ 0x80) << 24)
        | ((unsigned) (s[2] ^ 0x80) << 18)
        | ((unsigned) (s[3] ^ 0x80) << 12)
        | ((unsigned) (s[4] ^ 0x80) << 6)
        | (unsigned) (s[5] ^ 0x80);
      n -= 6;
      s += 6;
    } else
      return -1;
  }
  return pwc - pwc_original;
}

/* Conversion to and from hex */

static unsigned char num2hex(unsigned char c) {
  if (c < 10) return '0' + c;
  else return ('a' - 10) + c;
}

static unsigned char hex2num(unsigned char c) {
  if (c - '0' >= 0 && c - '0' < 10) return c - '0';
  else return c - ('a' - 10);
}

void conv_to_hex(unsigned char *dest, unsigned char *from, int num_chars) {
  int i;
  for (i=0;i<num_chars;i++) {
    const unsigned char thec = from[i];
    dest[i*2] = num2hex(thec/16);
    dest[i*2+1] = num2hex(thec%16);
  }
}

void conv_from_hex(unsigned char *dest, unsigned char *from, int num_chars) {
  int i;
  for (i=0;i<num_chars;i++)
    dest[i] = hex2num(from[i*2])*16 + hex2num(from[i*2+1]);
}

