%  Copyright (C) 2002-2003,2005 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs resolve}
\begin{code}
module Resolve ( resolve ) where
import System ( ExitCode(..), exitWith )
import SignalHandler ( withSignalsBlocked )
import Control.Monad ( when )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag( AnyOrder ),
                        ignoretimes, verbose, working_repo_dir,
                      )
import Repository ( withRepoLock, amInRepository,
                    read_repo, sync_repo, get_unrecorded, with_new_pending,
                    )
import Patch ( join_patches, invert, apply, is_null_patch )
import Resolution ( patchset_conflict_resolutions )
import SelectChanges ( promptChar )
#include "impossible.h"
\end{code}
\begin{code}
resolve_description :: String
resolve_description =
 "Mark any conflicts to the working copy for manual resolution."
\end{code}

\options{resolve}

\haskell{resolve_help}

\begin{code}
resolve_help :: String
resolve_help =
 "Resolve is used to mark and resolve any conflicts that may exist in a\n"++
 "repository.  Note that this trashes any unrecorded changes in the working\n"++
 "copy.\n"
\end{code}
\begin{code}
resolve :: DarcsCommand
resolve = DarcsCommand {command_name = "resolve",
                        command_help = resolve_help,
                        command_description = resolve_description,
                        command_extra_args = 0,
                        command_extra_arg_help = [],
                        command_command = resolve_cmd,
                        command_prereq = amInRepository,
                        command_get_arg_possibilities = return [],
                        command_argdefaults = nodefaults,
                        command_darcsoptions = [verbose, ignoretimes,
                                                working_repo_dir]}
\end{code}

\begin{code}
resolve_cmd :: [DarcsFlag] -> [String] -> IO ()
resolve_cmd opts [] = withRepoLock $ \repository -> do
  pend <- get_unrecorded repository (AnyOrder:opts)
  r <- read_repo repository
  let res = join_patches $ map head $ patchset_conflict_resolutions r
  when (is_null_patch res) $ do
      putStrLn "No conflicts to resolve."
      exitWith ExitSuccess
  case pend of
    Nothing -> return ()
    Just p -> do yorn <- promptChar
                         ("This will trash any unrecorded changes"++
                          " in the working directory.\nAre you sure? ") "yn"
                 when (yorn /= 'y') $ exitWith ExitSuccess
                 apply opts (invert p) `catch` \e ->
                    bug ("Can't undo pending changes!" ++ show e)
                 sync_repo repository
  withSignalsBlocked $ with_new_pending repository res $
      apply opts res `catch` \e ->
      bug ("Problem resolving conflicts in resolve!" ++ show e)
  putStrLn "Finished resolving."
resolve_cmd _ _ = impossible
\end{code}

