/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "Darkroom.h"

#include <unistd.h>

#include <QDockWidget>
#include <QDropEvent>
#include <QPainter>
#include <QPrinter>
#include <QProgressBar>
#include <QTimeLine>

#include <KAction>
#include <KActionCollection>
#include <KDebug>
#include <KFileDialog>
#include <KMessageBox>
#include <KMenu>
#include <KLocale>
#include <KStandardAction>
#include <KStatusBar>
#include <KStandardDirs>

#include <threadweaver/WeaverObserver.h>
#include <threadweaver/ThreadWeaver.h>

#include <libkdcraw/dcrawsettingswidget.h>
#include <libkdcraw/rawfiles.h>

#include "DarkroomView.h"
#include "JobExport.h"
#include "JobPreview.h"
#include "JobThumbnail.h"
#include "ListRawFileModel.h"
#include "RawImageInfo.h"
#include "PostProcessor.h"

#include "ui_BatchProcessWidget.h"
#include "ui_ExistingFilesDialog.h"
#include "ui_PostProcessorWidget.h"

struct Darkroom::Private {
  KDcrawIface::DcrawSettingsWidget* rawOptionsWidget;
  DarkroomView *view;
  QListView* fileListView;
  KToggleAction *toolbarAction;
  KToggleAction *statusbarAction;
  RawImageInfo* currentRawFile;
  QProgressBar* progressBar;
  QTimeLine progressBarTimeLine;
  Ui::PostProcessorWidget postProcessorWidget;
};

Darkroom::Darkroom()
    : KXmlGuiWindow(), d(new Private)
{
    KGlobal::mainComponent().dirs()->addResourceType("icc_profiles", 0, "share/color/icc/");
 
    d->rawOptionsWidget = 0;
    d->view = new DarkroomView(this);
    d->currentRawFile = 0;
    
    // accept dnd
    setAcceptDrops(true);

    // tell the KXmlGuiWindow that this is indeed the main widget
    setCentralWidget(d->view);

    // then, setup our actions
    setupActions();

    // add a status bar
    statusBar()->show();
    
    // Create the progress bar
    d->progressBar = new QProgressBar(statusBar());
    d->progressBar->setMinimum( 0 );
    d->progressBar->setMaximum( 100 );
    d->progressBar->setVisible( false );
    d->progressBar->setTextVisible( false );
    d->progressBar->setMaximumWidth( 200 );
    d->progressBarTimeLine.setDuration( 10000 );
    d->progressBarTimeLine.setFrameRange( 0, 100 );
    d->progressBarTimeLine.start();
    d->progressBarTimeLine.setLoopCount( 0 );
    connect( &d->progressBarTimeLine, SIGNAL(frameChanged ( int ) ), d->progressBar, SLOT(setValue( int ) ) );
    statusBar()->addPermanentWidget( d->progressBar );

    // a call to KXmlGuiWindow::setupGUI() populates the GUI
    // with actions, using KXMLGUI.
    // It also applies the saved mainwindow settings, if any, and ask the
    // mainwindow to automatically save settings if changed: window size,
    // toolbar position, icon size, etc.
    setupGUI();
    
    // Initialize dockers
    setupDockers();
    setCorner(Qt::TopLeftCorner, Qt::LeftDockWidgetArea );
    setCorner(Qt::BottomLeftCorner, Qt::LeftDockWidgetArea );
    setCorner(Qt::TopRightCorner, Qt::RightDockWidgetArea );
    setCorner(Qt::BottomRightCorner, Qt::RightDockWidgetArea );
    
    // Initialize weaver observer
    observeWeaver( JobPreview::weaver() );
    observeWeaver( JobExport::weaver() );
    observeWeaver( JobThumbnail::weaver() );
}

Darkroom::~Darkroom()
{
}

void Darkroom::setupActions()
{
    KStandardAction::open(this, SLOT(fileOpen()), actionCollection());

    // custom menu and menu item - the slot is in the class DarkroomView
    KAction *custom = new KAction(KIcon("document-open-folder"), i18n("Open &directory..."), this);
    actionCollection()->addAction( QLatin1String("open_directory_action"), custom );
    connect(custom, SIGNAL(triggered(bool)), this, SLOT(fileOpenDir()));

    KStandardAction::saveAs(d->view, SLOT(fileSaveAs()), actionCollection())->setShortcut( KShortcut( Qt::CTRL + Qt::Key_S ) );
    KStandardAction::quit(qApp, SLOT(closeAllWindows()), actionCollection());

    // Zoom actions
    KStandardAction::zoomIn(d->view, SLOT(zoomIn()), actionCollection());
    KStandardAction::zoomOut(d->view, SLOT(zoomOut()), actionCollection());
    KStandardAction::fitToPage(d->view, SLOT(fitToPage()), actionCollection());
    KStandardAction::actualSize(d->view, SLOT(actualSize()), actionCollection());
    
    // Refresh preview
    KAction* refreshPreviewAction = new KAction(KIcon("view-refresh"), i18n("Refresh preview"), this );
    actionCollection()->addAction( QLatin1String("refresh_preview"), refreshPreviewAction );
    connect(refreshPreviewAction, SIGNAL(triggered(bool)), d->view, SLOT(updatePreview()));
}

void Darkroom::setupDockers()
{
  // Setup RAW options dockers
  QDockWidget* rawOptionsDockWidget = new QDockWidget(i18n("Decoding Room"), this);
  d->rawOptionsWidget = new KDcrawIface::DcrawSettingsWidget( rawOptionsDockWidget, true, true, true );
  rawOptionsDockWidget->setWidget( d->rawOptionsWidget );
  d->rawOptionsWidget->setDefaultSettings();
  d->rawOptionsWidget->setSixteenBits( true );
  addDockWidget(Qt::LeftDockWidgetArea, rawOptionsDockWidget);
  
  // Setup Post Processor dockers
  QDockWidget* postProcessorDockWidget = new QDockWidget(i18n("Post Processing Room"), this);
  QWidget* postProcessWidget = new QWidget( postProcessorDockWidget );
  postProcessorDockWidget->setWidget( postProcessWidget );
  d->postProcessorWidget.setupUi( postProcessWidget );
  d->postProcessorWidget.exposureInput->setRange(-10, 10);
  addDockWidget( Qt::LeftDockWidgetArea, postProcessorDockWidget );
  tabifyDockWidget( rawOptionsDockWidget, postProcessorDockWidget );
  
  // Setup batchProcessor widget
  QDockWidget* batchProcessDockWidget = new QDockWidget(i18n("Batch Process"), this);
  QWidget* batchProcessWidget = new QWidget( batchProcessDockWidget );
  batchProcessDockWidget->setWidget( batchProcessWidget );
  Ui::BatchProcessWidget* bpw = new Ui::BatchProcessWidget;
  bpw->setupUi( batchProcessWidget );
  d->fileListView = bpw->listRawFiles;
  d->fileListView->setContextMenuPolicy( Qt::CustomContextMenu );
  connect( bpw->listRawFiles, SIGNAL(activated(const QModelIndex&)), SLOT(rawFileActivated(const QModelIndex& ) ) );
  connect( d->fileListView, SIGNAL(customContextMenuRequested( const QPoint & ) ), this, SLOT(customFileViewContextMenuRequested( const QPoint & ) ));
  addDockWidget(Qt::TopDockWidgetArea, batchProcessDockWidget);
}

void Darkroom::rawFileActivated ( const QModelIndex & index )
{
  delete d->currentRawFile;
  d->currentRawFile = 0;
  QVariant variant = d->fileListView->model()->data( index, 100);
  RawImageInfo* fpo = variant.value<RawImageInfo*>();
  d->view->setRawImageInfo( fpo );
}

void Darkroom::fileOpenDir()
{
  QString directoryName = KFileDialog::getExistingDirectory();
  if( not directoryName.isEmpty() and not directoryName.isNull())
  {
    openDir( directoryName );
  }
}

void Darkroom::openDir( const QString& _directoryName )
{
  d->fileListView->setEnabled( true );
  QDir directory( _directoryName );
  QStringList files = directory.entryList( QString(raw_file_extentions).split(" "), QDir::Readable | QDir::Files, QDir::Name | QDir::IgnoreCase );
  QList<RawImageInfo*> entries;
  foreach(QString file, files)
  {
    entries.push_back( new RawImageInfo( QFileInfo( directory, file ) ) );
  }
  d->fileListView->setModel( new ListRawFileModel( entries) );
}

void Darkroom::fileOpen()
{
  QString file = KFileDialog::getOpenFileName( KUrl(), raw_file_extentions );
  if( not file.isEmpty() and not file.isNull() )
  {
    openFile( file );
  }
}

void Darkroom::openFile( const QString& _file )
{
  delete d->currentRawFile;
  d->currentRawFile = new RawImageInfo( QFileInfo( _file ) );
  d->fileListView->setEnabled( false );
  d->view->setRawImageInfo( d->currentRawFile );
}

void Darkroom::openUrl(const QString& _url)
{
  QFileInfo fi(_url);
  if( fi.isDir() )
  {
    openDir( _url );
  } else if( fi.isFile() )
  {
    openFile( _url );
  } else {
    KMessageBox::error( this, i18n("Can't open file %1.", _url) );
  }
}

KDcrawIface::RawDecodingSettings Darkroom::rawDecodingSettings()
{
  Q_ASSERT( d->rawOptionsWidget );
  KDcrawIface::RawDecodingSettings settings;
  settings.sixteenBitsImage = d->rawOptionsWidget->sixteenBits();
  settings.brightness = d->rawOptionsWidget->brightness();
  settings.RAWQuality = d->rawOptionsWidget->quality();
  settings.outputColorSpace = d->rawOptionsWidget->outputColorSpace();
  settings.RGBInterpolate4Colors = d->rawOptionsWidget->useFourColor();
  settings.DontStretchPixels = d->rawOptionsWidget->useDontStretchPixels();
  settings.unclipColors = d->rawOptionsWidget->unclipColor();
  settings.whiteBalance = d->rawOptionsWidget->whiteBalance();
  settings.customWhiteBalance = d->rawOptionsWidget->customWhiteBalance();
  settings.customWhiteBalanceGreen = d->rawOptionsWidget->customWhiteBalanceGreen();
  
  settings.enableBlackPoint = d->rawOptionsWidget->useBlackPoint();
  settings.blackPoint = d->rawOptionsWidget->blackPoint();
  
  settings.enableNoiseReduction = d->rawOptionsWidget->useNoiseReduction();
  settings.NRThreshold = d->rawOptionsWidget->NRThreshold();
  
  settings.enableCACorrection = d->rawOptionsWidget->useCACorrection();
  settings.caMultiplier[0] = d->rawOptionsWidget->caRedMultiplier();
  settings.caMultiplier[1] = d->rawOptionsWidget->caBlueMultiplier();

  
  return settings;
}

PostProcessor Darkroom::postProcessor()
{
  return PostProcessor( d->postProcessorWidget.exposureInput->value(), d->postProcessorWidget.convertToSRGB->isChecked() );
}

void Darkroom::closeEvent(QCloseEvent *event)
{
  if( not idling() )
  {
    QMessageBox messageBox(QMessageBox::Question, i18n("Exit Darkroom"), i18n("Darkroom is still processing some images, are you sure you want to quit ? All unfinished processing will be lost."), QMessageBox::Yes | QMessageBox::No, this);
    messageBox.addButton("Wait", QMessageBox::YesRole );
    messageBox.setDefaultButton( QMessageBox::No );
    int result = messageBox.exec();
    switch( result )
    {
      case QMessageBox::No:
        event->ignore();
        return;
        break;
      case QMessageBox::Yes:
        break;
      default:
      {
        setEnabled( false );
        while(not idling() )
        {
          QApplication::processEvents( );
          usleep(100);
        }
      }
    }
  }
  event->accept();
}

void Darkroom::customFileViewContextMenuRequested( const QPoint& pos )
{
  KMenu menu(d->fileListView);
  menu.addAction(i18n("Apply current settings."), this, SLOT(applyCurrentSettings()));
  menu.exec(d->fileListView->mapToGlobal(pos ));
}

void Darkroom::weaverStatusChanged()
{
  if( idling() )
  {
    d->progressBar->setVisible( false );
  } else {
    d->progressBar->setVisible( true );
  }
}

#define MAKE_FILE_NAME \
  QFileInfo info = fpo->fileInfo(); \
  QString dstFileName = info.absolutePath() + "/" + info.baseName() + ".png";


void Darkroom::applyCurrentSettings()
{
  kDebug() << "Darkroom";
  QList<RawImageInfo*> riis;
  QStringList existingFiles;
  foreach( QModelIndex index, d->fileListView->selectionModel()->selectedIndexes () )
  {
    QVariant variant = d->fileListView->model()->data( index, 100);
    RawImageInfo* fpo = variant.value<RawImageInfo*>();
    MAKE_FILE_NAME
    if(fpo) {
      riis.push_back( fpo );
      if(QFileInfo( dstFileName).exists())
      {
        existingFiles.push_back( dstFileName );
      }
    }
  }
  if( not existingFiles.empty() )
  {
    QDialog dialog(this);
    Ui::ExistingFilesDialog efd;
    efd.setupUi( &dialog );
    foreach( QString existingFile, existingFiles)
    {
      efd.listExistingFiles->addItem( existingFile );
    }
    if( dialog.exec() == QDialog::QDialog::Rejected )
    {
      return;
    }
  }
  foreach( RawImageInfo* fpo, riis )
  {
    MAKE_FILE_NAME
    JobExport::weaver()->enqueue( new JobExport( fpo, dstFileName, rawDecodingSettings(), postProcessor() ) );
  }
}

bool Darkroom::idling()
{
  return  JobPreview::weaver()->isIdle() and
          JobExport::weaver()->isIdle() and
          JobThumbnail::weaver()->isIdle();
}

void Darkroom::observeWeaver( ThreadWeaver::Weaver* weaver )
{
  ThreadWeaver::WeaverObserver* observer = new ThreadWeaver::WeaverObserver( this );
  weaver->registerObserver( observer );
  connect( observer, SIGNAL(threadExited (ThreadWeaver::Thread *)), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(threadStarted (ThreadWeaver::Thread *)), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(weaverStateChanged (ThreadWeaver::State *)), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(threadSuspended (ThreadWeaver::Thread *)), SLOT(weaverStatusChanged()));
}

#include "Darkroom.moc"
