#include <string>
#include <vector>

using std::string;
using std::vector;

class DBPooledConnection;
class DBPostgresBackend;
class DBPostgresFrontend;
class DBBalancerConfig;
class DBHostConfig;

class DBPostgresWriterConnection: public virtual DBPooledConnection {

public:

  DBPostgresWriterConnection(DBBalancerConfig* config, vector<DBHostConfig *> hosts, string id);
  ~DBPostgresWriterConnection();

  // Implementacin del interfaz DBPooledConnection

  bool handleConnection(int socket);
  bool checkConnection();
  bool isFree();
  bool isValid();
  DBPooledConnection *usable();
  DBPooledConnection *testable();

  void closeConnection();

private:

  // Methods implementing the Postgres protocol;
  
  // Initializes the connection readying it for receiving connection.
  void cleanUp();
  bool interceptFrontEndMessage(char *message, int size,  bool first);
  // Error handling
  bool handleConnectionFrontendError(string message, bool state, int socket);
  bool handleConnectionBackendError(string message, DBPostgresBackend *be, int socket);
  bool handleConnectionSyncError(string message, int socket);
  // Connection close handle
  bool handleConnectionClose(int socket);
  // Consolidate data from BEs and return to FE
  int checkAndSendData(int fe_socket);
  bool processBackendExecutionError(int fe_socket);
  bool processBackendExecutionErrorWhenEqual(int fe_socket);
  bool processBackendExecutionErrorWhenNotEqual(int fe_socket);

  // Returns the first character of the BE buffers, or NULL if they're not the same.
  char getBEBuffersFirstCharacter();
  // Clears all the backends buffers;
  void resetBEBuffers();
  // Broadcast a transaction command to the BEs
  unsigned int sendTransactionCommand(string command, int socket);

  // Private attributes.

  const static unsigned int SEND_FLAGS = 0;
  const static unsigned int RECV_FLAGS = 0;
  const static unsigned int BUFFER_SIZE = 4096;

  const static unsigned int MAX_DISCARDED_READS = 2000;


  // FE connection data.
  DBPostgresFrontend* _fe;

  // BE connection data.
  vector<DBPostgresBackend *> _be;
  vector<int> _valid_connections;

  // Connections state
  int _nhosts;

  // State of the whole connection.
  bool _valid;
  bool _free;
  bool _new_command; // A new command has been received in the FrontEnd
  
  // Connection global lock.
  ACE_Thread_Mutex _mutex;

  // Highest socket descriptor for use in "select".
  int _highest_socket_descriptor;

  // Whether to use a transaction for each connection
  bool _writerTransactionPerRequest;

};

