(*
   Parser.pas - Log file parser unit

   Copyright (C) 2013 - 2015 Martin Bittermann (martinbittermann@gmx.de)

   This file is part of ddrescueview.

   ddrescueview is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   ddrescueview is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with ddrescueview.  If not, see <http://www.gnu.org/licenses/>.
*)

unit Parser;

interface
uses Classes, Shared;

type
  TSimpleParser = class(TObservablePersistent)
  private
    FLog : TLog;
    FRescueStatus : TRescueStatus;
    FComments : TStringList; // comment lines of logfile
    FVersion : String; // ddrescue version in first comment line / for future use?
    FLogStream : TStream;
    FFileName : String;
    procedure logMsg(msg : String);
    function isEmptyLine(strLine : String) : boolean;
    function isCommentLine(strLine : String) : boolean;
  public
    constructor Create;
    destructor Destroy; override;
    procedure OpenFile(filename : String);
    procedure CloseFile;
    procedure parse();
    function hasFile() : boolean;
    property rescueStatus : TRescueStatus read FRescueStatus;
    property log : TLog read FLog;
    property CommentLines : TStringList read FComments;
    property Version : String read FVersion;
    property LogFileName :String read FFileName;
  end;

  // This parser can also embed a domain log file parser
  TLogParser = class(TObservablePersistent)
  private
    FLogParser : TSimpleParser;
    FDomParser : TSimpleParser;
    FLog: TLog;
    FRescueStatus : TRescueStatus;
    FContiguous : Boolean;
    FHasFile : Boolean;
    function getLog(): TLog;
    procedure postParse();
    procedure setContiguous(mode: Boolean);
  public
    constructor Create;
    destructor Destroy; override;
    procedure OpenFile(filename : String);
    procedure OpenDomainFile(filename : String);
    procedure CloseFile;
    procedure parse();
    function hasFile() : boolean;
    function hasDomFile() : Boolean;
    property rescueStatus : TRescueStatus read FRescueStatus;
    property log : TLog read getLog;
    property CommentLines : TStringList read FLogParser.FComments;
    property Version : String read FLogParser.FVersion;
    property LogFileName : String read FLogParser.FFileName;
    property DomFileName : String read FDomParser.FFileName;
    property ContiguousDomain : Boolean read FContiguous write setContiguous;
  end;

implementation
uses SysUtils, Math, GUI;

{ TLogParser }

constructor TLogParser.Create;
begin
  inherited;
  FRescueStatus := emptyRescueStatus;
  FLogParser:=TSimpleParser.Create;
  FDomParser:=TSimpleParser.Create;
end;

destructor TLogParser.Destroy;
begin
  CloseFile();
  FreeAndNil(FDomParser);
  FreeAndNil(FLogParser);
  inherited Destroy;
end;

procedure TLogParser.OpenFile(filename: String);
begin
  CloseFile;
  FLogParser.OpenFile(filename);
  postParse;
end;

procedure TLogParser.OpenDomainFile(filename: String);
begin
  FDomParser.OpenFile(filename);
  if FDomParser.hasFile and FContiguous then begin
    FHasFile:=false;
    FPONotifyObservers(self, ooDeleteItem, nil);
  end;
  postParse;
end;

procedure TLogParser.CloseFile;
begin
  FDomParser.CloseFile;
  FLogParser.CloseFile;
  FRescueStatus := emptyRescueStatus;
  SetLength(FLog, 0);
  FHasFile:=false;
  FPONotifyObservers(self, ooDeleteItem, nil);
end;

procedure TLogParser.postParse;
var
  ResLog, DomLog: TLog;
  i, iRes, iDom : Longint;
  newEntry : TLogEntry;
  logCurOffset : Int64;
begin
  if FDomParser.hasFile then begin // post-process the logs
    FRescueStatus:=emptyRescueStatus;
    ResLog:=FLogParser.log;
    DomLog:=FDomParser.log;
    //build log and rescuestatus
    i:=0; iRes:=0; iDom:=0; // current log indices
    logCurOffset:=0; // needed for contiguous mode, but also used for normal mode
    while iDom < Length(DomLog) do begin
      if (DomLog[iDom].status<>'+') then begin // outside domain entry
        if not FContiguous then begin
          newEntry.status:='d';
          newEntry.offset:=DomLog[iDom].offset;
          newEntry.length:=DomLog[iDom].length;
          inc(FRescueStatus.outsidedomain, DomLog[iDom].length);
          logCurOffset:=newEntry.offset+newEntry.length;
          if InRange(FLogParser.rescueStatus.pos, newEntry.offset, logCurOffset-1) then
             FRescueStatus.pos:=FLogParser.rescueStatus.pos;
          if Length(FLog) <= i then SetLength(FLog, i+1024);
          FLog[i]:=newEntry;
          Inc(i);
        end;
      end else begin // iterate over the ResEntries intersecting current DomEntry
        while iRes < Length(ResLog) do begin
          // increase iRes until intersecting dom block
          if ResLog[iRes].offset+ResLog[iRes].length <= DomLog[iDom].offset then begin
            Inc(iRes);
            Continue;
          end;
          // if intersecting current dom block, create new block
          newEntry:=IntersectEntries(ResLog[iRes], DomLog[iDom]);
          if newEntry.length > 0 then begin //can be 0 for rescue logfiles with gaps
            if InRange(FLogParser.rescueStatus.pos, newEntry.offset,
                       newEntry.offset+newEntry.length-1) then
              FRescueStatus.pos:=logCurOffset+FLogParser.rescueStatus.pos-newEntry.offset;
            newEntry.offset:=logCurOffset;
            logCurOffset+=newEntry.length;
            if Length(FLog) <= i then SetLength(FLog, i+1024);
            FLog[i]:=newEntry;
            Inc(i);
            case newEntry.status of
              '?' : inc(FRescueStatus.nontried, newEntry.length);
              '+' : inc(FRescueStatus.rescued, newEntry.length);
              '*' : inc(FRescueStatus.nontrimmed, newEntry.length);
              '/' : inc(FRescueStatus.nonscraped, newEntry.length);
              '-' : begin
                      inc(FRescueStatus.bad, newEntry.length);
                      inc(FRescueStatus.errors);
                    end;
            end;
          end;
          // ensure iRes is set to last intersecting entry
          if ResLog[iRes].offset+ResLog[iRes].length >=
             DomLog[iDom].offset+DomLog[iDom].length then break;
          Inc(iRes);
        end;
      end;
      Inc(iDom);
    end;
    SetLength(FLog, i);
    FRescueStatus.curOperation:=FLogParser.rescueStatus.curOperation;
    FRescueStatus.strCurOperation:=FLogParser.rescueStatus.strCurOperation;
    if i > 0 then begin  // calculate device's size
      FRescueStatus.devicesize:=FLog[i-1].offset + FLog[i-1].length;
    end;
    FRescueStatus.suggestedBlockSize:=FLogParser.rescueStatus.suggestedBlockSize;
  end else begin
    FRescueStatus:=FLogParser.rescueStatus;
  end;
  if (not FHasFile) and FLogParser.hasFile then begin  // just been opened
    FHasFile:=true;
    FPONotifyObservers(self, ooAddItem, nil);
  end else FPONotifyObservers(self, ooChange, nil)
end;

procedure TLogParser.setContiguous(mode: Boolean);
begin
  if mode <> FContiguous then begin
    FContiguous:=mode;
    if FDomParser.hasFile then begin
      FHasFile:=false;
      FPONotifyObservers(self, ooDeleteItem, nil);
      postParse;
    end;
  end;
end;

procedure TLogParser.parse;
begin
  if FLogParser.hasFile() then FLogParser.parse();
  if FDomParser.hasFile() then FDomParser.parse();
  postParse;
end;

function TLogParser.hasFile: boolean;
begin
  result:=FHasFile;
end;

function TLogParser.hasDomFile: Boolean;
begin
  result:=FHasFile and FDomParser.hasFile;
end;

function TLogParser.getLog: TLog;
begin
  if not FDomParser.hasFile then result:=FLogParser.log
  else result:=FLog;
end;


{ TSimpleParser }

constructor TSimpleParser.Create;
begin
  inherited;
  FRescueStatus := emptyRescueStatus;
end;

destructor TSimpleParser.Destroy;
begin
  CloseFile();
  inherited;
end;

procedure TSimpleParser.OpenFile(filename : String);
begin
  try
    CloseFile(); // close already open file
    // open the log file using shared read access, so ddrescue can still write to it.
    FLogStream := TFileStream.Create(filename, fmOpenRead or fmShareDenyNone);
    FComments := TStringList.Create;
    FFileName := filename;
    parse();  // start parsing after opening
  except
    on E: Exception do begin
        CloseFile;
        logMsg('Parser error: '+E.Message);
    end;
  end;
end;

procedure TSimpleParser.CloseFile;
begin
  FFileName := '';
  FreeAndNil(FComments);
  FreeAndNil(FLogStream);
  FRescueStatus := emptyRescueStatus;
  SetLength(FLog, 0);
  FPONotifyObservers(self, ooDeleteItem, nil);
end;

(* returns true if a line is empty *)
function TSimpleParser.isEmptyLine(strLine : String) : boolean;
begin
  strLine := TrimLeft(strLine);
  result := (Length(strLine) = 0); // empty line
end;

(* returns true if a line is a comment line *)
function TSimpleParser.isCommentLine(strLine : String) : boolean;
begin
  result := (not isEmptyLine(strLine)) and (strLine[1] = '#'); // comment line
end;

(* add a log line to the application log *)
procedure TSimpleParser.logMsg(msg : String);
begin
  MainForm.DbgLog.Lines.Add(msg);
end;

(* Parse the log file.
   After parsing, the results are available in
   the properties log, rescueStatus and Comments *)
procedure TSimpleParser.parse();
var
  line : string;
  token : array[0..2] of string;
  i, logEntry, lineIdx, idx : Integer;
  logStrings : TStringList = nil;
  comments_end, prevHadFile : boolean;
begin
  // make sure the file is open
  if not hasFile then begin
    logMsg('Parser: No log file opened.');
    FPONotifyObservers(self, ooChange, nil);
    exit;
  end;

  // read file contents into string list
  try
    logStrings := TStringList.Create;
    if FLogStream.Seek(0, soFromBeginning) <> 0 then
      raise Exception.Create('Seek error!');
    logStrings.LoadFromStream(FLogStream);
    if (FLogStream.Size = 0) or (logStrings.Count = 0) then begin
      logMsg('Parser: log file seems empty, trying to reopen.');
      FreeAndNil(FLogStream);
      FLogStream := TFileStream.Create(FFilename, fmOpenRead or fmShareDenyNone);
      FLogStream.Seek(0, soFromBeginning);
      logStrings.LoadFromStream(FLogStream);
    end;
    logMsg('Reading log file: ' +IntToStr(logStrings.Count) + ' lines.');
  except
    on E : Exception do begin
      logMsg('Error: Cannot read log file: '+E.Message+'('+E.ClassName+')');
      FreeAndNil(logStrings);
      FPONotifyObservers(self, ooChange, nil);
      exit;
    end;
  end;

  // initialize stuff to zero
  FRescueStatus := emptyRescueStatus;
  FComments.Clear;
  FVersion:='';
  lineIdx := 0;
  logEntry := 0;
  comments_end := false;
  prevHadFile := Length(log) > 0;

  // parse the logfile lines into log array, rescueStatus and Comments.
  try
    while lineIdx < logStrings.Count do begin
      line:=logStrings[lineIdx];
      if isCommentLine(line) then begin
        // copy comment lines into FComments
        if comments_end or (pos('# current_pos', line) > 0) then
          comments_end:=true
        else FComments.Add(line);
        // process comment info
        if pos('Command line:', line) > 0 then begin
           repeat // not a loop, but goto replacement
             // try to find ddrescue's block size argument
             idx := pos(' -b', line);
             if idx <> 0 then idx := idx+3 // point to start of number
             else begin
               idx := pos(' --block-size=', line);
               if idx <> 0 then idx := idx+14 // point to start of number
               else break; // argument not found, jump over the rest
             end;
             token[0] :=TrimLeft(Copy(line, idx, Length(line)));
             idx := pos(' ', token[0]); //space after argument
             if idx <> 0 then token[0] :=Copy(token[0], 1, idx-1);
             FRescueStatus.suggestedBlockSize:=StrToIntDef(token[0], DEF_BSIZE);
           until true;
        end;
        idx:=pos('ddrescue version ', line);
        if idx > 0 then FVersion := Copy(line, idx+17, 1337);
      end else if not isEmptyLine(line) then begin
        // split line into maximum of 3 tokens
        line:=Trim(line);
        for i:= 0 to 2 do begin
          idx:= Pos(' ', line);
          if idx = 0 then begin
            token[i] := line;
            line := '';
          end else begin
            token[i] := Copy(line, 1, idx-1);
            line := TrimLeft(Copy(line, idx, Length(line)-idx+1));
          end;
        end;
        if (token[0] <> '') and (token[1] <> '') then begin
          if token[2] <> '' then begin // standard block line (3 tokens)
            // allocate more space if needed
            if Length(FLog) = logEntry then SetLength(FLog, logEntry+1024);
            FLog[logEntry].offset:=StrToInt64(token[0]);
            FLog[logEntry].length:=StrToInt64(token[1]);
            FLog[logEntry].status:=token[2][1];
            case FLog[logEntry].status of
              '?' : inc(FRescueStatus.nontried, FLog[logEntry].length);
              '+' : inc(FRescueStatus.rescued, FLog[logEntry].length);
              '*' : inc(FRescueStatus.nontrimmed, FLog[logEntry].length);
              '/' : inc(FRescueStatus.nonscraped, FLog[logEntry].length);
              '-' : begin
                      inc(FRescueStatus.bad, FLog[logEntry].length);
                      inc(FRescueStatus.errors);
                    end;
            end;
            inc(logEntry);
          end else begin // found the status line (2 tokens)
            if FRescueStatus.pos = 0 then begin
              FRescueStatus.pos:=StrToInt64Def(token[0], 0);
              FRescueStatus.curOperation:=token[1][1];
              FRescueStatus.strCurOperation:=OperationToText(FRescueStatus.curOperation);
            end else begin
              logMsg('Parser: Not enough tokens in line '+inttostr(lineIdx));
            end;
          end;
        end;
      end;
      lineIdx+=1;
    end;
    SetLength(FLog, logEntry); // trim array to actually needed size
  except
    on E : Exception do begin
      logMsg('Error parsing log file: '+E.Message+'('+E.ClassName+')');
      FRescueStatus := emptyRescueStatus; // some fail-safe values
      SetLength(FLog, 1);
      FLog[0].offset:=0;
      FLog[0].length:=DEF_BSIZE*(1 shl 18); // one block of 2^18 bad sectors
      FLog[0].status:='-';
      FRescueStatus.bad:=FLog[0].length;
      FRescueStatus.devicesize:=FLog[0].length;
    end;
  end;
  FreeAndNil(logStrings);

  // notify listeners
  if Length(FLog) > 0 then begin
    // calculate device's size from last block's offset and length
    FRescueStatus.devicesize:=FLog[Length(FLog)-1].offset + FLog[Length(FLog)-1].length;
    if prevHadFile then FPONotifyObservers(self, ooChange, nil)
      else FPONotifyObservers(self, ooAddItem, nil); // notify of new file
  end else begin
    logMsg('Parser: No blocks in logfile!');
    FPONotifyObservers(self, ooChange, nil); // notify any observers of the changes
  end;
end;

function TSimpleParser.hasFile: boolean;
begin
  hasFile:=Assigned(FLogStream);
end;

end.
