#!/usr/bin/perl

#%# Copyright (C) 2014 Christoph Biedl <debian.axhn@manchmal.in-ulm.de>
#%# License: GPL-2.0-only

use 5.010;
use strict;
use warnings;

use Test::More;
use Test::Command;
use Test::Differences;

use File::Slurp;
use File::Temp qw(tempdir);

my $temp_dir = tempdir (
    "check-support-status.$$.XXXXX",
    'TMPDIR' => 1,
    'CLEANUP' => 1,
) or BAIL_OUT ('Cannot get a temporary directory');

my %status_mapping = (
    'ioi' => 'install ok installed',
    'doc' => 'deinstall ok config-files',
);

# create a file that looks like the output of
# (squeeze)
# dpkg --query --show-format '${Status}\t${Package}\t${Version}\t${Source}\n'
# (wheezy or later)
# dpkg --query --show-format '${Status}\t${binary:Package}\t${Version}\t${Source}\n'

# parameters:
# - file name to write
# - list of lists, where each list has
#   - status (short form, see %status_mapping above)
#   - other fields as plain text

sub mock_query_result ($$) {
    my ($file_name, $listref) = @_;

    my $buffer;
    my $fh;
    open ($fh, '>', \$buffer);
    foreach my $elem (@$listref) {
        my ($status_short, @elems) = @$elem;
        if (!exists ($status_mapping{$status_short})) {
            fail ("BUG: No status mapping for '$status_short'");
            exit 1;
        }
        print $fh join (
            "\t",
            $status_mapping{$status_short},
            @elems,
        ), "\n";
    }
    close ($fh);
    write_file ($file_name, $buffer);
}

sub create_exe_list ($) {
    my $params = shift;

    my %result = ();
    foreach my $awk (qw(gawk mawk original-awk)) {
        my $exe = "$temp_dir/$awk";
        if (! -f $exe) {
            my $content = read_file ('check-support-status.in');
            $content =~ s:\@\@DPKG\@\@:dpkg:g;
            $content =~ s:\@\@DPKG_QUERY\@\@:t/data/dpkg-query:g;
            $content =~ s:\@\@AWK\@\@:"/usr/bin/$awk":eg;
            write_file ($exe, $content);
            chmod (0755, $exe);
        }

        $result{$awk} = [
            $exe,
            @$params,
        ];
    }
    return %result;
}

# work around features missing in Test::Command before 0.11
sub stdout_n_stderr ($) {
    my $run = shift;
    return (
        $run->can ('stdout_value') ? $run->stdout_value :
            scalar read_file ($run->{'result'}{'stdout_file'}),
        $run->can ('stderr_value') ? $run->stderr_value :
            scalar read_file ($run->{'result'}{'stderr_file'}),
    );
}

my $list = "$temp_dir/list";
my $query_result = "$temp_dir/query-result";
my $statusdb_file = "$temp_dir/status-db";
$ENV{'DPKG_QUERY_FILE'} = $query_result;

diag ('security support ended checks');

my %EXEs = create_exe_list ([
    '--type', 'ended',
    '--no-heading',
    '--list', $list,
    '--status-db', $statusdb_file,
]);

for my $exe (sort keys %EXEs) {
    # a single binary package, binary name = source name
    diag ("Basic test ($exe)");

    unlink ($statusdb_file);
    write_file ($list, <<__EOS__);
iceweasel       3.5.16-20       2013-05-01
base-files      6.0squeeze9     2014-05-01  Some spaced  explanation
debconf         1.5.36.0        2014-05-02
__EOS__
    mock_query_result (
        $query_result,
        [
            [ 'ioi', 'base-files', '6.0squeeze9' ],
            [ 'ioi', 'debconf', '1.5.36.1' ],
            [ 'ioi', 'debconf-i18n', '1.5.36.1', 'debconf' ],
        ],
    );

    # run a first time
    my $run = Test::Command->new ('cmd' => $EXEs{$exe});
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    my $expect_stdout = <<__EOS__;

* Source:base-files, ended on 2014-05-01 at version 6.0squeeze9
  Details: Some spaced  explanation
  Affected binary package:
  - base-files (installed version: 6.0squeeze9)
__EOS__

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    if (ok (-f $statusdb_file, 'status db file was created')) {
        my $got = read_file ($statusdb_file);
        my $expect = "base-files/6.0squeeze9\n";
        eq_or_diff (
            $got,
            $expect,
            'status db file content',
        );
    }

    ## run a second time
    $run->run;
    $run->exit_is_num (0);
    ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    eq_or_diff (
        $stdout,
        '',
        'stdout'
    );

    # status db file should be unchanged
    if (ok (-f $statusdb_file, 'status db file exists')) {
        my $got = read_file ($statusdb_file);
        my $expect = <<__EOS__;
base-files/6.0squeeze9
__EOS__
        eq_or_diff (
            $got,
            $expect,
            'status db file content',
        );
    }

    ## run a third time

    # but create fake records in the status db file
    write_file ($statusdb_file, <<__EOS__);
base-files/6.0squeeze7
base-files/6.0squeeze8
__EOS__

    $run->run;
    $run->exit_is_num (0);
    ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    # status db file should have one line now
    if (ok (-f $statusdb_file, 'status db file exists')) {
        my $got = read_file ($statusdb_file);
        my $expect = "base-files/6.0squeeze9\n";
        eq_or_diff (
            $got,
            $expect,
            'status db file content',
        );
    }
}

for my $exe (sort keys %EXEs) {
    # several binary packages from same source
    diag ("several binary packages ($exe)");

    write_file ($list, <<__EOS__);
# comments are allowed, too
iceweasel       3.5.16-20       2013-05-01
debconf         1.5.36.1        2014-05-02
__EOS__
    mock_query_result (
        $query_result,
        [
            [ 'ioi', 'base-files', '6.0squeeze9' ],
            [ 'ioi', 'debconf', '1.5.36.1' ],
            [ 'ioi', 'debconf-i18n', '1.5.36.1', 'debconf' ],
        ],
    );
    unlink ($statusdb_file);

    # run a first time
    my $run = Test::Command->new ('cmd' => $EXEs{$exe});
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    my $expect_stdout = <<__EOS__;

* Source:debconf, ended on 2014-05-02 at version 1.5.36.1
  Affected binary packages:
  - debconf (installed version: 1.5.36.1)
  - debconf-i18n (installed version: 1.5.36.1)
__EOS__

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    if (ok (-f $statusdb_file, 'status db file was created')) {
        my $got = read_file ($statusdb_file);
        my $expect = <<__EOS__;
debconf/1.5.36.1
debconf-i18n/1.5.36.1
__EOS__
        eq_or_diff (
            $got,
            $expect,
            'status db file content',
        );
    }
}

for my $exe (sort keys %EXEs) {
    diag ("Spacing in explanation ($exe)");

    # Assert any kind of spacing at the beginning the the explanative
    # is handled as expected

    mock_query_result (
        $query_result,
        [
            [ 'ioi', 'openswan', '1:2.6.28+dfsg-5+squeeze2' ],
        ],
    );
    my %tests = (
        'space1' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31 Not supported in squeeze LTS\n",
        'space2' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31  Not supported in squeeze LTS\n",
        'space2' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31   Not supported in squeeze LTS\n",
        'tabbed' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31\tNot supported in squeeze LTS\n",
        'tabbed2' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31\t\tNot supported in squeeze LTS\n",
    );

    for my $test_name (sort keys %tests) {
        unlink ($statusdb_file);
        write_file ($list, $tests{$test_name});

        my $run = Test::Command->new ('cmd' => $EXEs{$exe});
        $run->run;
        $run->exit_is_num (0);
        my ($stdout, $stderr) = stdout_n_stderr ($run);

        $stderr and diag ("stderr:\n" . $stderr);

        my $expect_stdout = <<__EOS__;

* Source:openswan, ended on 2014-05-31 at version 1:2.6.28+dfsg-5+squeeze2
  Details: Not supported in squeeze LTS
  Affected binary package:
  - openswan (installed version: 1:2.6.28+dfsg-5+squeeze2)
__EOS__

        eq_or_diff (
            $stdout,
            $expect_stdout,
            "$test_name, stdout",
        );
    }
}


for my $exe (sort keys %EXEs) {
    diag ("not-purged ($exe)");

    # Packages uninstalled but not purge should be ignored. #749551

    write_file ($list, <<__EOS__);
iceweasel       3.5.16-20       2013-05-01
base-files      6.0squeeze9     2014-05-01  Some spaced  explanation
debconf         1.5.36.0        2014-05-02
__EOS__
    mock_query_result (
        $query_result,
        [
            [ 'doc', 'base-files', '6.0squeeze9' ],
            [ 'ioi', 'debconf', '1.5.36.1' ],
            [ 'ioi', 'debconf-i18n', '1.5.36.1', 'debconf' ],
        ],
    );

    my $run = Test::Command->new ('cmd' => $EXEs{$exe});
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    my $expect_stdout = '';

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );
}


diag ('limited support checks');

%EXEs = create_exe_list ([
    '--type', 'limited',
    '--no-heading',
    '--list', $list,
    '--status-db', $statusdb_file,
]);

for my $exe (sort keys %EXEs) {
    diag ("simple ($exe)");

    write_file ($list, <<__EOS__);
php5    See README.Debian.security for the PHP security policy
__EOS__
    mock_query_result (
        $query_result,
        [
            [ 'ioi', 'php5', '5.3.3-7+squeeze19' ],
        ],
    );
    unlink ($statusdb_file);

    # run a first time
    my $run = Test::Command->new ('cmd' => $EXEs{$exe});
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    my $expect_stdout = <<__EOS__;

* Source:php5
  Details: See README.Debian.security for the PHP security policy
  Affected binary package:
  - php5 (installed version: 5.3.3-7+squeeze19)
__EOS__

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    if (ok (-f $statusdb_file, 'status db file was created')) {
        my $got = read_file ($statusdb_file);
        my $expect = <<__EOS__;
php5/5.3.3-7+squeeze19
__EOS__
        eq_or_diff (
            $got,
            $expect,
            'status db file content',
        );
    }
}

for my $exe (sort keys %EXEs) {
    diag ("Spacing in explanation ($exe)");

    # Assert any kind of spacing at the beginning the the explanative
    # is handled as expected

    mock_query_result (
        $query_result,
        [
            [ 'ioi', 'php5', '5.3.3-7+squeeze19' ],
        ],
    );
    my %tests = (
        'space1' =>
"php5 See README.Debian.security for the PHP security policy\n",
        'space2' =>
"php5  See README.Debian.security for the PHP security policy\n",
        'space3' =>
"php5   See README.Debian.security for the PHP security policy\n",
        'tabbed' =>
"php5\tSee README.Debian.security for the PHP security policy\n",
        'tabbed2' =>
"php5\t\tSee README.Debian.security for the PHP security policy\n",
    );

    for my $test_name (sort keys %tests) {
        unlink ($statusdb_file);
        write_file ($list, $tests{$test_name});

        my $run = Test::Command->new ('cmd' => $EXEs{$exe});
        $run->run;
        $run->exit_is_num (0);
        my ($stdout, $stderr) = stdout_n_stderr ($run);

        $stderr and diag ("stderr:\n" . $stderr);

        my $expect_stdout = <<__EOS__;

* Source:php5
  Details: See README.Debian.security for the PHP security policy
  Affected binary package:
  - php5 (installed version: 5.3.3-7+squeeze19)
__EOS__

        eq_or_diff (
            $stdout,
            $expect_stdout,
            "$test_name, stdout",
        );
    }
}


done_testing;

exit 0;
